#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include <compiz.h>

#define WALLPAPER_SCREEN_OPTION_IMAGES	0
#define WALLPAPER_SCREEN_OPTION_NUM		1

#define WALLPAPER_DEFAULT_IMAGE    "/usr/share/compiz/background.png"

#define MAX_NUMBER_OF_WALLPAPERS		32

#define GET_WALLPAPER_DISPLAY(d)									\
	((WallpaperDisplay *) (d)->privates[displayPrivateIndex].ptr)

#define WALLPAPER_DISPLAY(d)										\
	WallpaperDisplay *wd = GET_WALLPAPER_DISPLAY (d)

#define GET_WALLPAPER_SCREEN(s, wd)									\
	((WallpaperScreen *) (s)->privates[(wd)->screenPrivateIndex].ptr)

#define WALLPAPER_SCREEN(s)											\
	WallpaperScreen *ws = GET_WALLPAPER_SCREEN (s, GET_WALLPAPER_DISPLAY (s->display))

#define NUM_OPTIONS(s) (sizeof ((s)->opt) / sizeof (CompOption))

static int displayPrivateIndex;

static char *wallpapersImages[] = {
	WALLPAPER_DEFAULT_IMAGE,
	WALLPAPER_DEFAULT_IMAGE,
	WALLPAPER_DEFAULT_IMAGE,
	WALLPAPER_DEFAULT_IMAGE
};
#define N_WALLPAPERS_IMAGES (sizeof (wallpapersImages) / sizeof (wallpapersImages[0]))

typedef struct _WallpaperDisplay {
	int			screenPrivateIndex;
	HandleEventProc handleEvent;
} WallpaperDisplay;

typedef struct _WallpaperScreen {
    PaintBackgroundProc paintBackground;
	CompOptionValue *curValue;
	CompTexture *bgTextures[MAX_NUMBER_OF_WALLPAPERS];
	int nTextures;

	CompOption opt[WALLPAPER_SCREEN_OPTION_NUM];
} WallpaperScreen;


static void
wallpaperPaintBackground (CompScreen *s, 
							Region region, 
							unsigned int mask)
{
	WALLPAPER_SCREEN (s);

	/* Not sure if what follows is the best way to do it, probably not... */
	if (s->backgroundTexture.name != ws->bgTextures[s->x]->name)
	{
		s->backgroundTexture = *(ws->bgTextures[s->x]);
		damageScreen (s);
	}
	//printf ("Wallpaper: region has %ld rects\n", region->numRects);

	UNWRAP (ws, s, paintBackground);
	(*s->paintBackground) (s, region, mask);
	WRAP (ws, s, paintBackground, wallpaperPaintBackground);
}


static Bool
wallpaperInit (CompPlugin *p)
{
	displayPrivateIndex = allocateDisplayPrivateIndex ();

	if (displayPrivateIndex < 0)
	{
		return FALSE;
	}

	return TRUE;
}

static void
wallpaperFini (CompPlugin *p)
{
	if (displayPrivateIndex >= 0)
		freeDisplayPrivateIndex (displayPrivateIndex);
}


static Bool
wallpaperInitDisplay (CompPlugin *p, 
					CompDisplay *d)
{
	WallpaperDisplay *wd;

	wd = malloc (sizeof (WallpaperDisplay));
	if (!wd)
		return FALSE;

	wd->screenPrivateIndex = allocateScreenPrivateIndex (d);
	if (wd->screenPrivateIndex < 0)
	{
		free(wd);
		return FALSE;
	}

	d->privates[displayPrivateIndex].ptr = wd;

	return TRUE;
}

static void
wallpaperFiniDisplay (CompPlugin *p, 
					CompDisplay *d)
{
	WALLPAPER_DISPLAY (d);

	freeScreenPrivateIndex (d, wd->screenPrivateIndex);

	free (wd);
}


static void
wallpaperScreenInitOptions (WallpaperScreen *ws, int virtualScreenSize)
{
	CompOption *o;
	int i;

	o = &ws->opt[WALLPAPER_SCREEN_OPTION_IMAGES];
	o->name			= "images";
	o->shortDesc		= "Sets individual wallpapers";
	o->longDesc		= "Sets individual wallpapers for each face of the cube";
	o->type			= CompOptionTypeList;
	o->value.list.type	= CompOptionTypeString;
	o->value.list.nValue	= virtualScreenSize;
	o->value.list.value	= calloc (virtualScreenSize, sizeof (CompOptionValue));
	/* Caution: the following loop assumes that virtualScreenSize = size of the array wallpaperImages */
	for (i = 0; i < virtualScreenSize; i++)
	{
		o->value.list.value[i].s = strdup (wallpapersImages[i]);
	}
	o->rest.s.string		= 0;
	o->rest.s.nString		= 0;
}

static Bool
wallpaperInitScreen (CompPlugin *p, 
					CompScreen *s)
{
	WallpaperScreen *ws;
	CompOption *o;
	int numOptions;
	int virtualScreenSize;
	int i;
	CompTexture *t;
	unsigned int width, height;

	WALLPAPER_DISPLAY (s->display);

	/* Will need the number of virtual screens */
	o = compGetScreenOptions (s, &numOptions);
	o = compFindOption (o, numOptions, "hsize", NULL);
	if (!o)
	{
		fprintf (stderr, "Fatal error : could not find screen size\n");
		return FALSE;
	}
	
	virtualScreenSize = o->value.i;

	ws = malloc (sizeof (WallpaperScreen));
	if (!ws)
		return FALSE;

	o = &ws->opt[WALLPAPER_SCREEN_OPTION_IMAGES];

	/* Main initializations */
	wallpaperScreenInitOptions (ws, virtualScreenSize);

	ws->nTextures = virtualScreenSize;
	for (i = 0; i < virtualScreenSize; i++)
	{
		ws->curValue = &o->value.list.value[i];
		t = malloc (sizeof (CompTexture));
		if (!t)
		{
			while (i > 0)
			{
				i--;
				finiTexture (s, ws->bgTextures[i]);
				free (ws->bgTextures[i]);
			}
			free (ws);
			return FALSE;
		}
		initTexture (s, t);
		if (!readImageToTexture (s, t, ws->curValue->s, &width, &height))
		{
			finiTexture (s, t);
			free (t);
			while (i > 0)
			{
				i--;
				finiTexture (s, ws->bgTextures[i]);
				free (ws->bgTextures[i]);
			}
			free (ws);
			return FALSE;
		}
		ws->bgTextures[i] = t;
	}

	s->privates[wd->screenPrivateIndex].ptr = ws;

	if (s->backgroundTexture.name)
		finiTexture (s, &s->backgroundTexture);

	s->backgroundTexture = *(ws->bgTextures[s->x]);
	damageScreen (s);

	WRAP (ws, s, paintBackground, wallpaperPaintBackground);

	return TRUE;
}

static void
wallpaperFiniScreen (CompPlugin *p, 
					CompScreen *s)
{
	int i;

	WALLPAPER_SCREEN (s);

	UNWRAP (ws, s, paintBackground);

	for (i = 0; i < ws->nTextures; i++)
	{
		finiTexture (s, ws->bgTextures[i]);
		free (ws->bgTextures[i]);
	}

	free (ws);
}


static CompOption *
wallpaperGetScreenOptions (CompScreen *screen, 
							int *count)
{
	WALLPAPER_SCREEN (screen);

	*count = NUM_OPTIONS (ws);
	return ws->opt;
}

static Bool
wallpaperSetScreenOption (CompScreen 		*screen, 
							char 			*name, 
							CompOptionValue *value)
{
	CompOption *o;
	int index;
	int i;
	unsigned int width, height;

	WALLPAPER_SCREEN (screen);

	o = compFindOption (ws->opt, NUM_OPTIONS (ws), name, &index);
	if (!o)
		return FALSE;

	switch (index) {
		case WALLPAPER_SCREEN_OPTION_IMAGES:
		if (value->list.nValue != ws->nTextures)
		{
			return FALSE;
		}
		if (compSetOptionList (o, value))
		{
			for (i = 0; i < ws->nTextures; i++)
			{
				ws->curValue = &o->value.list.value[i];
				finiTexture (screen, ws->bgTextures[i]);
				initTexture (screen, ws->bgTextures[i]);
				readImageToTexture (screen, ws->bgTextures[i], 
									ws->curValue->s, 
									&width, &height);
			}
		}
		return TRUE;
	}	
	return FALSE;
}

CompPluginDep wallpaperDeps[] = {
	{ CompPluginRuleAfter, "cube" }, 
	{ CompPluginRuleAfter, "rotate" }
};


static int
wallpaperGetVersion (CompPlugin *plugin,
		int	   version)
{
    return ABIVERSION;
}
static CompPluginVTable wallpaperVTable = {
	"wallpaper", 
	"Sets wallpapers", 
	"Sets individual wallpapers for each screen",
	wallpaperGetVersion,
	wallpaperInit, 
	wallpaperFini, 
	wallpaperInitDisplay, 
	wallpaperFiniDisplay, 
	wallpaperInitScreen, 
	wallpaperFiniScreen, 
	0, /* InitWindow */
	0, /* FiniWindow */
	0, /* GetDisplayOptions */
	0, /* SetDisplayOption */
	wallpaperGetScreenOptions, 
	wallpaperSetScreenOption, 
	wallpaperDeps, 
	sizeof(wallpaperDeps) / sizeof(wallpaperDeps[0])
};

CompPluginVTable *
getCompPluginInfo (void)
{
	return &wallpaperVTable;
}
