//plugin to handle widgets-as-programs

//uses a special atom set on widget apps
//keeps windows with that atom set invisible unless in widget mode
//yes this means that the app has to be written to handle this (or an app to
//set the widget atom, maybe an extension to state)

//the atom will be for now _COMPIZ_WIDGET

//semantics:
//      when not in widget mode, all _COMPIZ_WIDGET windows will be withdrawn
//      when in widget mode, all non _COMPIZ_WIDGET windows will be
//              unable to be interacted with (perhaps by an invisible fullscreen
//              window stacked below widgets but above everything else)
//      when in widget mode, all _COMPIZ_WIDGET windows will be stacked
//              above all non _COMPIZ_WIDGET windows
//      when entering widget mode, we:
//              1>create(map) the invisible blocking window
//              2>stack it above all other windows
//              ?>somehow tell other plugins not to start (maybe)
//              3>set all the _C_W windows to 0% opacity
//              4>map the _C_W windows
//              5>stack them above the blocking window
//              6>start the fade in of _C_W windows & dimming of non _C_W windows
//              ?>wait for that to finish, then remove the screen grab?
//      when leaving widget mode, we:
//              1>start the fade out of _C_W windows & un-dimming of non _C_W
//              ?>screen grab? wait for? ungrab?
//              2>unmap the _C_W windows
//              3>unmap the blocking window

#include <compiz.h>
#include <X11/Xatom.h>
#include <X11/cursorfont.h>
#include <stdlib.h>
#include <string.h>

#define WIDGET_TOGGLE_KEY_DEFAULT "F9"
#define WIDGET_TOGGLE_KEY_MODIFIERS_DEFAULT 0

#define WIDGET_SPEED_DEFAULT   1.5f
#define WIDGET_SPEED_MIN       0.1f
#define WIDGET_SPEED_MAX       50.0f
#define WIDGET_SPEED_PRECISION 0.1f

#define WIDGET_TIMESTEP_DEFAULT   1.2f
#define WIDGET_TIMESTEP_MIN       0.1f
#define WIDGET_TIMESTEP_MAX       50.0f
#define WIDGET_TIMESTEP_PRECISION 0.1f

#define WIDGET_DISPLAY_OPTION_TOGGLE_KEY 0
#define WIDGET_DISPLAY_OPTION_NUM        1

static int displayPrivateIndex;

typedef enum _WidgetScreenOptions
{
  WIDGET_SCREEN_OPTION_TIMESTEP,
  WIDGET_SCREEN_OPTION_SPEED,
  WIDGET_SCREEN_OPTION_NUM
} WidgetScreenOptions;

typedef enum _WidgetScreenState
{
  STATE_NORMAL,
  STATE_ENTERING,
  STATE_HUD,
  STATE_LEAVING,
  STATE_COUNT
} WidgetScreenState;

typedef struct _WidgetDisplay
{
  int             screenPrivateIndex;
  CompOption      opt[WIDGET_DISPLAY_OPTION_NUM];
  HandleEventProc handleEvent;
  Atom            compizWidgetAtom;
} WidgetDisplay;

typedef struct _WidgetScreen
{
  int                    windowPrivateIndex;
  CompOption             opt[WIDGET_SCREEN_OPTION_NUM];
  WidgetScreenState      state;
  float                  animStep; // goes from 0 for normal to OPAQUE for hud
  int                    opacity; // widget opacity, convenience
  int                    brightness; // non-widget brightness, convenience
  PaintWindowProc        paintWindow;
  PreparePaintScreenProc preparePaintScreen;
  DonePaintScreenProc    donePaintScreen;
  FocusWindowProc        focusWindow;
  WindowResizeNotifyProc windowResizeNotify;
  WindowMoveNotifyProc   windowMoveNotify;
  Cursor                 cursor;
  int                    grabIndex;
  Window                 blockWindow;
  
  float                  speed;
  float                  timestep;

} WidgetScreen;

typedef struct _WidgetWindow
{
  Bool          isWidget;
  Bool          isBlockWindow;
  int           lastx;
  int           lasty;
  Bool          hasPosition;
  Bool          wantsToHide;
} WidgetWindow;

typedef struct {
  unsigned long flags;
  unsigned long functions;
  unsigned long decorations;
} MwmHints;

#define MwmHintsDecorations (1L << 1)

#define GET_WIDGET_DISPLAY(d)                                  \
    ((WidgetDisplay *) (d)->privates[displayPrivateIndex].ptr)

#define WIDGET_DISPLAY(d)                    \
    WidgetDisplay *wd = GET_WIDGET_DISPLAY (d)

#define GET_WIDGET_SCREEN(s, wd)                                   \
    ((WidgetScreen *) (s)->privates[(wd)->screenPrivateIndex].ptr)

#define WIDGET_SCREEN(s)                                                  \
    WidgetScreen *ws = GET_WIDGET_SCREEN (s, GET_WIDGET_DISPLAY (s->display))

#define GET_WIDGET_WINDOW(w, ws)                                   \
    ((WidgetWindow *) (w)->privates[(ws)->windowPrivateIndex].ptr)

#define WIDGET_WINDOW(w)                                       \
    WidgetWindow *ww = GET_WIDGET_WINDOW  (w,                    \
                     GET_WIDGET_SCREEN  (w->screen,            \
                     GET_WIDGET_DISPLAY (w->screen->display)))

#define NUM_OPTIONS(s) (sizeof ((s)->opt) / sizeof (CompOption))

static CompOption *
widgetGetScreenOptions (CompScreen *screen,
                      int        *count)
{
    WIDGET_SCREEN (screen);

    *count = NUM_OPTIONS (ws);

    return ws->opt;
}

static Bool
widgetSetScreenOption (CompScreen      *screen,
                         char            *name,
                         CompOptionValue *value)
{
    CompOption *o;
    int        index;

    WIDGET_SCREEN (screen);

    o = compFindOption (ws->opt, NUM_OPTIONS (ws), name, &index);
    if (!o)
        return FALSE;
    switch(index)
    {
        case WIDGET_SCREEN_OPTION_SPEED:
          if (compSetFloatOption (o, value))
          {
              ws->speed = o->value.f;
              return TRUE;
          }
          break;
        case WIDGET_SCREEN_OPTION_TIMESTEP:
          if (compSetFloatOption (o, value))
          {
              ws->timestep = o->value.f;
              return TRUE;
          }
          break;
    }
  
  return FALSE;
}

static Bool
widgetFocusWindow (CompWindow *w)
{
    Bool status;
  
    WIDGET_SCREEN (w->screen);
    WIDGET_WINDOW (w);
  
    if (ws->state == STATE_HUD && !ww->isWidget)
        return FALSE;

    UNWRAP (ws, w->screen, focusWindow);
    status = (*w->screen->focusWindow) (w);
    WRAP (ws, w->screen, focusWindow, widgetFocusWindow);

    return status;
}

static void
widgetUpdateWindows(CompScreen * s, Bool ActivateAWidget)
{
  WIDGET_SCREEN(s);
  CompWindow * w, * wid, * act;
  wid=0;
  switch(ws->state)
  {
    case STATE_NORMAL:
      w=s->windows;
      while(w)
      {
        WIDGET_WINDOW(w);
        if (ww->isWidget && (w->attrib.map_state == IsViewable))
        {
          ww->lastx=w->serverX;
          while (ww->lastx < 0)
            ww->lastx+=w->screen->width;
          while (ww->lastx > w->screen->width)
            ww->lastx-=w->screen->width;
          ww->lasty=w->serverY;
          while (ww->lasty < 0)
            ww->lasty+=w->screen->height;
          while (ww->lasty > w->screen->height)
            ww->lasty-=w->screen->height;
          ww->hasPosition=TRUE;
          XUnmapWindow(w->screen->display->display,w->id);
        }
        w=w->next;
      }
      break;
    case STATE_HUD:
    case STATE_ENTERING:
    case STATE_LEAVING:
    //case STATE_HUD:
      //savestate=ws->state;
      //ws->state=STATE_COUNT;
      w=s->windows;
      act = 0;
      while(w)
      {
        WIDGET_WINDOW(w);
        if (ww)
        {
          if (ww->isWidget && !ww->wantsToHide)
          {
            if (ww->hasPosition && w->attrib.map_state!=IsViewable)
            {
              XMoveWindow(w->screen->display->display,
                          w->id,ww->lastx,ww->lasty);
              w->placed=TRUE;
              w->attrib.x = ww->lastx;
              w->attrib.y = ww->lasty;
            }
            XMapWindow(w->screen->display->display,w->id);
            if (!act)
              act=w;
            else if (act->mapNum < w->mapNum)
              act=w;
            wid=w;
          }
        }
        w=w->next;
      }
      if (!wid) return;
      w = s->windows;
      while(w)
      {
        WIDGET_WINDOW(w);
        if (ww)
        {
          if (!ww->isWidget)
          {
            //restackWindowBelow(w,wid);
            //lowerWindow(w);
            restackWindowAbove(wid,w);
          }
        }
        w=w->next;
      }
      //if (!act) return;
      w=s->windows;
      while(w)
      {
        WIDGET_WINDOW(w);
        if (ww)
        {
          if (ww->isWidget)
          {
            restackWindowAbove(w,wid);
            raiseWindow(w);
          }
          break;
        }
        w=w->next;
      }
      if (ActivateAWidget && act) 
      {
        activateWindow(act);
      }
      //ws->state = savestate;
      break;
    default:
      break;
  }
}

static void
widgetGrabScreen(CompScreen * s)
{
  WIDGET_SCREEN(s);
  if (!ws->grabIndex)
    ws->grabIndex = pushScreenGrab (s, ws->cursor, "widget");
}

static void
widgetUnGrabScreen(CompScreen * s)
{
  WIDGET_SCREEN(s);
  removeScreenGrab (s, ws->grabIndex, 0);
  ws->grabIndex=0;
  widgetUpdateWindows(s,TRUE);
}

static void
widgetPreparePaintScreen (CompScreen *s,
                             int         msSinceLastPaint)
{
  WIDGET_SCREEN (s);

  if (ws->state == STATE_ENTERING || ws->state == STATE_LEAVING)
  {
    //CompWindow *w;
    int        steps;
    float      amount, chunk;

    amount = msSinceLastPaint * 0.5f * ws->speed;
    steps  = amount / (0.5f * ws->timestep);
    if (!steps) steps = 1;
    chunk  = amount / (float) steps;

    while (steps--)
    {
      if (ws->state == STATE_ENTERING && (ws->animStep+chunk*200)>=OPAQUE)
      {
        ws->state = STATE_HUD;
        ws->animStep = OPAQUE;
        widgetUnGrabScreen(s);
        break;
      }
      else if (ws->state == STATE_LEAVING && chunk*200 > ws->animStep)
      {
        ws->state = STATE_NORMAL;
        ws->animStep = 0;
        widgetUnGrabScreen(s);
        //if (ws->blockWindow)
        //XUnmapWindow(s->display->display,ws->blockWindow);
        break;
      }
      else
      {
        if (ws->state == STATE_ENTERING)
                ws->animStep+=chunk*200;
        else
                ws->animStep-=chunk*200;
      }
      /*sw->tx += sw->xVelocity * chunk;*/

    }
    ws->opacity = ws->animStep;
    ws->brightness = BRIGHT - (0.5 * ws->animStep);

  }
  UNWRAP (ws, s, preparePaintScreen);
  (*s->preparePaintScreen) (s, msSinceLastPaint);
  WRAP (ws, s, preparePaintScreen, widgetPreparePaintScreen);
  
}

static void
widgetDonePaintScreen    (CompScreen * s)
{
  WIDGET_SCREEN(s);
  
  if (ws->state == STATE_ENTERING || ws->state == STATE_LEAVING)
  {
    damageScreen(s);
  }
  
  UNWRAP (ws, s, donePaintScreen);
  (*s->donePaintScreen) (s);
  WRAP (ws, s, donePaintScreen, widgetDonePaintScreen);
}

static Bool
widgetPaintWindow (CompWindow              *w,
                      const WindowPaintAttrib *attrib,
                      Region                  region,
                      unsigned int            mask)
{
  CompScreen * s = w->screen;
  Bool status;
  WIDGET_SCREEN(s);
  WIDGET_WINDOW(w);
  
  WindowPaintAttrib fAttrib = *attrib;
  
  if (!ww->isWidget)
  {
//fprintf(stderr, "I am a widget \n");
    if (!(w->type & CompWindowTypeUnknownMask))
	//fAttrib.opacity = 0;
        fAttrib.brightness *= ((float)ws->brightness/(float)BRIGHT);
  }
  else
  {
    fAttrib.opacity *= ((float)ws->opacity/(float)BRIGHT);
  }
  
  if (ww->isBlockWindow)
    fAttrib.opacity = 0;
  
  UNWRAP (ws, s, paintWindow);
  status = (*s->paintWindow) (w, &fAttrib, region, mask);
  WRAP (ws, s, paintWindow, widgetPaintWindow);
  return status;
}

/*static void widgetCreateBlockWindow(CompScreen * s)
{
  WIDGET_SCREEN(s);
  Atom state[4];
  int nState = 0;
  Atom mwmHintsAtom;
  MwmHints mwmHints;

  ws->blockWindow = XCreateSimpleWindow(
      s->display->display, s->root, 0,0,
  s->width,s->height,0,0,0);
    
  mwmHintsAtom = XInternAtom (s->display->display,
                              "_MOTIF_WM_HINTS", 0);

  memset (&mwmHints, 0, sizeof (mwmHints));

  mwmHints.flags = MwmHintsDecorations;
  mwmHints.decorations = 0;

  XChangeProperty (s->display->display,
                   ws->blockWindow, mwmHintsAtom, mwmHintsAtom,
                   8, PropModeReplace, (unsigned char *) &mwmHints,
                   sizeof (mwmHints));

  //state[nState++] = s->display->winStateAboveAtom;
  state[nState++] = s->display->winStateStickyAtom;
  state[nState++] = s->display->winStateSkipTaskbarAtom;
  state[nState++] = s->display->winStateSkipPagerAtom;

  XChangeProperty (s->display->display, ws->blockWindow,
  XInternAtom (s->display->display,
  "_NET_WM_STATE", 0),
  XA_ATOM, 32, PropModeReplace,
  (unsigned char *) state, nState);
}

static void widgetEnsureBlockerWindow(CompScreen * s)
{
  WIDGET_SCREEN(s);
  if (!ws->blockWindow)
  {
    widgetCreateBlockWindow(s);
  }
  XMoveResizeWindow(s->display->display,ws->blockWindow,0,0,
                s->width,s->height);
  XMapWindow(s->display->display,ws->blockWindow);
}*/

static void
widgetToggleScreen (CompScreen * s)
{
    WIDGET_SCREEN(s);

    switch(ws->state)
    {
      case STATE_NORMAL:
        ws->state = STATE_ENTERING;
        widgetUpdateWindows(s,TRUE);
        widgetGrabScreen(s);
      case STATE_LEAVING:
        ws->state = STATE_ENTERING;
        break;
      case STATE_HUD:
        ws->state = STATE_LEAVING;
        widgetGrabScreen(s);
      case STATE_ENTERING:
        ws->state = STATE_LEAVING;
        break;
      case STATE_COUNT:
        break;
    }
    //widgetEnsureBlockerWindow(s);
    damageScreen(s);
}

static Bool
widgetToggle (CompDisplay     *d,
		 CompAction      *action,
		 CompActionState state,
		 CompOption      *option,
		 int	         nOption)
{
    CompScreen *s;

    s = findScreenAtDisplay (d, getIntOptionNamed (option, nOption, "root", 0));

    widgetToggleScreen (s);

    return TRUE;
}

static void
widgetHandleEvent (CompDisplay *d,
                     XEvent      *event)
{
  WIDGET_DISPLAY(d);
  CompScreen * s;
  CompWindow * w;



  switch(event->type)
  {

    case FocusIn:
    //case FocusOut:
      if (event->xfocus.mode==NotifyNormal)
      {
        w = findWindowAtDisplay (d, event->xfocus.window);
        if (w)
        {
          s = w->screen;
          WIDGET_SCREEN(s);
          if (ws->state == STATE_HUD)
          {
            WIDGET_WINDOW(w);
            if (!ww->isWidget && w->placed)
              widgetToggleScreen(w->screen);
          }
        }
      }
      break;
    case UnmapNotify:
      w = findWindowAtDisplay (d, event->xunmap.window);
      if (w)
      {
          s = w->screen;
          WIDGET_WINDOW(w);
          WIDGET_SCREEN(s);
          if (ws->state!=STATE_NORMAL)
                ww->wantsToHide = ww->isWidget;
      }
      break;
    case MapNotify:
      w = findWindowAtDisplay (d, event->xmap.window);
      if (w)
      {
        WIDGET_WINDOW(w);
        ww->wantsToHide=FALSE;
      }
      break;
    case CreateNotify:
      w = findWindowAtDisplay (d, event->xcreatewindow.window);
      if(w)
      {
        s = w->screen;
        WIDGET_SCREEN(s);
        if (event->xcreatewindow.window == ws->blockWindow)
        {
          WIDGET_WINDOW(w);
          ww->isBlockWindow=TRUE;
        }
        widgetUpdateWindows(s,FALSE);
      }
      break;
    case KeyPress:
    case ButtonPress:
    case PropertyNotify:
      if (event->xproperty.atom == wd->compizWidgetAtom)
      {
        w = findWindowAtDisplay(d,event->xproperty.window);
        if (w)
        {
          WIDGET_WINDOW(w);
          if (ww)
          {
            ww->isWidget = (event->xproperty.state == PropertyNewValue);
            if (!ww->isWidget)
              XMapWindow(w->screen->display->display, w->id);
            widgetUpdateWindows(s, TRUE);
          }
        }
      }
      break;
    default:
      break;
  }
  
  UNWRAP (wd, d, handleEvent);
  (*d->handleEvent) (d, event);
  WRAP (wd, d, handleEvent, widgetHandleEvent);
}


static void
widgetScreenInitOptions (WidgetScreen *ws, Display * display)
{
    CompOption *o;
  
    o = &ws->opt[WIDGET_SCREEN_OPTION_SPEED];
    o->name             = "speed";
    o->shortDesc        = "Speed";
    o->longDesc         = "Widget speed";
    o->type             = CompOptionTypeFloat;
    o->value.f          = WIDGET_SPEED_DEFAULT;
    o->rest.f.min       = WIDGET_SPEED_MIN;
    o->rest.f.max       = WIDGET_SPEED_MAX;
    o->rest.f.precision = WIDGET_SPEED_PRECISION;

    o = &ws->opt[WIDGET_SCREEN_OPTION_TIMESTEP];
    o->name             = "timestep";
    o->shortDesc        = "Timestep";
    o->longDesc         = "Widget timestep";
    o->type             = CompOptionTypeFloat;
    o->value.f          = WIDGET_TIMESTEP_DEFAULT;
    o->rest.f.min       = WIDGET_TIMESTEP_MIN;
    o->rest.f.max       = WIDGET_TIMESTEP_MAX;
    o->rest.f.precision = WIDGET_TIMESTEP_PRECISION;

}

static void
widgetDisplayInitOptions (WidgetDisplay *wd, Display * display)
{
    CompOption *o;

    o = &wd->opt[WIDGET_DISPLAY_OPTION_TOGGLE_KEY];
    o->name			     = "toggle_widget_display";
    o->shortDesc		     = N_("Toggle Widget Display");
    o->longDesc			     = N_("Toggle Widget Display");
    o->type			     = CompOptionTypeAction;
    o->value.action.initiate	     = widgetToggle;
    o->value.action.terminate	     = 0;
    o->value.action.bell	     = FALSE;
    o->value.action.edgeMask	     = 0;
    o->value.action.type	     = CompBindingTypeKey;
    o->value.action.state	     = CompActionStateInitKey;
    o->value.action.key.modifiers    = WIDGET_TOGGLE_KEY_MODIFIERS_DEFAULT;
    o->value.action.key.keycode      = 
      XKeysymToKeycode (display,
                        XStringToKeysym (WIDGET_TOGGLE_KEY_DEFAULT));
}

static CompOption *
widgetGetDisplayOptions (CompDisplay *display,
			int	    *count)
{
    WIDGET_DISPLAY (display);

    *count = NUM_OPTIONS (wd);

    return wd->opt;
}


static Bool
widgetSetDisplayOption (CompDisplay     *display,
		       char	       *name,
		       CompOptionValue *value)
{
    CompOption *o;
    int	       index;

    WIDGET_DISPLAY (display);

    o = compFindOption (wd->opt, NUM_OPTIONS (wd), name, &index);
    if (!o)
	return FALSE;

    switch (index) {
    case WIDGET_DISPLAY_OPTION_TOGGLE_KEY:
	if (setDisplayAction (display, o, value))
	    return TRUE;
	break;
    default:
	break;
    }

    return FALSE;
}

static Bool
widgetInitDisplay (CompPlugin  *p,
                   CompDisplay *d)
{
    WidgetDisplay *wd;

    wd = malloc (sizeof (WidgetDisplay));
    if (!wd)
      return FALSE;

    wd->screenPrivateIndex = allocateScreenPrivateIndex (d);
    if (wd->screenPrivateIndex < 0)
    {
      free (wd);
      return FALSE;
    }

    wd->compizWidgetAtom = XInternAtom(d->display,"_COMPIZ_WIDGET",FALSE);

    widgetDisplayInitOptions (wd, d->display);

    d->privates[displayPrivateIndex].ptr = wd;


    WRAP (wd, d, handleEvent, widgetHandleEvent);

    return TRUE;
}

static void
    widgetFiniDisplay (CompPlugin *p,
                     CompDisplay *d)
{
    WIDGET_DISPLAY (d);

    freeScreenPrivateIndex (d, wd->screenPrivateIndex);
  
    UNWRAP (wd, d, handleEvent);

    free (wd);
}



static Bool
    widgetInitScreen (CompPlugin *p,
                    CompScreen *s)
{
    WidgetScreen *ws;

    WIDGET_DISPLAY (s->display);

    ws = malloc (sizeof (WidgetScreen));
    if (!ws)
        return FALSE;

    ws->windowPrivateIndex = allocateWindowPrivateIndex (s);
    if (ws->windowPrivateIndex < 0)
    {
        free (ws);
        return FALSE;
    }

    widgetScreenInitOptions (ws, s->display->display);
  
    ws->state = STATE_NORMAL;
    ws->animStep = 0; //0 is fully normal
    ws->opacity = ws->animStep;
    ws->brightness = BRIGHT - (0.5 * ws->animStep);

    s->privates[wd->screenPrivateIndex].ptr = ws;
  
    addScreenAction (s, &wd->opt[WIDGET_DISPLAY_OPTION_TOGGLE_KEY].value.action);

    ws->cursor = XCreateFontCursor (s->display->display, XC_watch);
    ws->grabIndex = 0;
    ws->speed    = WIDGET_SPEED_DEFAULT;
    ws->timestep = WIDGET_TIMESTEP_DEFAULT;
  
    ws->blockWindow = 0;
    //widgetCreateBlockWindow(s);

    WRAP (ws, s, paintWindow, widgetPaintWindow);
    WRAP (ws, s, preparePaintScreen, widgetPreparePaintScreen);
    WRAP (ws, s, donePaintScreen, widgetDonePaintScreen);
    WRAP (ws, s, focusWindow, widgetFocusWindow);

    return TRUE;
}

static void
widgetFiniScreen (CompPlugin *p,
                    CompScreen *s)
{
    WIDGET_SCREEN (s);
  
    UNWRAP (ws, s, paintWindow);
    UNWRAP (ws, s, preparePaintScreen);
    UNWRAP (ws, s, donePaintScreen);
    UNWRAP (ws, s, focusWindow);

    freeWindowPrivateIndex (s, ws->windowPrivateIndex);

    free (ws);
}

static Bool
widgetInitWindow (CompPlugin *p,
                    CompWindow *w)
{
    WidgetWindow *ww;

    WIDGET_SCREEN (w->screen);
    WIDGET_DISPLAY (w->screen->display);

    ww = malloc (sizeof (WidgetWindow));
    if (!ww)
        return FALSE;
  
    w->privates[ws->windowPrivateIndex].ptr = ww;

    Atom            retType;
    int             format;
    unsigned long   nitems,remain;
    unsigned char * dat;
  
    dat = NULL;
  
    XGetWindowProperty(w->screen->display->display,w->id,wd->compizWidgetAtom,0,
                       0,FALSE,AnyPropertyType,&retType,&format,&nitems,
                       &remain,&dat);
  
    if (dat)
        XFree(dat);
  
    ww->isWidget = (retType != None);

if (ww->isWidget)
fprintf(stderr, "Made widget");

    ww->isBlockWindow = FALSE;
    ww->hasPosition = FALSE;
    ww->wantsToHide=FALSE;
  
    return TRUE;
}

static void
widgetFiniWindow (CompPlugin *p,
                    CompWindow *w)
{
    WIDGET_WINDOW (w);

    free (ww);
}

static Bool
widgetInit (CompPlugin *p)
{
    displayPrivateIndex = allocateDisplayPrivateIndex ();
    if (displayPrivateIndex < 0)
        return FALSE;

    return TRUE;
}

static void
widgetFini (CompPlugin *p)
{
    if (displayPrivateIndex >= 0)
        freeDisplayPrivateIndex (displayPrivateIndex);
}

static int
widgetGetVersion (CompPlugin *plugin,
		     int	version)
{
    return ABIVERSION;
}

CompPluginDep widgetDeps[] = {
};

static CompPluginVTable widgetVTable = {
    "widget",
    "Widget Layer",
    "Enables special layer for 'widget' windows.",
    widgetGetVersion,
    widgetInit,
    widgetFini,
    widgetInitDisplay,
    widgetFiniDisplay,
    widgetInitScreen,
    widgetFiniScreen,
    widgetInitWindow,
    widgetFiniWindow,
    widgetGetDisplayOptions,
    widgetSetDisplayOption,
    widgetGetScreenOptions,
    widgetSetScreenOption,
    0,
    0,
    0,
    0
};

CompPluginVTable *
getCompPluginInfo (void)
{
    return &widgetVTable;
}
