/*
 * Copyright © 2005 Novell, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice
 * appear in supporting documentation, and that the name of
 * Novell, Inc. not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior permission.
 * Novell, Inc. makes no representations about the suitability of this
 * software for any purpose. It is provided "as is" without express or
 * implied warranty.
 *
 * NOVELL, INC. DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL NOVELL, INC. BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Author: David Reveman <davidr@novell.com>
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <sys/time.h>

#include <X11/cursorfont.h>

#include <compiz.h>

#define WIN_X(w) ((w)->attrib.x - (w)->input.left)
#define WIN_Y(w) ((w)->attrib.y - (w)->input.top)
#define WIN_W(w) ((w)->width + (w)->input.left + (w)->input.right)
#define WIN_H(w) ((w)->height + (w)->input.top + (w)->input.bottom)

#define SCALE_SPACING_DEFAULT 25
#define SCALE_SPACING_MIN     0
#define SCALE_SPACING_MAX     250

#define SCALE_SLOPPY_FOCUS_DEFAULT TRUE


#define SCALE_INITIATE_CURRENT_HEAD_KEY_DEFAULT "F8"
#define SCALE_INITIATE_CURRENT_HEAD_MODIFIERS_DEFAULT 0


#define SCALE_INITIATE_KEY_DEFAULT       "F12"
#define SCALE_INITIATE_MODIFIERS_DEFAULT 0

#define SCALE_INITIATE_APP_KEY_DEFAULT       "F11"
#define SCALE_INITIATE_APP_MODIFIERS_DEFAULT 0

#define SCALE_INITIATE_ALL_KEY_DEFAULT       "F10"
#define SCALE_INITIATE_ALL_MODIFIERS_DEFAULT 0

#define SCALE_SPEED_DEFAULT   1.5f
#define SCALE_SPEED_MIN       0.1f
#define SCALE_SPEED_MAX       50.0f
#define SCALE_SPEED_PRECISION 0.1f

#define SCALE_DARKEN_BACK_FACTOR_DEFAULT   0.67f
#define SCALE_DARKEN_BACK_FACTOR_MIN       0.00f
#define SCALE_DARKEN_BACK_FACTOR_MAX       1.00f
#define SCALE_DARKEN_BACK_FACTOR_PRECISION 0.01f

#define SCALE_TIMESTEP_DEFAULT   1.2f
#define SCALE_TIMESTEP_MIN       0.1f
#define SCALE_TIMESTEP_MAX       50.0f
#define SCALE_TIMESTEP_PRECISION 0.1f

#define SCALE_STATE_NONE 0
#define SCALE_STATE_OUT  1
#define SCALE_STATE_WAIT 2
#define SCALE_STATE_IN   3

#define SCALE_DARKEN_BACK_DEFAULT TRUE

#define SCALE_OPACITY_DEFAULT 75
#define SCALE_OPACITY_MIN     0
#define SCALE_OPACITY_MAX     100

#define SCALE_USE_CLASS_DEFAULT FALSE

#define SCALE_HEAD_DEFAULT 1
#define SCALE_HEAD_MIN     0
#define SCALE_HEAD_MAX     10

#define SCALE_ORGANIC_DEFAULT FALSE

static char *winType[] = {
    N_("Toolbar"),
    N_("Utility"),
    N_("Dialog"),
    N_("ModalDialog"),
    N_("Fullscreen"),
    N_("Normal")
};
#define N_WIN_TYPE (sizeof (winType) / sizeof (winType[0]))

typedef enum {
    ScaleIconNone = 0,
    ScaleIconEmblem,
    ScaleIconBig
} IconOverlay;

static char *iconOverlayString[] = {
    N_("None"),
    N_("Emblem"),
    N_("Big")
};

static IconOverlay iconOverlay[] = {
    ScaleIconNone,
    ScaleIconEmblem,
    ScaleIconBig
};
#define N_ICON_TYPE (sizeof (iconOverlayString) / sizeof (iconOverlayString[0]))
#define SCALE_ICON_DEFAULT (iconOverlayString[1])

static int displayPrivateIndex;

typedef struct _ScaleSlot {
    int x1, y1, x2, y2;
    int line;
} ScaleSlot;

#define SCALE_DISPLAY_OPTION_INITIATE_NORMAL 0
#define SCALE_DISPLAY_OPTION_INITIATE_APP 1
#define SCALE_DISPLAY_OPTION_INITIATE_ALL 2
#define SCALE_DISPLAY_OPTION_INITIATE_CURRENT_HEAD 3
#define SCALE_DISPLAY_OPTION_NUM      4

typedef struct _ScaleDisplay {
    int		    screenPrivateIndex;
    HandleEventProc handleEvent;

    CompOption opt[SCALE_DISPLAY_OPTION_NUM];

    unsigned int lastActiveNum;
    KeyCode	 leftKeyCode, rightKeyCode, upKeyCode, downKeyCode;
} ScaleDisplay;

#define SCALE_SCREEN_OPTION_SPACING                  0 
#define SCALE_SCREEN_OPTION_SLOPPY_FOCUS             1
#define SCALE_SCREEN_OPTION_ICON         2
#define SCALE_SCREEN_OPTION_SPEED	             3
#define SCALE_SCREEN_OPTION_TIMESTEP	             4
#define SCALE_SCREEN_OPTION_WINDOW_TYPE              5
#define SCALE_SCREEN_OPTION_DARKEN_BACK              6
#define SCALE_SCREEN_OPTION_OPACITY                  7
#define SCALE_SCREEN_OPTION_USE_CLASS                8
#define SCALE_SCREEN_OPTION_DARKEN_BACK_FACTOR       9
#define SCALE_SCREEN_OPTION_HEAD         10
#define SCALE_SCREEN_OPTION_ORGANIC	11
#define SCALE_SCREEN_OPTION_NUM                      12


typedef struct _ScaleScreen {
    int windowPrivateIndex;

    PreparePaintScreenProc preparePaintScreen;
    DonePaintScreenProc    donePaintScreen;
    PaintScreenProc        paintScreen;
    PaintWindowProc        paintWindow;
    DamageWindowRectProc   damageWindowRect;

    CompOption opt[SCALE_SCREEN_OPTION_NUM];

    int spacing;

    float speed;
    float timestep;
    float darkenBackFactor;

    unsigned int wMask;

    int grabIndex;

    int state;
    int moreAdjust;

    Cursor cursor;

    ScaleSlot *slots;
    int        slotsSize;
    int        nSlots;

    int *line;
    int lineSize;
    int nLine;

    /* only used for sorting */
    CompWindow **windows;
    int        windowsSize;
    int        nWindows;

    GLfloat scale;

    Bool     darkenBack;
    GLushort opacity;

    Bool     currentHead;
    Bool     organic;
    Bool     allScreensMode;
    Bool     onlyCurrent;
    Bool     useClass;
    CompWindow * currentWindow;
    Window   selectedWindow;
    
    int head;

    IconOverlay iconOverlay;
} ScaleScreen;

typedef struct _ScaleWindow {
    ScaleSlot *slot;

    Bool rescaled;
    GLfloat oldScale;
    int origX, origY;

    GLfloat xVelocity, yVelocity, scaleVelocity;
    GLfloat scale;
    GLfloat tx, ty;
    Bool    adjust;
} ScaleWindow;


#define GET_SCALE_DISPLAY(d)				      \
    ((ScaleDisplay *) (d)->privates[displayPrivateIndex].ptr)

#define SCALE_DISPLAY(d)		     \
    ScaleDisplay *sd = GET_SCALE_DISPLAY (d)

#define GET_SCALE_SCREEN(s, sd)					  \
    ((ScaleScreen *) (s)->privates[(sd)->screenPrivateIndex].ptr)

#define SCALE_SCREEN(s)							   \
    ScaleScreen *ss = GET_SCALE_SCREEN (s, GET_SCALE_DISPLAY (s->display))

#define GET_SCALE_WINDOW(w, ss)					  \
    ((ScaleWindow *) (w)->privates[(ss)->windowPrivateIndex].ptr)

#define SCALE_WINDOW(w)					       \
    ScaleWindow *sw = GET_SCALE_WINDOW  (w,		       \
		      GET_SCALE_SCREEN  (w->screen,	       \
		      GET_SCALE_DISPLAY (w->screen->display)))

#define NUM_OPTIONS(s) (sizeof ((s)->opt) / sizeof (CompOption))

static CompOption *
scaleGetScreenOptions (CompScreen *screen,
		       int	  *count)
{
    SCALE_SCREEN (screen);

    *count = NUM_OPTIONS (ss);
    return ss->opt;
}

static Bool
scaleSetHead ( ScaleScreen *ss,
               CompScreen *s,
               int head )
{
    if ( !s->useXinerama )
    {
        if ( head > 0 )
            return FALSE;
        else ss->head = 1;
    }
    else
    {
        if ( head == -1 )
            ss->head = -1;
        else if ( head == 0 )
            ss->head = s->headsCount + 1;
        else if ( head > s->headsCount )
            return FALSE;
        else ss->head = head;
    }

    return TRUE;
}

static Bool
scaleSetScreenOption (CompScreen      *screen,
		      char	      *name,
		      CompOptionValue *value)
{
    CompOption *o;
    int	       index;

    SCALE_SCREEN (screen);

    o = compFindOption (ss->opt, NUM_OPTIONS (ss), name, &index);

    if (!o)
	return FALSE;

    switch (index) {
    case SCALE_SCREEN_OPTION_SPACING:
	if (compSetIntOption (o, value))
	{
	    ss->spacing = o->value.i;
	    return TRUE;
	}
	break;
    case SCALE_SCREEN_OPTION_SLOPPY_FOCUS:
	if (compSetBoolOption (o, value))
	    return TRUE;
	break;
    case SCALE_SCREEN_OPTION_ORGANIC:
	if (compSetBoolOption (o, value))
	{
	    ss->organic = o->value.b;
	    return TRUE;
	}
	break;
    case SCALE_SCREEN_OPTION_SPEED:
	if (compSetFloatOption (o, value))
	{
	    ss->speed = o->value.f;
	    return TRUE;
	}
	break;    
    case SCALE_SCREEN_OPTION_DARKEN_BACK_FACTOR:
	if (compSetFloatOption (o, value))
	{
	    ss->darkenBackFactor = o->value.f;
	    return TRUE;
	}
	break;
    case SCALE_SCREEN_OPTION_TIMESTEP:
	if (compSetFloatOption (o, value))
	{
	    ss->timestep = o->value.f;
	    return TRUE;
	}
	break;
    case SCALE_SCREEN_OPTION_WINDOW_TYPE:
	if (compSetOptionList (o, value))
	{
	    ss->wMask = compWindowTypeMaskFromStringList (&o->value);
	    return TRUE;
	}
	break;
    case SCALE_SCREEN_OPTION_DARKEN_BACK:
	if (compSetBoolOption (o, value))
	{
	    ss->darkenBack = o->value.b;
	    return TRUE;
	}
    break;
      case SCALE_SCREEN_OPTION_USE_CLASS:
        if (compSetBoolOption (o, value))
        {
          ss->useClass = o->value.b;
          return TRUE;
        }
      case SCALE_SCREEN_OPTION_OPACITY:
	if (compSetIntOption (o, value))
	{
	    ss->opacity = (OPAQUE * o->value.i) / 100;
	    return TRUE;
	}
	break;
    case SCALE_SCREEN_OPTION_ICON:
	if (compSetStringOption (o, value))
	{
	    int i;
 
	    for (i = 0; i < N_ICON_TYPE; i++)
	    {
		if (strcmp (o->value.s, iconOverlayString[i]) == 0)
		{
		    ss->iconOverlay = iconOverlay[i];
		    return TRUE;
		}
	    }
	}
	break;
    case SCALE_SCREEN_OPTION_HEAD:
	if (compSetIntOption (o, value))
	{
	    return scaleSetHead ( ss, screen, o->value.i - 1 );
	}
	break;

      default:
	break;
    }

    return FALSE;
}

static void
scaleScreenInitOptions (ScaleScreen *ss)
{
    CompOption *o;
    int	       i;

    o = &ss->opt[SCALE_SCREEN_OPTION_SPACING];
    o->name	  = "spacing";
    o->shortDesc  = N_("Spacing");
    o->longDesc   = N_("Space between windows");
    o->type	  = CompOptionTypeInt;
    o->value.i	  = SCALE_SPACING_DEFAULT;
    o->rest.i.min = SCALE_SPACING_MIN;
    o->rest.i.max = SCALE_SPACING_MAX;

    o = &ss->opt[SCALE_SCREEN_OPTION_SLOPPY_FOCUS];
    o->name	  = "sloppy_focus";
    o->shortDesc  = N_("Sloppy Focus");
    o->longDesc   = N_("Focus window when mouse moves over them");
    o->type	  = CompOptionTypeBool;
    o->value.b	  = SCALE_SLOPPY_FOCUS_DEFAULT;

    o = &ss->opt[SCALE_SCREEN_OPTION_ORGANIC];
    o->name	  = "organic";
    o->shortDesc  = "Organic Scale Algorithm";
    o->longDesc   = "Use Organic Algorithm for Window Placement";
    o->type	  = CompOptionTypeBool;
    o->value.b	  = SCALE_ORGANIC_DEFAULT;

    o = &ss->opt[SCALE_SCREEN_OPTION_SPEED];
    o->name		= "speed";
    o->shortDesc	= N_("Speed");
    o->longDesc		= N_("Scale speed");
    o->type		= CompOptionTypeFloat;
    o->value.f		= SCALE_SPEED_DEFAULT;
    o->rest.f.min	= SCALE_SPEED_MIN;
    o->rest.f.max	= SCALE_SPEED_MAX;
    o->rest.f.precision = SCALE_SPEED_PRECISION;

    o = &ss->opt[SCALE_SCREEN_OPTION_DARKEN_BACK_FACTOR];
    o->name		= "darken_back_factor";
    o->shortDesc	= "Darken Back factor";
    o->longDesc		= "Set how dark the background will become (0.0 to 1.0)";
    o->type		= CompOptionTypeFloat;
    o->value.f		= SCALE_DARKEN_BACK_FACTOR_DEFAULT;
    o->rest.f.min	= SCALE_DARKEN_BACK_FACTOR_MIN;
    o->rest.f.max	= SCALE_DARKEN_BACK_FACTOR_MAX;
    o->rest.f.precision = SCALE_DARKEN_BACK_FACTOR_PRECISION;

    o = &ss->opt[SCALE_SCREEN_OPTION_TIMESTEP];
    o->name		= "timestep";
    o->shortDesc	= N_("Timestep");
    o->longDesc		= N_("Scale timestep");
    o->type		= CompOptionTypeFloat;
    o->value.f		= SCALE_TIMESTEP_DEFAULT;
    o->rest.f.min	= SCALE_TIMESTEP_MIN;
    o->rest.f.max	= SCALE_TIMESTEP_MAX;
    o->rest.f.precision = SCALE_TIMESTEP_PRECISION;

    o = &ss->opt[SCALE_SCREEN_OPTION_WINDOW_TYPE];
    o->name	         = "window_types";
    o->shortDesc         = N_("Window Types");
    o->longDesc	         = N_("Window types that should scaled in scale mode");
    o->type	         = CompOptionTypeList;
    o->value.list.type   = CompOptionTypeString;
    o->value.list.nValue = N_WIN_TYPE;
    o->value.list.value  = malloc (sizeof (CompOptionValue) * N_WIN_TYPE);
    for (i = 0; i < N_WIN_TYPE; i++)
	o->value.list.value[i].s = strdup (winType[i]);
    o->rest.s.string     = windowTypeString;
    o->rest.s.nString    = nWindowTypeString;

    ss->wMask = compWindowTypeMaskFromStringList (&o->value);

    o = &ss->opt[SCALE_SCREEN_OPTION_DARKEN_BACK];
    o->name      = "darken_back";
    o->shortDesc = "Darken Background";
    o->longDesc  = "Darken background when scaling windows";
    o->type      = CompOptionTypeBool;
    o->value.b   = SCALE_DARKEN_BACK_DEFAULT;

    o = &ss->opt[SCALE_SCREEN_OPTION_OPACITY];
    o->name	  = "opacity";
    o->shortDesc  = "Opacity";
    o->longDesc	  = "Amount of opacity in percent";
    o->type	  = CompOptionTypeInt;
    o->value.i    = SCALE_OPACITY_DEFAULT;
    o->rest.i.min = SCALE_OPACITY_MIN;
    o->rest.i.max = SCALE_OPACITY_MAX;
    
    o = &ss->opt[SCALE_SCREEN_OPTION_USE_CLASS];
    o->name      = "use_class";
    o->shortDesc = "Use WM_CLASS for only-current";
    o->longDesc  = "Use WM_CLASS to find out wether a window should be scaled in only-current";
    o->type      = CompOptionTypeBool;
    o->value.b   = SCALE_USE_CLASS_DEFAULT;

    o = &ss->opt[SCALE_SCREEN_OPTION_HEAD];
    o->name	  = "head";
    o->shortDesc  = "Head";
    o->longDesc	  = "Head on which windows are scaled";
    o->type	  = CompOptionTypeInt;
    o->value.i    = SCALE_HEAD_DEFAULT;
    o->rest.i.min = SCALE_HEAD_MIN;
    o->rest.i.max = SCALE_HEAD_MAX;

    o = &ss->opt[SCALE_SCREEN_OPTION_ICON];
    o->name	      = "overlay_icon";
    o->shortDesc      = N_("Overlay Icon");
    o->longDesc	      = N_("Overlay an icon on windows once they are scaled");
    o->type	      = CompOptionTypeString;
    o->value.s	      = strdup (SCALE_ICON_DEFAULT);
    o->rest.s.string  = iconOverlayString;
    o->rest.s.nString = N_ICON_TYPE;
}

static Bool
isScaleWin (CompWindow *w)
{
    SCALE_SCREEN (w->screen);

    if (!ss->allScreensMode && ss->currentHead
	&& !ss->onlyCurrent
	&& screenGetHeadForWindow (w) != screenGetCurrentHead (w->screen)){
    	return FALSE;
    }

    if (ss->allScreensMode)
    {
    	if ((!w->mapNum || w->attrib.map_state != IsViewable)) 
            // && ((!w->minimized && !(w->state & CompWindowStateHiddenMask))))
            return FALSE;
    }
    else
    {
        if (!(*w->screen->focusWindow) (w))
        	return FALSE;
    }

    if (w->state & CompWindowStateShadedMask)
	return FALSE;

    if (!(ss->wMask & w->type))
	return FALSE;

    if (w->state & CompWindowStateSkipPagerMask)
	return FALSE;

    if (ss->onlyCurrent && ss->currentWindow)
    {
      if (!ss->useClass && (w->clientLeader != ss->currentWindow->clientLeader))
        return FALSE;
      if (ss->useClass && (strcmp(w->resClass,ss->currentWindow->resClass)!=0))
        return FALSE;
    }

    if (!(ss->currentWindow))
      return TRUE;
    if (ss->onlyCurrent && ss->currentWindow != w)
      restackWindowBelow(w,ss->currentWindow);
    return TRUE;
}

static Bool
scalePaintWindow (CompWindow		  *w,
		  const WindowPaintAttrib *attrib,
		  Region		  region,
		  unsigned int		  mask)
{
    CompScreen *s = w->screen;
    Bool       status;

    SCALE_SCREEN (s);

    if (ss->grabIndex)
    {
	WindowPaintAttrib sAttrib = *attrib;

	SCALE_WINDOW (w);

	if (sw->adjust || sw->slot)
	{
	    mask |= PAINT_WINDOW_TRANSFORMED_MASK;

	    if (w->id	    != s->display->activeWindow &&
		ss->opacity != OPAQUE			&&
		ss->state   != SCALE_STATE_IN)
	    {
		/* modify opacity of windows that are not active */
  		sAttrib.opacity = (sAttrib.opacity * ss->opacity) >> 16;
	    }
	}
	else if (ss->darkenBack && ss->state != SCALE_STATE_IN)
	{
	    /* modify brightness of the other windows */
 	    sAttrib.brightness = sAttrib.brightness / 2;
	}

	UNWRAP (ss, s, paintWindow);
	status = (*s->paintWindow) (w, &sAttrib, region, mask);
	WRAP (ss, s, paintWindow, scalePaintWindow);
 
	if ((ss->iconOverlay != ScaleIconNone) && (sw->adjust || sw->slot))
	{
	    CompIcon *icon;
 
	    icon = getWindowIcon (w, 96, 96);
	    if (!icon)
		icon = w->screen->defaultIcon;
 
	    if (icon && (icon->texture.name || iconToTexture (w->screen, icon)))
	    {
		REGION	    iconReg;
		CompMatrix  matrix;
		int	    wx, wy, width, height;
		int	    scaledWinWidth, scaledWinHeight;
		float	    ds;
 
		scaledWinWidth  = w->width  * sw->scale;
		scaledWinHeight = w->height * sw->scale;
 
		switch (ss->iconOverlay) {
		case ScaleIconNone:
		case ScaleIconEmblem:
		    sAttrib.xScale = sAttrib.yScale = 1.0f;
		    break;
		case ScaleIconBig:
		    sAttrib.opacity /= 3;
		    sAttrib.xScale = sAttrib.yScale =
			MIN (((float) scaledWinWidth / (float) icon->width),
			     ((float) scaledWinHeight / (float) icon->height));
		default:
		    break;
		}
 
		width  = icon->width  * sAttrib.xScale;
		height = icon->height * sAttrib.yScale;
  
		switch (ss->iconOverlay) {
		case ScaleIconNone:
		case ScaleIconEmblem:
		    wx = w->attrib.x + (w->width  * sw->scale) - icon->width;
		    wy = w->attrib.y + (w->height * sw->scale) - icon->height;
		    break;
		case ScaleIconBig:
		    wx = w->attrib.x + ((scaledWinWidth - width) / 2) /
			sAttrib.xScale;
		    wy = w->attrib.y + ((scaledWinHeight - height) / 2) /
			sAttrib.yScale;
		    break;
		default:
		    wx = wy = 0;
		    break;
		}
  
		ds = 1.0f - ss->scale;
		if (ds)
		{
		    sAttrib.opacity =
			(fabs (1.0f - sw->scale) * sAttrib.opacity) / ds;
		}
		else if (!sw->slot)
		{
		    sAttrib.opacity = 0;
		}
 
		mask |= PAINT_WINDOW_TRANSLUCENT_MASK;
 
		iconReg.rects    = &iconReg.extents;
		iconReg.numRects = 1;
 
		iconReg.extents.x1 = wx;
		iconReg.extents.y1 = wy;
		iconReg.extents.x2 = iconReg.extents.x1 + width;
		iconReg.extents.y2 = iconReg.extents.y1 + height;
 
		matrix = icon->texture.matrix;
		matrix.x0 -= wx * icon->texture.matrix.xx;
		matrix.y0 -= wy * icon->texture.matrix.yy;
 
		w->vCount = 0;
		if (iconReg.extents.x1 < iconReg.extents.x2 &&
		    iconReg.extents.y1 < iconReg.extents.y2)
		    (*w->screen->addWindowGeometry) (w, &matrix, 1, &iconReg,
						     &iconReg);
 
		if (w->vCount)
		    (*w->screen->drawWindowTexture) (w,
						     &icon->texture, &sAttrib,
						     mask);
	    }
	}
   }
    else
    {
	UNWRAP (ss, s, paintWindow);
	status = (*s->paintWindow) (w, attrib, region, mask);
	WRAP (ss, s, paintWindow, scalePaintWindow);
   }
  
    return status;

}


static int
compareWindows (const void *elem1,
		const void *elem2)
{
    CompWindow *w1 = *((CompWindow **) elem1);
    CompWindow *w2 = *((CompWindow **) elem2);

    if (!(WIN_X(w1) - WIN_X(w2)))
      return WIN_Y(w1) - WIN_Y(w2);
    return WIN_X(w1) - WIN_X(w2);
}

/**
 * experimental organic layout method
 * inspired by smallwindows (smallwindows.sf.net) by Jens Egeblad
 * */
#define ORGANIC_STEP 0.05

static int
organicCompareWindows (const void *elem1,
		const void *elem2)
{
    CompWindow *w1 = *((CompWindow **) elem1);
    CompWindow *w2 = *((CompWindow **) elem2);

    return (WIN_X(w1)+WIN_Y(w1)) - (WIN_X(w2)+WIN_Y(w2));
}


static double
layoutOrganicCalculateOverlap (CompScreen *s, int win, int x, int y)
{
    SCALE_SCREEN (s);
    int i;
    int x1 = x;
    int y1 = y;
    int x2 = x1 + WIN_W(ss->windows[win])*ss->scale;
    int y2 = y1 + WIN_H(ss->windows[win])*ss->scale;
    int overlapx = 0, overlapy = 0;
    int xmin, xmax;
    int ymin, ymax;

    double result = -0.01;

    for (i = 0; i < ss->nWindows; i++)
    {
        if (i == win)
            continue;
        overlapx = overlapy = 0;
        xmax = MAX(ss->slots[i].x1, x1);
        xmin = MIN(ss->slots[i].x1+WIN_W(ss->windows[i])*ss->scale,x2);
        if (xmax <= xmin)
            overlapx = xmin - xmax;

        ymax = MAX(ss->slots[i].y1, y1);
        ymin = MIN(ss->slots[i].y1+WIN_H(ss->windows[i])*ss->scale,y2);

        if (ymax <= ymin)
            overlapy = ymin - ymax;

        result += (double) overlapx * overlapy;
    }
    return result;
}


static double
layoutOrganicFindBestHorizontalPosition (CompScreen *s, int win, int *bestx, int areaWidth)
{
    SCALE_SCREEN (s);
    int i;
    int y1 = ss->slots[win].y1;
    int y2 = ss->slots[win].y1 + WIN_H(ss->windows[win])*ss->scale;

    double bestoverlap = 1e31, overlap;
    int w = WIN_W(ss->windows[win])*ss->scale;
    *bestx = ss->slots[win].x1;

    for (i = 0; i < ss->nWindows; i++)
    {
        if (i == win)
            continue;

        if (ss->slots[i].y1 < y2 && ss->slots[i].y1 + WIN_H(ss->windows[i]) * ss->scale > y1)
        {
            if (ss->slots[i].x1 - w >= 0)
            {
                double overlap =  layoutOrganicCalculateOverlap(s, win, ss->slots[i].x1 - w, y1);
                if (overlap < bestoverlap)
                {
                    *bestx = ss->slots[i].x1 - w;
                    bestoverlap = overlap;
                }
            }
            if (WIN_W(ss->windows[i])*ss->scale + ss->slots[i].x1 + w < areaWidth)
            {
                double overlap = layoutOrganicCalculateOverlap(s, win, ss->slots[i].x1 + WIN_W(ss->windows[i])*ss->scale,y1);
                if (overlap < bestoverlap)
                {
                    *bestx = ss->slots[i].x1 + WIN_W(ss->windows[i])*ss->scale; 
                    bestoverlap = overlap;
                }
            }
        }
    }

    overlap = layoutOrganicCalculateOverlap (s, win, 0, y1);
    if (overlap < bestoverlap)
    {
        *bestx = 0;
        bestoverlap = overlap;
    }

    overlap = layoutOrganicCalculateOverlap(s,win, areaWidth - w, y1);
    if (overlap < bestoverlap)
    {
        *bestx = areaWidth - w;
        bestoverlap = overlap;
    }

    return bestoverlap;
}

static double
layoutOrganicFindBestVerticalPosition (CompScreen *s, int win, int *besty, int areaHeight)
{
    SCALE_SCREEN (s);
    int i;
    int x1 = ss->slots[win].x1;
    int x2 = ss->slots[win].x1 + WIN_W(ss->windows[win])*ss->scale;

    double bestoverlap = 1e31, overlap;
    int h = WIN_H(ss->windows[win])*ss->scale;
    *besty = ss->slots[win].y1;

    for (i = 0; i < ss->nWindows; i++)
    {
        if (i == win)
            continue;

        if (ss->slots[i].x1 < x2 && ss->slots[i].x1 + WIN_W(ss->windows[i]) * ss->scale > x1)
        {
            if (ss->slots[i].y1 - h >= 0 && ss->slots[i].y1 < areaHeight)
            {
                double overlap =  layoutOrganicCalculateOverlap(s, win, x1, ss->slots[i].y1-h);
                if (overlap < bestoverlap)
                {
                    *besty = ss->slots[i].y1 - h;
                    bestoverlap = overlap;
                }
            }
            if (WIN_H(ss->windows[i])*ss->scale + ss->slots[i].y1 > 0 && WIN_H(ss->windows[i])*ss->scale + h + ss->slots[i].y1 < areaHeight)
            {
                double overlap = layoutOrganicCalculateOverlap(s, win, x1, WIN_H(ss->windows[i])*ss->scale + ss->slots[i].y1);
                if (overlap < bestoverlap)
                {
                    *besty = ss->slots[i].y1 + WIN_H(ss->windows[i])*ss->scale;
                    bestoverlap = overlap;
                }
            }
        }
    }

    overlap = layoutOrganicCalculateOverlap (s, win, x1, 0);
    if (overlap < bestoverlap)
    {
        *besty = 0;
        bestoverlap = overlap;
    }

    overlap = layoutOrganicCalculateOverlap(s,win, x1, areaHeight - h);
    if (overlap < bestoverlap)
    {
        *besty = areaHeight - h;
        bestoverlap = overlap;
    }

    return bestoverlap;
}


static Bool
layoutOrganicLocalSearch (CompScreen *s, int areaWidth, int areaHeight)
{
    SCALE_SCREEN (s);
    Bool improvement = FALSE;
    int i;
    double totaloverlap;

    do {
        improvement = FALSE;
        for (i = 0; i < ss->nWindows; i++) {
            Bool improved = FALSE;
            do
            {
                int newx, newy;
                double oldoverlap, overlaph, overlapv;
                improved = FALSE;
                oldoverlap = layoutOrganicCalculateOverlap (s, i, ss->slots[i].x1, ss->slots[i].y1);
                overlaph = layoutOrganicFindBestHorizontalPosition (s, i, &newx, areaWidth);
                overlapv = layoutOrganicFindBestVerticalPosition (s, i, &newy, areaHeight);
                if (overlaph < oldoverlap-0.1 || overlapv < oldoverlap-0.1) {
                    improved = TRUE;
                    improvement = TRUE;
                    if (overlapv > overlaph)
                        ss->slots[i].x1 = newx;
                    else
                        ss->slots[i].y1 = newy;
                }

            } while (improved);
        }
    } while (improvement);

    totaloverlap = 0.0;
    for (i = 0; i < ss->nWindows; i++) {
        totaloverlap += layoutOrganicCalculateOverlap (s, i, ss->slots[i].x1, ss->slots[i].y1);
    }
    return (totaloverlap > 0.1);
}

static void 
layoutOrganicRemoveOverlap (CompScreen *s, int areaWidth, int areaHeight)
{
    int i;
    SCALE_SCREEN (s);
    CompWindow *w;
    while (layoutOrganicLocalSearch (s, areaWidth, areaHeight))
    {
        for (i = 0; i < ss->nWindows; i++)
        {
            int centerX, centerY;
            int newX, newY, newWidth, newHeight;

            w = ss->windows[i];

            centerX = ss->slots[i].x1 + WIN_W (w) / 2;
            centerY = ss->slots[i].y1 + WIN_H (w) / 2;

            newWidth  = (int)( (1.0 - ORGANIC_STEP) * (double)WIN_W (w) ) - ss->spacing/2;
            newHeight = (int)( (1.0 - ORGANIC_STEP) * (double)WIN_H (w) ) - ss->spacing/2;
            newX = centerX - (newWidth / 2);
            newY = centerY - (newHeight / 2);

            ss->slots[i].x1 = newX;
            ss->slots[i].y1 = newY;
            ss->slots[i].x2 = newX + WIN_W (w);
            ss->slots[i].y2 = newY + WIN_H (w);
        }
        ss->scale -= ORGANIC_STEP;
    }
}

static Bool
layoutOrganicThumbs (CompScreen *s)
{
    CompWindow *w;
    
    int i;
    int cx, cy;
    int areaX, areaY, areaWidth, areaHeight;
    int head;

    SCALE_SCREEN (s);

    if (ss->head == -1)
        head = screenGetCurrentHead (s);
    else
        head = ss->head;

    ss->scale = 1.0f;

    areaX = screenGetHeadAreaX (s, head);
    areaY = screenGetHeadAreaY (s, head);
    areaWidth = screenGetHeadAreaWidth (s, head);
    areaHeight = screenGetHeadAreaHeight (s, head);

    cx = cy = ss->nWindows = 0;
    for (w = s->windows; w; w = w->next)
    {
        SCALE_WINDOW (w);

        if (sw->slot)
            sw->adjust = TRUE;

        sw->slot = 0;

        if (!isScaleWin (w))
            continue;

        if (ss->windowsSize <= ss->nWindows)
        {
            ss->windows = realloc (ss->windows,
                    sizeof (CompWindow *) * (ss->nWindows + 32));
            if (!ss->windows)
                return FALSE;

            ss->windowsSize = ss->nWindows + 32;
        }
        
        ss->windows[ss->nWindows++] = w;
    }
    if (ss->nWindows == 0)
        return FALSE;

    qsort (ss->windows, ss->nWindows, sizeof (CompWindow *), organicCompareWindows);

    if (ss->slotsSize <= ss->nWindows)
    {
        ss->slots = realloc (ss->slots, sizeof (ScaleSlot) *
                (ss->nWindows + 1));

        if (!ss->slots)
            return FALSE;

        ss->slotsSize = ss->nWindows + 1;
    }

    for (i = 0; i < ss->nWindows; i++)
    {
        SCALE_WINDOW (ss->windows[i]);
        w = ss->windows[i];

        sw->slot = &ss->slots[i];
        ss->slots[i].x1 = WIN_X(w) - areaX;
        ss->slots[i].y1 = WIN_Y(w) - areaY;
        ss->slots[i].x2 = WIN_X(w) + WIN_W (w) - areaX;
        ss->slots[i].y2 = WIN_Y(w) + WIN_H (w) - areaY;

        if (ss->slots[i].x1 < 0)
        {
            ss->slots[i].x2 += abs(ss->slots[i].x1);
            ss->slots[i].x1 = 0;
        }
        if (ss->slots[i].x2 > areaWidth-areaX)
        {
            ss->slots[i].x1 -= abs(ss->slots[i].x2-areaWidth);
            ss->slots[i].x2 = areaWidth-areaX;
        }

        if (ss->slots[i].y1 < 0)
        {
            ss->slots[i].y2 += abs(ss->slots[i].y1);
            ss->slots[i].y1 = 0;
        }
        if (ss->slots[i].y2 > areaWidth-areaY)
        {
            ss->slots[i].y1 -= abs(ss->slots[i].y2-areaHeight-areaY);
            ss->slots[i].y2 = areaHeight-areaY;
        }


    }

    ss->nSlots = ss->nWindows;

    layoutOrganicRemoveOverlap(s, areaWidth-areaX, areaHeight-areaY);
    for (i = 0; i < ss->nWindows; i++)
    {
        SCALE_WINDOW (ss->windows[i]);
        ss->slots[i].x1 += ss->windows[i]->input.left + areaX;
        ss->slots[i].x2 += ss->windows[i]->input.left + areaX;
        ss->slots[i].y1 += ss->windows[i]->input.top + areaY;
        ss->slots[i].y2 += ss->windows[i]->input.top + areaY;
        sw->adjust = TRUE;
    }

    return TRUE;
}

static void
zoomWindow (CompScreen *s, CompWindow *w)
{
	SCALE_SCREEN(s);
	SCALE_WINDOW(w);

	int headHeight, headWidth, head, headX, headY;

	head = screenGetCurrentHead ( s );

	
       	headWidth = screenGetHeadAreaWidth ( s, head );
	headHeight = screenGetHeadAreaHeight ( s, head );
	headX = screenGetHeadAreaX ( s, head);
	headY = screenGetHeadAreaY ( s, head);

	
	
	if (sw->rescaled == FALSE)
	{
		raiseWindow (w);
		//backup old values
                sw->oldScale = ss->scale;
		sw->origX = sw->slot->x1;
		sw->origY = sw->slot->y1;

                ss->scale = 1.0f;
		sw->adjust = TRUE;
		
		sw->rescaled = TRUE;
		
		sw->slot->x1 = (headWidth/2) - ((w->attrib.width*ss->scale)/2) + headX;
		sw->slot->y1 = (headHeight/2) - ((w->attrib.height*ss->scale)/2) + headY;
	}else
	{
		sw->rescaled = FALSE; 
		ss->scale = sw->oldScale;
		sw->slot->x1 = sw->origX;
		sw->slot->y1 = sw->origY;
		sw->adjust = TRUE;
	}
}

static void
resetZoomValues ( CompScreen *s)
{	
	CompWindow *w;
	
	for (w = s->windows; w; w = w->next)
	{	
		SCALE_WINDOW (w);
		if (sw->rescaled && isScaleWin (w))
			sw->rescaled = FALSE;
	}
}


/* TODO: Place window thumbnails at smarter positions */
static Bool
layoutThumbs (CompScreen *s)
{
    CompWindow *w;
    int	       i, j, y2;
    int        cx, cy;
    int        lineLength, itemsPerLine;
    float      scaleW, scaleH;
    int        totalWidth, totalHeight;
    int        areaX, areaY, areaWidth, areaHeight, headWidth, headHeight, head;
    Bool       useOrganic;

    SCALE_SCREEN (s);

    useOrganic = ss->organic;

    //organic doesnt play nice with currentHead only always
    if (ss->currentHead) useOrganic = FALSE;

    if (useOrganic)
        return layoutOrganicThumbs (s);


    if ( ss->head == -1 ) head = screenGetCurrentHead ( s );
    else head = ss->head;

    headWidth = screenGetHeadWidth ( s, head );
    headHeight = screenGetHeadHeight ( s, head );
    areaX = screenGetHeadAreaX ( s, head );
    areaY = screenGetHeadAreaY ( s, head );
    areaWidth = screenGetHeadAreaWidth ( s, head );
    areaHeight = screenGetHeadAreaHeight ( s, head );

    cx = cy = ss->nWindows = 0;

	for (w = s->windows; w; w = w->next)
	{
		SCALE_WINDOW (w);
	
		if (sw->slot)
		sw->adjust = TRUE;
	
		sw->slot = 0;

	
		if (!isScaleWin (w))
		continue;
	
		if (ss->windowsSize <= ss->nWindows)
		{
		ss->windows = realloc (ss->windows,
					sizeof (CompWindow *) * (ss->nWindows + 32));
		if (!ss->windows)
			return FALSE;
	
		ss->windowsSize = ss->nWindows + 32;
		}
	
		ss->windows[ss->nWindows++] = w;
	}

    if (ss->nWindows == 0)
	return FALSE;

    qsort (ss->windows, ss->nWindows, sizeof (CompWindow *), compareWindows);

    itemsPerLine = (sqrt (ss->nWindows) * headWidth) / headHeight;
    if (itemsPerLine < 1)
	itemsPerLine = 1;

    if (ss->lineSize <= ss->nWindows / itemsPerLine + 1)
    {
	ss->line = realloc (ss->line, sizeof (int) *
			    (ss->nWindows / itemsPerLine + 2));
	if (!ss->line)
	    return FALSE;

	ss->lineSize = ss->nWindows / itemsPerLine + 2;
    }

    totalWidth = totalHeight = 0;

    ss->line[0] = 0;
    ss->nLine = 1;
    lineLength = itemsPerLine;

    if (ss->slotsSize <= ss->nWindows)
    {
	ss->slots = realloc (ss->slots, sizeof (ScaleSlot) *
			     (ss->nWindows + 1));
	if (!ss->slots)
	    return FALSE;

	ss->slotsSize = ss->nWindows + 1;
    }
    ss->nSlots = 0;

    for (i = 0; i < ss->nWindows; i++)
    {
	SCALE_WINDOW (ss->windows[i]);

	w = ss->windows[i];

	raiseWindow (w);

	/* find a good place between other elements */
	for (j = 0; j < ss->nSlots; j++)
	{
	    y2 = ss->slots[j].y2 + ss->spacing + WIN_H (w);
	    if (w->width < ss->slots[j].x2 - ss->slots[j].x1 &&
		y2 <= ss->line[ss->slots[j].line])
		break;
	}

	/* otherwise append or start a new line */
	if (j == ss->nSlots)
	{
	    if (lineLength < itemsPerLine)
	    {
		lineLength++;

		ss->slots[ss->nSlots].x1 = cx;
		ss->slots[ss->nSlots].y1 = cy;
		ss->slots[ss->nSlots].x2 = cx + WIN_W (w);
		ss->slots[ss->nSlots].y2 = cy + WIN_H (w);
		ss->slots[ss->nSlots].line = ss->nLine - 1;

		ss->line[ss->nLine - 1] = MAX (ss->line[ss->nLine - 1],
					       ss->slots[ss->nSlots].y2);
	    }
	    else
	    {
		lineLength = 1;

		cx = ss->spacing;
		cy = ss->line[ss->nLine - 1] + ss->spacing;

		ss->slots[ss->nSlots].x1 = cx;
		ss->slots[ss->nSlots].y1 = cy;
		ss->slots[ss->nSlots].x2 = cx + WIN_W (w);
		ss->slots[ss->nSlots].y2 = cy + WIN_H (w);
		ss->slots[ss->nSlots].line = ss->nLine - 1;

		ss->line[ss->nLine] = ss->slots[ss->nSlots].y2;

		ss->nLine++;
	    }

	    if (ss->slots[ss->nSlots].y2 > totalHeight)
		totalHeight = ss->slots[ss->nSlots].y2;
	}
	else
	{
	    ss->slots[ss->nSlots].x1 = ss->slots[j].x1;
	    ss->slots[ss->nSlots].y1 = ss->slots[j].y2 + ss->spacing;
	    ss->slots[ss->nSlots].x2 = ss->slots[ss->nSlots].x1 + WIN_W (w);
	    ss->slots[ss->nSlots].y2 = ss->slots[ss->nSlots].y1 + WIN_H (w);
	    ss->slots[ss->nSlots].line = ss->slots[j].line;

	    ss->slots[j].line = 0;
	}

	cx = ss->slots[ss->nSlots].x2;
	if (cx > totalWidth)
	    totalWidth = cx;

	cx += ss->spacing;

	sw->slot = &ss->slots[ss->nSlots];
	sw->adjust = TRUE;

	ss->nSlots++;
    }

    totalWidth  += ss->spacing;
    totalHeight += ss->spacing;

    scaleW = (float) areaWidth / totalWidth;
    scaleH = (float) areaHeight / totalHeight;

    ss->scale = MIN (MIN (scaleH, scaleW), 1.0f);

    for (i = 0; i < ss->nWindows; i++)
    {
	SCALE_WINDOW (ss->windows[i]);

	if (sw->slot)
	{
	    ss->slots[i].y1 += ss->windows[i]->input.top;
	    ss->slots[i].x1 += ss->windows[i]->input.left;
 	    ss->slots[i].y2 += ss->windows[i]->input.top;
 	    ss->slots[i].x2 += ss->windows[i]->input.left;
	    ss->slots[i].y1 = (float) ss->slots[i].y1 * ss->scale;
	    ss->slots[i].x1 = (float) ss->slots[i].x1 * ss->scale;
 	    ss->slots[i].y2 = (float) ss->slots[i].y2 * ss->scale;
 	    ss->slots[i].x2 = (float) ss->slots[i].x2 * ss->scale;
	    ss->slots[i].x1 += areaX;
	    ss->slots[i].y1 += areaY;
 	    ss->slots[i].x2 += areaX;
 	    ss->slots[i].y2 += areaY;
	}
    }

    return TRUE;
}

static int
adjustScaleVelocity (CompWindow *w)
{
    float dx, dy, ds, adjust, amount;
    float x1, y1, scale;

    SCALE_SCREEN (w->screen);
    SCALE_WINDOW (w);

    if (sw->slot)
    {
	x1 = sw->slot->x1;
	y1 = sw->slot->y1;
	scale = ss->scale;
    }
    else
    {
	x1 = w->serverX;
	y1 = w->serverY;
	scale = 1.0f;
    }

    dx = x1 - (w->serverX + sw->tx);

    adjust = dx * 0.15f;
    amount = fabs (dx) * 1.5f;
    if (amount < 0.5f)
	amount = 0.5f;
    else if (amount > 5.0f)
	amount = 5.0f;

    sw->xVelocity = (amount * sw->xVelocity + adjust) / (amount + 1.0f);

    dy = y1 - (w->serverY + sw->ty);

    adjust = dy * 0.15f;
    amount = fabs (dy) * 1.5f;
    if (amount < 0.5f)
	amount = 0.5f;
    else if (amount > 5.0f)
	amount = 5.0f;

    sw->yVelocity = (amount * sw->yVelocity + adjust) / (amount + 1.0f);

    ds = scale - sw->scale;

    adjust = ds * 0.1f;
    amount = fabs (ds) * 7.0f;
    if (amount < 0.01f)
	amount = 0.01f;
    else if (amount > 0.15f)
	amount = 0.15f;

    sw->scaleVelocity = (amount * sw->scaleVelocity + adjust) /
	(amount + 1.0f);

    if (fabs (dx) < 0.1f && fabs (sw->xVelocity) < 0.2f &&
	fabs (dy) < 0.1f && fabs (sw->yVelocity) < 0.2f &&
	fabs (ds) < 0.001f && fabs (sw->scaleVelocity) < 0.002f)
    {
	sw->xVelocity = sw->yVelocity = sw->scaleVelocity = 0.0f;
	sw->tx = x1 - w->serverX;
	sw->ty = y1 - w->serverY;
	sw->scale = scale;

	return 0;
    }

    return 1;
}

static Bool
scalePaintScreen (CompScreen		  *s,
		  const ScreenPaintAttrib *sAttrib,
		  Region		  region,
		  unsigned int		  mask)
{
    Bool status;

    SCALE_SCREEN (s);

    if (ss->grabIndex)
	mask |= PAINT_SCREEN_WITH_TRANSFORMED_WINDOWS_MASK;

    UNWRAP (ss, s, paintScreen);
    status = (*s->paintScreen) (s, sAttrib, region, mask);
    WRAP (ss, s, paintScreen, scalePaintScreen);

    return status;
}

static void
scalePreparePaintScreen (CompScreen *s,
			 int	     msSinceLastPaint)
{
    SCALE_SCREEN (s);

    if (ss->grabIndex ) // && ss->state != SCALE_STATE_WAIT)
    {
	CompWindow *w;
	int        steps, dx, dy;
	float      amount, chunk;

	amount = msSinceLastPaint * 0.05f * ss->speed;
	steps  = amount / (0.5f * ss->timestep);
	if (!steps) steps = 1;
	chunk  = amount / (float) steps;

	while (steps--)
	{
	    ss->moreAdjust = 0;

	    for (w = s->windows; w; w = w->next)
	    {
            SCALE_WINDOW (w);

		if (sw->adjust)
		{
		    sw->adjust = adjustScaleVelocity (w);

		    ss->moreAdjust |= sw->adjust;

		    sw->tx += sw->xVelocity * chunk;
		    sw->ty += sw->yVelocity * chunk;
		    sw->scale += sw->scaleVelocity * chunk;

		    dx = (w->serverX + sw->tx) - w->attrib.x;
		    dy = (w->serverY + sw->ty) - w->attrib.y;

		    moveWindow (w, dx, dy, FALSE, FALSE);

		    (*s->setWindowScale) (w, sw->scale, sw->scale);
		}

		if (!sw->adjust)
		{
			if (ss->selectedWindow != None)
			{
				if(ss->selectedWindow == w->id)
				{
					if(ss->grabIndex > 0 && ss->state == SCALE_STATE_NONE)
					{
						removeScreenGrab (s, ss->grabIndex, 0);
						ss->grabIndex = 0;
					}
					sendWindowActivationRequest (w->screen, w->id);
					addWindowDamage (w);
					activateWindow (w);
					ss->selectedWindow=0;
					damageScreen (s);
				}
			}
		}

	    }

	    if (!ss->moreAdjust)
		break;
    	}
    }

    UNWRAP (ss, s, preparePaintScreen);
    (*s->preparePaintScreen) (s, msSinceLastPaint);
    WRAP (ss, s, preparePaintScreen, scalePreparePaintScreen);
}

static void
scaleDonePaintScreen (CompScreen *s)
{
    SCALE_SCREEN (s);

    if (ss->grabIndex)
    {
	if (ss->moreAdjust)
	{
	    damageScreen (s);
	}
	else
	{
	    if (ss->state == SCALE_STATE_IN)
	    {
		removeScreenGrab (s, ss->grabIndex, 0);
		ss->grabIndex = 0;
	    }
	    else if (ss->state == SCALE_STATE_OUT)
		ss->state = SCALE_STATE_WAIT;
	}
    }

    UNWRAP (ss, s, donePaintScreen);
    (*s->donePaintScreen) (s);
    WRAP (ss, s, donePaintScreen, scaleDonePaintScreen);
}

static CompWindow *
scaleCheckForWindowAt (CompScreen *s,
		       int        x,
		       int        y)
{
    int        x1, y1, x2, y2;
    CompWindow *w;

    for (w = s->reverseWindows; w; w = w->prev)
    {
	SCALE_WINDOW (w);

	if (sw->slot)
	{
	    x1 = w->attrib.x - w->input.left * sw->scale;
	    y1 = w->attrib.y - w->input.top  * sw->scale;
	    x2 = w->attrib.x + (w->width  + w->input.right)  * sw->scale;
	    y2 = w->attrib.y + (w->height + w->input.bottom) * sw->scale;

	    if (x1 <= x && y1 <= y && x2 > x && y2 > y)
		return w;
	}
    }

    return 0;
}

static Bool
scaleTerminate (CompDisplay     *d,
		CompAction      *action,
		CompActionState state,
		CompOption      *option,
		int	        nOption)
{
    CompScreen *s;
    Window     xid;
  
    SCALE_DISPLAY (d);

    xid = getIntOptionNamed (option, nOption, "root", 0);

    for (s = d->screens; s; s = s->next)
    {
	SCALE_SCREEN (s);
    ss->currentWindow=0;
    ss->allScreensMode=FALSE;
    ss->onlyCurrent=FALSE;

	if (xid && s->root != xid)
	    continue;

	//clean up any sw->rescaled values that may still be true
	//causes odd behavior on next scale/zoom otherwise
    	resetZoomValues (s);

	if (ss->grabIndex)
	{
	    if (ss->state == SCALE_STATE_NONE)
	    {
		removeScreenGrab (s, ss->grabIndex, 0);
		ss->grabIndex = 0;
	    }
	    else
	    {
		CompWindow *w;

		for (w = s->windows; w; w = w->next)
		{
		    SCALE_WINDOW (w);

		    if (sw->slot)
		    {
			sw->slot = 0;
			sw->adjust = TRUE;
		    }
		}

		ss->state = SCALE_STATE_IN;

		damageScreen (s);
	    }

	    sd->lastActiveNum = None;
	}
    }

    return FALSE;
}
#define SCALE_KIND_NORMAL 0
#define SCALE_KIND_APP 1
#define SCALE_KIND_ALL 2
#define SCALE_KIND_CURRENT_HEAD 3

static Bool
scaleInitiateReal (CompDisplay     *d,
	       CompAction      *action,
	       CompActionState state,
	       CompOption      *option,
	       int	       nOption,
           int         scaleKind)
{
    CompScreen *s;
    CompWindow *w;
    Window     xid;
    Window     xid2;

    xid = getIntOptionNamed (option, nOption, "root", 0);
    s = findScreenAtDisplay (d, xid);
    if (s)
    {
    w = findTopLevelWindowAtScreen(s,s->display->activeWindow);
    xid2=w?w->id:0;

	SCALE_DISPLAY (s->display);
	SCALE_SCREEN (s);
 
	if (ss->state != SCALE_STATE_WAIT &&
	    ss->state != SCALE_STATE_OUT)
	{
        ss->onlyCurrent=(scaleKind==SCALE_KIND_APP);
        if (ss->onlyCurrent)
        {
            ss->currentWindow=findWindowAtDisplay(d,xid2);
        }
        ss->allScreensMode=(scaleKind==SCALE_KIND_ALL);
	ss->currentHead=(scaleKind==SCALE_KIND_CURRENT_HEAD);
	    if (!layoutThumbs (s))
		return FALSE;
 
	    if (!ss->grabIndex)
	    {
		if (otherScreenGrabExist (s, "scale", 0))
		    return FALSE;
 
		ss->grabIndex = pushScreenGrab (s, ss->cursor, "scale");
	    }
 
	    if (ss->grabIndex)
	    {
		if (!sd->lastActiveNum)
		    sd->lastActiveNum = s->activeNum - 1;
 
		ss->state = SCALE_STATE_OUT;
 
		damageScreen (s);
	    }
	}
	else
	{
	    scaleTerminate (d, action, state, option, nOption);
	}
    }
 
    return FALSE;
}
static Bool
scaleInitiate (CompDisplay     *d,
	       CompAction      *action,
	       CompActionState state,
	       CompOption      *option,
	       int	       nOption)
{
    return scaleInitiateReal(d,action,state,option,nOption,SCALE_KIND_NORMAL);
}

static Bool
scaleInitiateApp (CompDisplay     *d,
	       CompAction      *action,
	       CompActionState state,
	       CompOption      *option,
	       int	       nOption)
{
    return scaleInitiateReal(d,action,state,option,nOption,SCALE_KIND_APP);
}

static Bool
scaleInitiateAll (CompDisplay     *d,
	       CompAction      *action,
	       CompActionState state,
	       CompOption      *option,
	       int	       nOption)
{
    return scaleInitiateReal(d,action,state,option,nOption,SCALE_KIND_ALL);
}

static Bool
scaleInitiateCurrentHead (CompDisplay     *d,
	       CompAction      *action,
	       CompActionState state,
	       CompOption      *option,
	       int	       nOption)
{
    return scaleInitiateReal(d,action,state,option,nOption,SCALE_KIND_CURRENT_HEAD);
}


static void
scaleSelectWindow (CompWindow *w)
{
    activateWindowLight (w);
}

static Bool
scaleHoverWindowAt (CompScreen *s,
		     int	 x,
		     int	 y)
	
{
    CompWindow *w;

    w = scaleCheckForWindowAt (s, x, y);
    if (w && isScaleWin (w))
    {
	scaleSelectWindow (w);
	return TRUE;
    }

    return FALSE;
}

static Bool
scaleSelectWindowAt (CompScreen *s,
		     int	 x,
		     int	 y)
	
{
    CompWindow *w;
    SCALE_SCREEN(s);

    w = scaleCheckForWindowAt (s, x, y);
    if (w && isScaleWin (w))
    {
	raiseWindow (w);
    scaleSelectWindow (w);
	ss->selectedWindow = w->id;
	return TRUE;
    }

    return FALSE;
}

static void
scaleMoveFocusWindow (CompScreen *s,
		      int	 dx,
		      int	 dy)

{
    CompWindow *active;

    active = findWindowAtScreen (s, s->display->activeWindow);
    if (active)
    {
	CompWindow *w, *focus = NULL;
	ScaleSlot  *slot;
	int	   x, y, cx, cy, d, min = MAXSHORT;

	SCALE_SCREEN (s);
	SCALE_WINDOW (active);

	if (!sw->slot)
	    return;

	cx = (sw->slot->x1 + sw->slot->x2) / 2;
	cy = (sw->slot->y1 + sw->slot->y2) / 2;

	for (w = s->windows; w; w = w->next)
	{
        if (w->id == active->id)
            continue;
	    slot = GET_SCALE_WINDOW (w, ss)->slot;
	    if (!slot)
		continue;

	    x = (slot->x1 + slot->x2) / 2;
	    y = (slot->y1 + slot->y2) / 2;

	    d = sqrt((x - cx)*(x-cx) + (y - cy)*(y-cy));
	    if (d < min)
	    {
 		if ((dx > 0 && slot->x1 < sw->slot->x2) ||
 		    (dx < 0 && slot->x2 > sw->slot->x1) ||
 		    (dy > 0 && slot->y1 < sw->slot->y2) ||
 		    (dy < 0 && slot->y2 > sw->slot->y1))
/*
		if ((dx > 0 && x < cx) ||
		    (dx < 0 && x > cx) ||
		    (dy > 0 && y < cy) ||
		    (dy < 0 && y > cy))*/
		    continue;

		min   = d;
		focus = w;
	    }
	}

	if (focus)
	{
	    SCALE_DISPLAY (s->display);

	    sd->lastActiveNum = focus->activeNum;

        scaleSelectWindow (focus);
	}
    }
}

static void
scaleWindowRemove (CompDisplay *d,
		   Window      id)
{
    CompWindow *w;

    w = findWindowAtDisplay (d, id);
    if (w)
    {
	SCALE_SCREEN (w->screen);

	if (ss->grabIndex && ss->state != SCALE_STATE_IN)
	{
	    int i;

	    for (i = 0; i < ss->nWindows; i++)
	    {
		if (ss->windows[i] == w)
		{
		    if (layoutThumbs (w->screen))
		    {
			ss->state = SCALE_STATE_OUT;
			damageScreen (w->screen);
			break;
		    }
		}
	    }
	}
    }
}

static void
scaleHandleEvent (CompDisplay *d,
		  XEvent      *event)
{
    CompScreen *s;

    SCALE_DISPLAY (d);

    switch (event->type) {
    case KeyPress:
	s = findScreenAtDisplay (d, event->xkey.root);
	if (s)
	{
	    SCALE_SCREEN (s);

	    if (ss->grabIndex)
	    {
		if (event->xkey.keycode == sd->leftKeyCode)
		    scaleMoveFocusWindow (s, -1, 0);
		else if (event->xkey.keycode == sd->rightKeyCode)
		    scaleMoveFocusWindow (s, 1, 0);
		else if (event->xkey.keycode == sd->upKeyCode)
		    scaleMoveFocusWindow (s, 0, -1);
		else if (event->xkey.keycode == sd->downKeyCode)
		    scaleMoveFocusWindow (s, 0, 1);
	    }
	}
	break;
    case ButtonPress:
	if (event->xbutton.button == Button2)
	{
		s = findScreenAtDisplay (d, event->xbutton.root);

		if (s)
		{		    
		    SCALE_SCREEN (s);
		    if (ss->grabIndex && ss->state != SCALE_STATE_IN)
		    {
		
			    if( scaleCheckForWindowAt (s,
						    event->xbutton.x_root,
						    event->xbutton.y_root))
			    {
			        CompWindow *w;
				w = scaleCheckForWindowAt (s,
						event->xbutton.x_root,
						event->xbutton.y_root);
			    	closeWindow (w, event->xkey.time);
			    }
		    }
		}
	}
        if (event->xbutton.button == Button3)
	{
		s = findScreenAtDisplay (d, event->xbutton.root);

		if (s)
		{		    
		    SCALE_SCREEN (s);
		    if (ss->grabIndex && ss->state != SCALE_STATE_IN)
		    {
		
			    if( scaleCheckForWindowAt (s,
						    event->xbutton.x_root,
						    event->xbutton.y_root))
			    {
			        CompWindow *w;
				w = scaleCheckForWindowAt (s,
						event->xbutton.x_root,
						event->xbutton.y_root);
			    	zoomWindow (s, w);
			    }
		    }
		}
	}
	if (event->xbutton.button == Button1)
	{
	    s = findScreenAtDisplay (d, event->xbutton.root);
	    if (s)
	    {
		CompAction *action =
		    &sd->opt[SCALE_DISPLAY_OPTION_INITIATE_NORMAL].value.action;

		SCALE_SCREEN (s);
		

		if (ss->grabIndex && ss->state != SCALE_STATE_IN)
		{
		    CompOption o;

		    o.type    = CompOptionTypeInt;
		    o.name    = "root";
		    o.value.i = s->root;

		    if (scaleSelectWindowAt (s,
					     event->xbutton.x_root,
					     event->xbutton.y_root))
		    {
			scaleTerminate (d, action, 0, &o, 1);
		    }
		    /*else if (event->xbutton.x_root > s->workArea.x &&
			     event->xbutton.x_root < (s->workArea.x +
						      s->workArea.width) &&
			     event->xbutton.y_root > s->workArea.y &&
			     event->xbutton.y_root < (s->workArea.y +
						      s->workArea.height))
		    {
			scaleTerminate (d, action, 0, &o, 1);
		    }*/
            //disabled exit-scale-by-invalid-click-into-show-desktop
		}
	    }
      }
	break;
    case MotionNotify:
	s = findScreenAtDisplay (d, event->xmotion.root);
	if (s)
	{
	    SCALE_SCREEN (s);

	    if (ss->grabIndex		    &&
		ss->state != SCALE_STATE_IN &&
		ss->opt[SCALE_SCREEN_OPTION_SLOPPY_FOCUS].value.b)
		scaleHoverWindowAt (s,
				     event->xmotion.x_root,
				     event->xmotion.y_root);
	}
	break;
    default:
	break;
    }

    UNWRAP (sd, d, handleEvent);
    (*d->handleEvent) (d, event);
    WRAP (sd, d, handleEvent, scaleHandleEvent);

    switch (event->type) {
    case UnmapNotify:
	scaleWindowRemove (d, event->xunmap.window);
	break;
    case DestroyNotify:
	scaleWindowRemove (d, event->xdestroywindow.window);
	break;
    }

}

static Bool
scaleDamageWindowRect (CompWindow *w,
		       Bool	  initial,
		       BoxPtr     rect)
{
    Bool status;

    SCALE_SCREEN (w->screen);

    if (initial)
    {
	if (isScaleWin (w))
	{
	    if (ss->grabIndex && layoutThumbs (w->screen))
	    {
		ss->state = SCALE_STATE_OUT;
		damageScreen (w->screen);
	    }
	}
    }

    UNWRAP (ss, w->screen, damageWindowRect);
    status = (*w->screen->damageWindowRect) (w, initial, rect);
    WRAP (ss, w->screen, damageWindowRect, scaleDamageWindowRect);

    return status;
}

static CompOption *
scaleGetDisplayOptions (CompDisplay *display,
			int	    *count)
{
    SCALE_DISPLAY (display);

    *count = NUM_OPTIONS (sd);
    return sd->opt;
}

static Bool
scaleSetDisplayOption (CompDisplay     *display,
		       char	       *name,
		       CompOptionValue *value)
{
    CompOption *o;
    int	       index;

    SCALE_DISPLAY (display);

    o = compFindOption (sd->opt, NUM_OPTIONS (sd), name, &index);

    if (!o)
	return FALSE;

    switch (index) {
    case SCALE_DISPLAY_OPTION_INITIATE_NORMAL:
    case SCALE_DISPLAY_OPTION_INITIATE_APP:
    case SCALE_DISPLAY_OPTION_INITIATE_CURRENT_HEAD:
    case SCALE_DISPLAY_OPTION_INITIATE_ALL:
	if (setDisplayAction (display, o, value))
	    return TRUE;
    default:
	break;
    }

    return FALSE;
}

static void
scaleDisplayInitOptions (ScaleDisplay *sd,
			 Display      *display)
{
    CompOption *o;

    o = &sd->opt[SCALE_DISPLAY_OPTION_INITIATE_NORMAL];
    o->name			  = "initiate";
    o->shortDesc		  = N_("Initiate Window Picker");
    o->longDesc			  = N_("Layout and start transforming windows");
    o->type			  = CompOptionTypeAction;
    o->value.action.initiate	  = scaleInitiate;
    o->value.action.terminate	  = scaleTerminate;
    o->value.action.bell	  = FALSE;
    o->value.action.edgeMask	  = (1 << SCREEN_EDGE_TOPRIGHT);
    o->value.action.state	  = CompActionStateInitEdge;
    o->value.action.type	  = CompBindingTypeKey;
    o->value.action.state	 |= CompActionStateInitKey;
    o->value.action.key.modifiers = SCALE_INITIATE_MODIFIERS_DEFAULT;
    o->value.action.key.keycode   =
	XKeysymToKeycode (display,
			  XStringToKeysym (SCALE_INITIATE_KEY_DEFAULT));

    o = &sd->opt[SCALE_DISPLAY_OPTION_INITIATE_APP];
    o->name			  = "initiate_app";
    o->shortDesc		  = "Initiate Window Picker for Current App";
    o->longDesc			  = "Layout and start transforming windows of current application";
    o->type			  = CompOptionTypeAction;
    o->value.action.initiate	  = scaleInitiateApp;
    o->value.action.terminate	  = scaleTerminate;
    o->value.action.bell	  = FALSE;
    o->value.action.edgeMask	  = (1 << SCREEN_EDGE_TOPLEFT);
    o->value.action.state	  = CompActionStateInitEdge;
    o->value.action.type	  = CompBindingTypeKey;
    o->value.action.state	  |= CompActionStateInitKey;
    o->value.action.key.modifiers = SCALE_INITIATE_APP_MODIFIERS_DEFAULT;
    o->value.action.key.keycode   =
	XKeysymToKeycode (display,
			  XStringToKeysym (SCALE_INITIATE_APP_KEY_DEFAULT));

    o = &sd->opt[SCALE_DISPLAY_OPTION_INITIATE_ALL];
    o->name			  = "initiate_all";
    o->shortDesc		  = "Initiate Window Picker for All Workspaces";
    o->longDesc			  = "Layout and start transforming windows from all workspaces";
    o->type			  = CompOptionTypeAction;
    o->value.action.initiate	  = scaleInitiateAll;
    o->value.action.terminate	  = scaleTerminate;
    o->value.action.bell	  = FALSE;
    o->value.action.edgeMask	  = (1 << SCREEN_EDGE_BOTTOMLEFT);
    o->value.action.state	  = CompActionStateInitEdge;
    o->value.action.type	  = CompBindingTypeKey;
    o->value.action.state	  |= CompActionStateInitKey;
    o->value.action.key.modifiers = SCALE_INITIATE_ALL_MODIFIERS_DEFAULT;
    o->value.action.key.keycode   =
	XKeysymToKeycode (display,
			  XStringToKeysym (SCALE_INITIATE_ALL_KEY_DEFAULT));

    o = &sd->opt[SCALE_DISPLAY_OPTION_INITIATE_CURRENT_HEAD];
    o->name			  = "initiate_current_head";
    o->shortDesc		  = "Initiate current head scale only";
    o->longDesc			  = "Only scale windows on the current monitor";
    o->type			  = CompOptionTypeAction;
    o->value.action.initiate	  = scaleInitiateCurrentHead;
    o->value.action.terminate	  = scaleTerminate;
    o->value.action.bell	  = FALSE;
    o->value.action.edgeMask	  = 0;
    o->value.action.state	  = CompActionStateInitEdge;
    o->value.action.type	  = CompBindingTypeKey;
    o->value.action.state	  |= CompActionStateInitKey;
    o->value.action.key.modifiers = SCALE_INITIATE_CURRENT_HEAD_MODIFIERS_DEFAULT;
    o->value.action.key.keycode   =
	XKeysymToKeycode (display,
			  XStringToKeysym (SCALE_INITIATE_CURRENT_HEAD_KEY_DEFAULT));
}

static Bool
scaleInitDisplay (CompPlugin  *p,
		  CompDisplay *d)
{
    ScaleDisplay *sd;

    sd = malloc (sizeof (ScaleDisplay));
    if (!sd)
	return FALSE;

    sd->screenPrivateIndex = allocateScreenPrivateIndex (d);
    if (sd->screenPrivateIndex < 0)
    {
	free (sd);
	return FALSE;
    }

    sd->lastActiveNum = None;

    scaleDisplayInitOptions (sd, d->display);

    sd->leftKeyCode  = XKeysymToKeycode (d->display, XStringToKeysym ("Left"));
    sd->rightKeyCode = XKeysymToKeycode (d->display, XStringToKeysym ("Right"));
    sd->upKeyCode    = XKeysymToKeycode (d->display, XStringToKeysym ("Up"));
    sd->downKeyCode  = XKeysymToKeycode (d->display, XStringToKeysym ("Down"));

    WRAP (sd, d, handleEvent, scaleHandleEvent);

    d->privates[displayPrivateIndex].ptr = sd;

    return TRUE;
}

static void
scaleFiniDisplay (CompPlugin  *p,
		  CompDisplay *d)
{
    SCALE_DISPLAY (d);

    freeScreenPrivateIndex (d, sd->screenPrivateIndex);

    UNWRAP (sd, d, handleEvent);

    free (sd);
}

static Bool
scaleInitScreen (CompPlugin *p,
		 CompScreen *s)
{
    ScaleScreen *ss;

    SCALE_DISPLAY (s->display);

    ss = malloc (sizeof (ScaleScreen));
    if (!ss)
	return FALSE;

    ss->windowPrivateIndex = allocateWindowPrivateIndex (s);
    if (ss->windowPrivateIndex < 0)
    {
	free (ss);
	return FALSE;
    }

    ss->grabIndex = 0;

    ss->state = SCALE_STATE_NONE;
    ss->useClass = SCALE_USE_CLASS_DEFAULT;

    ss->slots = 0;
    ss->slotsSize = 0;

    ss->windows = 0;
    ss->windowsSize = 0;

    ss->line = 0;
    ss->lineSize = 0;

    ss->scale = 1.0f;

    ss->spacing  = SCALE_SPACING_DEFAULT;
    ss->speed    = SCALE_SPEED_DEFAULT;
    ss->timestep = SCALE_TIMESTEP_DEFAULT;
    ss->opacity  = (OPAQUE * SCALE_OPACITY_DEFAULT) / 100;
    ss->darkenBackFactor = SCALE_DARKEN_BACK_FACTOR_DEFAULT;

    ss->onlyCurrent = FALSE;
    ss->currentWindow = 0;
    ss->selectedWindow = None;
    ss->allScreensMode = FALSE;
    ss->organic = FALSE;
    
    ss->darkenBack = SCALE_DARKEN_BACK_DEFAULT;
    ss->currentHead = FALSE;

    ss->head = 0;

    scaleSetHead ( ss, s, SCALE_HEAD_DEFAULT - 1 );

    ss->iconOverlay = ScaleIconEmblem;

    scaleScreenInitOptions (ss);

    addScreenAction (s, &sd->opt[SCALE_DISPLAY_OPTION_INITIATE_NORMAL].value.action);
    addScreenAction (s, &sd->opt[SCALE_DISPLAY_OPTION_INITIATE_APP].value.action);
    addScreenAction (s, &sd->opt[SCALE_DISPLAY_OPTION_INITIATE_ALL].value.action);
    addScreenAction (s, &sd->opt[SCALE_DISPLAY_OPTION_INITIATE_CURRENT_HEAD].value.action);

    WRAP (ss, s, preparePaintScreen, scalePreparePaintScreen);
    WRAP (ss, s, donePaintScreen, scaleDonePaintScreen);
    WRAP (ss, s, paintScreen, scalePaintScreen);
    WRAP (ss, s, paintWindow, scalePaintWindow);
    WRAP (ss, s, damageWindowRect, scaleDamageWindowRect);

    ss->cursor = XCreateFontCursor (s->display->display, XC_left_ptr);

    s->privates[sd->screenPrivateIndex].ptr = ss;

    return TRUE;
}

static void
scaleFiniScreen (CompPlugin *p,
		 CompScreen *s)
{
    SCALE_SCREEN (s);

    UNWRAP (ss, s, preparePaintScreen);
    UNWRAP (ss, s, donePaintScreen);
    UNWRAP (ss, s, paintScreen);
    UNWRAP (ss, s, paintWindow);
    UNWRAP (ss, s, damageWindowRect);

    if (ss->slotsSize)
	free (ss->slots);

    if (ss->lineSize)
	free (ss->line);

    if (ss->windowsSize)
	free (ss->windows);

    free (ss);
}

static Bool
scaleInitWindow (CompPlugin *p,
		 CompWindow *w)
{
    ScaleWindow *sw;

    SCALE_SCREEN (w->screen);

    sw = malloc (sizeof (ScaleWindow));
    if (!sw)
	return FALSE;

    sw->slot = 0;
    sw->scale = 1.0f;
    sw->tx = sw->ty = 0.0f;
    sw->adjust = FALSE;
    sw->xVelocity = sw->yVelocity = 0.0f;
    sw->scaleVelocity = 1.0f;
    sw->rescaled = FALSE;
    sw->oldScale = -1.0f;

    w->privates[ss->windowPrivateIndex].ptr = sw;

    return TRUE;
}

static void
scaleFiniWindow (CompPlugin *p,
		 CompWindow *w)
{
    SCALE_WINDOW (w);

    free (sw);
}

static Bool
scaleInit (CompPlugin *p)
{
    displayPrivateIndex = allocateDisplayPrivateIndex ();
    if (displayPrivateIndex < 0)
	return FALSE;

    return TRUE;
}

static void
scaleFini (CompPlugin *p)
{
    if (displayPrivateIndex >= 0)
	freeDisplayPrivateIndex (displayPrivateIndex);
}

CompPluginVTable scaleVTable = {
    "scale",
    N_("Scale"),
    N_("Scale windows"),
    scaleInit,
    scaleFini,
    scaleInitDisplay,
    scaleFiniDisplay,
    scaleInitScreen,
    scaleFiniScreen,
    scaleInitWindow,
    scaleFiniWindow,
    scaleGetDisplayOptions,
    scaleSetDisplayOption,
    scaleGetScreenOptions,
    scaleSetScreenOption,
    0, /* Deps */
    0  /* nDeps */
};

CompPluginVTable *
getCompPluginInfo (void)
{
    return &scaleVTable;
}
