#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include <limits.h>

#include <compiz.h>

#include <glib.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>

#define STATE_USE_PUT_DEFAULT FALSE



static int displayPrivateIndex;

typedef struct _StateDisplay {
    int		        screenPrivateIndex;
    HandleEventProc handleEvent;
    Atom            wmPidAtom;
    Atom            wmVisibleNameAtom;
    Atom            wmNameAtom;
    Atom            utf8String;
} StateDisplay;

typedef enum _StateOption
{
  STATE_SCREEN_OPTION_OPACITY,
  STATE_SCREEN_OPTION_SATURATION,
  STATE_SCREEN_OPTION_BRIGHTNESS,
  STATE_SCREEN_OPTION_VIEWPORT,
  STATE_SCREEN_OPTION_WIDGET,
  STATE_SCREEN_OPTION_POSITION,
  STATE_SCREEN_OPTION_USE_PUT,
  STATE_SCREEN_OPTION_NUM
} StateOption;

#define STATE_SCREEN_OPTION_OPACITY_MAX 100
#define STATE_SCREEN_OPTION_OPACITY_MIN 0

#define STATE_SCREEN_OPTION_BRIGHTNESS_MAX 100
#define STATE_SCREEN_OPTION_BRIGHTNESS_MIN 0

#define STATE_SCREEN_OPTION_SATURATION_MAX 100
#define STATE_SCREEN_OPTION_SATURATION_MIN 0

typedef enum _StateType {
    NIL, WINDOW, TITLE, PROGRAM, CLASS,
} StateType;

typedef struct _StateIntValue {
    StateType type;
    char *    name;
    int       value;
} StateIntValue;

typedef struct _StateIntValues {
    StateIntValue ** values;
    int            count;
} StateIntValues;

typedef struct _StateScreen {
    CompOption opt[STATE_SCREEN_OPTION_NUM];
    DamageWindowRectProc damageWindowRect;

    StateIntValues * opacities;
    StateIntValues * brightnesses;
    StateIntValues * saturations;
    StateIntValues * viewports;
    StateIntValues * widgets;
    StateIntValues * positions;
} StateScreen;

#define GET_STATE_DISPLAY(d)				     \
    ((StateDisplay *) (d)->privates[displayPrivateIndex].ptr)

#define STATE_DISPLAY(d)			   \
    StateDisplay *fd = GET_STATE_DISPLAY (d)

#define GET_STATE_SCREEN(s, fd)					 \
    ((StateScreen *) (s)->privates[(fd)->screenPrivateIndex].ptr)

#define STATE_SCREEN(s)							\
    StateScreen *fs = GET_STATE_SCREEN (s, GET_STATE_DISPLAY (s->display))

#define NUM_OPTIONS(s) (sizeof ((s)->opt) / sizeof (CompOption))

static int
stateClipValue(int value, int min, int max)
{
    if (value < min)
      value=min;
    if (value > max)
        value=max;
    return value;
}

static StateIntValue * 
stateParseIntValue(char * s)
{
    StateIntValue * value = NULL;
    char * sep1, * sep2;
    if (s && *s)
    {
        value = (StateIntValue*) malloc (sizeof (StateIntValue));
        value->name = NULL;
        if (s[0] == 'w')
            value->type = WINDOW;
        else if (s[0] == 't')
            value->type = TITLE;
        else if (s[0] == 'p')
            value->type = PROGRAM;
        else if (s[0] == 'c')
          value->type = CLASS;
        else
            goto err;

        sep1 = strchr (s, ':');
        if (!sep1) 
            goto err;

        sep1 ++;
        sep2 = strchr (sep1, ':');
        if (!sep2) 
            goto err;

        value->name = (char*) malloc (sizeof (char) * (sep2 - sep1 + 1));
        strncpy (value->name, sep1, sep2 - sep1);
        value->name[sep2 - sep1] = 0;

        value->value = atoi (sep2 + 1);

        return value;
    }
err:
    if (value->name)
        free (value->name);
    free (value);
    return NULL;
}

static StateIntValues * 
stateLoadIntValuesFromStringList(CompOptionValue *value,
                                   int min, int max)
{
    StateIntValue * ivalue;
    StateIntValues * values;
    int i, count = 0;

    if (value->list.nValue > 0)
    {
        values = (StateIntValues *)malloc(sizeof(StateIntValues));
        values->values = (StateIntValue **)malloc(
            sizeof(StateIntValue*) * value->list.nValue);
        for (i = 0; i < value->list.nValue; i++)
        {
            ivalue = stateParseIntValue(value->list.value[i].s);
            if (ivalue)
            {
                ivalue->value = stateClipValue(ivalue->value,min,max);
                values->values[count++] = ivalue;
            }
        }
        if (count == 0)
        {
            free(values->values);
            free(values);
            return NULL;
        } else
        {
            values->count = count;
            return values;
        }
    }
    return NULL;
}

static void
stateFreeIntValues(StateIntValues * values)
{
    int i;

    if (values != NULL)
    {
        for (i = 0; i < values->count; ++i)
        {
            free(values->values[i]->name);
            free(values->values[i]);
        }
        free(values);
    }
}

static CompOption *
stateGetScreenOptions (CompScreen *screen,
                       int	 *count)
{
    STATE_SCREEN (screen);

    *count = NUM_OPTIONS (fs);
    return fs->opt;
}

static void
stateScreenInitOptions (StateScreen *fs)
{
    CompOption *o;

    o = &fs->opt[STATE_SCREEN_OPTION_OPACITY];
    o->name	         = "opacity";
    o->shortDesc         = "Window Opacity";
    o->longDesc	         = "Set window opacity by conditions";
    o->type	         = CompOptionTypeList;
    o->value.list.type   = CompOptionTypeString;
    o->value.list.nValue = 0;
    o->value.list.value  = 0; /* NULL; */
    o->rest.s.string     = 0; /* windowTypeString; */
    o->rest.s.nString    = 0; /* nWindowTypeString; */
    
    o = &fs->opt[STATE_SCREEN_OPTION_BRIGHTNESS];
    o->name              = "brightness";
    o->shortDesc         = "Window Brightness";
    o->longDesc          = "Set window brightness by conditions";
    o->type              = CompOptionTypeList;
    o->value.list.type   = CompOptionTypeString;
    o->value.list.nValue = 0;
    o->value.list.value  = 0; /* NULL; */
    o->rest.s.string     = 0; /* windowTypeString; */
    o->rest.s.nString    = 0; /* nWindowTypeString; */
    
    o = &fs->opt[STATE_SCREEN_OPTION_SATURATION];
    o->name              = "saturation";
    o->shortDesc         = "Window Saturation";
    o->longDesc          = "Set window saturation by conditions";
    o->type              = CompOptionTypeList;
    o->value.list.type   = CompOptionTypeString;
    o->value.list.nValue = 0;
    o->value.list.value  = 0; /* NULL; */
    o->rest.s.string     = 0; /* windowTypeString; */
    o->rest.s.nString    = 0; /* nWindowTypeString; */
    
    o = &fs->opt[STATE_SCREEN_OPTION_VIEWPORT];
    o->name              = "viewport";
    o->shortDesc         = "Initial Viewport";
    o->longDesc          = "Set initial viewport for windows by conditions";
    o->type              = CompOptionTypeList;
    o->value.list.type   = CompOptionTypeString;
    o->value.list.nValue = 0;
    o->value.list.value  = 0; /* NULL; */
    o->rest.s.string     = 0; /* windowTypeString; */
    o->rest.s.nString    = 0; /* nWindowTypeString; */

    o = &fs->opt[STATE_SCREEN_OPTION_WIDGET];
    o->name              = "widget";
    o->shortDesc         = "Widget Hack";
    o->longDesc          = "Set to 2 to set widget state, 1 to clear.";
    o->type              = CompOptionTypeList;
    o->value.list.type   = CompOptionTypeString;
    o->value.list.nValue = 0;
    o->value.list.value  = 0; /* NULL; */
    o->rest.s.string     = 0; /* windowTypeString; */
    o->rest.s.nString    = 0; /* nWindowTypeString; */

    o = &fs->opt[STATE_SCREEN_OPTION_POSITION];
    o->name              = "position";
    o->shortDesc         = "Position";
    o->longDesc          = "Initial position. 1-9 arranged like the numeric keypad";
    o->type              = CompOptionTypeList;
    o->value.list.type   = CompOptionTypeString;
    o->value.list.nValue = 0;
    o->value.list.value  = 0; /* NULL; */
    o->rest.s.string     = 0; /* windowTypeString; */
    o->rest.s.nString    = 0; /* nWindowTypeString; */

    o = &fs->opt[STATE_SCREEN_OPTION_USE_PUT];
    o->name              = "use_put";
    o->shortDesc         = "Use Put";
    o->longDesc          = "Use put to move windows to viewports";
    o->type              = CompOptionTypeBool;
    o->value.b           = STATE_USE_PUT_DEFAULT;
}

static void
stateChangeWindowOpacity (CompWindow *w,
                          int	opacity)
{
    opacity = OPAQUE * opacity / 100;

        w->paint.opacity = opacity;
        
        setWindowProp32 (w->screen->display, w->id,
                                w->screen->display->winOpacityAtom,
                                w->paint.opacity);
        addWindowDamage (w);
}

static void
    stateChangeWindowBrightness (CompWindow *w,
                              int   brightness)
{
  brightness = BRIGHT * brightness / 100;

        w->paint.brightness = brightness;
        
        addWindowDamage (w);
}

static void
    stateChangeWindowSaturation (CompWindow *w,
                                 int   saturation)
{
  saturation = COLOR * saturation / 100;

        w->paint.saturation = saturation;
        
        addWindowDamage (w);
}

static unsigned int
stateWindowTypeMaskFromString (char * value)
{
    unsigned int mask = 0;

    if (strcasecmp (value, "desktop") == 0)
        mask |= CompWindowTypeDesktopMask;
    else if (strcasecmp (value, "dock") == 0)
        mask |= CompWindowTypeDockMask;
    else if (strcasecmp (value, "toolbar") == 0)
        mask |= CompWindowTypeToolbarMask;
    else if (strcasecmp (value, "menu") == 0)
        mask |= CompWindowTypeMenuMask;
    else if (strcasecmp (value, "utility") == 0)
        mask |= CompWindowTypeUtilMask;
    else if (strcasecmp (value, "splash") == 0)
        mask |= CompWindowTypeSplashMask;
    else if (strcasecmp (value, "dialog") == 0)
        mask |= CompWindowTypeDialogMask;
    else if (strcasecmp (value, "modaldialog") == 0)
        mask |= CompWindowTypeModalDialogMask;
    else if (strcasecmp (value, "normal") == 0)
        mask |= CompWindowTypeNormalMask;
    else if (strcasecmp (value, "fullscreen") == 0)
        mask |= CompWindowTypeFullscreenMask;
    else if (strcasecmp (value, "unknown") == 0)
        mask |= CompWindowTypeUnknownMask;
    else if (strcasecmp (value, "dropdownmenu") == 0)
        mask |= CompWindowTypeDropdownMenuMask;
    else if (strcasecmp (value, "popupmenu") == 0)
        mask |= CompWindowTypePopupMenuMask;
    else if (strcasecmp (value, "tooltip") == 0)
        mask |= CompWindowTypeTooltipMask;
    else if (strcasecmp (value, "notification") == 0)
        mask |= CompWindowTypeNotificationMask;
    else if (strcasecmp (value, "combo") == 0)
        mask |= CompWindowTypeComboMask;
    else if (strcasecmp (value, "dnd") == 0)
        mask |= CompWindowTypeDndMask;

    return mask;
}

static char *
stateGetProgramPathFromPid(int pid)
{
    int    len;
    char * path;
    if (pid >= 0) 
    {
        path = (char *)malloc(sizeof(char) * 256);
        sprintf(path, "/proc/%d/exe", pid);
        if (access(path, F_OK) == 0 &&
            (len=readlink(path, path, 256)) > 0)
        {
            path[len] = 0;
            return path;
        }
        free(path);
    }
    return NULL;
}

static Bool
stateGetXPropertyCardinal (CompWindow *win,
                           Atom       atom,
                           int        *val)
{
    Atom type;
    int format;
    unsigned long nitems;
    unsigned long bytes_after;
    unsigned long *num;
    int result;
    
    *val = 0;
    
    type = None;
    result = XGetWindowProperty (win->screen->display->display,
                                 win->id,
                                 atom,
                                 0, LONG_MAX,
                                 False, XA_CARDINAL, &type, &format, &nitems,
                                 &bytes_after, (void*)&num);  
    if (result != Success)
      return FALSE;

    if (type != XA_CARDINAL)
    {
      XFree (num);
      return FALSE;
    }

    *val = *num;
    XFree (num);
    return TRUE;
}

static char *
stateGetXPropertyUtf8 (CompWindow *win,
                       Atom       atom)
{
    Atom type;
    int format;
    unsigned long nitems;
    unsigned long bytes_after;
    unsigned long *val;
    int result;
    char *retval;
    Atom utf8_string;

    STATE_DISPLAY(win->screen->display);

    utf8_string = fd->utf8String;

    type = None;
    val = NULL;
    result = XGetWindowProperty (win->screen->display->display,
                  			     win->id,
                  			     atom,
                  			     0, LONG_MAX,
                  			     False, utf8_string,
                  			     &type, &format, &nitems,
                  			     &bytes_after, (unsigned char **)&val);  
  
    if (result != Success)
      return NULL;
    
    if (type != utf8_string ||
        format != 8 ||
        nitems == 0)
      {
        if (val)
          XFree (val);
        return NULL;
      }


    retval = (char *)malloc(sizeof(char) * (nitems+1));
    strncpy(retval, (char *)val, nitems);
    retval[nitems] = '\0';

    XFree (val);
    
    return retval;
}

static char *
stateGetWindowTitle(CompWindow *w)
{
    char *title;

    STATE_DISPLAY(w->screen->display);

    title = stateGetXPropertyUtf8(w, fd->wmVisibleNameAtom);

    if (title == NULL)
        title = stateGetXPropertyUtf8(w, fd->wmNameAtom);
    return title;
}

static int
stateGetWindowPid(CompWindow *w)
{
    int val;

    STATE_DISPLAY(w->screen->display);

    if (!stateGetXPropertyCardinal (w, fd->wmPidAtom, &val))
        return -1;
    else
        return val;
}

static char * 
strchrr(char * chars, char ch)
{
    int pos = -1;
    char * tmp = chars;

    if (chars == NULL)
        return NULL;
    while(*tmp)
    {
        if (*tmp == ch)
            pos = tmp - chars; 
        tmp ++;
    }
    if (pos != -1)
        return chars + pos + 1;
    return NULL;
}

static signed int
stateGetParamForWindow(CompWindow * w, StateIntValues * params)
{
  signed int ret = -1;
  StateType    changed;
  StateType    type;
  char *       name;
  char *       title;
  char *       progr;
  char *       sname;
  int          dret;
  int          i;
  unsigned int mask;
  
  
  if (params)
  {
    changed = NIL;
    title = stateGetWindowTitle(w);
    progr = stateGetProgramPathFromPid(
        stateGetWindowPid(w));

    sname = strchrr(progr, '/');
    for (i = 0; i < params->count; ++i)
    {
      type = params->values[i]->type;
      name = params->values[i]->name;
      dret = params->values[i]->value-1;
      if (type == WINDOW)
      {
        mask = stateWindowTypeMaskFromString(name);
        if ((w->type & mask) && changed <= WINDOW)
        {
          ret=dret;
          changed = WINDOW;
        }
      } else if (type == TITLE)
      {
        if (title && strcasecmp(title, name) == 0 && changed <= TITLE)
        {
          ret=dret;
          changed = TITLE;
        }
                
      } else if (type == PROGRAM)
      {
        if (progr && (strcasecmp(progr, name) == 0 || 
            strcasecmp(sname, name) == 0))
        {
          ret=dret;
          changed = PROGRAM;
        }
      } else if (type == CLASS)
      {
        if (w->resClass && (strcmp(w->resClass,name)==0 && changed<=CLASS))
        {
          ret=dret;
          changed=CLASS;
        }
      }
    }
    if (title) free(title);
    if (progr) free(progr);
  }
  return ret;
}

static void 
    stateAdjustWindowPaintParams(CompWindow * w)
{
  STATE_SCREEN(w->screen);
  signed int val;
  val=stateGetParamForWindow(w,fs->opacities);
  if (val!=-1)
    stateChangeWindowOpacity(w,val);
  val=stateGetParamForWindow(w,fs->saturations);
  if (val!=-1)
    stateChangeWindowSaturation(w,val);
  val=stateGetParamForWindow(w,fs->brightnesses);
  if (val!=-1)
    stateChangeWindowBrightness(w,val);
}

static void
    stateAdjustAllWindowsPaintParams(CompScreen *screen)
{
  CompWindow * w = screen->windows;
  while(w)
  {
    stateAdjustWindowPaintParams(w);
    w=w->next;
  }
}

static void
    updateWidgetStatusForWindow(CompWindow *w)
{
  Atom a = XInternAtom(w->screen->display->display,"_COMPIZ_WIDGET",
                       FALSE);
  STATE_SCREEN(w->screen);
  signed int val = stateGetParamForWindow(w,fs->widgets);
  if (val!=-1)
  {
    if (val==0)
    {
      XDeleteProperty(w->screen->display->display,w->id,a);
    }
    else
    {
        if (w->inShowDesktopMode || w->mapNum || 
                w->attrib.map_state == IsViewable || w->minimized)
        {
            if (w->minimized || w->inShowDesktopMode)
                unminimizeWindow(w);
      XChangeProperty(w->screen->display->display,w->id,a,
                      XA_STRING,8,PropModeReplace,(unsigned char*)&val,1);
        }
    }
  }
}
static void
    updateWidgetStatus(CompScreen *screen)
{
  CompWindow * w = screen->windows;
  while(w)
  {
    updateWidgetStatusForWindow(w);
    w=w->next;
  }
}

static Bool
    stateSetScreenOption (CompScreen      *screen,
                          char            *name,
                          CompOptionValue *value)
{
  CompOption *o;
  int        index;

  STATE_SCREEN (screen);

  o = compFindOption (fs->opt, NUM_OPTIONS (fs), name, &index);
  if (!o)
    return FALSE;

  switch (index) {
    case STATE_SCREEN_OPTION_OPACITY:
      if (compSetOptionList (o, value))
      {
        stateFreeIntValues(fs->opacities);
        fs->opacities = 
            stateLoadIntValuesFromStringList 
            (&o->value, STATE_SCREEN_OPTION_OPACITY_MIN,
            STATE_SCREEN_OPTION_OPACITY_MAX);
        stateAdjustAllWindowsPaintParams(screen);

        return TRUE;
      }
      break;
    case STATE_SCREEN_OPTION_BRIGHTNESS:
      if (compSetOptionList (o, value))
      {
        stateFreeIntValues(fs->brightnesses);
        fs->brightnesses = 
            stateLoadIntValuesFromStringList 
            (&o->value, STATE_SCREEN_OPTION_BRIGHTNESS_MIN,
              STATE_SCREEN_OPTION_BRIGHTNESS_MAX);
        stateAdjustAllWindowsPaintParams(screen);

        return TRUE;
      }
      break;
    case STATE_SCREEN_OPTION_SATURATION:
      if (compSetOptionList (o, value))
      {
        stateFreeIntValues(fs->saturations);
        fs->saturations = 
            stateLoadIntValuesFromStringList 
            (&o->value, STATE_SCREEN_OPTION_SATURATION_MIN,
              STATE_SCREEN_OPTION_SATURATION_MAX);
        stateAdjustAllWindowsPaintParams(screen);

        return TRUE;
      }
      break;
    case STATE_SCREEN_OPTION_VIEWPORT:
      if (compSetOptionList (o, value))
      {
        stateFreeIntValues(fs->viewports);
        fs->viewports = 
            stateLoadIntValuesFromStringList 
            (&o->value, 1, screen->size);
        return TRUE;
      }
      break;
    case STATE_SCREEN_OPTION_WIDGET:
      if (compSetOptionList (o, value))
      {
        stateFreeIntValues(fs->widgets);
        fs->widgets = 
            stateLoadIntValuesFromStringList 
            (&o->value, 1, 2);
        updateWidgetStatus(screen);
        return TRUE;
      }
      break;
    case STATE_SCREEN_OPTION_POSITION:
      if (compSetOptionList (o, value))
      {
        stateFreeIntValues (fs->positions);
        fs->positions = stateLoadIntValuesFromStringList (&o->value, 1, 9);

        return TRUE;
      }
      break;
    default:
      break;
  }

  return FALSE;
}

static Bool
stateDamageWindowRect (CompWindow *w, Bool initial, BoxPtr rect)
{
    Bool status;
    Bool wasPlaced;

    STATE_SCREEN (w->screen);
  
    wasPlaced = w->placed;

    UNWRAP (fs, w->screen, damageWindowRect);
    status = (*w->screen->damageWindowRect) (w, initial, rect);
    WRAP (fs, w->screen, damageWindowRect, stateDamageWindowRect);

    if (initial && !w->attrib.override_redirect && w->placed && !wasPlaced)
    {
    
        signed int v = stateGetParamForWindow (w, fs->viewports);
        if (v != -1)
        {
            //check to see first if the window is already
            //on the right viewport.
            if (defaultViewportForWindow (w) != v)            
            {

                if (fs->opt[STATE_SCREEN_OPTION_USE_PUT].value.b)
                {

                    // send an event to put and tell it to move this window to a viewport

                    XClientMessageEvent xev;

                    xev.type         = ClientMessage;
                    xev.window       = w->id;
                    xev.message_type = XInternAtom (w->screen->display->display, "_COMPIZ_PUT_WINDOW", 0);
                    xev.format       = 32;
                    xev.data.l[0]    = 0; 
                    xev.data.l[1]    = 0; 
                    xev.data.l[2]    = v + 1;  
                    xev.data.l[3]    = 11;

                    XSendEvent (w->screen->display->display, w->screen->root, 0, StructureNotifyMask, (XEvent *) &xev);            
                }
                else
                {
                    moveWindow (w, (v - w->screen->x) * w->screen->width, 0, TRUE, TRUE);
                    syncWindowPosition (w);
                }
            }
        }

        int p = stateGetParamForWindow (w, fs->positions);
        if (p != -1)
        {
            // send window to numbered position
            XClientMessageEvent xev;

            xev.type         = ClientMessage;
            xev.window       = w->id;
            xev.message_type = XInternAtom (w->screen->display->display, "_COMPIZ_PUT_WINDOW", 0);
            xev.format       = 32;
            xev.data.l[0]    = 0; 
            xev.data.l[1]    = 0; 
            xev.data.l[2]    = 0;  
            xev.data.l[3]    = p + 1;

            XSendEvent (w->screen->display->display, w->screen->root, 0, StructureNotifyMask, (XEvent *) &xev);                
        }
    }
    return status;
}

static void
stateHandleEvent (CompDisplay *d,
                  XEvent      *event)
{
    CompWindow *w;

    STATE_DISPLAY (d);
    w = NULL;
    switch (event->type) {
    /*case CreateNotify:
        printf("CreateNotify.1\n");
        w = findWindowAtDisplay (d, event->xcreatewindow.window);
        if (w) 
        {
        printf("CreateNotify.2\n");
            stateAdjustWindowOpacity(w);
        }
        printf("CreateNotify.3\n");
        break;
    case DestroyNotify:
        break;*/
    case MapNotify:
        w = findWindowAtDisplay (d, event->xmap.window);
        if (w && (!w->placed)){
            stateAdjustWindowPaintParams(w);
            updateWidgetStatusForWindow(w);}
        break;
    default:
        break;
    }

    UNWRAP (fd, d, handleEvent);
    (*d->handleEvent) (d, event);
    WRAP (fd, d, handleEvent, stateHandleEvent);
}

static Bool
stateInitDisplay (CompPlugin  *p,
		 CompDisplay *d)
{
    StateDisplay *fd;

    fd = malloc (sizeof (StateDisplay));
    if (!fd)
	return FALSE;

    fd->screenPrivateIndex = allocateScreenPrivateIndex (d);
    if (fd->screenPrivateIndex < 0)
    {
	free (fd);
	return FALSE;
    }

    fd->utf8String = XInternAtom(d->display, "UTF8_STRING", 0);
    fd->wmPidAtom  = XInternAtom(d->display, "_NET_WM_PID", 0);
    fd->wmNameAtom = XInternAtom(d->display, "_NET_WM_NAME", 0);
    fd->wmVisibleNameAtom = XInternAtom(d->display, "_NET_WM_VISIBLE_NAME", 0);

    WRAP (fd, d, handleEvent, stateHandleEvent);

    d->privates[displayPrivateIndex].ptr = fd;

    return TRUE;
}

static void
stateFiniDisplay (CompPlugin *p,
		 CompDisplay *d)
{
    STATE_DISPLAY (d);

    freeScreenPrivateIndex (d, fd->screenPrivateIndex);

    UNWRAP (fd, d, handleEvent);

    free (fd);
}

static Bool
stateInitScreen (CompPlugin *p,
		CompScreen *s)
{
    StateScreen *fs;

    STATE_DISPLAY (s->display);

    fs = malloc (sizeof (StateScreen));
    if (!fs)
	return FALSE;

    stateScreenInitOptions (fs);

    fs->opacities = NULL;
    fs->brightnesses = NULL;
    fs->saturations = NULL;
    fs->viewports = NULL;
    fs->widgets = NULL;
    fs->positions = NULL;

    s->privates[fd->screenPrivateIndex].ptr = fs;
    
    WRAP (fs, s, damageWindowRect, stateDamageWindowRect);

    return TRUE;
}

static void
stateFiniScreen (CompPlugin *p,
		CompScreen *s)
{
    STATE_SCREEN (s);

    stateFreeIntValues(fs->opacities);
    fs->opacities=NULL;
    stateFreeIntValues(fs->brightnesses);
    fs->brightnesses=NULL;
    stateFreeIntValues(fs->saturations);
    fs->saturations=NULL;
    stateFreeIntValues(fs->viewports);
    fs->viewports=NULL;
    stateFreeIntValues(fs->widgets);
    fs->widgets = NULL;
    stateFreeIntValues(fs->positions);
    fs->positions = NULL;

    UNWRAP (fs, s, damageWindowRect);

    free (fs);
}

static Bool
stateInit (CompPlugin *p)
{
    displayPrivateIndex = allocateDisplayPrivateIndex ();
    if (displayPrivateIndex < 0)
	return FALSE;

    return TRUE;
}

static void
stateFini (CompPlugin *p)
{
    if (displayPrivateIndex >= 0)
	freeDisplayPrivateIndex (displayPrivateIndex);
}

static Bool 
stateInitWindow (CompPlugin *plugin,
			     CompWindow *window)
{
    /*if (window)
        stateAdjustWindowOpacity(window);
      */  
    return True;
}

static void 
stateFiniWindow (CompPlugin *plugin,
				 CompWindow *window)
{

}

CompPluginDep stateDeps[] = {
    /*{ CompPluginRuleAfter, "decoration" },
    { CompPluginRuleAfter, "fade" },
    { CompPluginRuleAfter, "switcher" }*/
    { CompPluginRuleAfter, "place" },
    /*{ CompPluginRuleBefore, "fade" },
    { CompPluginRuleBefore, "cube" },*/
    { CompPluginRuleBefore, "widget" }
};

static CompPluginVTable stateVTable = {
    "state",
    "Set Window Attribs by various criteria",
    "w = window type, c = window class, t = window title, p = owning program",
    stateInit,
    stateFini,
    stateInitDisplay,
    stateFiniDisplay,
    stateInitScreen,
    stateFiniScreen,
    stateInitWindow,
    stateFiniWindow,
    0, /* GetDisplayOptions */
    0, /* SetDisplayOption */
    stateGetScreenOptions,
    stateSetScreenOption,
    stateDeps,
    sizeof (stateDeps) / sizeof (stateDeps[0])
};

CompPluginVTable *
getCompPluginInfo (void)
{
    return &stateVTable;
}

