/**
 * 
 * trailfocus.c:
 * author: casey langen, 2006
 * version: 0.1.2
 * purpose: leave a "trail" of focused windows
 * 
 * ---------------------------------------------
 * 
 * based on...
 *   
 * trailFocus.c v0.03
 *
 * Author: François Ingelrest
 * E-Mail: Athropos@gmail.com
 *
**/

#include <compiz.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>


#undef __DEBUG


/* option identifiers ---------------------------------------------------- */


#define TRAILFOCUS_SCREEN_OPTION_MINIMUM_OPACITY_LEVEL     0
#define TRAILFOCUS_SCREEN_OPTION_MAXIMUM_OPACITY_LEVEL     1 
#define TRAILFOCUS_SCREEN_OPTION_MINIMUM_SATURATION_LEVEL  2
#define TRAILFOCUS_SCREEN_OPTION_MAXIMUM_SATURATION_LEVEL  3
#define TRAILFOCUS_SCREEN_OPTION_MINIMUM_BRIGHTNESS_LEVEL  4
#define TRAILFOCUS_SCREEN_OPTION_MAXIMUM_BRIGHTNESS_LEVEL  5
#define TRAILFOCUS_SCREEN_OPTION_MAXIMUM_TRAIL_COUNT       6
#define TRAILFOCUS_SCREEN_OPTION_EXCLUDE_LIST              7
#define TRAILFOCUS_SCREEN_OPTION_NUM                       8 


/* macros --------------------------------------------------------------- */


#define GET_TRAILFOCUS_DISPLAY(display)    \
    ((TrailFocusDisplay*)display->privates[mDisplayPrivateIndex].ptr)

#define GET_TRAILFOCUS_SCREEN(screen)      \
    ((TrailFocusScreen*)screen->privates[GET_TRAILFOCUS_DISPLAY(screen->display)->screenPrivateIndex].ptr)

#define GET_TRAILFOCUS_WINDOW(w, tfs)				         \
    ((TrailFocusWindow *) (w)->privates[(tfs)->windowPrivateIndex].ptr)

#define NUM_OPTIONS(screen) (sizeof((screen)->opt) / sizeof(CompOption))


/* structs -------------------------------------------------------------- */


typedef struct _TrailFocusDisplay
{
    int screenPrivateIndex;
    HandleEventProc handleEvent;
} TrailFocusDisplay;

typedef struct _TrailFocusScreen
{
    int windowPrivateIndex;
    int wmask;                                      // Which kind of windows are we looking at?
    
    int minSaturationLevel;
    int minBrightnessLevel;
    int minOpacityLevel;
    int maxSaturationLevel;
    int maxBrightnessLevel;
    int maxOpacityLevel;
    int maxTrailCount;
    
    CompOption opt[TRAILFOCUS_SCREEN_OPTION_NUM];

    PaintWindowProc	paintWindow;
} TrailFocusScreen;

typedef struct _TrailFocusWindow
{
  GLushort saturation;
  GLushort brightness;
  GLushort opacity;
} TrailFocusWindow;


/* locals ---------------------------------------------------------------- */


/*static char *winType[] = {
    "Splash",
    "Normal",
    "Dialog",
    "ModalDialog"
};

define N_WIN_TYPE (sizeof(winType) / sizeof(winType[0]))*/

static int mDisplayPrivateIndex;
static CompWindow** sWindows = 0;
static int sWindowCount = 0;
static int sWindowMax = 0;


/* compiz prototypes ----------------------------------------------------- */


static Bool trailFocusInit(CompPlugin *p);
static void trailFocusFini(CompPlugin *p);
static Bool trailFocusInitDisplay(CompPlugin *p, CompDisplay *d);
static void trailFocusFiniDisplay(CompPlugin *p, CompDisplay *d);
static Bool trailFocusInitScreen(CompPlugin *p, CompScreen *s);
static void trailFocusFiniScreen(CompPlugin *p, CompScreen *s);
static Bool trailFocusInitWindow(CompPlugin *p, CompWindow *w);
static void trailFocusFiniWindow(CompPlugin *p, CompWindow *w);
static Bool trailFocusSetScreenOption(CompScreen *screen, char *name, CompOptionValue *value);
static CompOption* trailFocusGetScreenOptions(CompScreen *screen, int *count);
static void trailFocusHandleEvent(CompDisplay *d, XEvent *event);
static Bool trailFocusPaintWindow(CompWindow *w, const WindowPaintAttrib *attrib, Region region, unsigned int mask);


/* trailfocus prototypes ------------------------------------------------- */


static void reallocWindowList(int count);
static void freeWindowList();
static int windowListContains(CompWindow* targetWindow);
static void recalcWindowEffects();
static void removeWindowFromList(CompWindow* remove, Bool recalcEffects);
static void rotateWindowFocus(CompWindow* newFocus);

#ifdef __DEBUG
char debugStr[512];
static void dumpWindowList();
#endif


/* compiz definitions  --------------------------------------------------- */


/* compiz requests this table in the entry point */
static CompPluginVTable trailFocusVTable = {
    "trailfocus",
    "Window focus leaves a trail",
    "Window focus leaves a trail",
    trailFocusInit,
    trailFocusFini,
    trailFocusInitDisplay,
    trailFocusFiniDisplay,
    trailFocusInitScreen,
    trailFocusFiniScreen,
    trailFocusInitWindow, 
    trailFocusFiniWindow,
    0, /* GetDisplayOptions */
    0, /* SetDisplayOption */
    trailFocusGetScreenOptions,
    trailFocusSetScreenOption,
    NULL,
    0
};


/* entry point */
CompPluginVTable* getCompPluginInfo(void)
{
    return &trailFocusVTable;
}

/* plugin init */
static Bool trailFocusInit(CompPlugin *p)
{
    reallocWindowList(0);
    mDisplayPrivateIndex = allocateDisplayPrivateIndex();
    if(mDisplayPrivateIndex < 0)
    {
        return FALSE;
    }
    
    return TRUE;
}

/* plugin exit */
static void trailFocusFini(CompPlugin *p)
{
    if(mDisplayPrivateIndex >= 0)
    {
        freeDisplayPrivateIndex(mDisplayPrivateIndex);
    }

    freeWindowList();
}

/* display init */
static Bool trailFocusInitDisplay(CompPlugin *p, CompDisplay *d)
{
    TrailFocusDisplay *tfd;

    tfd = malloc(sizeof(TrailFocusDisplay));
    if(!tfd)
    {
        return FALSE;
    }
    
    tfd->screenPrivateIndex = allocateScreenPrivateIndex(d);
    if(tfd->screenPrivateIndex < 0)
    {
        free (tfd);
        return FALSE;
    }

    WRAP(tfd, d, handleEvent, trailFocusHandleEvent);

    d->privates[mDisplayPrivateIndex].ptr = tfd;

    return TRUE;
}

/* display exit */
static void trailFocusFiniDisplay(CompPlugin *p, CompDisplay *d)
{
    TrailFocusDisplay *tfd = GET_TRAILFOCUS_DISPLAY(d);

    freeScreenPrivateIndex(d, tfd->screenPrivateIndex);

    UNWRAP(tfd, d, handleEvent);

    free (tfd);
}

/* screen init */
static Bool trailFocusInitScreen(CompPlugin *p, CompScreen *s)
{
    CompOption *o;
    TrailFocusScreen *tfs;
    TrailFocusDisplay *tfd;

    tfd = GET_TRAILFOCUS_DISPLAY(s->display);

    tfs = malloc(sizeof(TrailFocusScreen));
    if(!tfs)
    {
      return FALSE;
    }

    tfs->windowPrivateIndex = allocateWindowPrivateIndex(s);
    if(tfs->windowPrivateIndex < 0)
    {
      free(tfs);
      return FALSE;
    }
    
    /* Options for this plugin */
    o = &tfs->opt[TRAILFOCUS_SCREEN_OPTION_MINIMUM_OPACITY_LEVEL];
    o->name		    = "minimum_window_opacity_level";
    o->shortDesc	= "Opacity level of completely unfocused windows";
    o->longDesc		= "Opacity level of completely unfocused windows";
    o->type		    = CompOptionTypeInt;
    o->value.i		= 90;
    o->rest.i.min	= 25;
    o->rest.i.max	= 100;
    tfs->minOpacityLevel = 90;
    
    o = &tfs->opt[TRAILFOCUS_SCREEN_OPTION_MINIMUM_SATURATION_LEVEL];
    o->name		    = "minimum_window_saturation_level";
    o->shortDesc	= "Saturation level of completely unfocused windows";
    o->longDesc		= "Saturation level of completely unfocused windows";
    o->type		    = CompOptionTypeInt;
    o->value.i		= 100;
    o->rest.i.min	= 0;
    o->rest.i.max	= 100;
    tfs->minSaturationLevel = 100;

    o = &tfs->opt[TRAILFOCUS_SCREEN_OPTION_MINIMUM_BRIGHTNESS_LEVEL];
    o->name		    = "minimum_window_brightness_level";
    o->shortDesc	= "Brightness level of unfocused windows";
    o->longDesc		= "Brightness level of unfocused windows";
    o->type		    = CompOptionTypeInt;
    o->value.i		= 90;
    o->rest.i.min	= 0;
    o->rest.i.max	= 100;
    tfs->minBrightnessLevel = 90;
    
    o = &tfs->opt[TRAILFOCUS_SCREEN_OPTION_MAXIMUM_OPACITY_LEVEL];
    o->name		    = "maximum_window_opacity_level";
    o->shortDesc	= "Opacity level of completely focused windows";
    o->longDesc		= "Opacity level of completely focused windows";
    o->type		    = CompOptionTypeInt;
    o->value.i		= 100;
    o->rest.i.min	= 25;
    o->rest.i.max	= 100;
    tfs->maxOpacityLevel = 100;
    
    o = &tfs->opt[TRAILFOCUS_SCREEN_OPTION_MAXIMUM_SATURATION_LEVEL];
    o->name		    = "maximum_window_saturation_level";
    o->shortDesc	= "Saturation level of completely focused windows";
    o->longDesc		= "Saturation level of completely focused windows";
    o->type		    = CompOptionTypeInt;
    o->value.i		= 100;
    o->rest.i.min	= 0;
    o->rest.i.max	= 100;
    tfs->maxSaturationLevel = 100;

    o = &tfs->opt[TRAILFOCUS_SCREEN_OPTION_MAXIMUM_BRIGHTNESS_LEVEL];
    o->name		    = "maximum_window_brightness_level";
    o->shortDesc	= "Brightness level of focused windows";
    o->longDesc		= "Brightness level of focused windows";
    o->type		    = CompOptionTypeInt;
    o->value.i		= 100;
    o->rest.i.min	= 0;
    o->rest.i.max	= 100;
    tfs->maxBrightnessLevel = 100;
        
    o = &tfs->opt[TRAILFOCUS_SCREEN_OPTION_MAXIMUM_TRAIL_COUNT];
    o->name	    	= "maximum_trail_count";
    o->shortDesc	= "Number of recently used windows to keep track of";
    o->longDesc		= "Number of recently used windows to keep track of";
    o->type		    = CompOptionTypeInt;
    o->value.i		= 6;
    o->rest.i.min	= 1;
    o->rest.i.max	= 64;
    tfs->maxTrailCount = 6;
    
    /* Window types cannot be changed with options, bit we're using CompOption to create the mask */
    /* Perhaps is there an easier way? */
    /*wtypes.value.list.nValue = N_WIN_TYPE;
    wtypes.value.list.value  = malloc(sizeof(CompOptionValue) * N_WIN_TYPE);
    for(i = 0; i < N_WIN_TYPE; i++)
    {
	    wtypes.value.list.value[i].s = strdup(winType[i]);
    }
    tfs->wmask = compWindowTypeMaskFromStringList(&wtypes.value);
    free(wtypes.value.list.value);*/
    tfs->wmask = CompWindowTypeNormalMask | CompWindowTypeSplashMask |
        CompWindowTypeDialogMask | CompWindowTypeModalDialogMask | 
        CompWindowTypeUtilMask;
    
    o = &tfs->opt[TRAILFOCUS_SCREEN_OPTION_EXCLUDE_LIST];
    o->name              = "exclude";
    o->shortDesc         = "WM_CLASS to exclude";
    o->longDesc          = "WM_Classes to exclude from trailfocusing";
    o->type              = CompOptionTypeList;
    o->value.list.type   = CompOptionTypeString;
    o->value.list.nValue = 0;
    o->value.list.value  = 0;
    o->rest.s.string     = 0;
    o->rest.s.nString    = 0;

/*static char *winType[] = {
        "Splash",
        "Normal",
        "Dialog",
        "ModalDialog"
};*/

    s->privates[tfd->screenPrivateIndex].ptr = tfs;

//    WRAP (tfs, s, paintWindow, trailFocusPaintWindow);
    
    return TRUE;
}

/* screen exit */
static void trailFocusFiniScreen(CompPlugin *p, CompScreen *s)
{
    TrailFocusScreen *tfs = GET_TRAILFOCUS_SCREEN(s);

    UNWRAP(tfs, s, paintWindow);
    freeWindowPrivateIndex(s, tfs->windowPrivateIndex);
    
    free (tfs);
}

/* a screen option has changed */
static Bool trailFocusSetScreenOption(CompScreen *screen, char *name, CompOptionValue *value)
{
    int index;
    CompOption *o;
    TrailFocusScreen *tfs;

    tfs = GET_TRAILFOCUS_SCREEN(screen);
    o  = compFindOption(tfs->opt, NUM_OPTIONS(tfs), name, &index);
    
    if(!o)
    {
      return FALSE;
    }
    
    /* todo: optimize with a dictionary (hashmap) */
    switch(index)
    {
        case TRAILFOCUS_SCREEN_OPTION_MINIMUM_OPACITY_LEVEL:
            if(compSetIntOption(o, value))
            {
                o->value.i = value->i;
                tfs->minOpacityLevel = o->value.i;
                return TRUE;
            }
            break;
        
        case TRAILFOCUS_SCREEN_OPTION_MAXIMUM_OPACITY_LEVEL:
            if(compSetIntOption(o, value))
            {
                o->value.i = value->i;
                tfs->maxOpacityLevel = o->value.i;
                return TRUE;
            }
            break;
        
        case TRAILFOCUS_SCREEN_OPTION_MINIMUM_SATURATION_LEVEL:
            {
                o->value.i = value->i;
                tfs->minSaturationLevel = value->i;
                return TRUE;
            }
            break;
        
        case TRAILFOCUS_SCREEN_OPTION_MAXIMUM_SATURATION_LEVEL:
            {
                o->value.i = value->i;
                tfs->maxSaturationLevel = value->i;
                return TRUE;
            }
            break;

        case TRAILFOCUS_SCREEN_OPTION_MINIMUM_BRIGHTNESS_LEVEL:
            {
                o->value.i = value->i;
                tfs->minBrightnessLevel = value->i;
                return TRUE;
            }
        
        case TRAILFOCUS_SCREEN_OPTION_MAXIMUM_BRIGHTNESS_LEVEL:
            {
                o->value.i = value->i;
                tfs->maxBrightnessLevel = value->i;
                return TRUE;
            }

        case TRAILFOCUS_SCREEN_OPTION_MAXIMUM_TRAIL_COUNT:
            {
                o->value.i = value->i;
              
                if(sWindowMax != value->i)
                {
                  reallocWindowList(value->i);
                }
                return TRUE;
            }
      case TRAILFOCUS_SCREEN_OPTION_EXCLUDE_LIST:
        if(compSetOptionList(o, value))
          return TRUE;
        break;
            
        default:
            break;
    }

    return FALSE;
}

/* return our options */
static CompOption* trailFocusGetScreenOptions(CompScreen *screen, int *count)
{
    TrailFocusScreen *tfs = GET_TRAILFOCUS_SCREEN(screen);
    
    *count = NUM_OPTIONS(tfs);
    return tfs->opt;
}

/* handle focus (and other related) events */
static void trailFocusHandleEvent(CompDisplay *d, XEvent *event)
{
    CompWindow *w;
    TrailFocusDisplay *tfd;
    TrailFocusScreen *tfs;

    tfd = GET_TRAILFOCUS_DISPLAY(d);

    switch(event->type)
    {
        case FocusIn:
        case FocusOut:

            /* Debugging 
             
                 if(event->type == FocusIn) printf("FocusIn (%X)\n", event->xfocus.window);
            else                            printf("FocusOut (%X)\n", event->xfocus.window);

                 if(event->xfocus.mode == NotifyNormal)       printf("NotifyNormal\n\n");
            else if(event->xfocus.mode == NotifyWhileGrabbed) printf("NotifyWhileGrabbed\n\n");
            else if(event->xfocus.mode == NotifyGrab)         printf("NotifyGrab\n\n");
            else if(event->xfocus.mode == NotifyUngrab)       printf("NotifyUngrab\n\n");
            else                                              printf("Unknown\n\n");
            */

            /* NotifyWhileGrabbed is generated when using F12 function (expose) */
            if(event->xfocus.mode == NotifyNormal || event->xfocus.mode == NotifyWhileGrabbed)
            {
                w = findWindowAtDisplay(d, event->xfocus.window);
                /* Don't touch to hidden windows */
                if(w && !(w->state & CompWindowStateHiddenMask))
                {
                    tfs = GET_TRAILFOCUS_SCREEN(w->screen);
                    if(tfs->wmask & w->type)
                    {
                      if(event->type == FocusIn)
                      {
                        rotateWindowFocus(w);
                      }
                    }
                }
            }
            break;

        case DestroyNotify: 
        case UnmapNotify:     /* iconified/hidden */    
            {
              w = findWindowAtDisplay(d, event->xdestroywindow.window);
              int exists = windowListContains(w);
              if(exists >= 0)
              {
#ifdef __DEBUG
                system("echo >> ~/DEBUG '(INFO) destroyed window removed from list'");
#endif
                removeWindowFromList(w, True);
              }
            }
            break;
  
        default:
            break;
    }

    UNWRAP(tfd, d, handleEvent);
    (*d->handleEvent)(d, event);
    WRAP(tfd, d, handleEvent, trailFocusHandleEvent);
}

/* window begin */
static Bool trailFocusInitWindow (CompPlugin *p, CompWindow *w)
{
  TrailFocusScreen* tfs = GET_TRAILFOCUS_SCREEN(w->screen);
  
  TrailFocusWindow *tfw = malloc(sizeof(TrailFocusWindow));
  if(!tfw)
  {
    return FALSE;
  }

  tfw->saturation = 0;
  tfw->brightness = 0;
  tfw->opacity = 0;

  w->privates[tfs->windowPrivateIndex].ptr = tfw;

  return TRUE;
}

/* window end */
static void trailFocusFiniWindow (CompPlugin *p, CompWindow *w)
{
  TrailFocusScreen* tfs = GET_TRAILFOCUS_SCREEN(w->screen);
  TrailFocusWindow* tfw = GET_TRAILFOCUS_WINDOW(w, tfs);
 
  removeWindowFromList(w, True);
  
  free (tfw);
}

static Bool trailFocusPaintWindow (CompWindow *w, const WindowPaintAttrib *attrib, Region region, unsigned int mask)
{
  Bool status;
  CompScreen* s = w->screen;
  TrailFocusScreen* tfs = GET_TRAILFOCUS_SCREEN (w->screen);

  /*
  TrailFocusWindow* tfw = GET_TRAILFOCUS_WINDOW (w, tfs);

  int exists = windowListContains(w);
  if(exists == -1)
  {
    if(attrib->saturation != 50)
    {
      WindowPaintAttrib newAttrib = *attrib;
      newAttrib.saturation = 50;
    
      UNWRAP (tfs, s, paintWindow);
	    status = (*s->paintWindow) (w, &newAttrib, region, mask);
	    WRAP (tfs, s, paintWindow, trailFocusPaintWindow);

      addWindowDamage(w);
    }
  }
  else
  {
    UNWRAP (tfs, s, paintWindow);
	  status = (*s->paintWindow) (w, attrib, region, mask);
	  WRAP (tfs, s, paintWindow, trailFocusPaintWindow);
  }
  return status;
  */

  UNWRAP (tfs, s, paintWindow);
  status = (*s->paintWindow) (w, attrib, region, mask);
  WRAP (tfs, s, paintWindow, trailFocusPaintWindow);

  return status;
}


/* trailfocus definitions ------------------------------------------------ */


#ifdef __DEBUG
static void dumpWindowList()
{
  sprintf(debugStr, "echo >> ~/DEBUG '\tcount: %d", sWindowCount);
  system(debugStr); 
  int i; 
  for(i = 0; i < sWindowMax; i++)
  {
    sprintf(
        debugStr, 
        "echo >> ~/DEBUG '\t%d) %x'",
        i, (unsigned int)sWindows[i]
    );
    system(debugStr);
  }
}
#endif

static void reallocWindowList(int count)
{
#ifdef __DEBUG
  sprintf(debugStr, "echo >> ~/DEBUG 'reallocWindowList: %d'", count);
  system(debugStr);
#endif
  
  CompWindow** oldList = sWindows;
  
  sWindows = malloc(sizeof(CompWindow*) * count);
  
  int i;
  if(oldList != 0)
  {
    for(i = 0; i < count; i++)
    {
      if(i < sWindowCount)
      {
        sWindows[i] = oldList[i];
      }
      else
      {
        sWindows[i] = (CompWindow*)0;
      }
    }

    if(sWindowCount > count)
    {
      sWindowCount = count;
    }

    free(oldList);
  }
  else
  {
    for(i = 0; i < count; i++)
    {
      sWindows[i] = (CompWindow*)0;
    }

    sWindowCount = 0;
  }
  
  sWindowMax = count;
  
#ifdef __DEBUG
  dumpWindowList();
#endif
}

static void freeWindowList()
{
  if(sWindows)
  {
    free(sWindows);
    sWindows = 0;
  }
  
  sWindowCount = 0;
  sWindowMax = 0;
}

static int windowListContains(CompWindow* targetWindow)
{
  int i;
  for(i = 0; i < sWindowCount; i++)
  {
    if(sWindows[i] == targetWindow)
    {
      return i;
    }
  }

  return -1;
}

inline static GLushort getRealValue(int min, int max, float multiplier)
{
  GLushort realMin = (GLushort)((float)0xffff * ((float)min/(float)100.0));
  GLushort realMax = (GLushort)((float)0xffff * ((float)max/(float)100.0));
  GLushort offset = (GLushort)((float)abs(realMax - realMin) * multiplier);
  
  return ((GLushort)realMin+offset);
}

static void recalcWindowEffects()
{
  if(sWindowCount <= 0)
  {
    return;
  }
  
#ifdef __DEBUG  
  sprintf(debugStr, "echo >> ~/DEBUG 'start recalcWindowEffects(): %d'", sWindowCount);
  system(debugStr);
#endif

  TrailFocusScreen* tfs;
  //float trailStep = 1.0 / (float)sWindowCount;
  float multiplier;
  int i;
  GLushort sat, bright, opacity;
  
  for(i = 0; i < sWindowCount; i++)
  {
    if(sWindows[i]->state & CompWindowStateHiddenMask)
    {
      continue;
    } 

    /* todo: fix kludge for 1 window */
    if(sWindowCount == 1)
    {
      multiplier = 1.0;
    }
    else
    {
      multiplier = (float)(sWindowCount-i-1) / (float)(sWindowCount-1);    
    }
    tfs = GET_TRAILFOCUS_SCREEN (sWindows[i]->screen);
    sat = getRealValue(tfs->minSaturationLevel, tfs->maxSaturationLevel, multiplier);
    bright = getRealValue(tfs->minBrightnessLevel, tfs->maxBrightnessLevel, multiplier);
    opacity = getRealValue(tfs->minOpacityLevel, tfs->maxOpacityLevel, multiplier);
    
    if(    sat != sWindows[i]->paint.saturation
        || bright != sWindows[i]->paint.brightness
        || opacity != sWindows[i]->paint.opacity
      )
    {
      CompOption * o = &tfs->opt[TRAILFOCUS_SCREEN_OPTION_EXCLUDE_LIST];
      Bool handleThis = TRUE;
      int j;
      for (j=0;j<o->value.list.nValue;j++)
      {
        if (sWindows[i]->resClass)
        {
                if (!strcmp(o->value.list.value[j].s,sWindows[i]->resClass))
                {
                handleThis = FALSE;
                //break;
                }
        }
      }

      if (!sWindows[i]->alive)
      {
        handleThis = FALSE;
      }

      if (handleThis)
      {
        sWindows[i]->paint.saturation = sat;
        sWindows[i]->paint.brightness = bright;
        sWindows[i]->paint.opacity =  getWindowProp32(sWindows[i]->screen->display,
                                                      sWindows[i]->id,
                                                      sWindows[i]->screen->display->winOpacityAtom,
                                                      opacity);
        addWindowDamage(sWindows[i]);
      }
    }
  }
  
#ifdef __DEBUG  
  system("echo >> ~/DEBUG 'end recalcWindowEffects()'");
#endif
}

static void removeWindowFromList(CompWindow* remove, Bool recalcEffects)
{
  if(remove == 0)
  {
#ifdef __DEBUG
    system("echo >> ~/DEBUG '(WARNING) try to remove a null window'");
#endif
    return;
  }

#ifdef __DEBUG
  system("echo >> ~/DEBUG 'window list before removal'");
  dumpWindowList();
#endif
  
  int i;
  int location = windowListContains(remove);
  if(location >= 0)
  {
    for(i = location; i < sWindowMax; i++)
    {
      sWindows[i] = sWindows[i+1];
    }

    sWindows[sWindowCount-1] = 0;
    sWindowCount--;

#ifdef __DEBUG
    sprintf(debugStr, "echo >> ~/DEBUG 'removed window: %x'", (unsigned int)remove);
    system(debugStr);

    location = windowListContains(remove);
    if (location >= 0)
    {
      system("echo >> ~/DEBUG '(CRITICAL) window still exists'");
    }
    
    dumpWindowList();
#endif
    
    if(recalcEffects)
    {
      recalcWindowEffects();
    }
  }
}

static void rotateWindowFocus(CompWindow* newFocus)
{
  int i;
  int exists = windowListContains(newFocus);

  if(exists >= 0)
  {
    removeWindowFromList(newFocus, False);
  }
    
  for(i = sWindowMax - 1; i > 0; i--)
  {
    sWindows[i] = sWindows[i-1];
  }
  
  if(sWindowCount < sWindowMax)
  {
    sWindowCount++;
  }
  
  sWindows[0] = newFocus;

#ifdef __DEBUG
  system("echo >> ~/DEBUG 'window rotated'"); 
  dumpWindowList();
#endif
  
  recalcWindowEffects();
}
