#!/usr/bin/env python
#
#   ConVirt   -  Copyright (c) 2008 Jd & Hap Hazard
#   ======
#
# ConVirt is a Xen management tool with a GTK based graphical interface
# that allows for performing the standard set of domain operations
# (start, stop, pause, kill, shutdown, reboot, snapshot, etc...). It
# also attempts to simplify certain aspects such as the creation of
# domains, as well as making the consoles available directly within the
# tool's user interface.
#
#
# This software is subject to the GNU Lesser General Public License (LGPL)
# and for details, please consult it at:
#
#    http://www.fsf.org/licensing/licenses/lgpl.txt
#
# author : Jd <jd_jedi@users.sourceforge.net>
#

# Encapsulates some image store structure and convetions.
# 

import os, shutil
import urllib,urllib2, urlparse

import constants
from constants import *
from XenDomain import XenConfig
from utils import XMConfig, PyConfig, copyToRemote
from utils import read_python_conf, copyToRemote, mkdir2

import traceback

class ImageStore:
    """A class that manages the list of provisioning images.
      NOTE: This class is intended for use by a client application
      managing multiple DomU images.
      """

    STORE_CONF = "image_store.conf"
    VM_TEMPLATE = "vm_conf.template"
    SCRIPT_NAME  = "provision.sh"
    DEFAULT_STORE_LOCATION = "/var/cache/convirt/image_store"
    COMMON_DIR = 'common'
    IMAGE_CONF = 'image.conf'
    IMAGE_DESC = 'description.htm'
    IMAGE_DESC_HTML = 'description.html'

    INVALID_CHARS = "(space,comma,dot,quotes,/,\|,<,>,!,&,%,.)"
    INVALID_CHARS_EXP = "( |/|,|<|>|\||\!|\&|\%|\.|\'|\"|\\\\)"
    
    def __init__(self, config):
        self._config = config
        self.__initialize()

    def __initialize(self):
        self._list = []
        self._default = None # default image for provisioning
        self._excludes = {}  # dirs like common to be excluded
        
        self._store_location = self._config.get(XMConfig.PATHS,
                                               prop_image_store)
        if self._store_location is None or self._store_location is '':
            self._store_location = self.DEFAULT_STORE_LOCATION
        
        self.__read_store_conf()

        for file in os.listdir(self._store_location):
            if file not in self._excludes and file[0] != ".":
                full_file = os.path.join(self._store_location, file)
                if os.path.isdir(full_file):
                    self._list.append(file)

                    
    def __read_store_conf(self):
        conf_file = os.path.join(self._store_location, self.STORE_CONF)
        conf = read_python_conf(conf_file)
        if conf is not None:
            if conf.has_key("default"):
                self._default = conf["default"]
            if conf.has_key("excludes"):    
                self._excludes = conf["excludes"]


    def re_initialize(self):
        self.__initialize()
        
    def clone_image(self, source_name, target_name):
        shutil.copytree(
            os.path.join(self._store_location,source_name),
            os.path.join(self._store_location,target_name)
            )
        self._list.append(target_name)

    def delete_image(self, image_name):
        shutil.rmtree(
            os.path.join(self._store_location,image_name)
            )
        if image_name in self._list:
            self._list.remove(image_name)

    
    def get_default_image(self):
        return self._default

    def list(self):
        return self._list

    def get_store_location(self):
        return self._store_location
    
    def get_image_dir(self, image_name, new=False):
        if new or image_name in self.list():
            return os.path.join(self._store_location,
                                image_name)
        else:
            raise Exception("Invalid image name " + image_name)


    def get_vm_template(self, image_name, new=False):
        if new or image_name in self.list():
            return os.path.join(self._store_location,
                                image_name,
                                self.VM_TEMPLATE)
        else:
            raise Exception("Invalid image name " + image_name)

    def get_image_conf(self, image_name, new=False):
        if new or image_name in self.list():
            return os.path.join(self._store_location,
                                image_name,
                                self.IMAGE_CONF)
        else:
            raise Exception("Invalid image name " + image_name)
        
    def get_provisioning_script(self, image_name, new=False):
        if new or image_name in self.list():
            return os.path.join(self._store_location,
                                image_name,
                                self.SCRIPT_NAME)
        else:
            raise Exception("Invalid image name " + image_name)

    def get_provisioning_script_dir(self, image_name, new=False):
        if new or image_name in self.list():
            return os.path.join(self._store_location,
                                image_name)
        else:
            raise Exception("Invalid image name " + image_name)
        
    def get_image_desc(self, image_name, new=False):
        if new or image_name in self.list():
            return  os.path.join(self._store_location,
                                image_name,
                                self.IMAGE_DESC)
        else:
            raise Exception("Invalid image name " + image_name)
        
    def get_image_desc_html(self, image_name, new=False):
        if new or image_name in self.list():
            return  os.path.join(self._store_location,
                                image_name,
                                self.IMAGE_DESC_HTML)
        else:
            raise Exception("Invalid image name " + image_name)
        

    def get_common_dir(self):
        return os.path.join(self._store_location, self.COMMON_DIR)

    # Prepare environment for executing script on given managed node
    def prepare_env(self, managed_node, image_name, domconfig, image_conf):
        """ prepare execution environment on the remote node"""

        # prepare directory names
        remote_image_store = managed_node.config.get(XMConfig.PATHS,
                                                     prop_image_store)
        # if not specified, assume similar to client node
        if remote_image_store is None:
            remote_image_store = self.DEFAULT_STORE_LOCATION

        scripts_dest = os.path.join(remote_image_store , image_name)
        common_dest  = os.path.join(remote_image_store , self.COMMON_DIR)

        local_image_store = self.get_store_location()
        
        scripts_src_dir = self.get_provisioning_script_dir(image_name)
        common_src  = self.get_common_dir()

        copyToRemote(common_src, managed_node,remote_image_store)
        copyToRemote(scripts_src_dir, managed_node, remote_image_store)
        
        # prepare the log area where the instantiated image.conf and
        # args file would be placed, along with the log dir.

        log_dir = managed_node.config.get(XMConfig.PATHS,
                                          prop_log_dir)
        if log_dir is None or log_dir == '':
            log_dir = DEFAULT_LOG_DIR

        log_location = os.path.join(log_dir, 'image_store', image_name)
        mkdir2(managed_node, log_location)
        
        img_conf_filename = None
        name = domconfig["name"]
        # write the config on the remote node.
        if image_conf is not None:
            img_conf_base = self.get_image_filename(image_name,name)
            img_conf_filename = os.path.join(log_location,img_conf_base)
            # adjust the pyconfig with correct filename and managed_node
            image_conf.set_managed_node(managed_node)
            image_conf.save(img_conf_filename)
        

        return (remote_image_store,
                scripts_dest,
                img_conf_filename,
                log_location)

    def execute_provisioning_script(self,
                                    managed_node,
                                    image_name,
                                    dom_config,
                                    image_conf):

        name = dom_config["name"]
        # prepare the environment to execute script
        (image_store_location,
         script_location,
         img_conf_filename,
         log_location) = self.prepare_env(managed_node,
                                          image_name,
                                          dom_config,
                                          image_conf)

        # get the script name
        script_name = self.SCRIPT_NAME

        # prepare script args
        script = os.path.join(script_location, script_name)
        script_args_filename = os.path.join(log_location,
                                            self.get_args_filename(image_name,
                                                                     name))
        
        #for now empty
        args = managed_node.node_proxy.open(script_args_filename, "w")
        args.close()

        # update the domconfig to have the reference to the image file
        # kind a kludge: ok for now.
        dom_config["image_conf"] = img_conf_filename
        dom_config.write()

        log_file_base = self.get_log_filename(image_name, name)
        log_filename = os.path.join(log_location,log_file_base)
        script_args=" -x " + dom_config.filename + \
                    " -p " + script_args_filename + \
                    " -s " + image_store_location + \
                    " -i " + image_name + \
                    " -l " + log_filename + \
                    " -c " + img_conf_filename
        
        cmd = script +  script_args

        # execute the script
        (out, exit_code) = managed_node.node_proxy.exec_cmd(cmd)

        #print "#### Done provisioning ", out, exit_code

        #if exit_code != 0:
        #    raise OSError("Provisioning script failed.\n" +  out)

        #managed_node.node_proxy.remove(script_args_filename)
        #managed_node.node_proxy.remove(image_filename)
        #managed_node.node_proxy.remove(log_filename)
        return (out, exit_code, log_filename)

    def get_args_filename(self,image_name, dom_name):
        name = image_name + "_" + dom_name +  ".args"
        #for t in time.gmtime():
        #    name = name + str(t) + "_"
        #name = name[:-1]
        return name

    def get_image_filename(self,image_name, dom_name):
        name = image_name + "_" + dom_name + "_" + "image.conf"
        return name

    def get_log_filename(self,image_name, dom_name):
        name = image_name + "_" + dom_name + ".log"
        return name


# helper function to find the appliance related templates
# this is to handle the descripency of tar ball vs rpm.
# there seems to be not an easy way to find
import sys
def get_template_location(template):
    for path in (os.path.dirname(sys.argv[0]),'.', '/usr/share/convirt/src'):
        filename = os.path.join(path , template)
        if os.path.exists(filename):
            return path
    else:
        msg=  "ERROR: Couldn't find appliance_template %s!" % template
        print msg
        raise Exception(msg)




# class for handling image imports
# Some utility functions
import string
class ImageUtils:

    compressed_ext = [".zip", ".gz"]

    # given a file system create an image
    APPLIANCE_TEMPLATE_LOCATION = \
        get_template_location("appliance_vm_conf.template")

    
    @classmethod
    def download_appliance(cls, local, appliance_url, image_dir, filename=None,
                           progress =  None):
        if appliance_url[0] == '/' : # fully specified path
            appliance_url = "file://" + appliance_url

        if appliance_url.find("file://") == 0:
            file_cp = True
            msg = "Copying"
        else:
            file_cp = False
            msg = "Downloading"

        fd = None
        # Tweaks to get around absence of filename and size
        try:
            opener = urllib2.build_opener()
            req = urllib2.Request(appliance_url)
            req.add_header("User-Agent", constants.fox_header)
            fd = opener.open(req)
            url = fd.geturl()
            path = urlparse.urlparse(urllib.url2pathname(url))[2]
            if not filename:
                filename = path.split('/')[-1]

            clen = fd.info().get("Content-Length")
            if clen is not None:
                content_len = int(clen)
            else:
                content_len = -1


            print url, filename, content_len
            ex = None
            download_file = os.path.join(image_dir, filename)
            if not local.node_proxy.file_exists(download_file):
                if file_cp:
                    try:
                        try:
                            src = path
                            dest = download_file
                            if progress:
                                progress.update(progress.START_PULSE,
                                                "Copying " + src +
                                                " to \n" + dest )  
                            (out, code) = local.node_proxy.exec_cmd("cp -a " + \
                                                                    src + \
                                                                    " " + \
                                                                    dest)
                            if code != 0:
                                raise Exception(out)

                            if progress and not progress.continue_op():
                                raise Exception("Canceled by user.")

                        except Exception, ex:
                            traceback.print_exc()
                            if progress:
                                progress.update(progress.CANCELED,str(ex))
                            raise
                    finally:
                        if progress and not ex:
                            progress.update(progress.STOP_PULSE, "Copying done")
                        if progress and not progress.continue_op():
                            local.node_proxy.remove(download_file)
                        
                else: # url download
                    df = None
                    try:
                        df = open(download_file,"wb")
                        chunk_size = 1024 * 64
                        chunks = content_len / chunk_size + 1
                        x = fd.read(chunk_size)
                        c = 1
                        p = 1.0 / chunks
                        if progress:
                            progress.update(progress.SET_FRACTION,
                                            msg,(p * c))
                        while  x is not None and x != "":
                            df.write(x)
                            #print "wrote ", c, chunks, p * c
                            if progress:
                                progress.update(progress.SET_FRACTION, None,(p * c))
                                if not progress.continue_op():
                                    raise Exception("Canceled by user.") 

                            c = c + 1
                            x = fd.read(chunk_size)
                    finally:
                        if df:
                            df.close()
                        if progress and not progress.continue_op():
                            local.node_proxy.remove(download_file)
        finally:
            if fd:
                fd.close()

        return download_file


    @classmethod
    def get_file_ext(cls, filename):
        # return filename and ext
        file = filename
        dot_index = filename.rfind(".")
        ext = ""
        if dot_index > -1:
            ext = filename[dot_index:]
            file = filename[:dot_index]

        return (file, ext)



    @classmethod
    def open_package(cls, local, downloaded_filename, image_dir, progress=None):
        uzip = None
        utar = None

        (f, e) = ImageUtils.get_file_ext(downloaded_filename)
        if e in cls.compressed_ext:
            if e == ".gz":
                uzip = "gunzip -f"
            elif e == ".zip":
                uzip = "unzip -o -d " + image_dir

        if uzip:
            if progress:
                progress.update(progress.START_PULSE, "Unzipping " + downloaded_filename)
            msg = None
            ex = None
            try:
                try:
                    (output, code) = local.node_proxy.exec_cmd(uzip + " " + \
                                                               downloaded_filename)
                    if code != 0 :
                        msg = "Error unzipping " + \
                              downloaded_filename + ":" + \
                              output
                        print msg
                        raise Exception(msg)

                    if e == ".zip":
                        local.node_proxy.remove(downloaded_filename)

                    if progress and not progress.continue_op():
                        raise Exception("Canceled by user.")
                    
                except Exception, ex:
                    if progress:
                        progress.update(progress.CANCELED,str(ex))
                    raise
            finally:
                if progress and not ex:
                    progress.update(progress.STOP_PULSE, "Unzipping done")
            
        # untar if required
        if downloaded_filename.find(".tar") > -1:
            ex = None
            untar = "tar xvf "
            msg = None

            try:
                try:
                    tar_file = downloaded_file[0:downloaded_file.find(".tar") + 4]
                    if progress:
                        progress.update(progress.START_PULSE,
                                        "Opening archive " + tar_file)
                    (output, code) = local.node_proxy.exec_cmd(untar + " " +
                                                               tar_file)  
                    if code != 0:
                        print "Error untaring ", tar_file
                        raise Exception("Error untaring " +  tar_file + " " +
                                        output)
                
                    if progress and not progress.continue_op():
                        raise Exception("Canceled by user.")
                
                except Exception, ex:
                    if progress:
                        progress.update(progress.CANCELED,str(ex))
                    raise
                
            finally:
                if progress and not ex:
                    progress.update(progress.STOP_PULSE,"Opening archive done.")


    @classmethod
    def get_vm_conf_template(cls, node, appliance_entry, cfg,disk_info):
        vm_template = XenConfig(node, 
                                os.path.join(cls.APPLIANCE_TEMPLATE_LOCATION,
                                             'appliance_vm_conf.template'))

        if appliance_entry.get("is_hvm") and \
               str(appliance_entry["is_hvm"]).lower() == "true" :
            vm_template = XenConfig(node, 
                                os.path.join(cls.APPLIANCE_TEMPLATE_LOCATION,
                                             'appliance_hvm_conf.template'))
        
        value_map = {"MEMORY" : 256,
                     "VCPUS" : 1,
                     "RAMDISK" : '',
                     "EXTRA" : '',
                     "KERNEL" : ''
                     }

        if cfg is not None:
            value_map["MEMORY"] = cfg.get("memory") or 256
            value_map["VCPUS"] = cfg.get("vcpus") or 1
            value_map["RAMDISK"] = ""
            if cfg.get("extra") :
                value_map["EXTRA"] = str(cfg.get("extra"))
            else:
                value_map["EXTRA"] = ""
                
        if appliance_entry.get("is_hvm") and \
               str(appliance_entry["is_hvm"]).lower() == "true" :
            pass
            ### Taken care by the hvm template now.
            ### Still issue of computed kernel is tricky to fix.
##             value_map["BOOT_LOADER"]=""
                                
##             vm_template["vnc"] = 1
##             vm_template["sdl"] = 0
##             vm_template["builder"] = "hvm"
##             # special handing for these :Kludge
##             vm_template.lines.append('device_model="/usr/" + arch_libdir + "/xen/bin/qemu-dm"\n')
##             del vm_template["kernel"]
##             vm_template.lines.append('kernel="/usr/" + arch_libdir + "/xen/boot/hvmloader"')
##             # make it a customizable option
##             computed_options = vm_template.get_computed_options()
##             computed_options.append("kernel")
##             computed_options.append("device_model")
##             vm_template.set_computed_options(computed_options)
        else:
            value_map["BOOT_LOADER"]="/usr/bin/pygrub"
            value_map["KERNEL"] = ""

        disks_directive = []
        # now lets generate the disk entries
        for di in disk_info:
            de, dpes = di
            (proto, device, mode) = de

            disk_entry = proto + "$VM_DISKS_DIR" + "/$VM_NAME." + device+ \
                         ".disk.xm"
            disk_entry += "," + device+ ","  + mode
            
            disks_directive.append(str(disk_entry))

        vm_template["disk"] = disks_directive
        if appliance_entry.get("provider_id"):
            vm_template["provider_id"] = str(appliance_entry.get("provider_id"))
    
        vm_template.instantiate_config(value_map)
  
        return vm_template


    @classmethod
    def get_image_config(cls, node, appliance_entry, disk_info, image_dir):

        image_conf_template = PyConfig(node, 
                                       os.path.join(cls.APPLIANCE_TEMPLATE_LOCATION,
                                                    'appliance_image_conf.template'))
        disks = []
        ndx = 0
        for di in disk_info:
            de, dpes = di
            (proto,device, mode) = de

            for dpe in dpes:
                dpe_name, dpe_val = dpe
                image_conf_template[dpe_name] = dpe_val

            # adjust the dev_image_src in the template form
            src = image_conf_template.get(device + "_image_src")
            if src:
                pos = src.find(image_dir)
                if pos == 0 :
                    src = '$IMAGE_STORE/$IMAGE_NAME/' + \
                          src[(len(image_dir) + 1):]
                    image_conf_template[device + "_image_src"] = str(src)

        return image_conf_template

    @classmethod
    def create_files(cls, local, appliance_entry, image_store,
                     image_name, vm_template, image_conf, force):
        vm_conf_file =image_store.get_vm_template(image_name, new=True)
        image_conf_file =image_store.get_image_conf(image_name, new=True)
        prov_script = image_store.get_provisioning_script(image_name, new=True)
        desc_file = image_store.get_image_desc_html(image_name, new=True)
        
        if force or not local.node_proxy.file_exists(vm_conf_file):
            vm_template.save(vm_conf_file)
            print "Created ", vm_conf_file
        if force or  not local.node_proxy.file_exists(image_conf_file):  
            image_conf.save(image_conf_file)
            print "Created ", image_conf_file
        if force or  not local.node_proxy.file_exists(prov_script):
            src_script = os.path.join(image_store.get_common_dir(), 
                                      ImageStore.SCRIPT_NAME)
            shutil.copy(src_script, prov_script)
            print "Created ", prov_script
        if force or  not local.node_proxy.file_exists(desc_file):
            cls.create_description(local, image_store, image_name,
                                   appliance_entry)

    @classmethod
    def create_description(cls, local,
                           image_store,image_name, appliance_entry,
                           desc_meta_template=None):
        # basically read the template, instantiate it and write it
        # as desc file.
        if not desc_meta_template:
            desc_meta_template = os.path.join(cls.APPLIANCE_TEMPLATE_LOCATION,
                                              "appliance_desc.template")

        content = None
        try:
            f = open(desc_meta_template, "r")
            content = f.read()
        finally:
            if f: f.close()

        if not content : # should always find the template
            return
        
        val_map = {}
        for key, ae_key in  (("NAME", "title"),
                             ("URL", "link"),
                             ("PROVIDER", "provider"),
                             ("PROVIDER_URL", "provider_url"),
                             ("PROVIDER_LOGO_URL", "provider_logo_url"),
                             ("DESCRIPTION","description")):
            v = appliance_entry.get(ae_key)
            if v :
                v = str(v)
                val_map[key] = v

        val_map["IMAGE_NAME"] = image_name

        # add extra requirements
        e_req = ""
        if appliance_entry.get('is_hvm') and \
           str(appliance_entry.get('is_hvm')).lower() == "true":
            e_req = e_req + ", " + "HVM / VT Enabled h/w"

        if appliance_entry.get("PAE") :  
            if str(appliance_entry.get("PAE")).lower()=="true" :
                e_req = e_req + ", " + "PAE"
            else:
                e_req = e_req +"," + "NON-PAE"
                
        if appliance_entry.get("arch"):
            e_req = e_req + ", " + appliance_entry.get("arch")

        val_map["EXTRA_REQUIREMENTS"] = e_req

        # We are putting some template specific stuff.
        provider_href = ""
        provider_logo_href = ""
        provider_str = ""

        if val_map.get("PROVIDER_URL"):
            provider_href = '<a href="$PROVIDER_URL">$PROVIDER</a>'
        else:
            provider_href = "$PROVIDER"
            
        if val_map.get("PROVIDER_LOGO_URL"):
            provider_logo_href = '<img src="$PROVIDER_LOGO_URL"/>'
            
        provider_str = provider_href + provider_logo_href
        if provider_str == "":
            provider_str = "Unknown (Manually Imported)"

        provider_str = string.Template(provider_str).safe_substitute(val_map)
        val_map["PROVIDER_STR"] = provider_str

        appliance_contact = ""
        
        if val_map.get("URL"):
            appliance_contact = 'Visit <a href="$URL">$NAME</a> for more information on the appliance. <br/>'
        appliance_contact = string.Template(appliance_contact).safe_substitute(val_map)
            
        val_map["APPLIANCE_CONTACT"] = appliance_contact
        template_str = string.Template(content)
        new_content = template_str.safe_substitute(val_map)
        desc_file = image_store.get_image_desc(image_name,True)

        try:
            fout = open(desc_file, "w")
            fout.write(new_content)
        finally:
            if fout:
                fout.close()

    @classmethod
    def import_fs(cls, local,
                  appliance_entry,
                  image_store,
                  image_name,
                  force, progress = None):

            appliance_url = appliance_entry["href"]

            image_dir = image_store.get_image_dir(image_name, new=True)

            if not local.node_proxy.file_exists(image_dir):
                mkdir2(local, image_dir)

            # fetch the image
            filename = appliance_entry.get("filename")
            downloaded_filename = ImageUtils.download_appliance(local,
                                                                appliance_url,
                                                                image_dir,
                                                                filename,
                                                                progress)
            
            # TBD : need to formalize this in to package handlers.
            if appliance_entry["type"] =="FILE_SYSTEM":
                disk_info = []
                di = get_disk_info("hda", downloaded_filename, "w")
                disk_info.append(di)
            elif appliance_entry["type"] == "JB_ARCHIVE":
                # gunzip/unzip the archive
                ImageUtils.open_package(local, downloaded_filename, image_dir,
                                        progress)
                
                disk_location = search_disks(image_dir)
                # clean up vmdk and other files.
                adjust_jb_image(local, disk_location, progress)
                disk_info = get_jb_disk_info(disk_location)

            vm_template = ImageUtils.get_vm_conf_template(local,
                                                          appliance_entry,
                                                          None, disk_info)
            
            image_conf  = ImageUtils.get_image_config(local, appliance_entry,
                                                    disk_info,
                                                    image_dir)

            ImageUtils.create_files(local, appliance_entry,
                                    image_store, image_name,
                                    vm_template, image_conf, force)

            return True

import glob
def search_disks(image_dir):
  disk_location = glob.glob(image_dir + '/disks')

  if len(disk_location) <= 0:
    disk_location = glob.glob(image_dir + '/*/disks')

  if len(disk_location) <= 0:
    disk_location = glob.glob(image_dir + '/*/*/disks')

  if len(disk_location) <= 0:
    raise Exception("disk directory not found under " + image_dir)

  disk_location = disk_location[0]

  return disk_location


# clean up vmdk files and compress the root hdd
def adjust_jb_image(local, disk_location, progress = None):
    for file in ("root.hdd", "root/root.hdd", "var.hdd", "swap.hdd"):
        root_fs = os.path.join(disk_location, file)
        if os.path.exists(root_fs):
            ex =None
            try:
                try:
                    # compress it
                    if progress:
                        progress.update(progress.START_PULSE,
                                        "Compressing " + root_fs)
                    (output, code) = local.node_proxy.exec_cmd("gzip " + root_fs)
                    if code !=0 :
                        raise Exception("Could not gzip " + root_fs + ":" +  output)
                    if progress and not progress.continue_op():
                        raise Exception("Canceled by user.")
                
                except Exception, ex:
                    if progress:
                        progress.update(progress.CANCELED,str(ex))
                    raise
                
            finally:
                if progress and not ex:
                    progress.update(progress.STOP_PULSE,"Compressing %s done." % root_fs)

        
    # the .hdd file that is shipped is not suitable for Xen.
    # the data.hdd.gz is what we are looking for.
    # remove all other files
    data_disk = os.path.join(disk_location, "data")
    if os.path.exists(data_disk):
        for exp in ("/*.vmdk", "/*.vhd", "/*.hdd"):
            files  = glob.glob(data_disk + exp)   
            for f in files:
                local.node_proxy.remove(f)

# return disk info from the unpacked JumpBox archive
def get_jb_disk_info(disk_location):
    disk_info = []

    for file in ("root.hdd", "root.hdd.gz",
                 "root/root.hdd", "root/root.hdd.gz"):
        root_fs = os.path.join(disk_location, file)
        if os.path.exists(root_fs):
            di = get_disk_info("hda", root_fs, "w")
            disk_info.append(di)
            break

    for file in ("swap.hdd", "swap.hdd.gz"):
        swap_fs = os.path.join(disk_location, file)
        if os.path.exists(swap_fs):
            di = get_disk_info("hdb", swap_fs, "w")
            disk_info.append(di)
            break

    var_found = False
    for file in ("var.hdd", "var.hdd.gz"):
        var_fs = os.path.join(disk_location, file)
        if os.path.exists(var_fs):
            di = get_disk_info("hdd", var_fs, "w") # hdc reserved for cdrom
            disk_info.append(di)
            var_found = True
            break
        
    if not var_found:
        # for new version of jumpbox. (get exact version here)
        data_fs = os.path.join(disk_location, "data/data.xen.tgz")
        if os.path.exists(data_fs):
            di = get_disk_info("hdd", data_fs, "w") # hdc reserved for cdrom
            disk_info.append(di)
        else:
            # generate de for hdd

            # generate dpe so as to satify the following
            
            # need to create a data dir with 10GB sparse file
            # ext3 file system
            # label as 'storage'
            pass

    if len(disk_info) <=0 :
        raise Exception("No disks found from JumpBox Archive.")

    return disk_info

    
# utility function to generate disk info, i.e. disk entry for
# vm config as well as for image conf.
def get_disk_info(device_name, filename, mode, proto="file:"):
    (uncompressed_file, ext) = ImageUtils.get_file_ext(filename)

    de = (proto,device_name, mode)
    d = device_name
    dpes = [ ("%s_disk_create" % d, "yes"),
             ("%s_image_src" % d, filename),
             ("%s_image_src_type" % d, "disk_image")]

    if ext in (".gz", ):
        dpes.append(("%s_image_src_format" % d, "gzip"))
    elif ext in (".tgz", ):
        dpes.append(("%s_image_src_format" % d, "tar_gzip"))
    elif ext in (".bz2", ):
        dpes.append(("%s_image_src_format" % d, "bzip"))
    elif ext in (".tbz2", ):
        dpes.append(("%s_image_src_format" % d, "tar_bzip"))

    return (de, dpes)


