/*
 * cowbell
 * Copyright (c) 2005 Brad Taylor
 *
 * This file is part of cowbell.
 *
 * cowbell is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * cowbell is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with cowbell; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using Gtk;
using Cowbell.Base;
using System.Reflection;
using System.Collections;

namespace Cowbell.Gui
{
	public class MetadataProxy
	{
		/* public methods */
		public MetadataProxy ()
		{
			Assembly asm = Assembly.GetCallingAssembly ();
			foreach (Type t in asm.GetTypes ())
			{
				foreach (Attribute a in Attribute.GetCustomAttributes (t))
				{
					if (a.GetType () == typeof (MetadataProxyPriorityAttribute)) {
						Insert (((MetadataProxyPriorityAttribute)a).Priority, t);
					}
				}
			}
		}

		public void Import ()
		{
			foreach (DictionaryEntry d in proxies)
			{
				IMetadataProxy p = d.Value as IMetadataProxy;
				p.Import ();

				progress = new ProgressWindow ();

				if (ResponseType.Cancel == (ResponseType)progress.Run ()) {
					/*
					 * FIXME: We should revert to a previous
					 * version of the database here
					 */
					progress.Hide ();
					return;
				}
			}
		}

		/* private fields */
		ProgressWindow progress;
		ArrayList proxies = new ArrayList ();
		string text;

		/* private methods */
		private void Insert (int priority, Type t)
		{
			IMetadataProxy p = (IMetadataProxy)Activator.CreateInstance (t);
			
			p.Heartbeat += new MetadataProxyHeartbeatHandler (OnProxyPulse);
			p.ConnectionFailed += new EventHandler (OnConnectionFailed);
			p.ConnectionRejected += new EventHandler (OnConnectionRejected);
			p.Completed += new EventHandler (OnProxyCompleted);
			proxies.Add (new DictionaryEntry (priority, p));

			// Basic insertion sort algorithm
			int i, j, index;
			for (i = 1; i < proxies.Count; i++)
			{
				index = (int)((DictionaryEntry)proxies[i]).Key;
				j = i;

				while (j > 0 && (int)((DictionaryEntry)proxies[j-1]).Key > index)
				{
					proxies[j] = proxies[j-1];
					j = j - 1;
				}

				proxies[j] = proxies[i];
			}
		}

		private void OnProxyPulse (string text)
		{
			this.text = text;
			GLib.Timeout.Add (0, new GLib.TimeoutHandler (OnProxyPulseTimeout));
		}

		private bool OnProxyPulseTimeout ()
		{
			if (progress != null) {
				progress.Pulse (text);
			}
			return false;
		}

		private void OnConnectionFailed (object o, EventArgs args)
		{
			GLib.Timeout.Add (0, new GLib.TimeoutHandler (OnConnectionFailedTimeout));
		}

		private bool OnConnectionFailedTimeout ()
		{
			ErrorDialog d = new ErrorDialog (progress.Window,
			                                 Catalog.GetString ("<span size='x-large' weight='bold'>Connection Failed</span>\n\n"
			                                                    + "The connection to http://soap.amazon.com failed.  Please check your "
			                                                    + "internet connection and try again later.")); 
			d.Run ();
			d.Hide ();
			progress.Hide ();

			return false;
		}

		private void OnConnectionRejected (object o, EventArgs args)
		{
			GLib.Timeout.Add (0, new GLib.TimeoutHandler (OnConnectionRejectedTimeout));
		}

		private bool OnConnectionRejectedTimeout ()
		{
			ErrorDialog d = new ErrorDialog (progress.Window,
			                                 Catalog.GetString ("<span size='x-large' weight='bold'>Couldn't find album.</span>\n\n"
			                                                    + "Amazon couldn't find the album that you specified.  Please verify the Album "
			                                                    + "and Artist names and try again."));
			d.Run ();
			d.Hide ();
			progress.Hide ();

			return false;
		}

		private void OnProxyCompleted (object o, EventArgs args)
		{
			GLib.Timeout.Add (0, new GLib.TimeoutHandler (OnProxyCompletedTimeout));
		}

		private bool OnProxyCompletedTimeout ()
		{
			progress.Hide ();
			return false;
		}
	}
}
