/**************************************************************************/
/*                                                                        */
/*  cpbk - a mirroring utility for backing up your files                  */
/*  Copyright (C) 1998 Kevin Lindsay <klindsay@mkintraweb.com>            */
/*  Copyright (C) 2001 Yuuki NINOMIYA <gm@debian.or.jp>                   */
/*                                                                        */
/*  This program is free software; you can redistribute it and/or modify  */
/*  it under the terms of the GNU General Public License as published by  */
/*  the Free Software Foundation; either version 2, or (at your option)   */
/*  any later version.                                                    */
/*                                                                        */
/*  This program is distributed in the hope that it will be useful,       */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of        */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         */
/*  GNU General Public License for more details.                          */
/*                                                                        */
/*  You should have received a copy of the GNU General Public License     */
/*  along with this program; if not, write to the                         */
/*  Free Software Foundation, Inc., 59 Temple Place - Suite 330,          */
/*  Boston, MA 02111-1307, USA.                                           */
/*                                                                        */
/**************************************************************************/

/* $Id: order.c,v 1.6 2001/03/30 13:16:36 gm Exp $ */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>

#include "intl.h"
#include "strlib.h"
#include "variable.h"
#include "proto.h"


static Order *src_order = NULL;
static Order *dest_order = NULL;
static int max_src_order = 0;
static int max_dest_order = 0;



static void generate_order(FileData *src_file_data, FileData *dest_file_data, const char *src_dir, const char *dest_dir);

static void compare_src_recursively(FileData *src_file_data, FileData *dest_file_data, int src_path_num, int dest_path_num, const char *src_dir, const char *dest_dir);
static void compare_src_file(FileData *src_file_data, FileData *dest_file_data, int src_path_num, int dest_path_num, int src_file_num, const char *src_dir, const char *dest_dir);
static void compare_src_dir(FileData *src_file_data, FileData *dest_file_data, int src_path_num, int dest_path_num, int src_file_num, const char *src_dir, const char *dest_dir);
static void search_child_src_dir(FileData *src_file_data, FileData *dest_file_data, int src_path_num, int dest_path_num, int src_file_num, const char *src_dir, const char *dest_dir);

static void compare_dest_recursively(FileData *src_file_data, FileData *dest_file_data, int src_path_num, int dest_path_num, const char *src_dir, const char *dest_dir);
static void compare_dest_file(FileData *src_file_data, FileData *dest_file_data, int src_path_num, int dest_path_num, int dest_file_num, const char *src_dir, const char *dest_dir);
static void compare_dest_dir(FileData *src_file_data, FileData *dest_file_data, int src_path_num, int dest_path_num, int dest_file_num, const char *src_dir, const char *dest_dir);
static void search_child_dest_dir(FileData *src_file_data, FileData *dest_file_data, int src_path_num, int dest_path_num, int dest_file_num, const char *src_dir, const char *dest_dir);

static void add_src_action(const FileState *file_state, const char *src_dir, const char *dest_dir, Action action);
static void add_dest_action(const FileState *file_state, const char *src_dir, const char *dest_dir, Action action);

static Order *optimize_order(void);

static int get_int_max_digit(int a, int b, int c);
static int get_offt_max_digit(off_t a, off_t b, off_t c);


/* --- PUBLIC FUNCTIONS --- */

Order *compare_files_and_generate_order(FileData *local_file_data, FileData *remote_file_data, const char *local_top_dir, const char *remote_top_dir)
{
	generate_order(local_file_data, remote_file_data, local_top_dir, remote_top_dir);

	add_src_action(NULL, NULL, NULL, END);
	add_dest_action(NULL, NULL, NULL, END);

	return (optimize_order());
}


void free_order(Order *order)
{
	int i;

	for (i = 0; order[i].action != END; i++) {
		free(order[i].file);
		free(order[i].src_dir);
		free(order[i].dest_dir);
	}
	free(order);
}


void put_listing_of_updated_file(Order *order)
{
	int i;

	printf(_("New Files\n"));

	for (i = 0; order[i].action != END; i++) {
		if (order[i].action == NEW) {
			printf("%s/%s\n", order[i].src_dir, order[i].file);
		}
	}

	printf(_("\nUpdated Files\n"));

	for (i = 0; order[i].action != END; i++) {
		if (order[i].action == UPDATE) {
			printf("%s/%s\n", order[i].src_dir, order[i].file);
		}
	}

	printf(_("\nRemoved Files\n"));
	for (i = 0; order[i].action != END; i++) {
		if (order[i].action == REMOVE) {
			printf("%s/%s\n", order[i].dest_dir, order[i].file);
		}
	}

	printf(_("\nFiles/Directories changed the access permissions or the owner\n"));
	for (i = 0; order[i].action != END; i++) {
		if (order[i].action == CHMOD) {
			printf("%s/%s\n", order[i].dest_dir, order[i].file);
		}
	}

	printf(_("\nNew Directories\n"));
	for (i = 0; order[i].action != END; i++) {
		if (order[i].action == MKDIR) {
			printf("%s/%s\n", order[i].dest_dir, order[i].file);
		}
	}

	printf(_("\nRemoved Directories\n"));
	for (i = 0; order[i].action != END; i++) {
		if (order[i].action == RMDIR) {
			printf("%s/%s\n", order[i].dest_dir, order[i].file);
		}
	}
}


void put_the_number_of_updated_file(Order *order)
{
	int i;
	int new_num = 0;
	int update_num = 0;
	int remove_num = 0;
	int mkdir_num = 0;
	int rmdir_num = 0;
	int chmod_num = 0;
	off_t new_size = 0;
	off_t update_size = 0;
	off_t remove_size = 0;
	int num_digit;
	int size_digit;

	for (i = 0; order[i].action != END; i++) {
		switch (order[i].action) {
		case NEW:
			new_num++;
			new_size += (order[i].status.st_size / 1000);
			break;
		case UPDATE:
			update_num++;
			update_size += (order[i].status.st_size / 1000);
			break;
		case REMOVE:
			remove_num++;
			remove_size += (order[i].status.st_size / 1000);
			break;
		case MKDIR:
			mkdir_num++;
			break;
		case RMDIR:
			rmdir_num++;
			break;
		case CHMOD:
			chmod_num++;
			break;
		case ENTER:
		case LEAVE:
			break;
		default:
			internal_error(__FILE__, __LINE__);
		}
	}

	num_digit = get_int_max_digit(new_num, update_num, remove_num);
	size_digit = get_offt_max_digit(new_size, update_size, remove_size);

	printf(_("New Files     : %*d (%*ldKB)\n"), num_digit, new_num, size_digit, new_size);
	printf(_("Updated Files : %*d (%*ldKB)\n"), num_digit, update_num, size_digit, update_size);
	printf(_("Removed Files : %*d (%*ldKB)\n"), num_digit, remove_num, size_digit, remove_size);
	printf(_("New Directories     : %d\n"), mkdir_num);
	printf(_("Removed Directories : %d\n"), rmdir_num);
	printf(_("Files/Directories changed the access permissions or the owner : %d\n"), chmod_num);
}


/* TODO: using binary search */
int search_path_num(PathState *path_state, int num_paths, const char *path)
{
	int i;

	for (i = 0; i < num_paths; i++) {
		if (strcmp(path_state[i].path, path) == 0) {
			return (i);
		}
	}
	return (-1);
}


int search_file_num(PathState *path_state, int path_num, const char *name)
{
	int i;

	for (i = 0; i < path_state[path_num].num_files; i++) {
		if (strcmp(path_state[path_num].file_state[i].name, name) == 0) {
			return (i);
		}
	}
	return (-1);
}


/* --- PRIVATE FUNCTIONS --- */

static void generate_order(FileData *src_file_data, FileData *dest_file_data, const char *src_dir, const char *dest_dir)
{
	int src_path_num;
	int dest_path_num;

	src_path_num = search_path_num(src_file_data->path_state, src_file_data->num_paths, src_dir);
	dest_path_num = search_path_num(dest_file_data->path_state, dest_file_data->num_paths, dest_dir);

	if (src_path_num >= 0) {
		compare_src_recursively(src_file_data, dest_file_data, src_path_num, dest_path_num, src_dir, dest_dir);
	}
	if (dest_path_num >= 0) {
		compare_dest_recursively(src_file_data, dest_file_data, src_path_num, dest_path_num, src_dir, dest_dir);
	}
}


static void compare_src_recursively(FileData *src_file_data, FileData *dest_file_data, int src_path_num, int dest_path_num, const char *src_dir, const char *dest_dir)
{
	int i;

	for (i = 0; i < src_file_data->path_state[src_path_num].num_files; i++) {
		if (src_file_data->path_state[src_path_num].file_state[i].isdir) {
			compare_src_dir(src_file_data, dest_file_data, src_path_num, dest_path_num, i, src_dir, dest_dir);
		} else {
			compare_src_file(src_file_data, dest_file_data, src_path_num, dest_path_num, i, src_dir, dest_dir);
		}
	}
}


static void compare_src_file(FileData *src_file_data, FileData *dest_file_data, int src_path_num, int dest_path_num, int src_file_num, const char *src_dir, const char *dest_dir)
{
	int dest_file_num;

	if (dest_path_num >= 0) {
		dest_file_num = search_file_num(dest_file_data->path_state, dest_path_num, src_file_data->path_state[src_path_num].file_state[src_file_num].name);
		if (dest_file_num >= 0) { /* same name already exists */
			if (dest_file_data->path_state[dest_path_num].file_state[dest_file_num].isdir) { /* src is file, but dest is dir, so dest dir is removed */
				search_child_dest_dir(src_file_data, dest_file_data, src_path_num, dest_path_num, dest_file_num, src_dir, dest_dir);
				add_dest_action(dest_file_data->path_state[dest_path_num].file_state + dest_file_num, src_dir, dest_dir, RMDIR);
				add_dest_action(src_file_data->path_state[src_path_num].file_state + src_file_num, src_dir, dest_dir, NEW);
			} else if (is_src_newer(src_file_data->path_state[src_path_num].file_state + src_file_num, dest_file_data->path_state[dest_path_num].file_state + dest_file_num, src_dir, dest_dir)) { /* updated file */
				add_src_action(src_file_data->path_state[src_path_num].file_state + src_file_num, src_dir, dest_dir, UPDATE);
			} else if (is_src_mode_changed(src_file_data->path_state[src_path_num].file_state + src_file_num, dest_file_data->path_state[dest_path_num].file_state + dest_file_num, src_dir, dest_dir)) {
				add_src_action(src_file_data->path_state[src_path_num].file_state + src_file_num, src_dir, dest_dir, CHMOD);
			}
			return;
		}
	}

	/* new file || in new directory, so all files are sent */
	add_src_action(src_file_data->path_state[src_path_num].file_state + src_file_num, src_dir, dest_dir, NEW);
}


static void compare_src_dir(FileData *src_file_data, FileData *dest_file_data, int src_path_num, int dest_path_num, int src_file_num, const char *src_dir, const char *dest_dir)
{
	int dest_file_num;

	if (dest_path_num >= 0) {
		dest_file_num = search_file_num(dest_file_data->path_state, dest_path_num, src_file_data->path_state[src_path_num].file_state[src_file_num].name);
		if (dest_file_num >= 0) { /* same name already exists */
			if (dest_file_data->path_state[dest_path_num].file_state[dest_file_num].isdir) {
				if (is_src_mode_changed(src_file_data->path_state[src_path_num].file_state + src_file_num, dest_file_data->path_state[dest_path_num].file_state + dest_file_num, src_dir, dest_dir)) {
					add_src_action(src_file_data->path_state[src_path_num].file_state + src_file_num, src_dir, dest_dir, CHMOD);
				}
				search_child_src_dir(src_file_data, dest_file_data, src_path_num, dest_path_num, src_file_num, src_dir, dest_dir);
			} else { /* same name exists, but it's a file. so remove it and create new directory */
				add_src_action(dest_file_data->path_state[dest_path_num].file_state + dest_file_num, src_dir, dest_dir, REMOVE);
				add_src_action(src_file_data->path_state[src_path_num].file_state + src_file_num, src_dir, dest_dir, MKDIR);
				search_child_src_dir(src_file_data, dest_file_data, src_path_num, dest_path_num, src_file_num, src_dir, dest_dir);
			}
			return;
		}
	}

	/* new directory || in new directory, so all directories are made */
	add_src_action(src_file_data->path_state[src_path_num].file_state + src_file_num, src_dir, dest_dir, MKDIR);
	search_child_src_dir(src_file_data, dest_file_data, src_path_num, dest_path_num, src_file_num, src_dir, dest_dir);
}


static void search_child_src_dir(FileData *src_file_data, FileData *dest_file_data, int src_path_num, int dest_path_num, int src_file_num, const char *src_dir, const char *dest_dir)
{
	char *new_src_dir;
	char *new_dest_dir;
	int new_src_path_num;
	int new_dest_path_num;

	new_src_dir = str_concat(src_dir, "/", src_file_data->path_state[src_path_num].file_state[src_file_num].name, NULL);
	new_dest_dir = str_concat(dest_dir, "/", src_file_data->path_state[src_path_num].file_state[src_file_num].name, NULL);

	new_src_path_num = search_path_num(src_file_data->path_state, src_file_data->num_paths, new_src_dir);
	new_dest_path_num = search_path_num(dest_file_data->path_state, dest_file_data->num_paths, new_dest_dir);

	add_src_action(src_file_data->path_state[src_path_num].file_state + src_file_num, src_dir, dest_dir, ENTER);
	compare_src_recursively(src_file_data, dest_file_data, new_src_path_num, new_dest_path_num, new_src_dir, new_dest_dir);
	add_src_action(src_file_data->path_state[src_path_num].file_state + src_file_num, src_dir, dest_dir, LEAVE);

	free(new_dest_dir);
	free(new_src_dir);
}


static void compare_dest_recursively(FileData *src_file_data, FileData *dest_file_data, int src_path_num, int dest_path_num, const char *src_dir, const char *dest_dir)
{
	int i;

	for (i = 0; i < dest_file_data->path_state[dest_path_num].num_files; i++) {
		if (dest_file_data->path_state[dest_path_num].file_state[i].isdir) {
			compare_dest_dir(src_file_data, dest_file_data, src_path_num, dest_path_num, i, src_dir, dest_dir);
		} else {
			compare_dest_file(src_file_data, dest_file_data, src_path_num, dest_path_num, i, src_dir, dest_dir);
		}
	}
}


static void compare_dest_file(FileData *src_file_data, FileData *dest_file_data, int src_path_num, int dest_path_num, int dest_file_num, const char *src_dir, const char *dest_dir)
{
	int src_file_num;
	char *src_path_file;
	char *dest_path_file;

	if (src_path_num >= 0) {
		src_file_num = search_file_num(src_file_data->path_state, src_path_num, dest_file_data->path_state[dest_path_num].file_state[dest_file_num].name);
		if (src_file_num >= 0) { /* file exist (not remove) */
			return;
		}
	}

	src_path_file = str_concat(src_dir, "/", dest_file_data->path_state[dest_path_num].file_state[dest_file_num].name, NULL);
	dest_path_file = str_concat(dest_dir, "/", dest_file_data->path_state[dest_path_num].file_state[dest_file_num].name, NULL);
	if (config.save_exclude_path && is_exclude_path(src_path_file, dest_path_file)) {
		free(src_path_file);
		free(dest_path_file);
		return;
	}
	free(src_path_file);
	free(dest_path_file);

	/* obsolete file || in obsolete directory, so all files are removed */
	add_dest_action(dest_file_data->path_state[dest_path_num].file_state + dest_file_num, src_dir, dest_dir, REMOVE);
}


static void compare_dest_dir(FileData *src_file_data, FileData *dest_file_data, int src_path_num, int dest_path_num, int dest_file_num, const char *src_dir, const char *dest_dir)
{
	int src_file_num;
	char *src_path_dir;
	char *dest_path_dir;

	if (src_path_num >= 0) {
		src_file_num = search_file_num(src_file_data->path_state, src_path_num, dest_file_data->path_state[dest_path_num].file_state[dest_file_num].name);
		if (src_file_num >= 0) { /* same name already exists */
			if (src_file_data->path_state[src_path_num].file_state[src_file_num].isdir) {
				search_child_dest_dir(src_file_data, dest_file_data, src_path_num, dest_path_num, dest_file_num, src_dir, dest_dir);
			}
			return;
		}
	}

	src_path_dir = str_concat(src_dir, "/", dest_file_data->path_state[dest_path_num].file_state[dest_file_num].name, NULL);
	dest_path_dir = str_concat(dest_dir, "/", dest_file_data->path_state[dest_path_num].file_state[dest_file_num].name, NULL);
	if (config.save_exclude_path && is_exclude_path(src_path_dir, dest_path_dir)) {
		free(src_path_dir);
		free(dest_path_dir);
		return;
	}
	free(src_path_dir);
	free(dest_path_dir);

	/* obsolete directory || in obsolete directory, so all directories are removed */
	search_child_dest_dir(src_file_data, dest_file_data, src_path_num, dest_path_num, dest_file_num, src_dir, dest_dir);
	add_dest_action(dest_file_data->path_state[dest_path_num].file_state + dest_file_num, src_dir, dest_dir, RMDIR);
}

static void search_child_dest_dir(FileData *src_file_data, FileData *dest_file_data, int src_path_num, int dest_path_num, int dest_file_num, const char *src_dir, const char *dest_dir)
{
	char *new_src_dir;
	char *new_dest_dir;
	int new_src_path_num;
	int new_dest_path_num;

	new_src_dir = str_concat(src_dir, "/", dest_file_data->path_state[dest_path_num].file_state[dest_file_num].name, NULL);
	new_dest_dir = str_concat(dest_dir, "/", dest_file_data->path_state[dest_path_num].file_state[dest_file_num].name, NULL);

	new_src_path_num = search_path_num(src_file_data->path_state, src_file_data->num_paths, new_src_dir);
	new_dest_path_num = search_path_num(dest_file_data->path_state, dest_file_data->num_paths, new_dest_dir);

	add_dest_action(dest_file_data->path_state[dest_path_num].file_state + dest_file_num, src_dir, dest_dir, ENTER);
	compare_dest_recursively(src_file_data, dest_file_data, new_src_path_num, new_dest_path_num, new_src_dir, new_dest_dir);
	add_dest_action(dest_file_data->path_state[dest_path_num].file_state + dest_file_num, src_dir, dest_dir, LEAVE);

	free(new_dest_dir);
	free(new_src_dir);
}


static void add_src_action(const FileState *file_state, const char *src_dir, const char *dest_dir, Action action)
{
	src_order = str_realloc(src_order, sizeof(*src_order) * (max_src_order + 1));
	src_order[max_src_order].action = action;

	if (action != END) {
		src_order[max_src_order].file = str_dup(file_state->name);
		src_order[max_src_order].src_dir = str_dup(src_dir);
		src_order[max_src_order].dest_dir = str_dup(dest_dir);
		src_order[max_src_order].isdir = file_state->isdir;
		src_order[max_src_order].status = file_state->status;
	}
	max_src_order++;
}


static void add_dest_action(const FileState *file_state, const char *src_dir, const char *dest_dir, Action action)
{
	dest_order = str_realloc(dest_order, sizeof(*dest_order) * (max_dest_order + 1));
	dest_order[max_dest_order].action = action;

	if (action != END) {
		dest_order[max_dest_order].file = str_dup(file_state->name);
		dest_order[max_dest_order].src_dir = str_dup(src_dir);
		dest_order[max_dest_order].dest_dir = str_dup(dest_dir);
		dest_order[max_dest_order].isdir = file_state->isdir;
		dest_order[max_dest_order].status = file_state->status;
	}
	max_dest_order++;
}


static Order *optimize_order(void)
{
	Order *order;
	int max = 0;
	int dest = 0;
	int i, j;

	order = str_malloc(sizeof(*order) * (max_src_order + max_dest_order + 1));

	for (i = 0; src_order[i].action != END; i++) {
		if (src_order[i].action == MKDIR){
			for (j = i; !(src_order[j].action == LEAVE && strcmp(src_order[i].dest_dir, src_order[j].dest_dir) == 0 && strcmp(src_order[i].file, src_order[j].file) == 0); j++) {
				order[max] = src_order[j];
				max++;
			}
			order[max] = src_order[j];
			max++;
			i = j;
			continue;
		}
		if (src_order[i].action == ENTER){
			while (!(dest_order[dest].action == ENTER && strcmp(src_order[i].dest_dir, dest_order[dest].dest_dir) == 0 && strcmp(src_order[i].file, dest_order[dest].file) == 0)) {
				order[max] = dest_order[dest];
				max++;
				dest++;
			}
			free(dest_order[dest].file);
			free(dest_order[dest].src_dir);
			free(dest_order[dest].dest_dir);
			dest++;
		} else if (src_order[i].action == LEAVE) {
			while (!(dest_order[dest].action == LEAVE && strcmp(src_order[i].dest_dir, dest_order[dest].dest_dir) == 0 && strcmp(src_order[i].file, dest_order[dest].file) == 0)) {
				order[max] = dest_order[dest];
				max++;
				dest++;
			}
			free(dest_order[dest].file);
			free(dest_order[dest].src_dir);
			free(dest_order[dest].dest_dir);
			dest++;
		}
		order[max] = src_order[i];
		max++;
	}
	for (i = dest; dest_order[i].action != END; i++) {
		order[max] = dest_order[i];
		max++;
	}

	free(src_order);
	free(dest_order);
	src_order = NULL;
	dest_order = NULL;
	max_src_order = 0;
	max_dest_order = 0;

	order[max].action = END;
	order[max].file = NULL;
	order[max].src_dir = NULL;
	order[max].dest_dir = NULL;

	return (order);
}


static int get_int_max_digit(int a, int b, int c)
{
	int max;
	char tmp[20];

	max = a;
	if (b > max) {
		max = b;
	}
	if (c > max) {
		max = c;
	}

	sprintf(tmp, "%d", max);
	return (strlen(tmp));
}


static int get_offt_max_digit(off_t a, off_t b, off_t c)
{
	off_t max;
	char tmp[20];

	max = a;
	if (b > max) {
		max = b;
	}
	if (c > max) {
		max = c;
	}

	sprintf(tmp, "%ld", max);
	return (strlen(tmp));
}
