#include "print_backend_file.h"

int main()
{
  cpdbInit();
  GMainLoop *loop = g_main_loop_new(NULL, FALSE);
  b = get_new_BackendObj();
  acquire_session_bus_name(BUS_NAME);
  g_main_loop_run(loop);
}

static void
on_name_acquired(GDBusConnection *connection,
                 const gchar *name,
                 gpointer user_data)
{
  b->dbus_connection = connection;
  b->skeleton = print_backend_skeleton_new();
  connect_to_signals();
  connect_to_dbus(b, CPDB_BACKEND_OBJ_PATH);
}

static void
acquire_session_bus_name(char *bus_name)
{
  g_bus_own_name(G_BUS_TYPE_SESSION,
                 bus_name,
                 G_BUS_NAME_OWNER_FLAGS_NONE,
                 NULL,
                 on_name_acquired,
                 NULL,
                 NULL,
                 NULL);
}

static gboolean on_handle_get_printer_list(PrintBackend *interface,
                                           GDBusMethodInvocation *invocation,
                                           gpointer user_data)
{
  int num_printers;
  GVariant *printers, *printer;
  GVariantBuilder builder;

  const char *dialog_name = g_dbus_method_invocation_get_sender(invocation);

  add_frontend(b, dialog_name);
  num_printers = 1;
  g_variant_builder_init(&builder, G_VARIANT_TYPE_ARRAY);
  FilePrinter *fp = get_FilePrinter();
  printer = g_variant_new(CPDB_PRINTER_ARGS, fp->name, fp->name, fp->info,
                          fp->location, fp->name, fp->is_accepting_jobs,
                          fp->state, BACKEND_NAME);
  g_variant_builder_add(&builder, "(v)", printer);
  printers = g_variant_builder_end(&builder);
  
  free(fp);
  print_backend_complete_get_printer_list(interface, invocation, num_printers, printers);
  return TRUE;
}

static gboolean on_handle_print_file(PrintBackend *interface,
                                     GDBusMethodInvocation *invocation,
                                     const gchar *printer_name,
                                     const gchar *file_path,
                                     int num_settings,
                                     GVariant *settings,
                                     const gchar *final_file_path,
                                     gpointer user_data)
{
  const char *dialog_name = g_dbus_method_invocation_get_sender(invocation);
  Dialog *d = find_dialog(b, dialog_name);
  if(d == NULL)
  {
    MSG_LOG("Invalid dialog name.\n", ERR);
    return FALSE;
  }
  
  if(strcmp(d->printers->name, printer_name))
  {
    printf("Printer '%s' does not exist for the dialog %s.\n", printer_name, dialog_name);
    return FALSE;
  }
  
  print_file(file_path, final_file_path);
  print_backend_complete_print_file(interface, invocation, "File printed");

  /* (Currently Disabled) Printing a file must be the last operation. */
  //d->cancel = 1;
  //remove_frontend(b, dialog_name);
  if(b->num_frontends == 0)
  {
    g_message("No frontends connected .. exiting backend.\n");
    exit(EXIT_SUCCESS);
  }
  return TRUE;
}

static void on_stop_backend(GDBusConnection *connection,
                            const gchar *sender_name,
                            const gchar *object_path,
                            const gchar *interface_name,
                            const gchar *signal_name,
                            GVariant *parameters,
                            gpointer user_data)
{
  g_message("Stop backend signal from %s\n", sender_name);

  Dialog *d = find_dialog(b, sender_name);
  if(d != NULL)
  {
    if (d->keep_alive)
      return;

    d->cancel = 1;
    remove_frontend(b, sender_name);
  }

  if(b->num_frontends == 0)
  {
    g_message("No frontends connected .. exiting backend.\n");
    exit(EXIT_SUCCESS);
  }
}

static gboolean on_handle_get_default_printer(PrintBackend *interface,                                 
                                              GDBusMethodInvocation *invocation,
                                              gpointer user_data)
{
  char *printer = get_default_printer(b);
  printf("%s\n", printer);
  print_backend_complete_get_default_printer(interface, invocation, printer);
  return TRUE;
}

static gboolean on_handle_get_printer_state(PrintBackend *interface,
                                            GDBusMethodInvocation *invocation,
                                            const gchar *printer_name,
                                            gpointer user_data)
{
  const char *dialog_name = g_dbus_method_invocation_get_sender(invocation);
  Dialog *d = find_dialog(b, dialog_name);
  if(d == NULL)
  {
    MSG_LOG("Invalid dialog name.\n", ERR);
    return FALSE;
  }
  
  if(strcmp(d->printers->name, printer_name))
  {
    printf("Printer '%s' does not exist for the dialog %s.\n", printer_name, dialog_name);
    return FALSE;
  }

  const char *state = d->printers->state;
  printf("%s is %s\n", printer_name, state);
  print_backend_complete_get_printer_state(interface, invocation, state);
  return TRUE;
}

static gboolean on_handle_is_accepting_jobs(PrintBackend *interface,
                                            GDBusMethodInvocation *invocation,
                                            const gchar *printer_name,
                                            gpointer user_data)
{
  const char *dialog_name = g_dbus_method_invocation_get_sender(invocation);
  Dialog *d = find_dialog(b, dialog_name);
  if(d == NULL)
  {
    MSG_LOG("Invalid dialog name.\n", ERR);
    return FALSE;
  }
  
  if(strcmp(d->printers->name, printer_name))
  {
    printf("Printer '%s' does not exist for the dialog %s.\n", printer_name, dialog_name);
    return FALSE;
  }

  print_backend_complete_is_accepting_jobs(interface, invocation, d->printers->is_accepting_jobs);
  return TRUE;
}

static gboolean on_handle_get_all_options(PrintBackend *interface,
                                          GDBusMethodInvocation *invocation,
                                          const gchar *printer_name,
                                          gpointer user_data)
{
  const char *dialog_name = g_dbus_method_invocation_get_sender(invocation);
  Dialog *d = find_dialog(b, dialog_name);
  if(d == NULL)
  {
    MSG_LOG("Invalid dialog name.\n", ERR);
    return FALSE;
  }
  
  if(strcmp(d->printers->name, printer_name))
  {
    printf("Printer '%s' does not exist for the dialog %s.\n", printer_name, dialog_name);
    return FALSE;
  }

  int count, media_count;
  Option *options;
  GVariantBuilder *builder;
  GVariant *variant, *media_variant;

  media_count = 0;
  media_variant = g_variant_new_array(G_VARIANT_TYPE("(siiia(iiii))"), NULL, 0);

  count = d->printers->num_options;
  options = d->printers->options;
  if (count == 0)
  {
    variant = g_variant_new_array(G_VARIANT_TYPE("(sssia(s))"), NULL, 0);
  }
  else
  {
    builder = g_variant_builder_new(G_VARIANT_TYPE("a(sssia(s))"));
    for(int i=0; i<count ; i++)
    {
      GVariant *option = pack_option(&options[i]);
      g_variant_builder_add_value(builder, option);
    }
    variant = g_variant_builder_end(builder);
  }

  print_backend_complete_get_all_options(interface, invocation, count, variant, media_count, media_variant);
  return TRUE;
}

static gboolean on_handle_get_active_jobs_count(PrintBackend *interface,
                                                GDBusMethodInvocation *invocation,
                                                const gchar *printer_name,
                                                gpointer user_data)
{
  const char *dialog_name = g_dbus_method_invocation_get_sender(invocation);
  Dialog *d = find_dialog(b, dialog_name);
  if(d == NULL)
  {
    MSG_LOG("Invalid dialog name.\n", ERR);
    return FALSE;
  }
  
  if(strcmp(d->printers->name, printer_name))
  {
    printf("Printer '%s' does not exist for the dialog %s.\n", printer_name, dialog_name);
    return FALSE;
  }

  print_backend_complete_get_active_jobs_count(interface, invocation, 0);
  return TRUE;
}

static gboolean on_handle_get_all_jobs(PrintBackend *interface,
                                       GDBusMethodInvocation *invocation,
                                       gboolean active_only,
                                       gpointer user_data)
{
  GVariant *variant;
  GVariantBuilder *builder;

  builder = g_variant_builder_new(G_VARIANT_TYPE(CPDB_JOB_ARRAY_ARGS));
  variant = g_variant_builder_end(builder);

  print_backend_complete_get_all_jobs(interface, invocation, 0, variant);
  return TRUE;
}

static gboolean on_handle_cancel_job(PrintBackend *interface,
                                     GDBusMethodInvocation *invocation,
                                     const gchar *job_id,
                                     const gchar *printer_name,
                                     gpointer user_data)
{
  const char *dialog_name = g_dbus_method_invocation_get_sender(invocation);
  Dialog *d = find_dialog(b, dialog_name);
  if(d == NULL)
  {
    MSG_LOG("Invalid dialog name.\n", ERR);
    return FALSE;
  }
  
  if(strcmp(d->printers->name, printer_name))
  {
    printf("Printer '%s' does not exist for the dialog %s.\n", printer_name, dialog_name);
    return FALSE;
  }

  print_backend_complete_cancel_job(interface, invocation, FALSE);
  return TRUE;
}

static gboolean on_handle_keep_alive(PrintBackend *interface,
                                     GDBusMethodInvocation *invocation,
                                     gpointer user_data)
{
  const char *dialog_name = g_dbus_method_invocation_get_sender(invocation);
  Dialog *d = find_dialog(b, dialog_name);
  if(d == NULL)
  {
    MSG_LOG("Invalid dialog name.\n", ERR);
    return FALSE;
  }

  d->keep_alive = TRUE;
  print_backend_complete_keep_alive(interface, invocation);
  return TRUE;
}

static gboolean on_handle_replace(PrintBackend *interface,
                                  GDBusMethodInvocation *invocation,
                                  const gchar *previous_name,
                                  gpointer user_data)
{
  const char *dialog_name = g_dbus_method_invocation_get_sender(invocation);
  Dialog *d = find_dialog(b, dialog_name);
  if(d == NULL)
  {
    MSG_LOG("Invalid dialog name.\n", ERR);
    return FALSE;
  }

  g_hash_table_steal(b->dialogs, previous_name);
  g_hash_table_insert(b->dialogs, cpdbGetStringCopy(dialog_name), d);
  g_message("Replaced %s --> %s\n", previous_name, dialog_name);

  print_backend_complete_replace(interface, invocation);
  return TRUE;
}

static gboolean on_handle_get_option_translation(PrintBackend *interface,
                                                 GDBusMethodInvocation *invocation,
                                                 const gchar *printer_name,
                                                 const gchar *option_name,
                                                 const gchar *locale,
                                                 gpointer user_data)
{
  print_backend_complete_get_option_translation(interface, invocation, option_name);
  return TRUE;
}

static gboolean on_handle_get_choice_translation(PrintBackend *interface,
                                                 GDBusMethodInvocation *invocation,
                                                 const gchar *printer_name,
                                                 const gchar *option_name,
                                                 const gchar *choice_name,
                                                 const gchar *locale,
                                                 gpointer user_data)
{
  print_backend_complete_get_choice_translation(interface, invocation, choice_name);
  return TRUE;
}

static gboolean on_handle_get_group_translation(PrintBackend *interface,
                                                GDBusMethodInvocation *invocation,
                                                const gchar *printer_name,
                                                const gchar *group_name,
                                                const gchar *locale,
                                                gpointer user_data)
{
  char *translation = cpdbGetGroupTranslation2(group_name, locale);
  print_backend_complete_get_group_translation(interface, invocation, translation);
  free(translation);
  return TRUE;
}

static gboolean on_handle_get_all_translations(PrintBackend *interface,
                                               GDBusMethodInvocation *invocation,
                                               const gchar *printer_name,
                                               const gchar *locale,
                                               gpointer user_data)
{
  GVariant *translations = g_variant_new_array(G_VARIANT_TYPE(CPDB_TL_ARGS), NULL, 0);
  print_backend_complete_get_all_translations(interface, invocation, translations);
  return TRUE;
}

void connect_to_signals()
{
  PrintBackend *skeleton = b->skeleton;
  
  g_signal_connect(skeleton,
                   "handle-get-printer-list",
                   G_CALLBACK(on_handle_get_printer_list),
                   NULL);

  g_signal_connect(skeleton,
                   "handle-print-file",
                   G_CALLBACK(on_handle_print_file),
                   NULL);

  g_signal_connect(skeleton,
                   "handle-get-default-printer",
                   G_CALLBACK(on_handle_get_default_printer),
                   NULL);

  g_signal_connect(skeleton,
                   "handle-get-printer-state",
                   G_CALLBACK(on_handle_get_printer_state),
                   NULL);

  g_signal_connect(skeleton,
                   "handle-is-accepting-jobs",
                   G_CALLBACK(on_handle_is_accepting_jobs),
                   NULL);

  g_signal_connect(skeleton,
                   "handle-get-all-options",
                   G_CALLBACK(on_handle_get_all_options),
                   NULL);

  g_signal_connect(skeleton,
                   "handle-get-active-jobs-count",
                   G_CALLBACK(on_handle_get_active_jobs_count),
                   NULL);

  g_signal_connect(skeleton,
                   "handle-get-all-jobs",
                   G_CALLBACK(on_handle_get_all_jobs),
                   NULL);

  g_signal_connect(skeleton,
                   "handle-cancel-job",
                   G_CALLBACK(on_handle_cancel_job),
                   NULL);

  g_signal_connect(skeleton,
                   "handle-keep-alive",
                   G_CALLBACK(on_handle_keep_alive),
                   NULL);

  g_signal_connect(skeleton,
                   "handle-replace",
                   G_CALLBACK(on_handle_replace),
                   NULL);

  g_signal_connect(skeleton,
                   "handle-get-option-translation",
                   G_CALLBACK(on_handle_get_option_translation),
                   NULL);

  g_signal_connect(skeleton,
                   "handle-get-choice-translation",
                   G_CALLBACK(on_handle_get_choice_translation),
                   NULL);

  g_signal_connect(skeleton,
                   "handle-get-group-translation",
                   G_CALLBACK(on_handle_get_group_translation),
                   NULL);
  g_signal_connect(skeleton,
                   "handle-get-all-translations",
                   G_CALLBACK(on_handle_get_all_translations),
                   NULL);

  g_dbus_connection_signal_subscribe(b->dbus_connection,
                                     NULL,
                                     "org.openprinting.PrintFrontend",
                                     CPDB_SIGNAL_STOP_BACKEND,
                                     NULL,
                                     NULL,
                                     0,
                                     on_stop_backend,
                                     NULL,
                                     NULL);

}
