/* $Id: hdrl_bpm-test.c,v 1.3 2013-09-24 14:58:54 jtaylor Exp $
 *
 * This file is part of the HDRL
 * Copyright (C) 2013 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: jtaylor $
 * $Date: 2013-09-24 14:58:54 $
 * $Revision: 1.3 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/

#include "hdrl_bpm.h"
#include <cpl.h>

#ifndef ARRAY_LEN
#define ARRAY_LEN(a) sizeof((a))/sizeof((a)[0])
#endif


/*----------------------------------------------------------------------------*/
/**
 * @defgroup hdrl_bpm_test   Testing of the HDRL bad pixel handling
 */
/*----------------------------------------------------------------------------*/

cpl_error_code test_bpm_to_mask(void)
{
    cpl_size nx = 20;
    cpl_size ny = 20;
    {
        cpl_mask * mask = hdrl_bpm_to_mask(NULL, 0);
        cpl_test_error(CPL_ERROR_NULL_INPUT);
        cpl_test_null(mask);
    }
    /* non int input */
    {
        cpl_image * bpm = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
        cpl_mask * mask = hdrl_bpm_to_mask(bpm, 0);
        cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
        cpl_test_null(mask);
        cpl_image_delete(bpm);

        /* too big mask */
        bpm = cpl_image_new(nx, ny, CPL_TYPE_INT);
        mask = hdrl_bpm_to_mask(bpm, ~0LLU);
        cpl_test_null(mask);
        cpl_test_error(CPL_ERROR_UNSUPPORTED_MODE);

        cpl_image_delete(bpm);
    }
    /* empty bpm */
    {
        cpl_image * bpm = cpl_image_new(nx, ny, CPL_TYPE_INT);
        cpl_mask * mask = NULL;
        mask = hdrl_bpm_to_mask(bpm, 0);
        cpl_test_nonnull(mask);
        cpl_test_eq(cpl_mask_count(mask), 0);
        cpl_image_delete(bpm);
        cpl_mask_delete(mask);
    }
    /* range of codes */
    {
        cpl_image * bpm = cpl_image_new(nx, ny, CPL_TYPE_INT);
        cpl_mask * mask = NULL;
        cpl_image_set(bpm, 1, 1, 1);
        cpl_image_set(bpm, 1, 2, 2);
        cpl_image_set(bpm, 1, 3, 3);
        cpl_image_set(bpm, 1, 4, 4);

        mask = hdrl_bpm_to_mask(bpm, 1);
        cpl_test_nonnull(mask);
        cpl_test_eq(cpl_mask_count(mask), 2);
        cpl_mask_delete(mask);

        mask = hdrl_bpm_to_mask(bpm, ~0u);
        cpl_test_error(CPL_ERROR_NONE);
        cpl_test_nonnull(mask);
        cpl_test_eq(cpl_mask_count(mask), 4);
        cpl_mask_delete(mask);

        cpl_image_delete(bpm);
    }
    return cpl_error_get_code();
}


cpl_error_code test_mask_to_bpm(void)
{
    cpl_size nx = 20;
    cpl_size ny = 20;
    {
        cpl_image * bpm = hdrl_mask_to_bpm(NULL, 0);
        cpl_test_error(CPL_ERROR_NULL_INPUT);
        cpl_test_null(bpm);
    }
    /* empty mask */
    {
        cpl_image * bpm;
        cpl_mask * mask = cpl_mask_new(nx, ny);
        bpm = hdrl_mask_to_bpm(mask, 0);
        cpl_test_nonnull(bpm);
        cpl_test_eq(cpl_image_get_flux(bpm), 0);
        cpl_image_delete(bpm);
        cpl_mask_delete(mask);
    }
    /* non-empty mask */
    {
        cpl_image * bpm = NULL;
        cpl_mask * mask = cpl_mask_new(nx, ny);
        cpl_mask_set(mask, 1, 1, CPL_BINARY_1);
        cpl_mask_set(mask, 1, 2, CPL_BINARY_1);
        cpl_mask_set(mask, 1, 3, CPL_BINARY_1);
        cpl_mask_set(mask, 1, 4, CPL_BINARY_1);

        bpm = hdrl_mask_to_bpm(mask, 1);
        cpl_test_nonnull(bpm);
        cpl_test_eq(cpl_image_get_flux(bpm), 4);
        cpl_image_delete(bpm);

        bpm = hdrl_mask_to_bpm(mask, 5);
        cpl_test_nonnull(bpm);
        cpl_test_eq(cpl_image_get_flux(bpm), 5 * 4);
        cpl_image_delete(bpm);

        cpl_mask_delete(mask);
    }
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief   Unit tests of bias module
 **/
/*----------------------------------------------------------------------------*/
int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_bpm_to_mask();
    test_mask_to_bpm();

    cpl_test_error(CPL_ERROR_NONE);

    return cpl_test_end(0);
}
