/* $Id: hdrl_imagelist_view-test.c,v 1.3 2013-10-22 08:26:11 jtaylor Exp $
 *
 * This file is part of the HDRL
 * Copyright (C) 2013 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: jtaylor $
 * $Date: 2013-10-22 08:26:11 $
 * $Revision: 1.3 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/

#include "hdrl_imagelist.h"
#include "hdrl_test.h"
#include "hdrl_imagelist_view.h" /* TODO put in public api? */

#include <cpl.h>

#define IMG_PTR(a) \
    cpl_image_get_data(hdrl_image_get_image(a))
#define ERR_PTR(a) \
    cpl_image_get_data(hdrl_image_get_image(a))
#define MSK_PTR(a) \
    cpl_mask_get_data(hdrl_image_get_mask(a))

/*----------------------------------------------------------------------------*/
/**
 * @defgroup hdrl_imagelist_view_test   
            Testing of hdrl_imagelist_view module
 */
/*----------------------------------------------------------------------------*/

#define YSIZE 73
#define XSIZE 50

cpl_error_code test_row_view(void)
{
    hdrl_imagelist  *   hlist;
    hdrl_image      *   himg;
    hdrl_image      *   himg2;
    cpl_image       *   contrib;
    cpl_image       *   contrib2;
    cpl_image       *   ima;
    cpl_image       *   ima_err;
    cpl_size            nimages = 20;
    hdrl_imagelist * view;
    cpl_imagelist * clist;

    /* Create an image list */
    hlist = hdrl_imagelist_new() ;
    clist = cpl_imagelist_new() ;
    for (cpl_size i = 0 ; i < nimages ; i++) {

        ima = cpl_image_new(XSIZE, YSIZE, HDRL_TYPE_DATA);
        ima_err = cpl_image_new(XSIZE, YSIZE, HDRL_TYPE_ERROR);
        cpl_image_reject(ima, 1, 5);
        if ((i % 5) == 0) {
            cpl_image_reject(ima, 2, 5);
        }
        himg = hdrl_image_create(ima, ima_err);
        cpl_image_delete(ima_err);
        hdrl_imagelist_set(hlist, himg, i);
        cpl_imagelist_set(clist, ima, i);
    }

    view = hdrl_imagelist_row_view(NULL, 10, 20);
    cpl_test_null(view);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    view = hdrl_imagelist_row_view(hlist, 20, 10);
    cpl_test_null(view);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    view = hdrl_imagelist_row_view(hlist, 0, 10);
    cpl_test_null(view);
    cpl_test_error(CPL_ERROR_ACCESS_OUT_OF_RANGE);
    view = hdrl_imagelist_row_view(hlist, 1, YSIZE + 11);
    cpl_test_null(view);
    cpl_test_error(CPL_ERROR_ACCESS_OUT_OF_RANGE);
   
    // Mean collapse
    hdrl_imagelist_collapse_mean(hlist, &himg, &contrib);

    view = hdrl_imagelist_row_view(hlist, 1, YSIZE);
    hdrl_imagelist_collapse_mean(view, &himg2, &contrib2);

    hdrl_test_image_abs(himg, himg2, 0);
    cpl_test_image_abs(contrib, contrib2, 0);
    cpl_image * ccontrib = cpl_image_new_from_accepted(clist);
    cpl_test_image_abs(ccontrib, contrib2, 0);

    cpl_image_delete(ccontrib);
    cpl_image_delete(contrib);
    hdrl_image_delete(himg);
    cpl_image_delete(contrib2);
    hdrl_image_delete(himg2);

    hdrl_image * iorig = hdrl_imagelist_get(hlist, 0);
    hdrl_image * iview = hdrl_imagelist_get(view, 0);
    cpl_test_eq_ptr(IMG_PTR(iorig), IMG_PTR(iview));
    cpl_test_eq_ptr(ERR_PTR(iorig), ERR_PTR(iview));
    cpl_test_eq_ptr(MSK_PTR(iorig), MSK_PTR(iview));

    hdrl_image * icopy = hdrl_image_duplicate(iview);
    cpl_test_eq(hdrl_image_get_size_y(icopy), YSIZE);
    cpl_test_noneq_ptr(IMG_PTR(icopy), IMG_PTR(iview));
    cpl_test_noneq_ptr(ERR_PTR(icopy), ERR_PTR(iview));
    cpl_test_noneq_ptr(MSK_PTR(icopy), MSK_PTR(iview));
    hdrl_image_delete(icopy);

    hdrl_imagelist * lcopy = hdrl_imagelist_duplicate(view);
    hdrl_imagelist_delete(view);
    cpl_test_eq(hdrl_imagelist_get_size(lcopy), nimages);
    cpl_test_eq(hdrl_image_get_size_y(hdrl_imagelist_get(lcopy, 0)), YSIZE);
    hdrl_imagelist_delete(lcopy);

    cpl_size nsizey = YSIZE - 17 - 5 + 1;
    view = hdrl_imagelist_row_view(hlist, 5, YSIZE - 17);
    hdrl_image * vimg = hdrl_imagelist_get(view, 0);
    hdrl_image_add_scalar(vimg, 5, 1);

    cpl_test_eq(hdrl_image_get_size_x(vimg), XSIZE);
    cpl_test_eq(hdrl_image_get_size_y(vimg), nsizey);

    double v, e;
    hdrl_image * orig = hdrl_imagelist_get(hlist, 0);
    /* mean of original image (mix of 5s and zeros) */
    hdrl_image_get_mean(orig, &v, &e);
    cpl_test_noneq(v, 5);

    vimg = hdrl_imagelist_get(view, 0);
    hdrl_image_get_mean(vimg, &v, &e);
    cpl_test_rel(v, 5, HDRL_EPS_DATA * XSIZE * YSIZE);

    hdrl_imagelist_delete(view);

    hdrl_imagelist_delete(hlist);
    cpl_imagelist_delete(clist);

    return cpl_error_get_code();
}

cpl_error_code test_image_view(void)
{
    hdrl_imagelist * hl = hdrl_imagelist_new();
    hdrl_imagelist * view;

    view = hdrl_imagelist_image_view(hl, 5, 1);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    cpl_test_null(view);

    view = hdrl_imagelist_image_view(hl, 1, 1);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    cpl_test_null(view);

    view = hdrl_imagelist_image_view(hl, 0, 1);
    cpl_test_error(CPL_ERROR_ACCESS_OUT_OF_RANGE);
    cpl_test_null(view);

    hdrl_imagelist_set(hl, hdrl_image_new(5, 7), 0);

    view = hdrl_imagelist_image_view(hl, 1, 2);
    cpl_test_error(CPL_ERROR_ACCESS_OUT_OF_RANGE);
    cpl_test_null(view);

    view = hdrl_imagelist_image_view(hl, -1, 1);
    cpl_test_error(CPL_ERROR_ACCESS_OUT_OF_RANGE);
    cpl_test_null(view);

    view = hdrl_imagelist_image_view(hl, 0, 1);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(view);

    cpl_test_eq(hdrl_imagelist_get_size(view), hdrl_imagelist_get_size(hl));

    hdrl_imagelist_add_scalar(hl, 1, 1);

    hdrl_test_image_abs(hdrl_imagelist_get(view, 0),
                        hdrl_imagelist_get(hl, 0), 0);

    hdrl_image * n = hdrl_image_new(5, 7);
    hdrl_imagelist_set(hl, hdrl_image_new(5, 7), 1);

    hdrl_test_image_abs(hdrl_imagelist_get(view, 0),
                        hdrl_imagelist_get(hl, 0), 0);
    hdrl_test_image_abs(hdrl_imagelist_get(hl, 1), n, 0);
    hdrl_imagelist_delete(view);
    hdrl_image_delete(n);

    hdrl_imagelist_set(hl, hdrl_image_new(5, 7), 2);
    hdrl_imagelist_set(hl, hdrl_image_new(5, 7), 3);
    view = hdrl_imagelist_image_view(hl, 1, 2);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(hdrl_imagelist_get_size(view), 1);
    cpl_test_eq_ptr(IMG_PTR(hdrl_imagelist_get(view, 0)),
                    IMG_PTR(hdrl_imagelist_get(hl, 1)));
    cpl_test_eq_ptr(ERR_PTR(hdrl_imagelist_get(view, 0)),
                    ERR_PTR(hdrl_imagelist_get(hl, 1)));
    cpl_test_eq_ptr(MSK_PTR(hdrl_imagelist_get(view, 0)),
                    MSK_PTR(hdrl_imagelist_get(hl, 1)));
    hdrl_imagelist_delete(view);

    view = hdrl_imagelist_image_view(hl, 1, 3);
    cpl_test_eq(hdrl_imagelist_get_size(view), 2);
    cpl_test_eq_ptr(IMG_PTR(hdrl_imagelist_get(view, 0)),
                    IMG_PTR(hdrl_imagelist_get(hl, 1)));
    cpl_test_eq_ptr(ERR_PTR(hdrl_imagelist_get(view, 0)),
                    ERR_PTR(hdrl_imagelist_get(hl, 1)));
    cpl_test_eq_ptr(MSK_PTR(hdrl_imagelist_get(view, 0)),
                    MSK_PTR(hdrl_imagelist_get(hl, 1)));
    cpl_test_eq_ptr(IMG_PTR(hdrl_imagelist_get(view, 1)),
                    IMG_PTR(hdrl_imagelist_get(hl, 2)));
    cpl_test_eq_ptr(ERR_PTR(hdrl_imagelist_get(view, 1)),
                    ERR_PTR(hdrl_imagelist_get(hl, 2)));
    cpl_test_eq_ptr(MSK_PTR(hdrl_imagelist_get(view, 1)),
                    MSK_PTR(hdrl_imagelist_get(hl, 2)));

    /* test view of view */
    hdrl_imagelist * rview = hdrl_imagelist_row_view(view, 2, 7);
    cpl_test_eq(hdrl_imagelist_get_size(rview), hdrl_imagelist_get_size(view));
    hdrl_image * tmp = hdrl_imagelist_get(rview, 0);
    cpl_test_eq(hdrl_image_get_size_x(tmp), 5);
    cpl_test_eq(hdrl_image_get_size_y(tmp), 6);
    hdrl_imagelist_delete(rview);

    hdrl_imagelist_delete(hl);
    hdrl_imagelist_delete(view);

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief   Unit tests of hdrl_image_view
 **/
/*----------------------------------------------------------------------------*/
int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_row_view();
    test_image_view();

    return cpl_test_end(0);
}
