/* $Id: hdrl_sigclip-test.c,v 1.3 2013-09-24 14:58:54 jtaylor Exp $
 *
 * This file is part of the HDRL
 * Copyright (C) 2012,2013 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: jtaylor $
 * $Date: 2013-09-24 14:58:54 $
 * $Revision: 1.3 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/

#include "hdrl_sigclip.h"
#include "hdrl_utils.h"
#include <cpl.h>

#include <math.h>
#include <stdlib.h>

#ifndef ARRAY_LEN
#define ARRAY_LEN(a) sizeof((a))/sizeof((a)[0])
#endif


/*----------------------------------------------------------------------------*/
/**
 * @defgroup hdrl_sigclip_test   Testing of the HDRL Sigma Clipping module
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code hdrl_clip_kappa_sigma_test(void)
{
    double omean, omean_err, rej_low, rej_high;
    cpl_size naccepted;

    {
        double * dpixels = cpl_calloc(9, sizeof(double));

        cpl_vector * data = cpl_vector_wrap(9, dpixels);
        cpl_vector * errors = cpl_vector_new(9);
        cpl_vector_fill(errors, 1);

        /* null optional out params */
        hdrl_kappa_sigma_clip(data, errors, 3, 3, 3,
                             &omean, NULL, NULL, NULL, NULL);
        cpl_test_error(CPL_ERROR_NONE);

        cpl_test_eq(omean, 0.);

        /* all out params NULL makes no sense */
        hdrl_kappa_sigma_clip(data, errors, 3, 3, 3,
                             NULL, NULL, NULL, NULL, NULL);
        cpl_test_error(CPL_ERROR_NULL_INPUT);

        /* NULL data */
        hdrl_kappa_sigma_clip(NULL, errors, 3, 3, 3,
                             &omean, NULL, NULL, NULL, NULL);
        cpl_test_error(CPL_ERROR_NULL_INPUT);

        /* wrong iter */
        hdrl_kappa_sigma_clip(data, errors, 3, 3, 0,
                             &omean, NULL, NULL, NULL, NULL);
        cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

        cpl_vector_delete(data);
        cpl_vector_delete(errors);
    }

    {
        double * dpixels = cpl_calloc(9, sizeof(double));

        cpl_vector * data = cpl_vector_wrap(9, dpixels);
        cpl_vector * errors = cpl_vector_new(9);
        cpl_vector_fill(errors, 1);

        hdrl_kappa_sigma_clip(data, errors, 3, 3, 3,
                             &omean, &omean_err, &naccepted,
                             NULL, NULL);
        cpl_test_error(CPL_ERROR_NONE);

        cpl_test_eq(omean, 0.);
        cpl_test_rel(omean_err, 1 / sqrt(9), 0.001);
        cpl_test_eq(naccepted, 9);

        cpl_vector_delete(data);
        cpl_vector_delete(errors);
    }

    {
        /* iqr sigma ~1.5 median 6, check that 2 and 10 which are closer to
         * 6 -+ 3 than 4 and 6 but beyond the k*sig limit are not included */
        double values[] = {2.5, 6., 4., 6., 4., 6., 4., 6., 10.};
        const int n = ARRAY_LEN(values);

        cpl_vector * data = cpl_vector_wrap(n, values);
        cpl_vector * errors = cpl_vector_new(n);
        cpl_vector_fill(errors, 1);

        hdrl_kappa_sigma_clip(data, errors, 2., 2., 1,
                             &omean, &omean_err, &naccepted,
                             &rej_low, &rej_high);
        cpl_test_error(CPL_ERROR_NONE);

        cpl_test_rel(omean, (4. * 3 + 6 * 4.) / 7., 0.001);
        cpl_test_rel(omean_err, 1 / sqrt(n - 2), 0.001);
        cpl_test_rel(rej_low, 3, 0.02);
        cpl_test_rel(rej_high, 9, 0.02);
        cpl_test_eq(naccepted, n - 2);

        cpl_vector_unwrap(data);
        cpl_vector_delete(errors);
    }

    {
        /* special case of one remaining pixels */
        double values[] = {10.};
        const int n = ARRAY_LEN(values);

        cpl_vector * data = cpl_vector_wrap(n, values);
        cpl_vector * errors = cpl_vector_new(n);
        cpl_vector_fill(errors, 1);

        hdrl_kappa_sigma_clip(data, errors, 2., 2., 1,
                             &omean, &omean_err, &naccepted,
                             &rej_low, &rej_high);
        cpl_test_error(CPL_ERROR_NONE);

        cpl_test_rel(omean, values[0], 0.001);
        cpl_test_rel(omean_err, 1, 0.001);
        cpl_test_rel(rej_low, values[0], 0.02);
        cpl_test_rel(rej_high, values[0], 0.02);
        cpl_test_eq(naccepted, n);

        cpl_vector_unwrap(data);
        cpl_vector_delete(errors);
    }

    {
        /* gaus mean 100 sigma 3.5 */
        double values[] = {92, 93, 94, 94, 95, 95, 96, 96, 96, 97,
            97, 97, 97, 98, 98, 98, 98, 99, 99, 99,
            99, 100, 100, 100, 100, 100, 101, 101, 101, 101,
            102, 102, 102, 102, 103, 103, 103, 103, 104, 104,
            104, 105, 105, 106, 106, 107, 108 };
        const int n = ARRAY_LEN(values);

        cpl_vector * data = cpl_vector_wrap(n, values);
        cpl_vector * errors = cpl_vector_new(n);
        cpl_vector_fill(errors, 1);

        /* kappa 2 included by iqr 92 and 108 */
        hdrl_kappa_sigma_clip(data, errors, 2., 2., 3,
                             &omean, &omean_err, &naccepted,
                             &rej_low, &rej_high);
        cpl_test_error(CPL_ERROR_NONE);

        cpl_test_rel(omean, 100., 0.001);
        cpl_test_rel(omean_err, 1 / sqrt(n), 0.001);
        /* sigma overestimated by iqr */
        cpl_test_rel(rej_low, 91., 0.005);
        cpl_test_rel(rej_high, 109, 0.005);
        cpl_test_eq(naccepted, n);

        cpl_vector_unwrap(data);
        cpl_vector_delete(errors);
    }

    {
        /* gaus mean 100 sigma 3.5, 2 sigma range, 2 outliers */
        double values[] = {1, 150, 94, 94, 95, 95, 96, 96, 96, 97,
            97, 97, 97, 98, 98, 98, 98, 99, 99, 99,
            99, 100, 100, 100, 100, 100, 101, 101, 101, 101,
            102, 102, 102, 102, 103, 103, 103, 103, 104, 104,
            104, 105, 105, 106, 106, 107, 108 };
        const int n = ARRAY_LEN(values);

        cpl_vector * data = cpl_vector_wrap(n, values);
        cpl_vector * errors = cpl_vector_new(n);
        cpl_vector_fill(errors, 1);

        hdrl_kappa_sigma_clip(data, errors, 3, 3, 3,
                             &omean, &omean_err, &naccepted,
                             NULL, NULL);
        cpl_test_error(CPL_ERROR_NONE);

        cpl_test_rel(omean, 100., 0.005);
        cpl_test_rel(omean_err, 1 / sqrt(n - 2), 0.001);
        cpl_test_eq(naccepted, n - 2);

        cpl_vector_unwrap(data);
        cpl_vector_delete(errors);
    }

    /* test inputs are not modified */
    {
        double values[] = {54, 234. ,5,2, 343, 23 , 2, 0.21, 0.1232 , 1.2e3};
        const int n = ARRAY_LEN(values);

        cpl_vector * data = cpl_vector_wrap(n, values);
        cpl_vector * errors = cpl_vector_duplicate(data);
        cpl_vector * odata = cpl_vector_duplicate(data);
        cpl_vector * oerrors = cpl_vector_duplicate(errors);

        hdrl_kappa_sigma_clip(data, errors, 3, 3, 3,
                             &omean, &omean_err, &naccepted,
                             NULL, NULL);
        cpl_test_error(CPL_ERROR_NONE);

        cpl_test_vector_abs(data, odata, FLT_EPSILON);
        cpl_test_vector_abs(errors, oerrors, FLT_EPSILON);

        cpl_vector_unwrap(data);
        cpl_vector_delete(errors);
        cpl_vector_delete(odata);
        cpl_vector_delete(oerrors);
    }


    /* image test */
    {
        /* gaus mean 100 sigma 3.5, 2 sigma range, 2 outliers */
        double values[] = {1, 150, 94, 94, 95, 95, 96, 96, 96, 97,
            97, 97, 97, 98, 98, 98, 98, 99, 99, 99,
            99, 100, 100, 100, 100, 100, 101, 101, 101, 101,
            102, 102, 102, 102, 103, 103, 103, 103, 104, 104,
            104, 105, 105, 106, 106, 107, 108, 100, 100};
        const int n = ARRAY_LEN(values);

        cpl_image * data = cpl_image_wrap(sqrt(n), sqrt(n), CPL_TYPE_DOUBLE, 
                values);
        cpl_image * errors = cpl_image_new(sqrt(n), sqrt(n), CPL_TYPE_DOUBLE);
        cpl_image_add_scalar(errors, 1);

        hdrl_kappa_sigma_clip_image(data, errors, 3, 3, 3,
                                   &omean, &omean_err, &naccepted,
                                   NULL, NULL);
        cpl_test_error(CPL_ERROR_NONE);

        cpl_test_rel(omean, 100., 0.005);
        cpl_test_rel(omean_err, 1 / sqrt(n - 2), 0.001);
        cpl_test_eq(naccepted, n - 2);

        cpl_image_unwrap(data);
        cpl_image_delete(errors);
    }

    /* image test with bad pixels */
    {
        /* gaus mean 100 sigma 3.5, 2 sigma range, 2 outliers */
        float values[] = {1, 150, 94, 94, 95, 95, 96, 96, 96, 97,
            97, 97, 97, 98, 98, 98, 98, 99, 99, 99,
            99, 100, 100, 100, 100, 100, 101, 101, 101, 101,
            102, 102, 102, 102, 103, 103, 103, 103, 104, 104,
            104, 105, 105, 106, 106, 107, 108, 100, 100};
        const int n = sqrt(ARRAY_LEN(values));

        cpl_image * data = cpl_image_wrap(n, n, CPL_TYPE_FLOAT, values);
        cpl_image * errors = cpl_image_new(n, n, CPL_TYPE_FLOAT);
        cpl_image_add_scalar(errors, 1);
        /* set two bad pixels with really high error */
        cpl_image_reject(data, n, n);
        cpl_image_reject(data, n, n - 1);
        cpl_image_set(errors, n, n, 2343.e30);
        cpl_image_set(errors, n, n - 1, 2343.e30);
        cpl_image_reject_from_mask(errors, cpl_image_get_bpm(data));

        hdrl_kappa_sigma_clip_image(data, errors, 3, 3, 3,
                                   &omean, &omean_err, &naccepted,
                                   NULL, NULL);
        cpl_test_error(CPL_ERROR_NONE);

        cpl_test_rel(omean, 100., 0.005);
        cpl_test_rel(omean_err, 1 / sqrt(n * n - 4), 0.001);
        cpl_test_eq(naccepted, (n * n) - 4);

        cpl_image_unwrap(data);
        cpl_image_delete(errors);
    }

    /* test unequal bpms */
    {
        const int n = 5;
        cpl_image * data = cpl_image_new(n, n, CPL_TYPE_FLOAT);
        cpl_image * errors = cpl_image_new(n, n, CPL_TYPE_FLOAT);
        cpl_image_reject(data, n, n);
        cpl_image_reject(data, n, n - 1);

        hdrl_kappa_sigma_clip_image(data, errors, 3, 3, 3,
                                   &omean, &omean_err, &naccepted,
                                   NULL, NULL);
        cpl_test_error(CPL_ERROR_NONE);
        cpl_image_delete(data);
        cpl_image_delete(errors);
    }

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief   Unit tests of kappa sigma clipping
 **/
/*----------------------------------------------------------------------------*/
int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    hdrl_clip_kappa_sigma_test();

    return cpl_test_end(0);
}
