/*
 * This file is part of the FORS Data Reduction Pipeline
 * Copyright (C) 2002-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * response.cpp
 *
 *  Created on: 2014 3 28
 *      Author: cgarcia
 */

#include <response.h>
#include <vector_utils.h>

mosca::response::response()
{
    // TODO Auto-generated constructor stub

}

mosca::response::~response()
{
    // TODO Auto-generated destructor stub
}

void mosca::response::compute_response
(mosca::spectrum& obs_std_extcorrect, mosca::spec_std_star& std_star)
{
    std::vector<double> std_flux = std_star.flux();
    std::vector<double> std_wave = std_star.wave();
    std::vector<double> std_binsize = std_star.binsize();

    m_wave.clear();
    m_efficiency_raw.clear();
    m_efficiency_fit.clear();
    m_response_fit.clear();
    m_response_raw.clear();
    m_wave.clear();
    m_observed_flux.clear();

    for(size_t ibin = 0; ibin < std_flux.size() ; ++ibin)
    {
        //TODO: the 0.0026 is hardcoded, and depends on telescope area 
        //see moses.c   
        //Get the wavelength range
        double wave_bin_start = std_wave[ibin] - std_binsize[ibin] / 2.;
        double wave_bin_end   = std_wave[ibin] + std_binsize[ibin] / 2.;
        
        //Compute the integrated flux of the observed star in this wave range
        double obs_std_flux_bin =
            obs_std_extcorrect.integrate(wave_bin_start, wave_bin_end) /
               std_binsize[ibin];
        double wavelength = (wave_bin_start + wave_bin_end)/2.;
        double std_flux_bin_phys = std_flux[ibin] * 0.0026 * wavelength;
        double efficiency = obs_std_flux_bin / std_flux_bin_phys;
        double respon;
        if(obs_std_flux_bin != 0)
            respon =  std_flux[ibin] / obs_std_flux_bin;
        else
            respon = 0;

        m_efficiency_raw.push_back(efficiency);
        m_response_raw.push_back(respon);
        m_wave.push_back(wavelength);
        m_wave_bin.push_back(std_binsize[ibin]);
        m_observed_flux.push_back(obs_std_flux_bin);
        m_tabulated_flux.push_back(std_flux[ibin]);

    }
}

void mosca::response::fit_response
(int spline_knots,
 const std::vector<double>& ignored_waves,
 const std::vector<std::pair<double, double> >& ignored_wave_ranges)
{

    /* Filling the fit with the proper data */
    m_response_fit = m_response_raw;
    m_efficiency_fit = m_efficiency_raw;
    for(size_t ibin = 0; ibin < m_response_raw.size() ; ++ibin)
    {
        //Get the wavelength range
        double wave_bin_start = m_wave[ibin] - m_wave_bin[ibin] / 2.;
        double wave_bin_end   = m_wave[ibin] + m_wave_bin[ibin] / 2.;

        //Check whether this bin should be ignored or not
        bool ignore = false;
        for(size_t iline=0; iline < ignored_waves.size(); iline++)
            if(ignored_waves[iline]>= wave_bin_start && 
               ignored_waves[iline]<= wave_bin_end)
                ignore = true;
        for(size_t irange=0; irange < ignored_wave_ranges.size(); irange++)
        {
            double range_min = std::min(ignored_wave_ranges[irange].first,
                                        ignored_wave_ranges[irange].second);
            double range_max = std::max(ignored_wave_ranges[irange].first,
                                        ignored_wave_ranges[irange].second);
            
            if(wave_bin_start <= range_max && wave_bin_end >= range_min)
            {
                ignore = true;
                break;
            }
        }
        if(m_observed_flux[ibin] == 0)
            ignore = true;
        
        if(ignore)
        {
            m_response_fit[ibin] = 0;
            m_efficiency_fit[ibin] = 0;
        }
    }
    
    /* Fitting a cubic spline */
    mosca::vector_cubicspline_fit(m_response_fit, spline_knots, 0.01);
    mosca::vector_cubicspline_fit(m_efficiency_fit, spline_knots, 0.01);
}

std::vector<double>& mosca::response::wave()
{
    return m_wave;
}

std::vector<double>& mosca::response::efficiency_raw()
{
    return m_efficiency_raw;
}

std::vector<double>& mosca::response::efficiency_fit()
{
    return m_efficiency_fit;
}

std::vector<double>& mosca::response::response_fit()
{
    return m_response_fit;
}

std::vector<double>& mosca::response::response_raw()
{
    return m_response_raw;
}

std::vector<double>& mosca::response::observed_flux()
{
    return m_observed_flux;
}

std::vector<double>& mosca::response::tabulated_flux()
{
    return m_tabulated_flux;
}
