/* $Id: vector_utils.tcc,v 1.1 2013-07-24 07:44:56 cgarcia Exp $
 *
 * This file is part of the MOSCA library
 * Copyright (C) 2013 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*
 * $Author: cgarcia $
 * $Date: 2013-07-24 07:44:56 $
 * $Revision: 1.1 $
 * $Name: not supported by cvs2svn $
 */


#ifndef VECTOR_UTILS_TCC
#define VECTOR_UTILS_TCC

#include <stdexcept>
#include <algorithm>
#include "vector_utils.h"
#include "gsl/gsl_bspline.h"
#include "gsl/gsl_multifit.h"

/**
 * @brief
 *   Collapse a list of images in a container with median 
 *
 * @return The mean image
 *
 */
template<typename T>
void mosca::vector_divide(std::vector<T>& input,
                          std::vector<T>& input_err,
                          const std::vector<int>& dividend)
{
    if((input.size() != input_err.size()) || (input.size() != dividend.size()))
        throw std::invalid_argument("Vector sizes do not match");

    for(size_t i = 0 ; i < input.size(); ++i)
    {
        input[i] = input[i] / dividend[i];
        input_err[i] = input_err[i] / dividend[i];
    }    

}


/* TODO: It is wrong, it uses only the forward elements.
 * Use some cpl stuff here 
 */
template<typename T>
void mosca::vector_smooth(std::vector<T>& input,
                          std::vector<T>& input_err,
                          size_t smooth_size)
{
    if(input.size() != input_err.size())
        throw std::invalid_argument("Vector sizes do not match");

    mosca::vector_smooth(input, smooth_size);
    mosca::vector_smooth(input_err, smooth_size);
}


/* TODO: It is wrong, it uses only the forward elements.
 * Use some cpl stuff here 
 */
template<typename T>
void mosca::vector_smooth(std::vector<T>& input,
                          size_t smooth_size)
{
    if(smooth_size >= input.size())
        throw std::invalid_argument("Smooth size too large");

    cpl_vector * line = cpl_vector_new(input.size());
    for(size_t i = 0; i<input.size(); i++)
        cpl_vector_set(line, i, input[i]);
    
    //We smooth the line-image
    cpl_vector * smooth_line = 
            cpl_vector_filter_median_create(line, smooth_size);
    
    for(size_t i = 0; i<input.size(); i++)
        input[i]     = T(cpl_vector_get(smooth_line, i));
}

template<typename T>
void mosca::vector_cubicspline_fit(std::vector<T>& input, 
                                   size_t nknots, double threshold)
{
    
    //Allocate result
    size_t     nval = input.size();

    //This is valid for cubic splines, if not nbreak = ncoeffs + 2 - k, 
    //where k is the degree of the spline
    int ncoeffs = nknots + 2;
    
    /* Get the threshold in terms of the maximum */
    T max_value = *std::max_element(input.begin(), input.end());
    double thres_value = threshold * (double)max_value;
    
    /* Create a "mask" of pixels not to use */
    std::vector<bool> input_mask(nval);
    int nfit = 0;
    for (size_t i = 0; i < nval; ++i)
        if(input[i] < thres_value)
            input_mask[i] = false;
        else
        {
            input_mask[i] = true;
            nfit++;
        }

    /* allocate a cubic bspline workspace (k = 4) */
    gsl_bspline_workspace *bspl_wspc;
    gsl_vector *B;
    gsl_matrix * X;
    bspl_wspc = gsl_bspline_alloc(4, nknots);
    B = gsl_vector_alloc(ncoeffs);
    X = gsl_matrix_alloc(nfit, ncoeffs);
    
    /* allocate objects for the fitting */
    gsl_matrix *cov;
    gsl_vector * y_fit;
    gsl_vector * weigth;
    gsl_multifit_linear_workspace * mfit_wspc;
    gsl_vector *spl_coeffs;
    double chisq;
    y_fit = gsl_vector_alloc(nfit);
    weigth = gsl_vector_alloc(nfit);
    mfit_wspc = gsl_multifit_linear_alloc(nfit, ncoeffs);
    spl_coeffs = gsl_vector_alloc(ncoeffs);
    cov = gsl_matrix_alloc(ncoeffs, ncoeffs);
    
    /* use uniform breakpoints on [0, npix], which is the range of fitting */
    gsl_bspline_knots_uniform(0.0, (double)nval, bspl_wspc);
    

    /* construct the fit matrix X */
    for (size_t i = 0, ifit = 0; i < nval; ++i)
    {
        if(input_mask[i])
        {
            double xi = i;
            double yi = input[i];

            /* Fill the vector to fit */
            gsl_vector_set(y_fit, ifit, yi);
            gsl_vector_set(weigth, ifit, 1.);

            /* compute B_j(xi) for all j */
            gsl_bspline_eval(xi, B, bspl_wspc);

            /* fill in row i of X */
            for (int j = 0; j < ncoeffs; ++j)
            {
                double Bj = gsl_vector_get(B, j);
                gsl_matrix_set(X, ifit, j, Bj);
            }
            ifit++;
        }
    } 
    
    /* do the fit */
    gsl_multifit_wlinear(X, weigth, y_fit, spl_coeffs, cov, &chisq, mfit_wspc);
    
    /* output the fit */

    for(size_t i = 0; i < nval; i++)
    {
        double yi, yerr;
        gsl_bspline_eval((double)i, B, bspl_wspc);
        gsl_multifit_linear_est(B, spl_coeffs, cov, &yi, &yerr);
        input[i] = T(yi);
    }

    gsl_vector_free(y_fit);
    gsl_vector_free(weigth);
    gsl_vector_free(spl_coeffs);
    gsl_matrix_free(cov);
    gsl_multifit_linear_free(mfit_wspc);
}

/**
 * Smooth a vector fitting a polynomial 
 * @param input       Vector to be smoothed
 * @param polyorder   the order of the polynomial to fit
 * 
 * This function will smooth a MOS image in the dispersion direction 
 * (here assumed to be the X axis) fitting each row with a polynomial.
 * TODO: merge this with image_smooth_fit_1d_pol_spa() and place it in MOSCA.
 * TODO: It is named image_smooth_fit_1d_pol but it is not yet generalized
 * to any direction, only to spatial.   
 */
template<typename T>
void mosca::vector_pol_fit(std::vector<T>& input, int polyorder, 
                           double threshold)
{

    //Allocate result
    size_t     nval = input.size();

    /* Get the threshold in terms of the maximum */
    T max_value = *std::max_element(input.begin(), input.end());
    double thres_value = threshold * (double)max_value;
    
    /* Create a "mask" of pixels not to use */
    std::vector<bool> input_mask(nval);
    int nfit = 0;
    for (size_t i = 0; i < nval; ++i)
        if(input[i] < thres_value)
            input_mask[i] = false;
        else
        {
            input_mask[i] = true;
            nfit++;
        }

    cpl_vector * filtered_values = cpl_vector_new(nfit);
    cpl_vector * xpos = cpl_vector_new(nfit);
    for (size_t i = 0, ifit = 0; i < nval; ++i)
    {
        if(input_mask[i])
        {
            cpl_vector_set(filtered_values, (cpl_size)ifit, input[i]);
            cpl_vector_set(xpos, ifit, (double)i);
            ifit++;
        }
    }
 
    cpl_polynomial * pol_fit = cpl_polynomial_fit_1d_create
            (xpos, filtered_values, polyorder, NULL);
    
    if (pol_fit)
    {
        for(size_t i = 0; i < nval; i++)
            input[i] = T(cpl_polynomial_eval_1d(pol_fit, (double)i, NULL));
    }
    else 
        std::fill(input.begin(), input.end(), T()); 
    
    cpl_vector_delete(filtered_values);
    cpl_vector_delete(xpos);
    cpl_polynomial_delete(pol_fit);
}

#endif
