/* $Id: kmo_priv_multi_reconstruct.c,v 1.31 2013-09-17 12:12:36 aagudo Exp $
 *
 * This file is part of the KMOS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: aagudo $
 * $Date: 2013-09-17 12:12:36 $
 * $Revision: 1.31 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 *                              Includes
 *----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>

#ifdef __USE_XOPEN2K
#include <stdlib.h>
#define GGG
#else
#define __USE_XOPEN2K /* to get the definition for setenv in stdlib.h */
#include <stdlib.h>
#undef __USE_XOPEN2K
#endif

#include <cpl.h>

#include "kmclipm_constants.h"
#include "kmclipm_math.h"
#include "kmclipm_functions.h"
#include "kmclipm_priv_reconstruct.h"

#include "kmo_priv_reconstruct.h"
#include "kmo_priv_multi_reconstruct.h"
#include "kmo_functions.h"
#include "kmo_priv_functions.h"
#include "kmo_dfs.h"
#include "kmo_error.h"
#include "kmo_priv_fit_profile.h"
#include "kmo_priv_shift.h"
#include "kmo_priv_noise_map.h"

/*----------------------------------------------------------------------------*/
/**
    @defgroup kmos_priv_multi_reconstruct     Helper functions for recipe kmo_multi_reconstruct.

    @{
 */
/*----------------------------------------------------------------------------*/

/**

*/
double kmo_mr_get_rot_angle(cpl_frame *frame)
{
    double rot_angle;
    cpl_propertylist *header;

    KMO_TRY
    {
        KMO_TRY_EXIT_IF_NULL(
            header = kmclipm_propertylist_load(cpl_frame_get_filename(frame), 0));
        KMO_TRY_EXIT_IF_ERROR(
            rot_angle = cpl_propertylist_get_double(header, ROTANGLE));

        kmclipm_strip_angle(&rot_angle);
    } KMO_CATCH {
        KMO_CATCH_MSG();
    }

    cpl_propertylist_delete(header); header = NULL;

    return rot_angle;
}

/**

*/
cpl_imagelist** kmo_mr_create_datacubes(armNameStruct *arm_name_struct,
                                        int arm_id,
                                        cpl_frameset *frameset,
                                        const gridDefinition gd,
                                        int xcal_interpolation)
{
    cpl_imagelist       *cube_data              = NULL,
                        *cube_noise             = NULL,
                        **ret_cube_list         = NULL;
    cpl_frame           *xcal_frame             = NULL,
                        *ycal_frame             = NULL,
                        *lcal_frame             = NULL;
    cpl_propertylist    *main_header            = NULL;
    int                 ix                      = 0,
                        iy                      = 0,
                        ifu_nr                  = 0,
                        nr_frames               = 0,
                        cnt                     = 0,
                        *bounds                 = NULL;

    KMO_TRY
    {
        //
        // check inputs
        //
        KMO_TRY_ASSURE((frameset != NULL) ||
                       (arm_name_struct != NULL),
                       CPL_ERROR_ILLEGAL_INPUT,
                       "Some inputs are NULL!");

        nr_frames = arm_name_struct->namesCnt[arm_id-1];

        KMO_TRY_EXIT_IF_NULL(
            ret_cube_list = cpl_calloc(nr_frames, sizeof(cpl_imagelist*)));

        KMO_TRY_EXIT_IF_NULL(
            xcal_frame = kmo_dfs_get_frame(frameset, XCAL));
        KMO_TRY_EXIT_IF_NULL(
            ycal_frame = kmo_dfs_get_frame(frameset, YCAL));
        KMO_TRY_EXIT_IF_NULL(
            lcal_frame = kmo_dfs_get_frame(frameset, LCAL));

        // get left and right bounds of IFUs
        KMO_TRY_EXIT_IF_NULL(
            main_header = kmo_dfs_load_primary_header(frameset, XCAL));
        KMO_TRY_EXIT_IF_NULL(
            bounds = kmclipm_extract_bounds(main_header));
        cpl_propertylist_delete(main_header); main_header = NULL;

        for (iy = 0; iy < arm_name_struct->size; iy++) {
            for (ix = 0; ix < KMOS_NR_IFUS; ix++) {
                ifu_nr = ix+1;
                if (arm_name_struct->name_ids[ix+iy*KMOS_NR_IFUS] == arm_id) {
                    KMO_TRY_EXIT_IF_ERROR(
                        kmo_reconstruct_sci(ifu_nr,
                                            bounds[2*(ifu_nr-1)],
                                            bounds[2*(ifu_nr-1)+1],
                                            arm_name_struct->obj_sky_struct->table[iy].objFrame,
                                            SCIENCE,
                                            NULL,
                                            NULL,
                                            NULL,
                                            xcal_frame,
                                            ycal_frame,
                                            lcal_frame,
                                            NULL,
                                            NULL,
                                            &gd,
                                            &cube_data,
                                            &cube_noise,
                                            FALSE,
                                            FALSE,
                                            xcal_interpolation));

                    ret_cube_list[cnt] = cube_data;
                    cpl_imagelist_delete(cube_noise); cube_noise = NULL;
                    cnt++;
                }
            } // for (ix)
        } // for (iy)
    } KMO_CATCH {
        KMO_CATCH_MSG();
        cnt = 0;
        for (cnt = 0; cnt < nr_frames; cnt++) {
            cpl_imagelist_delete(ret_cube_list[cnt]); ret_cube_list[cnt] = NULL;
        }
        cpl_free(ret_cube_list); ret_cube_list = NULL;
    }

    cpl_free(bounds); bounds = NULL;

    return ret_cube_list;
}

/**

*/
cpl_propertylist** kmo_mr_get_headers(armNameStruct *arm_name_struct,
                                      int arm_id,
                                      cpl_frameset *frameset,
                                      const gridDefinition gd)
{
    int                 ix                  = 0,
                        iy                  = 0,
                        cnt                 = 0,
                        ifu_nr              = 0,
                        det_nr              = 0,
                        nr_frames           = 0;
    cpl_propertylist    *main_header        = 0,
                        **ret_sub_headers   = NULL;
    const char          *fn_obj             = NULL;

    KMO_TRY
    {
        //
        // check inputs
        //
        KMO_TRY_ASSURE((frameset != NULL) ||
                       (arm_name_struct != NULL),
                       CPL_ERROR_ILLEGAL_INPUT,
                       "Some inputs are NULL!");

        nr_frames = arm_name_struct->namesCnt[arm_id-1];

        KMO_TRY_EXIT_IF_NULL(
            ret_sub_headers = cpl_calloc(nr_frames, sizeof(cpl_propertylist*)));

        //
        // extract sub-headers for each exposures, calculate WCS
        //
        for (iy = 0; iy < arm_name_struct->size; iy++) {
            KMO_TRY_EXIT_IF_NULL(
                fn_obj = cpl_frame_get_filename(arm_name_struct->obj_sky_struct->table[iy].objFrame));
            KMO_TRY_EXIT_IF_NULL(
                main_header = kmclipm_propertylist_load(fn_obj, 0));

            for (ix = 0; ix < KMOS_NR_IFUS; ix++) {
                ifu_nr = ix + 1;
                det_nr = (ifu_nr - 1)/KMOS_IFUS_PER_DETECTOR + 1;

                if (arm_name_struct->name_ids[ix+iy*KMOS_NR_IFUS] == arm_id) {
                    KMO_TRY_EXIT_IF_NULL(
                        ret_sub_headers[cnt] = kmclipm_propertylist_load(fn_obj, det_nr));

                    KMO_TRY_EXIT_IF_ERROR(
                         kmclipm_update_property_int(ret_sub_headers[cnt],
                                                     NAXIS, 3, "number of data axes"));
                    KMO_TRY_EXIT_IF_ERROR(
                         kmclipm_update_property_int(ret_sub_headers[cnt],
                                                     NAXIS1, gd.x.dim, "length of data axis 1"));
                    KMO_TRY_EXIT_IF_ERROR(
                         kmclipm_update_property_int(ret_sub_headers[cnt],
                                                     NAXIS2, gd.y.dim, "length of data axis 2"));
                    KMO_TRY_EXIT_IF_ERROR(
                         kmclipm_update_property_int(ret_sub_headers[cnt],
                                                     NAXIS3, gd.l.dim, "length of data axis 3"));

                    KMO_TRY_EXIT_IF_ERROR(
                         kmo_calc_wcs_gd(main_header, ret_sub_headers[cnt], ifu_nr, gd));

                    cnt++;
                }
            } // for (ix)
            cpl_propertylist_delete(main_header); main_header = NULL;
        }  // for (iy)
    } KMO_CATCH {
        KMO_CATCH_MSG();
        cnt = 0;
        for (cnt = 0; cnt < nr_frames; cnt++) {
            cpl_propertylist_delete(ret_sub_headers[cnt]); ret_sub_headers[cnt] = NULL;
        }
        cpl_free(ret_sub_headers); ret_sub_headers = NULL;
    }

    return ret_sub_headers;
}

/**
    @brief
        Calculate shift offsets for all objects.

    @param arm_name_struct
    @param nr_science_frames
    @param gd
    @param method           Shift method (none, center, header, file)
    @param imethod
    @param neighborhoodRange
    @param filename         Filename for user-defined shifts
    @param frameset
    @param fmethod
    @param cmethod
    @param cpos_rej
    @param cneg_rej
    @param citer
    @param cmin
    @param cmax
    @param unused_ifus
    @param pix_scale
    @param no_subtract
    @param user_defined_ifu
    @param dev_cal
    @param xshifts          (Output) Has to be deallocated again
    @param yshifts          (Output) Has to be deallocated again

    @return
        a three dimensional array ([x][y][z] as defined in the grid definition) of neighbor lists

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT     if @c sampleList or is NULL.
    @li CPL_ERROR_ILLEGAL_INPUT  if the size of @c vec and @c index_vec isn't
                                 the same or if @c nr_min or @c nr_max are
                                negative.
*/
cpl_error_code kmo_mr_get_offsets(armNameStruct *arm_name_struct,
                                  int arm_id,
                                  const char *method,
                                  const char *imethod,
                                  const char *filename,
                                  cpl_frameset *frameset,
                                  cpl_imagelist **data_cube_list,
                                  cpl_propertylist **sub_headers,
                                  const char *fmethod,
                                  const char *cmethod,
                                  double cpos_rej,
                                  double cneg_rej,
                                  int citer,
                                  int cmin,
                                  int cmax,
                                  int dev_cal,
                                  char *mapping_mode,
                                  double **xshifts,
                                  double **yshifts)
{
    cpl_error_code      error                   = CPL_ERROR_NONE;
    double              *pxshifts               = NULL,
                        *pyshifts               = NULL,
                        xref                    = 0.,
                        yref                    = 0.,
                        cd1_1                   = 0.0,
                        cd1_2                   = 0.0,
                        ang1                    = 0.0,
                        ang2                    = 0.0;
    int                 ix                      = 0,
                        iy                      = 0,
                        ifu_nr                  = 0,
                        nr_frames               = 0,
                        cnt                     = 0,
                        *exposure_ifus          = NULL;
    const char          **exposure_filename     = NULL;
    const char          *fn_obj                 = NULL;
    cpl_matrix          *phys_ref               = NULL,
                        *world                  = NULL,
                        *phys                   = NULL;
    cpl_wcs             *wcs_ref                = NULL,
                        *wcs                    = NULL;
    cpl_array           *status                 = NULL;
    cpl_bivector        *shifts                 = NULL;
    cpl_vector          *identified             = NULL,
                        *fit_pars               = NULL;
    cpl_image           *tmp_img                = NULL,
                        *tmp_img2               = NULL;

    KMO_TRY
    {
        //
        // check inputs
        //
        KMO_TRY_ASSURE((xshifts != NULL) ||
                       (yshifts != NULL) ||
                       (method != NULL) ||
                       (imethod != NULL) ||
                       (frameset != NULL) ||
                       (arm_name_struct != NULL) ||
                       (fmethod != NULL) ||
                       (cmethod != NULL) ||
                       (xshifts != NULL) ||
                       (yshifts != NULL) ||
                       (sub_headers != NULL),
                       CPL_ERROR_ILLEGAL_INPUT,
                       "Some inputs are NULL!");

        nr_frames = arm_name_struct->namesCnt[arm_id-1];

        //
        // allocate arrays
        //
        KMO_TRY_EXIT_IF_NULL(
            *xshifts = cpl_calloc(nr_frames, sizeof(double)));
        KMO_TRY_EXIT_IF_NULL(
            *yshifts = cpl_calloc(nr_frames, sizeof(double)));
        KMO_TRY_EXIT_IF_NULL(
            exposure_filename = cpl_calloc(nr_frames, sizeof(char *)));
        KMO_TRY_EXIT_IF_NULL(
            exposure_ifus = cpl_calloc(nr_frames, sizeof(int)));

        //
        // extract sub-headers for each exposures, calculate WCS
        //
        for (iy = 0; iy < arm_name_struct->size; iy++) {
            KMO_TRY_EXIT_IF_NULL(
                fn_obj = cpl_frame_get_filename(arm_name_struct->obj_sky_struct->table[iy].objFrame));

            for (ix = 0; ix < KMOS_NR_IFUS; ix++) {
                ifu_nr = ix + 1;

                if (arm_name_struct->name_ids[ix+iy*KMOS_NR_IFUS] == arm_id) {
                    exposure_ifus[cnt] = ifu_nr;
                    KMO_TRY_EXIT_IF_NULL(
                        exposure_filename[cnt] = fn_obj);

                    cnt++;
                }
            } // for (ix)
        }  // for (iy)

        //
        // check rotation angle
        //
        if (dev_cal == FALSE) {
            cd1_1 = kmo_dfs_get_property_double(sub_headers[0], CD1_1);
            cd1_2 = kmo_dfs_get_property_double(sub_headers[0], CD1_2);
            KMO_TRY_CHECK_ERROR_STATE();

            ang1 = atan(cd1_2/cd1_1)*180/CPL_MATH_PI;

            for (cnt = 1; cnt < nr_frames; cnt++) {
                cd1_1 = kmo_dfs_get_property_double(sub_headers[cnt], CD1_1);
                cd1_2 = kmo_dfs_get_property_double(sub_headers[cnt], CD1_2);
                KMO_TRY_CHECK_ERROR_STATE();
                ang2 = atan(cd1_2/cd1_1)*180/CPL_MATH_PI;

                if (strcmp(method, "none") != 0) {
                    // center, header, user
                    KMO_TRY_ASSURE(fabs(ang1-ang2) <= 0.5,
                                   CPL_ERROR_ILLEGAL_INPUT,
                                   "Orientation of IFU %d in %s (%.1fdeg) "
                                   "and IFU %d in %s (%.1fdeg) differ! "
                                   "Align the orientation of this cube with "
                                   "kmo_rotate before applying this recipe.",
                                   exposure_ifus[0], exposure_filename[0], ang1,
                                   exposure_ifus[cnt], exposure_filename[cnt], ang2);
                } else {
                    // none
                    if (fabs(ang1-ang2) > 0.5) {
                        cpl_msg_warning("",
                                        "Orientation of IFU %d in %s (%.1fdeg) "
                                        "and IFU %d in %s (%.1fdeg) differ! "
                                        "Processing anyway.",
                                        exposure_ifus[0], exposure_filename[0], ang1,
                                        exposure_ifus[cnt], exposure_filename[cnt], ang2);
                    }
                }
            }
        }

        //
        // some preliminaries on shift vectors
        //
        if (strcmp(method, "header") == 0) {
            // fill shift vector
             phys_ref = cpl_matrix_new (1, 3);
             cpl_matrix_set(phys_ref, 0, 0, 1);  // lower left corner
             cpl_matrix_set(phys_ref, 0, 1, 1);
             cpl_matrix_set(phys_ref, 0, 2, 1);

             KMO_TRY_EXIT_IF_NULL(
                 wcs_ref = cpl_wcs_new_from_propertylist(sub_headers[0]));

             KMO_TRY_EXIT_IF_ERROR(
                 cpl_wcs_convert(wcs_ref, phys_ref, &world, &status, CPL_WCS_PHYS2WORLD));
             cpl_matrix_delete(phys_ref); phys_ref = NULL;
             cpl_array_delete(status); status = NULL;
        } else if (strcmp(method, "user") == 0) {
            KMO_TRY_EXIT_IF_NULL(
                shifts = cpl_bivector_read(filename));
            KMO_TRY_ASSURE(nr_frames - 1 == cpl_bivector_get_size(shifts),
                           CPL_ERROR_ILLEGAL_INPUT,
                           "Number of identified frames in sof-file (%d) "
                           "with identified objects doesn't "
                           "match the number of pairs shift values in "
                           "provided file (%lld)! For n pairs of shift "
                           "values, n+1 frames are expected.",
                           nr_frames, cpl_bivector_get_size(shifts));

            KMO_TRY_EXIT_IF_NULL(
                pxshifts = cpl_bivector_get_x_data(shifts));
            KMO_TRY_EXIT_IF_NULL(
                pyshifts = cpl_bivector_get_y_data(shifts));
        } else if (strcmp(method, "center") == 0) {
            KMO_TRY_ASSURE(data_cube_list != NULL,
                           CPL_ERROR_ILLEGAL_INPUT,
                           "Some inputs are NULL!");

            KMO_TRY_EXIT_IF_NULL(
                identified = cpl_vector_new(cpl_imagelist_get_size(data_cube_list[0])));

            KMO_TRY_EXIT_IF_ERROR(
                cpl_vector_fill(identified, 1.0));

            KMO_TRY_EXIT_IF_ERROR(
                kmclipm_make_image(data_cube_list[0],
                                   NULL,
                                   &tmp_img,
                                   NULL,
                                   identified,
                                   cmethod,
                                   cpos_rej,
                                   cneg_rej,
                                   citer,
                                   cmax,
                                   cmin));

            KMO_TRY_EXIT_IF_NULL(
                fit_pars = kmo_fit_profile_2D(tmp_img, NULL, fmethod, &tmp_img2, NULL));

            xref = cpl_vector_get(fit_pars, 2);
            yref = cpl_vector_get(fit_pars, 3);
            KMO_TRY_CHECK_ERROR_STATE();

            cpl_image_delete(tmp_img); tmp_img = NULL;
            cpl_image_delete(tmp_img2); tmp_img2 = NULL;
            cpl_vector_delete(fit_pars); fit_pars = NULL;
        }

        //
        // fill shift vectors
        //
        (*xshifts)[0] = 0.0;
        (*yshifts)[0] = 0.0;
        for (cnt = 1; cnt < nr_frames; cnt++) {
            if (strcmp(method, "none") == 0) {
                (*xshifts)[cnt] = 0.0;
                (*yshifts)[cnt] = 0.0;
            } else if (strcmp(method, "user") == 0) {
                (*xshifts)[cnt] = pxshifts[cnt-1];
                (*yshifts)[cnt] = pyshifts[cnt-1];
            } else if (strcmp(method, "header") == 0) {
                KMO_TRY_EXIT_IF_NULL(
                    wcs = cpl_wcs_new_from_propertylist(sub_headers[cnt]));

                KMO_TRY_EXIT_IF_ERROR(
                    cpl_wcs_convert(wcs, world, &phys , &status, CPL_WCS_WORLD2PHYS));
                cpl_array_delete(status); status = NULL;

                (*xshifts)[cnt] = cpl_matrix_get(phys, 0, 0) - 1;
                (*yshifts)[cnt] = -1 * (cpl_matrix_get(phys, 0, 1) - 1);
                cpl_wcs_delete(wcs); wcs = NULL;
                cpl_matrix_delete(phys); phys = NULL;
            } else if (strcmp(method, "center") == 0) {
                KMO_TRY_EXIT_IF_ERROR(
                    cpl_vector_fill(identified, 1.0));

                KMO_TRY_EXIT_IF_ERROR(
                    kmclipm_make_image(data_cube_list[cnt],
                                       NULL,
                                       &tmp_img,
                                       NULL,
                                       identified,
                                       cmethod,
                                       cpos_rej,
                                       cneg_rej,
                                       citer,
                                       cmax,
                                       cmin));

                KMO_TRY_EXIT_IF_NULL(
                    fit_pars = kmo_fit_profile_2D(tmp_img, NULL, fmethod, &tmp_img2, NULL));

                double x2 = cpl_vector_get(fit_pars, 2);
                double y2 = cpl_vector_get(fit_pars, 3);
                KMO_TRY_CHECK_ERROR_STATE();

                (*xshifts)[cnt] = x2 - xref;
                (*yshifts)[cnt] = yref - y2;

                cpl_image_delete(tmp_img); tmp_img = NULL;
                cpl_image_delete(tmp_img2); tmp_img2 = NULL;
                cpl_vector_delete(fit_pars); fit_pars = NULL;
            }
        }

        //
        // print info
        //
        cpl_msg_info("","Processing object name: '%s'", arm_name_struct->names[arm_id-1]);
        for (cnt = 0; cnt < nr_frames; cnt++) {
            cpl_msg_info("","   [%s, IFU %2d] x: %20.9g\t  y: %20.9g",
                             exposure_filename[cnt], exposure_ifus[cnt], (*xshifts)[cnt], (*yshifts)[cnt]);
            if (mapping_mode == NULL) {
                if (!(((*xshifts)[cnt] < KMOS_SLITLET_X) && ((*xshifts)[cnt] > -KMOS_SLITLET_X))) {
                    cpl_msg_warning("","      X-shift for this IFU is larger than 14 pix!");
                }
                if (!(((*yshifts)[cnt] < KMOS_SLITLET_Y) && ((*yshifts)[cnt] > -KMOS_SLITLET_Y))) {
                    cpl_msg_warning("","      Y-shift for this IFU is larger than 14 pix!");
                }
            }
        }
    } KMO_CATCH {
        KMO_CATCH_MSG();

        error = cpl_error_get_code();

        cpl_free(*xshifts); *xshifts = NULL;
        cpl_free(*yshifts); *yshifts = NULL;
    }

    if (exposure_ifus != NULL)     { cpl_free(exposure_ifus); exposure_ifus = NULL; }
    if (exposure_filename != NULL) { cpl_free(exposure_filename); exposure_filename = NULL; }
    if (shifts != NULL)            { cpl_bivector_delete(shifts); }
    if (wcs_ref != NULL)           { cpl_wcs_delete(wcs_ref); }
    if (world != NULL)             { cpl_matrix_delete(world); }
    if (identified != NULL)        { cpl_vector_delete(identified); }

    return error;
}

/**

*/
cpl_error_code kmo_mr_load_super_image(int ix,
                                       cpl_image *superImg,
                                       cpl_image *origImg,
                                       int *image_offsets,
                                       int ifu_nr,
                                       int *bounds)
{
    cpl_error_code  err         = CPL_ERROR_NONE;
    cpl_image       *extractImg = NULL;
    int             ysize       = 0,
                    left        = 0,
                    right       = 0;

    KMO_TRY
    {
        KMO_TRY_ASSURE((superImg != NULL) &&
                       (origImg != NULL),
                       CPL_ERROR_ILLEGAL_INPUT,
                       "Some inputs are NULL!");

        left = bounds[2*(ifu_nr-1)];
        right = bounds[2*(ifu_nr-1)+1];
        ysize = cpl_image_get_size_y(superImg);

        KMO_TRY_EXIT_IF_NULL(
            extractImg = cpl_image_extract(origImg, left, 1, right, ysize));

        KMO_TRY_EXIT_IF_ERROR(
            cpl_image_copy(superImg, extractImg, image_offsets[ix]+1, 1));
    }
    KMO_CATCH
    {
        KMO_CATCH_MSG();
        err = cpl_error_get_code();
    }

    if (extractImg != NULL) {
        cpl_image_delete(extractImg);
    }

    return err;
}

cpl_image* kmo_new_xcal_index(int new_ifu_nr,
                              cpl_image* xcalImg)
{
    float   *pxcalImg   = NULL,
            new_index   = 0;
    int     ix          = 0,
            iy          = 0,
            nx          = 0,
            ny          = 0;

    KMO_TRY
    {
        KMO_TRY_EXIT_IF_NULL(
            pxcalImg = cpl_image_get_data_float(xcalImg));

        if ((new_ifu_nr > 0) && (new_ifu_nr < 10)) {
            new_index = new_ifu_nr/10.;
        } else if ((new_ifu_nr >= 10) && (new_ifu_nr < 100)) {
            new_index = new_ifu_nr/100.;
        }

        nx = cpl_image_get_size_x(xcalImg);
        ny = cpl_image_get_size_y(xcalImg);

        for (ix = 0; ix < nx; ix++) {
            for (iy = 0; iy < ny; iy++) {
                if ((fabs(pxcalImg[ix+iy*nx]) > 0.0001) && !kmclipm_is_nan_or_inf(pxcalImg[ix+iy*nx])) {
                    // remove old index
                    pxcalImg[ix+iy*nx] = (int)(pxcalImg[ix+iy*nx]);

                    // add in new index
                    if (pxcalImg[ix+iy*nx] >= 0.0) {
                        pxcalImg[ix+iy*nx] += new_index;
                    } else {
                        pxcalImg[ix+iy*nx] = pxcalImg[ix+iy*nx]-new_index;
                    }
                }
            }
        }
    }
    KMO_CATCH
    {
        KMO_CATCH_MSG();
    }

    return xcalImg;
}

/** @{ */
