/* $Id: visir_util_spc_txt2fits.c,v 1.46 2009/01/29 08:56:58 llundin Exp $
 *
 * This file is part of the VISIR Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA
 */

/*
 * $Author: llundin $
 * $Date: 2009/01/29 08:56:58 $
 * $Revision: 1.46 $
 * $Name: visir-3_5_1 $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <string.h>

#include "visir_recipe.h"

/*-----------------------------------------------------------------------------
                            Recipe defines
 -----------------------------------------------------------------------------*/

#define RECIPE_STRING "visir_util_spc_txt2fits"

/*-----------------------------------------------------------------------------
                            Private Functions prototypes
 -----------------------------------------------------------------------------*/
static cpl_error_code visir_util_spc_txt2fits_save(cpl_frameset *,
                                                   const cpl_parameterlist *,
                                                   const char *,
                                                   const cpl_table *);

VISIR_RECIPE_DEFINE(visir_util_spc_txt2fits, VISIR_PARAM_QEFF,
                    "Generate spectrum calibration FITS tables",
                    "This recipe shall be used to generate spectrum "
                    "calibration tables.\n"
                    "The sof file shall consist of 1 line with the name of an "
                    "ASCII-file\n"
                    "currently tagged with either " VISIR_SPC_LINES_ASCII " or "
                    VISIR_SPC_QEFF_ASCII ".\n"
                    "The file must comprise two columns:\n"
                    "1st: Must be wavelengths in increasing order in units "
                    "of meter\n"
                    "2nd: For " VISIR_SPC_LINES_ASCII "-files must be the "
                    "atmospheric emission, "
                    "while\n"
                    "     for " VISIR_SPC_QEFF_ASCII "-files must be the "
                    "quantum efficiency of "
                    "the detector.\n"
                    "A " VISIR_SPC_LINES_ASCII "-file will generate a "
                    VISIR_CALIB_LINES_SPC
                    "-file, and \n"
                    "a " VISIR_SPC_QEFF_ASCII "-file will generate a "
                    VISIR_CALIB_QEFF_SPC
                    "-file.\n"
                    "The current " VISIR_CALIB_LINES_SPC "- and "
                    VISIR_CALIB_QEFF_SPC
                    "-files are\n"
                    "generated using the ASCII-files in the catalogs/ "
                    "directory of the VISIR\n"
                    "source-code distribution.");

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/

static struct {
    /* Inputs */
    double qeff;

} visir_util_spc_txt2fits_config;

/*----------------------------------------------------------------------------*/
/**
 * @defgroup visir_util_spc_txt2fits   Generate spectrum calibration FITS tables
 */
/*----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                Functions code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    The FITS file creation occurs here 
  @param    framelist   the frames list
  @return   0 iff everything is ok

  The recipe expects a text file with two columns, and will create a FITS file
  out of it. The default column labels and the PRO CATG may be modified on the
  command line. 
 */
/*----------------------------------------------------------------------------*/
static int visir_util_spc_txt2fits(cpl_frameset            * framelist,
                                   const cpl_parameterlist * parlist)
{
    const char       * label2 = NULL; /* Avoid uninit warning */
    const char       * procat;
    irplib_framelist * allframes = NULL;
    irplib_framelist * rawframes = NULL;
    const cpl_frame  * frame;
    cpl_bivector     * spectrum = NULL;
    cpl_table        * tab = NULL;
    FILE             * stream = NULL;
    int                nvals;
    

    if (cpl_error_get_code()) return cpl_error_get_code();

    /* Identify the RAW frames in the input frameset */
    skip_if (visir_dfs_set_groups(framelist));

    /* Objects observation */
    allframes = irplib_framelist_cast(framelist);
    skip_if(allframes == NULL);
    rawframes = irplib_framelist_extract_regexp(allframes, "^("
                                                VISIR_SPC_LINES_ASCII "|"
                                                VISIR_SPC_QEFF_ASCII ")$",
                                                CPL_FALSE);
    skip_if (rawframes == NULL);

    frame = irplib_framelist_get_const(rawframes, 0);

    stream = fopen(cpl_frame_get_filename(frame), "r");

    skip_if (stream == NULL);

    spectrum = cpl_bivector_new(2481); /* Some 1st guess at actual length */
    skip_if( visir_bivector_load(spectrum, stream));

    if (strcmp(cpl_frame_get_tag(frame), VISIR_SPC_LINES_ASCII)) {
        /* The file is a quantum efficiency file */

        /* Retrieve input parameters */
        visir_util_spc_txt2fits_config.qeff =
            visir_parameterlist_get_double(parlist, RECIPE_STRING, VISIR_PARAM_QEFF);

        if (visir_util_spc_txt2fits_config.qeff > 1) {
            cpl_msg_error(cpl_func, "--qeff efficiency may not exceed 1");
            skip_if (1);
        }
        if (visir_util_spc_txt2fits_config.qeff <= 0) {
            cpl_msg_error(cpl_func, "--qeff efficiency must be positive");
            skip_if (1);
        }

        procat = VISIR_SPEC_CAL_QEFF_PROCATG;
        label2 = "Efficiency";

       skip_if(cpl_vector_multiply_scalar(cpl_bivector_get_y(spectrum), 
                                          visir_util_spc_txt2fits_config.qeff));

    } else {
        /* The file is a sky lines file */
       procat = VISIR_SPEC_CAL_LINES_PROCATG;
       label2 = "Emission";
    }

    /* Get the number of values in the model */
    nvals = cpl_bivector_get_size(spectrum);

    /* Allocate the data container */
    tab = cpl_table_new(nvals);

    skip_if( tab == NULL );

    skip_if (cpl_table_wrap_double(tab, cpl_bivector_get_x_data(spectrum),
                                   "Wavelength"));
    skip_if (cpl_table_set_column_unit(tab, "Wavelength", "m"));

    skip_if (cpl_table_wrap_double(tab, cpl_bivector_get_y_data(spectrum),
                                   label2));

    skip_if (cpl_table_set_column_unit(tab, label2, "[0;1] (Unitless)"));

    /* Save the table */
    cpl_msg_info(cpl_func, "Saving the table with %d rows", nvals);
    skip_if (visir_util_spc_txt2fits_save(framelist, parlist, procat, tab));

    end_skip;

    if (stream != NULL) fclose(stream);
    cpl_bivector_delete(spectrum);
    if (tab) {
        cpl_table_unwrap(tab, "Wavelength");
        cpl_table_unwrap(tab, label2);
        cpl_table_delete(tab);
    }
    irplib_framelist_delete(allframes);
    irplib_framelist_delete(rawframes);

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Save the visir_util_spc_txt2fits recipe products on disk
  @param    set      the input frame set
  @param    parlist  the input list of parameters
  @param    procat   the value of the PROCATG card
  @param    tab      the table to save
  @return   0 iff everything is ok
 
  The table contains 2 columns.
 */
/*----------------------------------------------------------------------------*/
static
cpl_error_code visir_util_spc_txt2fits_save(cpl_frameset            * set,
                                            const cpl_parameterlist * parlist,
                                            const char              * procat,
                                            const cpl_table         * tab)
{

    cpl_propertylist * applist = cpl_propertylist_new();


    bug_if(cpl_propertylist_append_string(applist, "INSTRUME", "VISIR"));

    skip_if (irplib_dfs_save_table(set, parlist, set, tab, NULL, RECIPE_STRING, procat,
                               applist, NULL, visir_pipe_id,
                               RECIPE_STRING CPL_DFS_FITS));

    end_skip;

    cpl_propertylist_delete(applist);

    return cpl_error_get_code();
}
