/* $Id: cpl_array.h,v 1.20 2011/01/12 09:28:46 cizzo Exp $
 *
 * This file is part of the ESO Common Pipeline Library
 * Copyright (C) 2001-2006 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: cizzo $
 * $Date: 2011/01/12 09:28:46 $
 * $Revision: 1.20 $
 * $Name: cpl-5_3_0-BRANCH $
 */

#ifndef CPL_ARRAY_H
#define CPL_ARRAY_H

#include <stdio.h>
#include <complex.h>
#include <cpl_macros.h>
#include <cpl_type.h>
#include <cpl_error.h>

CPL_BEGIN_DECLS


typedef struct _cpl_array_ cpl_array;

cpl_array *cpl_array_new(int, cpl_type);
cpl_array *cpl_array_wrap_int(int *, int);
cpl_array *cpl_array_wrap_float(float *, int);
cpl_array *cpl_array_wrap_double(double *, int);
#ifdef _Complex_I
cpl_array *cpl_array_wrap_float_complex(float complex *, int);
cpl_array *cpl_array_wrap_double_complex(double complex *, int);
#endif
cpl_array *cpl_array_wrap_string(char **, int);

cpl_error_code cpl_array_copy_data(cpl_array *, const double *);
cpl_error_code cpl_array_copy_data_int(cpl_array *, const int *);
cpl_error_code cpl_array_copy_data_float(cpl_array *, const float *);
cpl_error_code cpl_array_copy_data_double(cpl_array *, const double *);
cpl_error_code cpl_array_copy_data_string(cpl_array *, const char **);
#ifdef _Complex_I
cpl_error_code cpl_array_copy_data_complex(cpl_array *, const double complex *);
cpl_error_code cpl_array_copy_data_float_complex(cpl_array *,
                                                 const float complex *);
cpl_error_code cpl_array_copy_data_double_complex(cpl_array *,
                                                  const double complex *);
#endif
void cpl_array_delete(cpl_array *);
void *cpl_array_unwrap(cpl_array *);
int cpl_array_get_size(const cpl_array *);
cpl_error_code cpl_array_set_size(cpl_array *, int);
cpl_type cpl_array_get_type(const cpl_array *);
int cpl_array_has_invalid(const cpl_array *);
int cpl_array_has_valid(const cpl_array *);
int cpl_array_is_valid(const cpl_array *, int);
int cpl_array_count_invalid(const cpl_array *);
int *cpl_array_get_data_int(cpl_array *);
const int *cpl_array_get_data_int_const(const cpl_array *);
float *cpl_array_get_data_float(cpl_array *);
const float *cpl_array_get_data_float_const(const cpl_array *);
double *cpl_array_get_data_double(cpl_array *);
const double *cpl_array_get_data_double_const(const cpl_array *);
#ifdef _Complex_I
float complex *cpl_array_get_data_float_complex(cpl_array *);
const float complex *cpl_array_get_data_float_complex_const(const cpl_array *);
const double complex *cpl_array_get_data_double_complex_const
                                                      (const cpl_array *);
double complex *cpl_array_get_data_double_complex(cpl_array *);
#endif
char **cpl_array_get_data_string(cpl_array *);
const char **cpl_array_get_data_string_const(const cpl_array *);
double cpl_array_get(const cpl_array *, int, int *);
int cpl_array_get_int(const cpl_array *, int, int *);
float cpl_array_get_float(const cpl_array *, int, int *);
double cpl_array_get_double(const cpl_array *, int, int *);
const char *cpl_array_get_string(const cpl_array *, int);
#ifdef _Complex_I
double complex cpl_array_get_complex(const cpl_array *, int, int *);
float complex cpl_array_get_float_complex(const cpl_array *, int, int *);
double complex cpl_array_get_double_complex(const cpl_array *, int, int *);
#endif
cpl_error_code cpl_array_set(cpl_array *array, int, double);
cpl_error_code cpl_array_set_int(cpl_array *, int, int);
cpl_error_code cpl_array_set_float(cpl_array *, int, float);
cpl_error_code cpl_array_set_double(cpl_array *, int, double);
cpl_error_code cpl_array_set_string(cpl_array *, int, const char *);
cpl_error_code cpl_array_set_invalid(cpl_array *, int);
#ifdef _Complex_I
cpl_error_code cpl_array_set_complex(cpl_array *, int, double complex);
cpl_error_code cpl_array_set_float_complex(cpl_array *, int, float complex);
cpl_error_code cpl_array_set_double_complex(cpl_array *, int, double complex);
#endif

cpl_error_code cpl_array_fill_window(cpl_array *, int, int, double);
cpl_error_code cpl_array_fill_window_int(cpl_array *, int, int, int);
cpl_error_code cpl_array_fill_window_float(cpl_array *, int, int, float);
cpl_error_code cpl_array_fill_window_double(cpl_array *, int, int, double);
cpl_error_code cpl_array_fill_window_string(cpl_array *, int, int,
                                            const char *);
#ifdef _Complex_I
cpl_error_code cpl_array_fill_window_complex(cpl_array *, 
                                             int, int, double complex);
cpl_error_code cpl_array_fill_window_float_complex(cpl_array *, int, int,
                                                   float complex);
cpl_error_code cpl_array_fill_window_double_complex(cpl_array *, int, int,
                                                    double complex);
#endif
cpl_error_code cpl_array_fill_window_invalid(cpl_array *, int, int);
cpl_array *cpl_array_duplicate(const cpl_array *);
cpl_array *cpl_array_extract(const cpl_array *, int start, int count);
cpl_array *cpl_array_extract_real(cpl_array *);
cpl_array *cpl_array_extract_imag(cpl_array *);
cpl_error_code cpl_array_insert_window(cpl_array *, int start, int count);
cpl_error_code cpl_array_erase_window(cpl_array *, int start, int count);
cpl_error_code cpl_array_insert(cpl_array *, const cpl_array *, int);

cpl_error_code cpl_array_add(cpl_array *, const cpl_array *);
cpl_error_code cpl_array_subtract(cpl_array *, const cpl_array *);
cpl_error_code cpl_array_multiply(cpl_array *, const cpl_array *);
cpl_error_code cpl_array_divide(cpl_array *, const cpl_array *);
cpl_error_code cpl_array_add_scalar(cpl_array *, double);
cpl_error_code cpl_array_subtract_scalar(cpl_array *, double);
cpl_error_code cpl_array_multiply_scalar(cpl_array *, double);
cpl_error_code cpl_array_divide_scalar(cpl_array *, double);
cpl_error_code cpl_array_abs(cpl_array *);
cpl_error_code cpl_array_arg(cpl_array *);
cpl_error_code cpl_array_logarithm(cpl_array *, double);
cpl_error_code cpl_array_power(cpl_array *, double);
cpl_error_code cpl_array_exponential(cpl_array *, double);
#ifdef _Complex_I
cpl_error_code cpl_array_add_scalar_complex(cpl_array *, double complex);
cpl_error_code cpl_array_subtract_scalar_complex(cpl_array *, double complex);
cpl_error_code cpl_array_multiply_scalar_complex(cpl_array *, double complex);
cpl_error_code cpl_array_divide_scalar_complex(cpl_array *, double complex);
#endif

#ifdef _Complex_I
double complex cpl_array_get_mean_complex(const cpl_array *);
#endif
double cpl_array_get_mean(const cpl_array *);
double cpl_array_get_median(const cpl_array *);
double cpl_array_get_stdev(const cpl_array *);

double cpl_array_get_max(const cpl_array *);
double cpl_array_get_min(const cpl_array *);
cpl_error_code cpl_array_get_maxpos(const cpl_array *, int *);
cpl_error_code cpl_array_get_minpos(const cpl_array *, int *);
void cpl_array_dump_structure(const cpl_array *, FILE *);
void cpl_array_dump(const cpl_array *, int start, int count, FILE *);
cpl_array *cpl_array_cast(cpl_array *, cpl_type);

CPL_END_DECLS

#endif
/* end of cpl_array.h */

