/* $Id: cpl_imagelist_io.c,v 1.84 2010/12/23 14:12:36 llundin Exp $
 *
 * This file is part of the ESO Common Pipeline Library
 * Copyright (C) 2001-2008 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: llundin $
 * $Date: 2010/12/23 14:12:36 $
 * $Revision: 1.84 $
 * $Name: cpl-5_3_0-BRANCH $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <math.h>
#include <assert.h>

#include <fitsio.h>

#include "cpl_tools.h"
#include "cpl_error_impl.h"
#include "cpl_propertylist_impl.h"
#include "cpl_memory.h"
#include "cpl_imagelist_io_impl.h"
#include "cpl_image_io_impl.h"

#include "cpl_imagelist_defs.h"

/*-----------------------------------------------------------------------------
                                Defines
 -----------------------------------------------------------------------------*/

static
cpl_imagelist * cpl_imagelist_load_one(const char *, cpl_type, int, cpl_boolean,
                                       int, int, int, int) CPL_ATTR_ALLOC;

/*----------------------------------------------------------------------------*/
/**
 * @defgroup cpl_imagelist_io Image list IO routines
 *
 * This module provides IO functions to handle @em cpl_imagelist.
 *
 * @par Synopsis:
 * @code
 *   #include "cpl_imagelist_io.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*-----------------------------------------------------------------------------
                            Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create an empty imagelist
  @return   1 newly allocated cpl_imagelist
  @see      cpl_imagelist_set()

  The returned cpl_imagelist must be deallocated using cpl_imagelist_delete()

 */
/*----------------------------------------------------------------------------*/
cpl_imagelist * cpl_imagelist_new(void)
{
    return (cpl_imagelist *) cpl_calloc(1, sizeof(cpl_imagelist));
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Load a FITS file extension into a list of images
  @param    filename   The FITS file name
  @param    im_type    Type of the images in the created image list
  @param    xtnum      The extension number (0 for primary HDU)
  @return   The loaded list of images or NULL on error.
  @see      cpl_image_load()

  This function loads all the images of a specified extension (NAXIS=2
  or 3) into an image list.

  Type can be CPL_TYPE_DOUBLE, CPL_TYPE_FLOAT or CPL_TYPE_INT.
  The loaded images have an empty bad pixel map.

  The returned cpl_imagelist must be deallocated using cpl_imagelist_delete()

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if xtnum is negative
  - CPL_ERROR_INVALID_TYPE if the passed type is not supported
  - CPL_ERROR_FILE_IO If the file cannot be opened or read, or if xtnum is
                      bigger than the number of extensions in the FITS file
  - CPL_ERROR_BAD_FILE_FORMAT if the file cannot be parsed
  - CPL_ERROR_DATA_NOT_FOUND if the data cannot be read from the file
 */
/*----------------------------------------------------------------------------*/
cpl_imagelist * cpl_imagelist_load(const char * filename,
                                   cpl_type     im_type,
                                   int          xtnum)
{
    cpl_imagelist * self = cpl_imagelist_load_one(filename, im_type, xtnum,
                                                  CPL_FALSE, 0, 0, 0, 0);

    if (self == NULL) cpl_error_set_where_();

    return self;
}

/*----------------------------------------------------------------------------*/
/**
  @brief  Load images windows from a FITS file extension into an image list
  @param  filename    The FITS file name
  @param  im_type     Type of the images in the created image list
  @param  xtnum       The extension number (0 for primary HDU)
  @param  llx         Lower left  x position (FITS convention, 1 for leftmost)
  @param  lly         Lower left  y position (FITS convention, 1 for lowest)
  @param  urx         Upper right x position (FITS convention)
  @param  ury         Upper right y position (FITS convention)
  @return The loaded list of image windows or NULL on error.
  @see      cpl_imagelist_load(), cpl_image_load_window()
  @note The returned cpl_imagelist must be deallocated using
        cpl_imagelist_delete()

  This function loads all the image windows of a specified extension in an
  image list.

  Type can be CPL_TYPE_DOUBLE, CPL_TYPE_FLOAT or CPL_TYPE_INT.

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if xtnum is negative
  - CPL_ERROR_INVALID_TYPE if the passed type is not supported
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if xtnum is bigger than the number of
    extensions in the FITS file
  - CPL_ERROR_BAD_FILE_FORMAT if the file cannot be parsed
  - CPL_ERROR_DATA_NOT_FOUND if the data cannot be read from the file
 */
/*----------------------------------------------------------------------------*/
cpl_imagelist * cpl_imagelist_load_window(const char * filename,
                                          cpl_type     im_type,
                                          int          xtnum,
                                          int          llx,
                                          int          lly,
                                          int          urx,
                                          int          ury)

{
    cpl_imagelist * self = cpl_imagelist_load_one(filename, im_type, xtnum,
                                                  CPL_TRUE, llx, lly, urx, ury);

    if (self == NULL) cpl_error_set_where_();

    return self;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the number of images in the imagelist
  @param    imlist    the list of image
  @return   The number of images or -1 on error

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
int cpl_imagelist_get_size(const cpl_imagelist * imlist)
{
    cpl_ensure(imlist != NULL, CPL_ERROR_NULL_INPUT,
               -1);

    assert( imlist->ni >= 0 );

    return imlist->ni;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get an image from a list of images
  @param    imlist    the image list
  @param    inum    the image id (from 0 to number of images-1)
  @return   A pointer to the image or NULL in error case.

  The returned pointer refers to already allocated data.

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if inum is bigger thant the list size
  - CPL_ERROR_ILLEGAL_INPUT if inum is negative
 */
/*----------------------------------------------------------------------------*/
cpl_image * cpl_imagelist_get(cpl_imagelist * imlist,
                              int             inum)
{

    cpl_ensure(imlist != NULL,    CPL_ERROR_NULL_INPUT,          NULL);
    cpl_ensure(inum >= 0,         CPL_ERROR_ILLEGAL_INPUT,       NULL);
    cpl_ensure(inum < imlist->ni, CPL_ERROR_ACCESS_OUT_OF_RANGE, NULL);

    return imlist->images[inum];
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get an image from a list of images
  @param    imlist    the image list
  @param    inum    the image id (from 0 to number of images-1)
  @return   A pointer to the image or NULL in error case.
  @see cpl_imagelist_get
 */
/*----------------------------------------------------------------------------*/
const cpl_image * cpl_imagelist_get_const(const cpl_imagelist * imlist,
                                          int                   inum)
{
    cpl_ensure(imlist != NULL,    CPL_ERROR_NULL_INPUT,          NULL);
    cpl_ensure(inum >= 0,         CPL_ERROR_ILLEGAL_INPUT,       NULL);
    cpl_ensure(inum < imlist->ni, CPL_ERROR_ACCESS_OUT_OF_RANGE, NULL);

    return imlist->images[inum];
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Insert an image into an imagelist
  @param    imlist   The imagelist
  @param    im       The image to insert
  @param    pos      The list position (from 0 to number of images)
  @return   CPL_ERROR_NONE or the relevant #_cpl_error_code_ on error

  It is allowed to specify the position equal to the number of images in the
  list. This will increment the size of the imagelist.

  No action occurs if an image is inserted more than once into the same
  position. It is allowed to insert the same image into two different
  positions in a list.

  The image is inserted at the position pos in the image list. If the image
  already there is only present in that one location in the list, then the
  image is deallocated.

  It is not allowed to insert images of different size into a list.

  You should not deallocate the image after it has been inserted into an
  imagelist, since the cpl_imagelist_delete() will deallocate its images.

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if pos is negative
  - CPL_ERROR_TYPE_MISMATCH if im and imlist are of different types
  - CPL_ERROR_INCOMPATIBLE_INPUT if im and imlist have different sizes
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if pos is bigger than the number of
    images in imlist
 */
/*----------------------------------------------------------------------------*/
cpl_error_code cpl_imagelist_set(
        cpl_imagelist * imlist,
        cpl_image     * im,
        int             pos)
{

    cpl_ensure_code(imlist,            CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(im,                CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pos >= 0,          CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(pos <= imlist->ni, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* Do nothing if the image is already there */
    if (pos < imlist->ni && im == imlist->images[pos]) return CPL_ERROR_NONE;

    if (pos > 0 || imlist->ni > 1) {
        /* Require images to have the same size and type */
        cpl_ensure_code(cpl_image_get_size_x(im) ==
                        cpl_image_get_size_x(imlist->images[0]),
                        CPL_ERROR_INCOMPATIBLE_INPUT);
        cpl_ensure_code(cpl_image_get_size_y(im) ==
                        cpl_image_get_size_y(imlist->images[0]),
                        CPL_ERROR_INCOMPATIBLE_INPUT);
        cpl_ensure_code(cpl_image_get_type(im) ==
                        cpl_image_get_type(imlist->images[0]),
                        CPL_ERROR_TYPE_MISMATCH);
    }

    if (pos == imlist->ni) {
        imlist->ni++;
        imlist->images = cpl_realloc(imlist->images,
                                 imlist->ni * sizeof(cpl_image*));
    } else {
        /* Check if the image at the position to be overwritten
           is present in only one position */
        int i;

        for (i = 0; i < imlist->ni; i++) {
            if (i != pos && imlist->images[i] == imlist->images[pos]) break;
        }
        if (i == imlist->ni) {
            /* The image at the position to be overwritten
               is present in only one position, so delete it */
            cpl_image_delete(imlist->images[pos]);
        }
    }

    imlist->images[pos] = im;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Remove an image from an imagelist
  @param    self   The imagelist
  @param    pos      The list position (from 0 to number of images-1)
  @return   The pointer to the removed image or NULL in error case

  The specified image is not deallocated, it is simply removed from the
  list. The pointer to the image is returned to let the user decide to
  deallocate it or not.
  Eventually, the image will have to be deallocated with cpl_image_delete().

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if pos is negative
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if pos is bigger than the number of
    images in self
 */
/*----------------------------------------------------------------------------*/
cpl_image * cpl_imagelist_unset(cpl_imagelist * self,
                                int             pos)
{
    cpl_image * out;
    int         i;

    cpl_ensure(self, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(pos >= 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(pos < self->ni, CPL_ERROR_ACCESS_OUT_OF_RANGE, NULL);

    /* Get pointer to image to be removed */
    out = self->images[pos];

    /* Move the following images one position towards zero */
    for (i=pos + 1; i < self->ni; i++) {
        self->images[i-1] = self->images[i];
    }

    /* Decrement of the size */
    self->ni--;

    return out;
}

/*----------------------------------------------------------------------------*/
/**
  @internal
  @brief    Empty an imagelist and deallocate all its images
  @param    self  The image list
 */
/*----------------------------------------------------------------------------*/
void cpl_imagelist_empty(cpl_imagelist * self)
{

    if (self != NULL) {

        while (self->ni > 0) { /* An iteration may unset more than 1 image */
            int i = self->ni - 1;
            cpl_image * del = cpl_imagelist_unset(self, i);

            cpl_image_delete(del);

            /* If this image was inserted more than once into the list,
               the other insertions must be unset without a delete. */
            while (--i >= 0) {
                if (self->images[i] == del) {
                    /* This image was inserted more than once in the list */
                    (void)cpl_imagelist_unset(self, i);
                }
            }
        }
    }

    return;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Free memory associated to a cpl_imagelist object.
  @param    imlist    the image list

  @return   Nothing

   The function deallocates the memory used by the imagelist @em imlist.
   If @em imlist is @c NULL, nothing is done, and no error is set.

 */
/*----------------------------------------------------------------------------*/
void cpl_imagelist_delete(cpl_imagelist * imlist)
{

    if (imlist != NULL) {

        cpl_imagelist_empty(imlist);

        cpl_free(imlist->images);
        cpl_free(imlist);
    }

    return;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Copy an image list
  @param    imlist    Source image list.
  @return   1 newly allocated image list, or NULL on error.

  Copy an image list into a new image list object.
  The returned image list must be deallocated using cpl_imagelist_delete().

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_imagelist * cpl_imagelist_duplicate(const cpl_imagelist * imlist)
{
    cpl_imagelist *   out;
    int               i;

    cpl_ensure(imlist != NULL, CPL_ERROR_NULL_INPUT, NULL);

    /* Create the new imagelist */
    out = cpl_imagelist_new();

    /* Duplicate the images */
    for (i=0; i<imlist->ni; i++) {
        cpl_imagelist_set(out, cpl_image_duplicate(imlist->images[i]), i);
    }

    return out;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Reject one or more images in a list according to an array of flags.
  @param    imlist  Non-empty imagelist to examine for image rejection.
  @param    valid   Vector of flags (>=-0.5: valid, <-0.5: invalid)
  @return   CPL_ERROR_NONE or the relevant #_cpl_error_code_ on error

  This function takes an imagelist and a vector of flags. The imagelist and
  vector must have equal lengths.

  Images flagged as invalid are removed from the list.

  The removal of image(s) will reduce the length of the list accordingly.

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_INCOMPATIBLE_INPUT if the vector size and the image list
    size are different
 */
/*----------------------------------------------------------------------------*/
cpl_error_code cpl_imagelist_erase(
        cpl_imagelist    * imlist,
        const cpl_vector * valid)
{
    int          nkeep = 0;
    int          i;

    /* Check entries */
    cpl_ensure_code(imlist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(valid,  CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(cpl_vector_get_size(valid) == imlist->ni,
                    CPL_ERROR_INCOMPATIBLE_INPUT);

    for (i=0; i < imlist->ni; i++) {
        if (cpl_vector_get(valid, i) >= -0.5) {
            /* image is to be kept, place it in the 1st free position */
            imlist->images[nkeep] = imlist->images[i];
            nkeep++;
        } else {
            /* image is to be erased, delete it */
            cpl_image_delete(imlist->images[i]);
        }
    }

    /* Update the size of the altered list */
    imlist->ni = nkeep;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Save an imagelist to disk in FITS format.
  @param    self      Imagelist to save
  @param    filename  Name of the FITS file to write
  @param    bpp       Bits per pixels
  @param    pl        Property list for the output header or NULL
  @param    mode      The desired output options (combined with bitwise or)
  @return   the #_cpl_error_code_ or CPL_ERROR_NONE
  @see cpl_image_save()

  This function saves an image list to a FITS file. If a
  property list is provided, it is written to the named file before the
  pixels are written.

  The requested pixel depth (bpp) follows the FITS convention. Possible values
  are CPL_BPP_8_UNSIGNED (8), CPL_BPP_16_SIGNED (16), CPL_BPP_16_UNSIGNED (-16),
  CPL_BPP_32_SIGNED (32), CPL_BPP_IEEE_FLOAT (-32), CPL_BPP_IEEE_DOUBLE (-64)

  In the case of CPL_BPP_16_UNSIGNED, the BITPIX stays 16, the data are
  stored as signed (32768 is subtracted to the images) and BZERO is set to
  32768 as specified by the FITS standard.

  Supported image lists types are CPL_TYPE_DOUBLE, CPL_TYPE_FLOAT, CPL_TYPE_INT.

  Supported output modes are CPL_IO_CREATE (create a new file),
  CPL_IO_EXTEND (extend an existing file with a new extension) and
  CPL_IO_APPEND (append list of images to the last data unit, which must
  already contain compatible image(s)).

  In extend and append mode, make sure that the file has write permissions.
  You may have problems if you create a file in your application and append
  something to it with the umask set to 222. In this case, the file created
  by your application would not be writable, and the append would fail.

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if the bpp value or the mode is not supported
  - CPL_ERROR_FILE_IO if the file cannot be written
  - CPL_ERROR_INVALID_TYPE if the passed image list type is not supported
 */
/*----------------------------------------------------------------------------*/
cpl_error_code cpl_imagelist_save(const cpl_imagelist    * self,
                                  const char             * filename,
                                  cpl_type_bpp             bpp,
                                  const cpl_propertylist * pl,
                                  unsigned                 mode)
{
    int               error=0;
    fitsfile       *  fptr;
    int               i;
    const char      * badkeys = mode & CPL_IO_EXTEND ?
    CPL_FITS_BADKEYS_EXT  "|" CPL_FITS_COMPRKEYS :
    CPL_FITS_BADKEYS_PRIM "|" CPL_FITS_COMPRKEYS;
    const cpl_image * first = cpl_imagelist_get_const(self, 0);
    const int         nx = cpl_image_get_size_x(first);
    const int         ny = cpl_image_get_size_y(first);
    const cpl_type    type = cpl_image_get_type(first);
    const int         cfitsiotype = type == CPL_TYPE_DOUBLE ? TDOUBLE
    : (type == CPL_TYPE_FLOAT ? TFLOAT : TINT);
    const long        naxes[3] = {nx, ny, cpl_imagelist_get_size(self)};
    long              plane1 = 0; /* First plane to write */

    /* Test entries */
    cpl_ensure_code(filename != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self     != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(bpp==CPL_BPP_8_UNSIGNED  || bpp==CPL_BPP_16_SIGNED ||
                    bpp==CPL_BPP_16_UNSIGNED || bpp==CPL_BPP_32_SIGNED ||
                    bpp==CPL_BPP_IEEE_FLOAT  || bpp==CPL_BPP_IEEE_DOUBLE,
                    CPL_ERROR_ILLEGAL_INPUT);

    cpl_ensure_code(((mode & CPL_IO_CREATE) != 0) +
                    ((mode & CPL_IO_EXTEND) != 0) +
                    ((mode & CPL_IO_APPEND) != 0) == 1, CPL_ERROR_ILLEGAL_INPUT);

    cpl_ensure_code(mode <= CPL_IO_MAX, CPL_ERROR_ILLEGAL_INPUT);

    cpl_ensure_code(cpl_imagelist_is_uniform(self)==0,
                    CPL_ERROR_ILLEGAL_INPUT);

    cpl_ensure_code(type == CPL_TYPE_DOUBLE ||
            type == CPL_TYPE_FLOAT ||
            type == CPL_TYPE_INT, CPL_ERROR_INVALID_TYPE);

    if (mode & (CPL_IO_EXTEND | CPL_IO_APPEND)) {
        /* Open the file */
        if (fits_open_diskfile(&fptr, filename, READWRITE, &error)) {
            return cpl_error_set_fits(CPL_ERROR_FILE_IO, error,
                                      fits_open_diskfile, "filename='%s', "
                                      "bpp=%d, mode=%d", filename, bpp, mode);
        }
    } else {
        /* Create the file */
        char * sval = cpl_sprintf("!%s", filename);
        fits_create_file(&fptr, sval, &error);
        cpl_free(sval);
        if (error != 0) {
            return cpl_error_set_fits(CPL_ERROR_FILE_IO, error,
                                      fits_create_file, "filename='%s', "
                                      "bpp=%d, mode=%d", filename, bpp, mode);
        }
    }

    if (mode & CPL_IO_APPEND) {
        int next;
        int exttype;
        cpl_boolean isok = CPL_FALSE;

        fits_get_num_hdus(fptr, &next, &error);

        fits_movabs_hdu(fptr, next, &exttype, &error);

        if (exttype == IMAGE_HDU) {

            int naxis;

            fits_get_img_dim(fptr, &naxis, &error);

            if (naxis == 2 || naxis == 3) {
                long fnaxes[3] = {nx, ny, 0}; /* Last element read on naxis=2 */
                int bitpix;

                fits_get_img_type(fptr, &bitpix, &error);

                if (bitpix == bpp) {

                    fits_get_img_size(fptr, naxis, fnaxes, &error);

                    if (naxes[0] == fnaxes[0] && naxes[1] == fnaxes[1]) {
                        plane1 = fnaxes[2];
                        fnaxes[2] += naxes[2];
                        fits_resize_img(fptr, bitpix, naxis, fnaxes, &error);
                        if (!error) isok = CPL_TRUE;
                    } else {
                    int error2 = 0;
                    fits_close_file(fptr, &error2);
                    return cpl_error_set_message_
                        (CPL_ERROR_ILLEGAL_INPUT, "Data-unit %d in file %s has "
                         "wrong NAXIS1/2 (%ldX%ld != %ldX%ld). bpp=%d, "
                         "mode=%d)", next, filename, fnaxes[0], fnaxes[1],
                         naxes[0], naxes[1], bpp, mode);
                    }
                } else {
                    int error2 = 0;
                    fits_close_file(fptr, &error2);
                    return cpl_error_set_message_
                        (CPL_ERROR_ILLEGAL_INPUT, "Data-unit %d in file %s has "
                         "wrong BITPIX (%d != %d). mode=%d)", next, filename,
                         bitpix, bpp, mode);
                }
            } else {
                int error2 = 0;
                fits_close_file(fptr, &error2);
                return cpl_error_set_message_
                    (CPL_ERROR_ILLEGAL_INPUT, "Data-unit %d in file %s has "
                     "wrong NAXIS (%d != 2/3). bpp=%d, mode=%d)", next,
                     filename, naxis, bpp, mode);
            }
        } else {
            int error2 = 0;
            fits_close_file(fptr, &error2);
            return cpl_error_set_message_
                (CPL_ERROR_ILLEGAL_INPUT, "Data-unit %d in file %s is not "
                 "image-type (%d != %d. bpp=%d, mode=%d)",
                 next, filename, IMAGE_HDU, exttype, bpp, mode);
        }

        if (error) {
            int error2 = 0;
            fits_close_file(fptr, &error2);
            return cpl_error_set_fits(CPL_ERROR_FILE_IO, error, "CFITSIO",
                                      "filename='%s', bpp=%d, mode=%d",
                                      filename, bpp, mode);
        } else if (!isok) {
            /* Not supposed to enter here */
            int error2 = 0;
            fits_close_file(fptr, &error2);
            return cpl_error_set_(CPL_ERROR_UNSPECIFIED);
        }
    } else {
        /* Create the imagelist in the file */
        fits_create_img(fptr, bpp, 3, (long*)naxes, &error);

        if (mode & CPL_IO_CREATE)
            fits_write_date(fptr, &error); /* Add Date */
    }

    if (error) {
        int error2 = 0;
        fits_close_file(fptr, &error2);
        return cpl_error_set_fits(CPL_ERROR_FILE_IO, error, "CFITSIO",
                                  "filename='%s', bpp=%d, mode=%d",
                                  filename, bpp, mode);
    }

    /* Add the property list */
    if (cpl_fits_add_properties(fptr, pl, badkeys)) {
        fits_close_file(fptr, &error);
        return cpl_error_set_(CPL_ERROR_ILLEGAL_INPUT);
    }

    /* Loop on the images */
    for (i=0; i < self->ni; i++) {
        const long        fpixel[3] = {1, 1, plane1 + i + 1};
    const cpl_image * image = cpl_imagelist_get_const(self, i);
    const void      * data  = cpl_image_get_data_const(image);

    /* Write the pixels */
    if (fits_write_pix(fptr, cfitsiotype, (long*)fpixel, nx*ny,
                           (void*)data, &error)) break;
    }

    /* Check  */
    if (error) {
        int error2 = 0;
        fits_close_file(fptr, &error2);
        return cpl_error_set_fits(CPL_ERROR_FILE_IO, error,
                                  "CFITSIO", "filename='%s', "
                                  "bpp=%d, mode=%d", filename, bpp, mode);
    }

    /* Close (and write to disk) */
    return fits_close_file(fptr, &error)
        ? cpl_error_set_fits(CPL_ERROR_FILE_IO, error,
                             fits_close_file, "filename='%s', "
                             "bpp=%d, mode=%d", filename, bpp, mode)
        : CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Determine if an imagelist contains images of equal size and type
  @param    imlist    The imagelist to check
  @return   Zero if uniform, positive if non-uniform and negative on error.

  The function returns 1 if the list is empty.

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
int cpl_imagelist_is_uniform(const cpl_imagelist * imlist)
{
    cpl_type     type;
    int          nx, ny;
    int          i;

    cpl_ensure(imlist != NULL, CPL_ERROR_NULL_INPUT, -1);
    if (imlist->ni == 0) return 1;

    /* Check the images */
    nx   = cpl_image_get_size_x(imlist->images[0]);
    ny   = cpl_image_get_size_y(imlist->images[0]);
    type = cpl_image_get_type  (imlist->images[0]);

    for (i=1; i < imlist->ni; i++) {
        if (cpl_image_get_size_x(imlist->images[i]) != nx)   return i+1;
        if (cpl_image_get_size_y(imlist->images[i]) != ny)   return i+1;
        if (cpl_image_get_type  (imlist->images[i]) != type) return i+1;
    }
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Dump structural information of images in an imagelist
  @param    self    Imagelist to dump
  @param    stream  Output stream, accepts @c stdout or @c stderr
  @return   CPL_ERROR_NONE or the relevant #_cpl_error_code_ on error

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if a write operation fails
 */
/*----------------------------------------------------------------------------*/
cpl_error_code cpl_imagelist_dump_structure(const cpl_imagelist * self,
                        FILE * stream)
{
    const char * msg    = "Imagelist with %d image(s)\n";
    const int    msgmin = strlen(msg) - 5;

    int i;

    cpl_ensure_code(self   != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(stream != NULL, CPL_ERROR_NULL_INPUT);

    cpl_ensure_code( fprintf(stream,  msg, self->ni) >= msgmin,
             CPL_ERROR_FILE_IO );

    for (i = 0; i < self -> ni; i++) {
    const cpl_image * image   = cpl_imagelist_get_const(self, i);
    const char      * imsg    = "Image nb %d of %d in imagelist\n";
    const int         imsgmin = strlen(imsg) - 5;

    cpl_ensure_code( fprintf(stream,  imsg, i, self->ni) >= imsgmin,
             CPL_ERROR_FILE_IO );

    cpl_ensure_code( !cpl_image_dump_structure(image, stream),
             cpl_error_get_code() );
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Dump pixel values of images in a CPL imagelist
  @param    self    Imagelist to dump
  @param    llx     Specifies the window position
  @param    lly     Specifies the window position
  @param    urx     Specifies the window position
  @param    ury     Specifies the window position
  @param    stream  Output stream, accepts @c stdout or @c stderr
  @return   CPL_ERROR_NONE or the relevant #_cpl_error_code_ on error

  Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if a write operation fails
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if the defined window is not in the image
  - CPL_ERROR_ILLEGAL_INPUT if the window definition is wrong (e.g llx > urx)
 */
/*----------------------------------------------------------------------------*/
cpl_error_code cpl_imagelist_dump_window(const cpl_imagelist * self,
                     int llx, int lly, int urx, int ury,
                     FILE * stream)
{
    int i;

    cpl_ensure_code(self   != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(stream != NULL, CPL_ERROR_NULL_INPUT);

    for (i = 0; i < self -> ni; i++) {
    const cpl_image * image   = cpl_imagelist_get_const(self, i);
    const char      * imsg    = "Image nb %d of %d in imagelist\n";
    const int         imsgmin = strlen(imsg) - 5;

    cpl_ensure_code( fprintf(stream,  imsg, i, self->ni) >= imsgmin,
             CPL_ERROR_FILE_IO );

    cpl_ensure_code( !cpl_image_dump_window(image, llx, lly, urx, ury,
                           stream),
             cpl_error_get_code() );
    }

    return CPL_ERROR_NONE;
}

/**@}*/

/*----------------------------------------------------------------------------*/
/**
  @internal
  @brief  Load an imagelist from a FITS file.
  @param  filename    Name of the file to load from.
  @param  im_type     Type of the created images
  @param  xtnum       Extension number in the file (0 for primary HDU)
  @param  do_window   True for (and only for) a windowed load
  @param  llx         Lower left  x position (FITS convention, 1 for leftmost)
  @param  lly         Lower left  y position (FITS convention, 1 for lowest)
  @param  urx         Upper right x position (FITS convention)
  @param  ury         Upper right y position (FITS convention)
  @return 1 newly allocated imagelist or NULL on error
  @see cpl_imagelist_load()
*/
/*----------------------------------------------------------------------------*/
static cpl_imagelist * cpl_imagelist_load_one(const char * filename,
                                              cpl_type     im_type,
                                              int          xtnum,
                                              cpl_boolean  do_window,
                                              int          llx,
                                              int          lly,
                                              int          urx,
                                              int          ury)
{

    /* Count number of images read - use also to indicate failure */
    int             selfsize;
    cpl_imagelist * self;
    cpl_image     * image;
    fitsfile      * fptr;
    int             error = 0;
    /* Initialize to indicate that they need to be read from the file */
    int             naxis = 0;
    long int        naxes[3] ={0, 0, 0};
    cpl_type        pix_type = im_type;

    /* FIXME: Version 3.2 of fits_open_diskfile() seg-faults on NULL.
       If fixed in CFITSIO, this check should be removed */
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT,    NULL);
    cpl_ensure(xtnum    >= 0,    CPL_ERROR_ILLEGAL_INPUT, NULL);

    if (fits_open_diskfile(&fptr, filename, READONLY, &error)) {
        (void)cpl_error_set_fits(CPL_ERROR_FILE_IO, error, fits_open_diskfile,
                                 "filename='%s', im_type=%u, xtnum=%d",
                                 filename, im_type, xtnum);
        return NULL;
    }

    /* Load 1st image from the extension. This will set naxis and naxes[]
       (and optionally the pixel type) for use in subsequent calls */
    image = cpl_image_load_(fptr, &naxis, naxes, &pix_type, filename, 0, xtnum,
                            do_window, llx, lly, urx, ury);

    self = cpl_imagelist_new();
    selfsize = 0;

    if (image == NULL || cpl_imagelist_set(self, image, selfsize)) {
        cpl_image_delete(image);
    } else {
        selfsize++;
    }

    if (selfsize > 0 && naxis == 3) {
        /* Handle other planes in this extension, if any */
        int iplane;

        for (iplane = 1; iplane < naxes[2]; iplane++) {
            image = cpl_image_load_(fptr, &naxis, naxes, &pix_type,
                                    filename, iplane, 0,
                                    do_window, llx, lly, urx, ury);
            if (image == NULL) break;

            if (cpl_imagelist_set(self, image, selfsize)) {
                cpl_image_delete(image);
                break;
            }

            selfsize++;
        }
        if (iplane < naxes[2]) {
            selfsize = 0; /* Indicate failure */
        }
    }

    if (fits_close_file(fptr, &error)) {
        (void)cpl_error_set_fits(CPL_ERROR_BAD_FILE_FORMAT, error,
                                 fits_close_file, "filename='%s', "
                                 "im_type=%u, xtnum=%d",
                                 filename, (unsigned)im_type, xtnum);
        selfsize = 0; /* Indicate failure */
    } else if (selfsize == 0) {
        (void)cpl_error_set_where_();
    }

    if (selfsize == 0) {
        cpl_imagelist_delete(self);
        self = NULL;
    }

    return self;
}
