/* this file is part of criawips, a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004,2005 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <rendering/cria-slide-renderer-priv.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <inttypes.h>
#include <string.h>

#include <glib/gi18n.h>
#include <libgnomecanvas/libgnomecanvas.h>

#define CDEBUG_TYPE cria_slide_renderer_get_type
#include <cdebug/cdebug.h>

#include <dom/cria-enums.h>
#include <dom/cria-slide-view.h>
#include <widgets/cria-slide-editor.h>
#include <rendering/cria-block-renderer.h>
#include <rendering/cria-image-pool.h>

enum {
	PROP_0,
};

enum {
	N_SIGNALS
};

static void csv_iface_init(CriaSlideViewIface* iface);
#ifdef WITH_CCC
G_DEFINE_TYPE_WITH_CODE(CriaSlideRenderer, cria_slide_renderer, CC_TYPE_NODE,
#else
G_DEFINE_TYPE_WITH_CODE(CriaSlideRenderer, cria_slide_renderer, CRIA_TYPE_ITEM,
#endif
			G_IMPLEMENT_INTERFACE(CRIA_TYPE_SLIDE_VIEW, csv_iface_init));

static void cria_slide_renderer_get_property   (GObject		* object,
						guint		  prop_id,
						GValue		* value,
						GParamSpec	* param_spec);
static void cria_slide_renderer_set_property   (GObject		* object,
						guint		  prop_id,
						const	GValue	* value,
						GParamSpec	* param_spec);
#if 0
/* enable these to add support for signals */
static	guint	cria_slide_renderer_signals[N_SIGNALS] = { 0 };

static	void	cria_slide_renderer_signal	       (CriaSlideRenderer	* self,
						const	gchar	* string);
#endif
#ifdef WITH_CCC
static void
csr_render(CcNode* node, cairo_t* cr) {
	/* render the shadow */
	cairo_save(cr);
#warning "FIXME: get the size from the slide"
#define OFFSET 25.0
	 cairo_rectangle(cr, 0.0 + OFFSET, 0.0 + OFFSET,
			 4650.0, 5670.0);
#undef OFFSET
	 cairo_set_source_rgba(cr, 0.0, 0.0, 0.0, 0.75);
	 cairo_fill(cr);
	cairo_restore(cr);

	/* render background */
#warning "FIXME: render the background"
	cairo_save(cr);
	 cairo_rectangle(cr, 0.0, 0.0, 4650.0, 5670.0);
	 cairo_set_source_rgba(cr, 1.0, 1.0, 1.0, 1.0);
	 cairo_fill(cr);
	cairo_restore(cr);

	/* render blocks */
#warning "FIXME: render the blocks"
}
#else
static void
csr_update_focus_layer_size(CriaSlideRenderer* self) {
	GoPoint* size;
	g_return_if_fail(GNOME_IS_CANVAS_ITEM(self->focus_layer));
	g_return_if_fail(CRIA_IS_SLIDE(self->slide));
	
	size = cria_slide_get_size(self->slide);
	gnome_canvas_item_set(self->focus_layer,
			"x1", 0.0,
			"y1", 0.0,
			"x2", 0.0 + size->x,
			"y2", 0.0 + size->y,
			NULL);
	g_free(size);
}

static void
csr_ensure_stacking(CriaSlideRenderer* self) {
	if(self->frame) {
		gnome_canvas_item_raise_to_top(self->frame);
	}

	if(self->focus_layer) {
		gnome_canvas_item_lower_to_bottom(self->focus_layer);
	}
	if(self->background_renderer) {
		gnome_canvas_item_lower_to_bottom(GNOME_CANVAS_ITEM(self->background_renderer));
	}
	if(self->shadow) {
		gnome_canvas_item_lower_to_bottom(self->shadow);
	}
}

static void
csr_ensure_focus_layer(CriaSlideRenderer* self) {
	g_return_if_fail(GNOME_CANVAS_ITEM(self)->canvas);
	if(!self->focus_layer) {
		self->focus_layer = gnome_canvas_item_new(GNOME_CANVAS_GROUP(self),
							  gnome_canvas_rect_get_type(),
							  NULL);
		csr_update_focus_layer_size(self);
		csr_ensure_stacking(self);
	}
}


static gdouble
csr_point(GnomeCanvasItem* self, gdouble x, gdouble y, int cx, int cy, GnomeCanvasItem** actual_item) {
	gdouble retval = FALSE;

	if(actual_item && *actual_item == GNOME_CANVAS_ITEM(CRIA_SLIDE_RENDERER(self)->background_renderer)) {
		/* don't focus the background renderer, focus ourselves */
		*actual_item = self;
	}

	if(!retval && GNOME_CANVAS_ITEM_CLASS(cria_slide_renderer_parent_class)->point) {
		retval = GNOME_CANVAS_ITEM_CLASS(cria_slide_renderer_parent_class)->point(self, x, y, cx, cy, actual_item);
	}
	
	return retval;
}

static gboolean
csr_button_press_event(CriaItem* item, GdkEventButton* ev) {
	gboolean retval = FALSE;

	if(ev->type == GDK_BUTTON_PRESS && ev->button == 1) {
		cria_item_grab_focus(item);
		retval = TRUE;
	}
	
	return retval;
}

static gboolean
csr_focus_in_event(CriaItem* item, GdkEventFocus* ev) {
	CriaSlideRenderer* self = CRIA_SLIDE_RENDERER(item);
	guint		   color;
	GdkColor           gdk_color;
	gdk_color = GTK_WIDGET(GNOME_CANVAS_ITEM(self)->canvas)->style->base[GTK_STATE_SELECTED];
	gnome_canvas_item_set(self->frame,
		     /* MEMO text would get style->text[GTK_STATE_SELECTED] */
		     "outline-color-gdk", &gdk_color,
		     "width-pixels", 2,
		     NULL);
	/* The focus layer doesn't work, because the canvas doesn't support semi-transparent rectangles */
	csr_ensure_focus_layer(self);
	color = GNOME_CANVAS_COLOR_A(gdk_color.red >> 2, gdk_color.green >> 2, gdk_color.blue >> 2, 0x11);
	gnome_canvas_item_set(self->focus_layer,
		     "fill-color-rgba", &color,
		     NULL);
	gnome_canvas_item_show(self->focus_layer);
	return TRUE;
}

static void
csr_frame_unfocused(CriaSlideRenderer* self) {
	gnome_canvas_item_set(self->frame,
		     /* MEMO text would get style->text[GTK_STATE_ACTIVE] */
		     "outline-color-gdk", &GTK_WIDGET(GNOME_CANVAS_ITEM(self)->canvas)->style->black,
		     "width-pixels", 1,
		     NULL);
}

static gboolean
csr_focus_out_event(CriaItem* item, GdkEventFocus* ev) {
	CriaSlideRenderer* self = CRIA_SLIDE_RENDERER(item);
	
	csr_frame_unfocused(self);
	/* The focus layer doesn't work because the canvas doesn't support semi-transparent rectangles */
/*	if(self->focus_layer) {
		gnome_canvas_item_hide(self->focus_layer);
	}
*/	return TRUE;
}
#endif
static void
csr_finalize(GObject* object) {
	CriaSlideRenderer* self = CRIA_SLIDE_RENDERER(object);
	
	cdebug("finalize()", "Start");
	if(self->slide) {
		cdebug("finalize()", "slide has %d references, removing one of them", G_OBJECT(self->slide)->ref_count);
		g_object_unref(self->slide);
		self->slide = NULL;
	}

	cdebug("finalize()", "parent_class->finalize()");
	G_OBJECT_CLASS(cria_slide_renderer_parent_class)->finalize(object);
	cdebug("finalize()", "End");
}

static void
cria_slide_renderer_class_init(CriaSlideRendererClass* self_class) {
	GObjectClass* g_object_class;
#ifdef WITH_CCC
	CcNodeClass * cn_class;
#else
	GnomeCanvasItemClass* canvas_item_class;
	CriaItemClass       * item_class;
#endif

	/* setting up the object class */
	g_object_class = G_OBJECT_CLASS(self_class);
	g_object_class->finalize = csr_finalize;
#if 0
	/* setting up signal system */
	cria_slide_renderer_class->signal = cria_slide_renderer_signal;

	cria_slide_renderer_signals[SIGNAL] = g_signal_new("signal",
						  CRIA_TYPE_SLIDE_RENDERER,
						  G_SIGNAL_RUN_LAST,
						  G_STRUCT_OFFSET(CriaSlideRendererClass,
							  	  signal),
						  NULL,
						  NULL,
						  g_cclosure_marshal_VOID__STRING,
						  G_TYPE_NONE,
						  0);
#endif
	/* setting up the object class */
	g_object_class->set_property = cria_slide_renderer_set_property;
	g_object_class->get_property = cria_slide_renderer_get_property;

#ifdef WITH_CCC
	/* Setting up CcNodeClass */
	cn_class = CC_NODE_CLASS(self_class);
	cn_class->render = csr_render;
#else
	/* setting up the canvas item class */
	canvas_item_class = GNOME_CANVAS_ITEM_CLASS(self_class);
	canvas_item_class->point = csr_point;

	/* setting up the cria item class */
	item_class = CRIA_ITEM_CLASS(self_class);
	item_class->button_press_event = csr_button_press_event;
	item_class->focus_in_event = csr_focus_in_event;
	item_class->focus_out_event = csr_focus_out_event;
#endif
	/* setting up the slide view interface */
	_cria_slide_view_install_properties(g_object_class);
}

#ifndef WITH_CCC
static void
csr_update_frame_size(CriaSlideRenderer* self) {
	GoPoint* size;
	g_return_if_fail(GNOME_IS_CANVAS_ITEM(self->frame));
	g_return_if_fail(CRIA_IS_SLIDE(self->slide));

	size = cria_slide_get_size(self->slide);
	gnome_canvas_item_set(self->frame,
		     "x1", 0.0,
		     "y1", 0.0,
		     "x2", 0.0 + size->x,
		     "y2", 0.0 + size->y,
		     NULL);
	g_free(size);
}

static void
csr_ensure_frame(CriaSlideRenderer* self) {
	if(!self->frame) {
		self->frame = gnome_canvas_item_new(GNOME_CANVAS_GROUP(self),
						    gnome_canvas_rect_get_type(),
						    NULL);
		csr_frame_unfocused(self);

		if(self->slide) {
			csr_update_frame_size(self);
		}
		gnome_canvas_item_raise_to_top(self->frame);
		gnome_canvas_item_show(self->frame);
	}
}

static void
csr_update_shadow_size(CriaSlideRenderer* self) {
	GoPoint* size;
	g_return_if_fail(GNOME_IS_CANVAS_ITEM(self->shadow));
	g_return_if_fail(CRIA_IS_SLIDE(self->slide));

	size = cria_slide_get_size(self->slide);
	gnome_canvas_item_set(self->shadow,
		     "x1", 0.0 + SLIDE_EDITOR_SHADOW_OFFSET,
		     "y1", 0.0 + SLIDE_EDITOR_SHADOW_OFFSET,
		     "x2", 0.0 + (size->x + SLIDE_EDITOR_SHADOW_OFFSET),
		     "y2", 0.0 + (size->y + SLIDE_EDITOR_SHADOW_OFFSET),
		     NULL);
	g_free(size);
}

static void
csr_ensure_shadow(CriaSlideRenderer* self) {
	if(!self->shadow) {
		/* add the shadow */
		self->shadow = gnome_canvas_item_new(GNOME_CANVAS_GROUP(self),
						   gnome_canvas_rect_get_type(),
						   "fill-color-rgba", GNOME_CANVAS_COLOR_A(0, 0, 0, 127),
						   NULL);

		if(self->slide) {
			csr_update_shadow_size(self);
		}
		gnome_canvas_item_lower_to_bottom(self->shadow);
		gnome_canvas_item_show(self->shadow);
	}
}
#endif

static void
cria_slide_renderer_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* param_spec) {
	CriaSlideRenderer	* self;

	self = CRIA_SLIDE_RENDERER(object);

	switch(prop_id) {
	case CRIA_SLIDE_VIEW_PROP_SLIDE:
		g_value_set_object(value, cria_slide_renderer_get_slide(self));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, param_spec);
		break;
	}
}

CriaSlide*
cria_slide_renderer_get_slide(CriaSlideRenderer const* self) {
	g_return_val_if_fail(CRIA_IS_SLIDE_RENDERER(self), NULL);
	
	return self->slide;
}

#ifndef WITH_CCC
static void
cb_interactive_changed(CriaSlideRenderer* self) {
	if(cria_item_is_interactive(CRIA_ITEM(self)) && GNOME_IS_CANVAS(GNOME_CANVAS_ITEM(self)->canvas)) {
		csr_ensure_frame(self);
		csr_ensure_shadow(self);
		csr_ensure_stacking(self);
	} else {
		if(self->frame) {
			gnome_canvas_item_hide(self->frame);
		}

		if(self->shadow) {
			gnome_canvas_item_hide(self->shadow);
		}
	}
}
#endif

static void
cria_slide_renderer_init(CriaSlideRenderer* self) {
	self->background_renderer = cria_background_renderer_new();
#ifndef WITH_CCC
	CRIA_ITEM_SET_FLAG(CRIA_ITEM(self), CRIA_CAN_FOCUS);

	g_signal_connect(self, "notify::interactive", G_CALLBACK(cb_interactive_changed), NULL);
#endif
}

/**
 * cria_slide_renderer_new:
 * @display: a #CriaCanvas
 * @interactive: specifies whether the renderer should allow editing
 *
 * Create a new renderer for a #CriaSlide.
 *
 * Returns the new slide renderer.
 */
#ifdef WITH_CCC
CcNode*
#else
CriaSlideRenderer*
#endif
cria_slide_renderer_new(gboolean interactive) {
	return g_object_new(CRIA_TYPE_SLIDE_RENDERER,
#ifndef WITH_CCC
			    "interactive", interactive,
#else
#warning "FIXME: enable the interactive stuff"
#endif
			    NULL);
}

static void
cria_slide_renderer_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* param_spec) {
	CriaSlideRenderer	* self;
	
	self = CRIA_SLIDE_RENDERER(object);
	
	switch(prop_id) {
	case CRIA_SLIDE_VIEW_PROP_SLIDE:
		cria_slide_renderer_set_slide(self, g_value_get_object(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, param_spec);
		break;
	}
}

static GList*
csr_add_slide_elements(GList* elements, GList* new_elements) {
	for(; new_elements; new_elements = new_elements->next) {
		gboolean overwrites = FALSE;
		gchar const* name_new = cria_slide_element_get_name(CRIA_SLIDE_ELEMENT(new_elements->data));
		GList* old_elem;

		if(name_new && strcmp(name_new, _("unnamed"))) {
			for(old_elem = elements; old_elem; old_elem = old_elem->next) {
				gchar const* name_old = cria_slide_element_get_name(CRIA_SLIDE_ELEMENT(old_elem->data));
				
				/* don't overwrite blocks with empty names */
				if(name_new && name_old && !strcmp(name_new, name_old)) {
					/* overwrite the old element if necessary */
					old_elem->data = new_elements->data;
					overwrites = TRUE;
					break;
				}
			}
		}
		
		if(!overwrites) {
			/* append new elements */
			elements = g_list_append(elements, new_elements->data);
		}
	}
	
	return elements;
}

static GList*
csr_get_elements(CriaSlideRenderer const* self) {
	GList    * slides = NULL;
	GList    * slide;
	GList    * elements = NULL;
	CriaSlide* master = NULL;

	for(master = self->slide; master; master = cria_slide_get_master_slide(master)) {
		slides = g_list_prepend(slides, master);
	}

	elements = cria_slide_get_elements(slides->data);
	for(slide = slides->next; slide; slide = slide->next) {
		GList* new_elements = cria_slide_get_elements(CRIA_SLIDE(slide->data));
		elements = csr_add_slide_elements(elements, new_elements);
		g_list_free(new_elements);
	}
	
	return elements;
}

static void
csr_update_block_renderers(CriaSlideRenderer* self) {
	/* This is the algorithm:
	 * 1. iterate over the existing renderers until
	 *    a) there are no renderers left
	 *    b) there are no blocks left
	 * 2. create new renderers if b) or hide renderers if a)
	 */
	GList	* renderer,
		* elements,
		* element;

	g_return_if_fail(CRIA_IS_SLIDE(self->slide));

	element = elements = csr_get_elements(self);

	renderer = self->block_renderers;
	cdebug("updateBlockRenderers()", "rendering %d blocks with %d renderers",
	       g_list_length(elements),
	       g_list_length(renderer));
	while(renderer && element) {
		cdebug("updateBlockRenderers()", "Rendering %d. block: %s",
		       g_list_position(elements, element) + 1,
		       cria_slide_element_get_name(CRIA_SLIDE_ELEMENT(element->data)));
		cria_block_renderer_set_block(CRIA_BLOCK_RENDERER(renderer->data), CRIA_BLOCK(element->data));
		cria_block_renderer_show(CRIA_BLOCK_RENDERER(renderer->data));

		renderer = renderer->next;
		element = element->next;
	}

	if(!element && renderer) {
		cdebug("updateBlockRenderers()", "need to hide %d renderers", g_list_length(renderer));
		for(; renderer; renderer = renderer->next) {
			cdebug("updateBlockRenderers()", "Hiding a block renderer");
			cria_block_renderer_hide(CRIA_BLOCK_RENDERER(renderer->data));
		}
	} else if (element && !renderer) {
		cdebug("updateBlockRenderers()", "need to create %d renderers", g_list_length(element));
		for(; element; element = element->next) {
			cdebug("updateBlockRenderers()", "Creating a%s block renderer for '%s'",
			       cria_item_is_interactive(CRIA_ITEM(self)) ? "n interactive" : "",
			       cria_slide_element_get_name(CRIA_SLIDE_ELEMENT(element->data)));
#ifndef WITH_CCC
			self->block_renderers = g_list_prepend(self->block_renderers,
							       cria_block_renderer_new(CRIA_ITEM(self),
										       cria_item_is_interactive(CRIA_ITEM(self))));
#endif
			cdebug("updateBlockRenderers()", "Setting the block of the renderer");
			cria_block_renderer_set_block(CRIA_BLOCK_RENDERER(self->block_renderers->data),
						      CRIA_BLOCK(element->data));
			cdebug("updateBlockRenderers()", "Done setting the block of the renderer");
		}
	} else {
		cdebug("updateBlockRenderers()", "no need to create/hide renderers");
	}

	g_list_free(elements);
	cdebug("updateBlockRenderers()", "Return");
}

#ifndef WITH_CCC
static void
csr_ensure_background_renderer(CriaSlideRenderer* self) {
	gnome_canvas_item_construct(GNOME_CANVAS_ITEM(self->background_renderer),
				    GNOME_CANVAS_GROUP(self),
				    NULL, NULL);
	csr_ensure_stacking(self);

	gnome_canvas_item_show(GNOME_CANVAS_ITEM(self->background_renderer));
}
#endif

static void
update_slide_size(CriaSlideRenderer* self) {
	GoPoint* size;
#ifdef WITH_CCC
	CcDRect box;
#endif
	g_return_if_fail(CRIA_IS_SLIDE_RENDERER(self));
	g_return_if_fail(CRIA_IS_SLIDE(self->slide));

	size = cria_slide_get_size(self->slide);
#ifdef WITH_CCC
	box.x1 = 0.0;
	box.y1 = 0.0;
	box.x2 = 0.0 + size->x;
	box.y2 = 0.0 + size->y;
	if(!cc_d_rect_equal(box, CC_NODE(self)->bounding_box)) {
		CC_NODE(self)->bounding_box = box;
		g_signal_emit_by_name(self, "bounds-changed");
	}
#else
	if(self->frame) {
		csr_update_frame_size(self);
	}
	if(self->shadow) {
		csr_update_shadow_size(self);
	}
	if(self->focus_layer) {
		csr_update_focus_layer_size(self);
	}
	if(self->background_renderer) {
		GoRect rect = {
			0, /* top */
			0, /* left */
			size->y, /* bottom */
			size->x  /* right */
		};
		
		cdebug("updateSlideSize()", "setting background position to: %llix%lli => %llix%lli", rect.left, rect.top, rect.right, rect.bottom);
		cria_background_renderer_set_position(self->background_renderer, &rect);
	}
#endif
	g_free(size);
}

void
cria_slide_renderer_set_slide(CriaSlideRenderer* self, CriaSlide* slide) {
	g_return_if_fail(CRIA_IS_SLIDE_RENDERER(self));
	g_return_if_fail(CRIA_IS_SLIDE(slide));

	if(self->slide != NULL) {
		cria_slide_view_unregister(CRIA_SLIDE_VIEW(self), self->slide);
		g_object_unref(self->slide);
		self->slide = NULL;
	}

	self->slide = g_object_ref(slide);
	cria_slide_view_register(CRIA_SLIDE_VIEW(self), slide);

	g_object_notify(G_OBJECT(self), "slide");
#ifndef WITH_CCC
	if(GNOME_CANVAS_ITEM(self)->canvas && !GNOME_CANVAS_ITEM(self->background_renderer)->canvas) {
		csr_ensure_background_renderer(self);
	}
#endif
	cria_background_renderer_set_background(self->background_renderer, cria_slide_get_background(self->slide, FALSE));
	update_slide_size(self);
	csr_update_block_renderers(self);
}

/* CriaSlideViewIface handling stuff */
static void
csr_on_slide_element_added(CriaSlideView* slide_view, CriaSlideElement* element, CriaSlide* slide) {
#warning "FIXME: this can be optimized very well: just split the lists of used and unused renderers, then we only need to add the new renderer"
	csr_update_block_renderers(CRIA_SLIDE_RENDERER(slide_view));
}

static void
csv_iface_init(CriaSlideViewIface* iface) {
	iface->on_slide_element_added = csr_on_slide_element_added;
}

