// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000-2004 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

///////////////////////////////////////////////////////////////////////
// combat.cpp
///////////////////////////////////////////////////////////////////////

#include "SDL_endian.h"

#include "combat.h"
#include "history.h"
#include "hexsup.h"

////////////////////////////////////////////////////////////////////////
// NAME       : Combat::Combat
// DESCRIPTION: Keep the combat data until end of turn when all clashes
//              will be resolved.
// PARAMETERS : att - attacking unit
//              def - defending unit
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

Combat::Combat( Unit *att, Unit *def ) :
  c_att( att ), c_def( def ), aamod(0), admod(0), damod(0), ddmod(0) {}

////////////////////////////////////////////////////////////////////////
// NAME       : Combat::Combat
// DESCRIPTION: Load combat data from a file.
// PARAMETERS : file - data file descriptor
//              game - pointer to game object
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

Combat::Combat( SDL_RWops *file, Game *game ) :
        aamod(0), admod(0), damod(0), ddmod(0) {
  c_att = game->GetUnitByID( SDL_ReadLE16( file ) );
  c_def = game->GetUnitByID( SDL_ReadLE16( file ) );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Combat::Save
// DESCRIPTION: Save combat data to a file.
// PARAMETERS : file - data file descriptor
// RETURNS    : 0 on success, -1 on error
////////////////////////////////////////////////////////////////////////

int Combat::Save( SDL_RWops *file ) const {
  SDL_WriteLE16( file, c_att->ID() );
  SDL_WriteLE16( file, c_def->ID() );
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Combat::CalcModifiers
// DESCRIPTION: Calculate attack and defence modifiers for this combat.
//              This should be done after all pairings for this turn
//              are known and units cannot be moved but before the first
//              shot is fired.
// PARAMETERS : map - pointer to the map object
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Combat::CalcModifiers( const Map *map ) {
  const Point &apos = c_att->Position(), &dpos = c_def->Position();
  bool nextto = NextTo( apos, dpos );

  // calculate modifiers; these are already calculated when initiating
  // combat because they are subject to change during the execution of
  // attack commands; terrain modifiers do not apply for aircraft, of
  // course, and only if it's not a ranged attack
  // if, however, it is a ranged attack give a uniform bonus because
  // we are not directly engaged
  if ( !c_att->IsAircraft() ) {
    if ( nextto ) aamod += map->AttackMod( apos );
    else aamod += 5;
    admod = map->DefenceMod( apos );
  }

  if ( !c_def->IsAircraft() ) {
    damod += map->AttackMod( dpos );
    ddmod += map->DefenceMod( dpos );
  }

  // check for units supporting our attack (wedging) or supporting defence (blocking),
  // only for non-ranged attacks
  if ( nextto ) {
    Point nbors[6];
    Unit *u;
    map->GetNeighbors( dpos, nbors );

    for ( int i = NORTH; i <= NORTHWEST; ++i ) {
      if ( nbors[i].x != -1 ) {
        u = map->GetUnit( nbors[i] );
        if ( u ) {
          if ( u != c_att ) {
            if ( u->Owner() == c_att->Owner() ) {
              if ( u->CouldHit( c_def ) ) aamod += 10;
              else aamod += 5;

              // the unit might also help defend against the retaliation attack
              if ( NextTo( nbors[i], apos ) ) admod += 10;
            }
          } else {
            // if there's a supporter in the defender's back get another 10% plus
            Direction behind = ReverseDir( i );
            if ( nbors[behind].x != -1 ) {
              Unit *stab = map->GetUnit( nbors[behind] );
              if ( stab && (stab->Owner() == c_att->Owner()) ) {
                if ( stab->CouldHit( c_def ) ) aamod += 10;
                else aamod += 5;
              }
            }
          }
        }
      }
    }

    // check for units supporting defence (blocking), only for non-ranged attacks
    Point upos;
    Direction attdir = Hex2Dir( dpos, apos );

    // any unit can help with blocking
    if ( !map->Dir2Hex( dpos, TurnLeft(attdir), upos ) &&
         (u = map->GetUnit( upos )) && (u->Owner() == c_def->Owner()) ) ddmod += 10;
    if ( !map->Dir2Hex( dpos, TurnRight(attdir), upos ) &&
         (u = map->GetUnit( upos )) && (u->Owner() == c_def->Owner()) ) ddmod += 10;
  } else ddmod -= 5;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Combat::Resolve
// DESCRIPTION: Pop up a window displaying the raging firefight between
//              two units.
// PARAMETERS : mapwin - pointer to the map window object
// RETURNS    : pointer to the combat window
//
// NOTE       : Should the terrain modifiers also apply for both
//              attack and defence? Currently they are used for attack
//              for the attacking unit and for defence for the defending
//              unit only.
////////////////////////////////////////////////////////////////////////

CombatWindow *Combat::Resolve( MapWindow *mapwin, View *view ) {
  if ( !c_att->IsAlive() || !c_def->IsAlive() ) return NULL;

  CombatWindow *comwin = NULL;
  if ( Gam->CurrentPlayer()->Type() == HUMAN )
    comwin = new CombatWindow( this, mapwin, view );

  Point p1 = c_att->Position(), p2 = c_def->Position();
  unsigned short numatts = c_att->GroupSize(), numdefs = c_def->GroupSize(),
    dist = Distance( p1.x, p1.y, p2.x, p2.y );
  unsigned char ahits, dhits;

  if ( !c_att->IsDummy() ) {
    int i;
    unsigned char atohit, dtohit, atodef, dtodef,
                  aapool = 0, adpool = 0, dapool = 0, ddpool = 0,
                  aastr, adstr, dastr, ddstr,
                  axp = c_att->XPLevel() * 2, dxp = c_def->XPLevel() * 2;
    ahits = dhits = 0;
    atohit = MAX( 25 + axp - (dist - 1) * 2 + aamod, 1 );
    atodef = 25 + axp + admod;
    dtodef = 25 + dxp + ddmod;
    aastr = c_att->OffensiveStrength( c_def ) + axp;
    adstr = c_att->DefensiveStrength() + axp;
    dastr = c_def->OffensiveStrength( c_att ) + dxp;
    ddstr = c_def->DefensiveStrength() + dxp;

    // can the defender return fire?
    if ( (dist == 1) && c_def->CanHit( c_att ) ) {
      dtohit = 25 + dxp + damod;
    } else dtohit = 0;

    for ( i = 0; i < numatts; ++i ) {
      if ( random( 1, 100 ) <= atohit )
         aapool += aastr * random( 80, 120 ) / 100;
      if ( random( 1, 100 ) <= atodef )
         adpool += adstr * random( 80, 120 ) / 100;
    }

    for ( i = 0; i < numdefs; ++i ) {
      if ( random( 1, 100 ) <= dtohit )
        dapool += dastr * random( 80, 120 ) / 100;
      if ( random( 1, 100 ) <= dtodef )
        ddpool += ddstr * random( 80, 120 ) / 100;
    }

    // set defence pools to a minimum of the enemy unit strength;
    // that avoids division by zero and leads to somewhat sane results
    if ( adpool <= dastr ) adpool = MAX( 1, dastr );
    if ( ddpool < aastr ) ddpool = aastr;   // attacker can't have 0 strength

    // generally, the number of hits is determined by comparing the attack and
    // defence pool values. To reduce the impact of luck, however, we take the
    // average of the "randomized" and a strictly deterministic number.
    ahits = (aapool/ddpool +
            (aastr * atohit)/(ddstr * dtodef) + 1) / 2;
    dhits = (dapool/adpool +
            (dastr * dtohit)/(adstr * atodef) + 1) / 2;

    // record event
    History *hist = Gam->GetHistory();
    if ( hist ) hist->RecordCombatEvent( *this, dhits, ahits );
  } else {
    // we only have dummy units; the modifiers tell us about the casualties
    ahits = aamod;
    dhits = ddmod;
  }

  Map *map = mapwin->GetMapView()->GetMap();
  if ( c_def->Hit( ahits ) ) map->SetUnit( NULL, p2 );
  if ( c_att->Hit( dhits ) ) map->SetUnit( NULL, p1 );

  // award experience: 1 if the enemy troops were reduced, 3 if they were destroyed
  if ( !c_att->IsMine() ) {
    if ( !c_att->IsAlive() ) c_def->AwardXP( 3 );
    else if ( c_att->GroupSize() < numatts ) c_def->AwardXP( 1 );
  }
  if ( !c_def->IsMine() ) {
    if ( !c_def->IsAlive() ) c_att->AwardXP( 3 );
    else if ( c_def->GroupSize() < numdefs ) c_att->AwardXP( 1 );
  }

  if ( comwin ) {
    comwin->Draw();
    comwin->Show();
  }
  return comwin;
}


////////////////////////////////////////////////////////////////////////
// NAME       : CombatWindow::CombatWindow
// DESCRIPTION: The CombatWindow visualizes the fight between two units.
// PARAMETERS : combat - pointer to a combat object
//              mapwin - pointer to the map window; required to update
//                       the display when a unit is destroyed
//              view   - view the window will belong to
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

CombatWindow::CombatWindow( Combat *combat, MapWindow *mapwin, View *view ) :
  Window( 0, view ), mapwin( mapwin ), att( combat->c_att ), def( combat->c_def ),
    apos( att->Position() ), dpos( def->Position() ),
    startgroup1( att->GroupSize() ), startgroup2( def->GroupSize() ) {

  // calculate window dimensions
  short wwidth = (MAX( sfont->TextWidth( att->Name() ),
                       sfont->TextWidth( def->Name() ) ) + XP_ICON_WIDTH + 25) * 2;
  wwidth = MAX( wwidth, GFX_WIDTH * 6 + 50 );
  Rect win( 0, 0, wwidth, GFX_HEIGHT * 3 + sfont->Height() * 2 + 50 );
  MapView *mv = mapwin->GetMapView();

  // place window on map window
  mv->CenterOnHex( Point( (apos.x + dpos.x) / 2, (apos.y + dpos.y) / 2 ) );
  mapwin->BoxAvoidHexes( win, apos, dpos );
  SetSize( win.x, win.y, win.w, win.h );

  // highlight the fighting units
  Rect clip;
  Point punit = mv->Hex2Pixel( apos );
  clip = Rect( punit.x, punit.y, GFX_WIDTH, GFX_HEIGHT );
  clip.Clip( *mv );
  mv->DrawTerrain( IMG_CURSOR_HIGHLIGHT, mapwin, punit.x, punit.y, clip );
  punit = mv->Hex2Pixel( dpos );
  clip = Rect( punit.x, punit.y, GFX_WIDTH, GFX_HEIGHT );
  clip.Clip( *mv );
  mv->DrawTerrain( IMG_CURSOR_HIGHLIGHT, mapwin, punit.x, punit.y, clip );

  // create button widget, message areas, and unit display areas
  Rect brect( 1, h - sfont->Height() - 10, w - 2, sfont->Height() + 10 );

  msgbar1 = Rect( 4, h - sfont->Height() - 14 - brect.h, (w - 16) / 2, sfont->Height() + 6 );
  msgbar2 = Rect( msgbar1.x + msgbar1.w + 8, msgbar1.y, msgbar1.w, msgbar1.h );

  att_anchor = Rect( w/4 - GFX_WIDTH/2, 10 + GFX_HEIGHT, GFX_WIDTH, GFX_HEIGHT );
  def_anchor = Rect( (w * 3)/4 - GFX_WIDTH/2, att_anchor.y, GFX_WIDTH, GFX_HEIGHT );

  clock[0][0] = Rect( att_anchor.x, att_anchor.y - GFX_HEIGHT, GFX_WIDTH, GFX_HEIGHT );
  clock[0][1] = Rect( att_anchor.x + GFX_WIDTH - GFX_OVERLAP_X, att_anchor.y - GFX_OVERLAP_Y, GFX_WIDTH, GFX_HEIGHT );
  clock[0][2] = Rect( att_anchor.x + GFX_WIDTH - GFX_OVERLAP_X, att_anchor.y + GFX_OVERLAP_Y, GFX_WIDTH, GFX_HEIGHT );
  clock[0][3] = Rect( att_anchor.x, att_anchor.y + GFX_HEIGHT, GFX_WIDTH, GFX_HEIGHT );
  clock[0][4] = Rect( att_anchor.x - GFX_WIDTH + GFX_OVERLAP_X, att_anchor.y + GFX_OVERLAP_Y, GFX_WIDTH, GFX_HEIGHT );
  clock[0][5] = Rect( att_anchor.x - GFX_WIDTH + GFX_OVERLAP_X, att_anchor.y - GFX_OVERLAP_Y, GFX_WIDTH, GFX_HEIGHT );
  clock[1][0] = Rect( def_anchor.x, def_anchor.y - GFX_HEIGHT, GFX_WIDTH, GFX_HEIGHT );
  clock[1][1] = Rect( def_anchor.x + GFX_WIDTH - GFX_OVERLAP_X, def_anchor.y - GFX_OVERLAP_Y, GFX_WIDTH, GFX_HEIGHT );
  clock[1][2] = Rect( def_anchor.x + GFX_WIDTH - GFX_OVERLAP_X, def_anchor.y + GFX_OVERLAP_Y, GFX_WIDTH, GFX_HEIGHT );
  clock[1][3] = Rect( def_anchor.x, def_anchor.y + GFX_HEIGHT, GFX_WIDTH, GFX_HEIGHT );
  clock[1][4] = Rect( def_anchor.x - GFX_WIDTH + GFX_OVERLAP_X, def_anchor.y + GFX_OVERLAP_Y, GFX_WIDTH, GFX_HEIGHT );
  clock[1][5] = Rect( def_anchor.x - GFX_WIDTH + GFX_OVERLAP_X, def_anchor.y - GFX_OVERLAP_Y, GFX_WIDTH, GFX_HEIGHT );

  button = new ButtonWidget( GUI_CLOSE, brect.x, brect.y, brect.w, brect.h, WIDGET_DEFAULT, "_OK", this );

  Draw();
  view->Refresh();
}

////////////////////////////////////////////////////////////////////////
// NAME       : CombatWindow::Draw
// DESCRIPTION: Draw the window.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void CombatWindow::Draw( void ) {
  Window::Draw();
  DrawBox( msgbar1, BOX_RECESSED );
  DrawBox( msgbar2, BOX_RECESSED );

  DrawBox( Rect( msgbar1.x, 4, msgbar1.w, msgbar1.y - 8 ), BOX_CARVED );
  DrawBox( Rect( msgbar2.x, 4, msgbar2.w, msgbar2.y - 8 ), BOX_CARVED );

  Images[ICON_XP_BASE + att->XPLevel()]->Draw( this, msgbar1.x + 4, msgbar1.y + (msgbar1.h - XP_ICON_HEIGHT)/2 );
  Images[ICON_XP_BASE + def->XPLevel()]->Draw( this, msgbar2.x + 4, msgbar2.y + (msgbar2.h - XP_ICON_HEIGHT)/2 );

  sfont->Write( att->Name(), this, msgbar1.x + XP_ICON_WIDTH + 8,
                msgbar1.y + (msgbar1.h - sfont->Height())/2 );
  sfont->Write( def->Name(), this, msgbar2.x + XP_ICON_WIDTH + 8,
                msgbar2.y + (msgbar2.h - sfont->Height())/2 );

  // draw terrain and unit image to the center of the 'clock'
  MapView *mv = mapwin->GetMapView();
  Map *map = mv->GetMap();
  mv->DrawTerrain( map->HexImage( apos ), this, att_anchor.x, att_anchor.y, att_anchor );
  mv->DrawTerrain( map->HexImage( dpos ), this, def_anchor.x, def_anchor.y, def_anchor );
  mv->DrawUnit( att->Image(), this, att_anchor.x, att_anchor.y, att_anchor );
  mv->DrawUnit( def->Image(), this, def_anchor.x, def_anchor.y, def_anchor );

  DrawState();
}

////////////////////////////////////////////////////////////////////////
// NAME       : CombatWindow::DrawState
// DESCRIPTION: Update the two 'clocks' to display the current standing.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void CombatWindow::DrawState( void ) {
  short group1 = att->GroupSize(), group2 = def->GroupSize();
  MapView *mv = mapwin->GetMapView();

  for ( int i = NORTH; i <= NORTHWEST; ++i ) {

    if ( i < startgroup1 ) {
      mv->DrawTerrain( IMG_RECESSED_HEX, this, clock[0][i].x, clock[0][i].y, clock[0][i] );
      mv->DrawUnit( att->BaseImage() + i, this, clock[0][i].x, clock[0][i].y, clock[0][i] );

      if ( i >= group1 )
        mv->DrawFog( this, clock[0][i].x, clock[0][i].y, clock[0][i] );
    }

    if ( i < startgroup2 ) {
      mv->DrawTerrain( IMG_RECESSED_HEX, this, clock[1][i].x, clock[1][i].y, clock[1][i] );
      mv->DrawUnit( def->BaseImage() + i, this, clock[1][i].x, clock[1][i].y, clock[1][i] );

      if ( i >= group2 )
        mv->DrawFog( this, clock[1][i].x, clock[1][i].y, clock[1][i] );
    }
  }
}

