// CoMET - The Crimson Fields Map Editing Tool
// Copyright (C) 2002-2003 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

///////////////////////////////////////////////////////////////
// mission.h - the data construct to contain all level info
///////////////////////////////////////////////////////////////

#ifndef _INCLUDE_ED_MISSION_H
#define _INCLUDE_ED_MISSION_H

#include <string>
#include <fstream>
using namespace std;

#include "map.h"

class Player {
public:
  Player( unsigned char id );

  int Load( SDL_RWops *file );
  int Save( SDL_RWops *file ) const;
  int Export( ofstream &file ) const;

  void SetID( unsigned char id ) { p_id = id; }
  void SetName( const char *name )
	{ strncpy( p_name, name, 15 ); p_name[strlen(name)] = '\0'; }
  void SetBriefing( signed char briefing ) { p_briefing = briefing; }

  const char *Name( void ) const { return p_name; }
  signed char Briefing( void ) const { return p_briefing; }

private:
  unsigned char p_id;
  unsigned char p_type;
  unsigned char p_level;
  unsigned char p_success;
  signed char p_briefing;
  char p_name[16];
  char p_password[8];
};

class Event : public Node {
public:
  Event( void ) {}
  Event( unsigned char id, unsigned char type, unsigned char trigger );

  int Load( SDL_RWops *file );
  int Save( SDL_RWops *file ) const;
  int Export( ofstream &file, const UnitSet *uset ) const;

  unsigned char ID( void ) const { return e_id; }
  const char *Name( void ) const;
  unsigned char Trigger( void ) const { return e_trigger; }
  const char *TriggerName( void ) const;
  unsigned char Type( void ) const { return e_type; }

  unsigned char Player( void ) const { return e_player; }
  void SetPlayer( unsigned char player ) { e_player = player; }
  short Title( void ) const { return e_title; }
  void SetTitle( short title ) { e_title = title; }
  short Message( void ) const { return e_message; }
  void SetMessage( short msg ) { e_message = msg; }
  unsigned short Flags( void ) const { return e_flags; }
  void ToggleFlags( unsigned short flags )
       { e_flags ^= flags; }
  signed char Dependency( void ) const { return e_depend; }
  void SetDependency( signed char dep ) { e_depend = dep; }

  short GetData( unsigned short index ) const { return e_data[index]; }
  void SetData( unsigned short index, short value )
       { e_data[index] = value; }
  short GetTData( unsigned short index ) const { return e_tdata[index]; }
  void SetTData( unsigned short index, short value )
       { e_tdata[index] = value; }

private:
  unsigned char e_id;
  unsigned char e_type;
  unsigned char e_trigger;
  signed char e_depend;
  short e_tdata[3];
  short e_data[3];
  short e_title;
  short e_message;
  unsigned short e_flags;
  unsigned char e_player;
};

class Message : public Node {
public:
  Message( string txt ) : text(txt) {}

  int Save( SDL_RWops *file ) const;
  int Export( ofstream &file ) const;
  string &GetText( void ) { return text; }

private:
  string text;
};


class Mission {
public:
  Mission( void ) : p1(PLAYER_ONE), p2(PLAYER_TWO),
                    unit_set(0), terrain_set(0) {};
  Mission( const Point &size, TerrainSet *ts, UnitSet *us );
  ~Mission( void );

  Map &GetMap( void ) { return map; }
  const string &GetTitle( void ) const { return last_file_name; }
  void SetTitle( const string &title ) { last_file_name = title; }
  const char *GetPassword( void ) const
       { return (strlen(password) == 0 ? NULL : password); }
  void SetPassword( const char *pass );
  const UnitSet &GetUnitSet( void ) const { return *unit_set; }
  signed char GetLevelInfoMsg( void ) const { return level_info; }
  void SetLevelInfoMsg( signed char msg ) { level_info = msg; }
  unsigned char GetNumPlayers( void ) const
                { return ((flags & GI_AI) == 0) ? 2 : 1; }
  void SetNumPlayers( unsigned char num );
  signed char GetSequel( void ) const { return next_map; }
  void SetSequel( signed char msg ) { next_map = msg; }

  int Load( const char *filename );
  int Save( const char *filename );
  int Export( const string &filename ) const;
  unsigned short Validate( string &errors );

  Unit *CreateUnit( const UnitType *type, unsigned char pid, const Point &pos );
  void DeleteUnit( Unit *u );
  Unit *GetUnitByID( unsigned short id ) const;

  Building *CreateBuilding( unsigned char pid, const Point &pos );
  void DeleteBuilding( Building *b );
  Building *GetBuildingByID( unsigned short id ) const;

  Event *CreateEvent( unsigned char type, unsigned char trigger );
  void DeleteEvent( Event *e );
  Event *GetEventByID( unsigned short id ) const;

  unsigned short StorageLeft( Unit &u ) const;

  List &GetEvents( void ) { return events; }
  List &GetUnits( void ) { return units; }
  List &GetBuildings( void ) { return buildings; }
  List &GetMessages( void ) { return messages; }
  const char *GetMessage( short id ) const;
  Player &GetPlayer( unsigned char id )
         { return (id == PLAYER_ONE) ? p1 : p2; }

private:
  unsigned short GetUnitID( void ) const;
  unsigned short GetBuildingID( void ) const;
  unsigned char GetEventID( void ) const;

  unsigned short ValidateEvent( Event &e, string &errors ) const;
  unsigned short ValidateMap( string &errors ) const;

  unsigned short flags;
  signed char level_info;
  signed char next_map;
  unsigned short msg_num;
  unsigned long msg_len;

  Map map;
  List units;
  List buildings;
  List events;
  List messages;
  Player p1;
  Player p2;

  UnitSet *unit_set;
  TerrainSet *terrain_set;

  string last_file_name;
  char password[8];
};

#endif  // _INCLUDE_ED_MISSION_H

