// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000-2003 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// fileio.cpp
//
//  * additional code for Win32 by Adam Gates <radad@xoasis.com>
//  * additional code for WinCE/Unicode by Maik Stohn <maik@stohn.de>
////////////////////////////////////////////////////////////////////////

#ifndef _WIN32_WCE
# include <sys/stat.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "fileio.h"
#include "textbox.h"      // TLWNode class
#include "globals.h"      // for CF_SHORTNAME

////////////////////////////////////////////////////////////////////////
// NAME       : Directory::Directory
// DESCRIPTION: Open a directory and initialize the first directory
//              entry for reading. Whether the directory was
//              successfully opened can be checked using
//              Directory::IsValid().
// PARAMETERS : dir - name of the directory to open
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

Directory::Directory( const char *dir ) {
#ifdef WIN32
  string search( dir );
  append_path_delim( search );
  search += "*.*";

# ifdef UNICODE                     // unicode handling for WindowsCE
  WCHAR *wsearch = new WCHAR[search.length()+1];

  size_t len = mbstowcs( wsearch, search.c_str(), search.length() );
  if ( len < 0 ) len = 0;
  wsearch[len] = 0;

  m_Dir = FindFirstFile( wsearch, &m_Entry );

  delete [] wsearch;
# else
  m_Dir = FindFirstFile( search.c_str(), &m_Entry );
# endif
#else
  m_Entry = NULL;
  m_Dir = opendir( dir );

  if ( m_Dir ) m_Entry = readdir( m_Dir );
#endif
}

////////////////////////////////////////////////////////////////////////
// NAME       : Directory::~Directory
// DESCRIPTION: Close a directory.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

Directory::~Directory( void ) {
#ifdef WIN32
  if ( m_Dir ) FindClose( m_Dir );
#else
  if ( m_Dir ) closedir( m_Dir );
#endif
}

////////////////////////////////////////////////////////////////////////
// NAME       : Directory::GetFileName
// DESCRIPTION: Get the name of the currently selected file in the
//              directory.
// PARAMETERS : -
// RETURNS    : pointer to file name or NULL if end of directory reached
//              (or not opened).
////////////////////////////////////////////////////////////////////////

const char *Directory::GetFileName( void ) const {
#ifdef WIN32
# ifdef UNICODE                            // unicode handling for WindowsCE
  size_t len = wcstombs( (char *)m_AsciiDir, m_Entry.cFileName, wcslen(m_Entry.cFileName) );
  if ( len < 0 ) len = 0;
  ((char *)m_AsciiDir)[len] = 0;
  return m_AsciiDir;
# else
  return m_Entry.cFileName;
# endif
#else
  if ( m_Entry ) return m_Entry->d_name;
  else return NULL;
#endif
}

////////////////////////////////////////////////////////////////////////
// NAME       : Directory::GetFileNameLen
// DESCRIPTION: Get the length of the currently selected file name.
// PARAMETERS : -
// RETURNS    : length of file name in characters
////////////////////////////////////////////////////////////////////////

size_t Directory::GetFileNameLen( void ) const {
#ifdef WIN32
# ifdef UNICODE
  return wcslen( m_Entry.cFileName );
# else
  return strlen( m_Entry.cFileName );
# endif
#else
  if ( m_Entry ) {
# ifdef HAVE_DIRENT_H
    return strlen( m_Entry->d_name );
# else
    return m_Entry->d_namlen;
# endif
  }
  return 0;
#endif
}

////////////////////////////////////////////////////////////////////////
// NAME       : Directory::IsFileHidden
// DESCRIPTION: Determine whether the currently selected file is a
//              hidden file.
// PARAMETERS : -
// RETURNS    : TRUE if file is a hidden file, FALSE otherwise
////////////////////////////////////////////////////////////////////////

bool Directory::IsFileHidden( void ) const {
#ifdef WIN32
  return (m_Entry.dwFileAttributes & FILE_ATTRIBUTE_HIDDEN) != 0;
#else
  if ( m_Entry ) return ( m_Entry->d_name[0] == '.' );
  else return false;
#endif
}

////////////////////////////////////////////////////////////////////////
// NAME       : Directory::NextFile
// DESCRIPTION: Initialize the next file in the directory for
//              examination.
// PARAMETERS : -
// RETURNS    : TRUE if another file was found, FALSE if the end of the
//              directory was reached.
////////////////////////////////////////////////////////////////////////

bool Directory::NextFile( void ) {
#ifdef WIN32
  return FindNextFile( m_Dir, &m_Entry ) != FALSE;
#else
  if ( m_Dir ) m_Entry = readdir( m_Dir );
  return m_Entry != NULL;
#endif
}


////////////////////////////////////////////////////////////////////////
// NAME       : append_path_delim
// DESCRIPTION: Append a path delimiter to the end of the string if
//              it isn't already there.
// PARAMETERS : path - string containing a path
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void append_path_delim( string &path ) {
  if ( path.at( path.length()  - 1 ) != PATHDELIM ) path += PATHDELIM;
}

////////////////////////////////////////////////////////////////////////
// NAME       : make_dir
// DESCRIPTION: Create a directory.
// PARAMETERS : dir - name of the directory to create
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void make_dir( const char *dir ) {
#ifdef WIN32
# ifdef UNICODE
  WCHAR *wdir = new WCHAR[strlen(dir)+1];
  size_t len = mbstowcs( wdir, dir, strlen(dir) );
  if ( len < 0 ) len = 0;
  wdir[len] = 0;
  CreateDirectory( wdir, NULL );                         
  delete [] wdir;
# else
  CreateDirectory( dir, NULL );
# endif
#else
  mkdir( dir, S_IRWXU|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH );
#endif
}

////////////////////////////////////////////////////////////////////////
// NAME       : create_config_dir
// DESCRIPTION: Create a configuration and save directory in the user's
//              home dir (UNIX & Co.) or the crimson program directory
//              (Win32).
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void create_config_dir( void ) {
  string confd = get_config_dir();

  // does the directory exist?
  Directory cfdir( confd.c_str() );
  if ( !cfdir.IsValid() ) make_dir( confd.c_str() );

  // now check for the saved games directory
  append_path_delim( confd );
  confd.append( "games" );

  Directory svdir( confd.c_str() );
  if ( !svdir.IsValid() ) make_dir( confd.c_str() );
}

////////////////////////////////////////////////////////////////////////
// NAME       : get_home_dir
// DESCRIPTION: Get the name of the user's home directory.
// PARAMETERS : -
// RETURNS    : the user's home directory if any
////////////////////////////////////////////////////////////////////////

string get_home_dir( void ) {
#ifdef WIN32
# ifdef _WIN32_WCE
  return "";
# else
  char dir[MAX_PATH];
  int length = GetEnvironmentVariable("HOME", dir, MAX_PATH);
  if ( length == 0 ) {
    length = GetEnvironmentVariable("HOMEDRIVE", dir, MAX_PATH);
    length += GetEnvironmentVariable("HOMEPATH", dir + length, MAX_PATH - length);
  }

  if ( length == 0 ) return "";
  else return dir;
# endif
#else
  return getenv( "HOME" );
#endif
}

////////////////////////////////////////////////////////////////////////
// NAME       : get_config_dir
// DESCRIPTION: Get the name of the configuration directory.
// PARAMETERS : -
// RETURNS    : config directory
////////////////////////////////////////////////////////////////////////

string get_config_dir( void ) {
  string confd;
  string homed( get_home_dir() );

#ifdef WIN32
  if ( homed.empty() ) confd = get_data_dir();
#else
  if ( homed.empty() ) confd.append( CURRENTDIR );
#endif
  else {
    confd.append( homed );
    append_path_delim( confd );
#ifndef WIN32
    confd += '.';
#endif
    confd.append( CF_SHORTNAME );
  }
  append_path_delim( confd );

  return confd;
}

////////////////////////////////////////////////////////////////////////
// NAME       : get_data_dir
// DESCRIPTION: Get the name of the directory containing the data files.
// PARAMETERS : -
// RETURNS    : data directory name
////////////////////////////////////////////////////////////////////////

string get_data_dir( void ) {

#ifdef WIN32
  char dir[MAX_PATH];

# ifdef UNICODE
  WCHAR wdir[MAX_PATH];
  GetModuleFileName( NULL, wdir, MAX_PATH );
  size_t len = wcstombs( dir, wdir, wcslen(wdir) );
  if( len < 0 ) len = 0;
  dir[len] = 0;
# else
  GetModuleFileName( NULL, dir, MAX_PATH );
# endif
  {
    // Remove the file name
    char *l = dir;
    char *c = dir;
    while( *c != '\0' ) {
      if ( *c == PATHDELIM ) l = c;
      ++c;
    }
    ++l;
    *l = '\0';
  }
  return dir;
#elif defined CF_DATADIR
  return CF_DATADIR;
#else
  return CURRENTDIR;
#endif
}

////////////////////////////////////////////////////////////////////////
// NAME       : get_save_dir
// DESCRIPTION: Get the name of the directory to save games in.
// PARAMETERS : -
// RETURNS    : saved games directory
////////////////////////////////////////////////////////////////////////

string get_save_dir( void ) {
  string saved( get_config_dir() );

  saved.append( "games" );
  append_path_delim( saved );

  return saved;
}

////////////////////////////////////////////////////////////////////////
// NAME       : get_sounds_dir
// DESCRIPTION: Get the name of the directory containing the sound
//              effects.
// PARAMETERS : -
// RETURNS    : sound effects directory
////////////////////////////////////////////////////////////////////////

string get_sounds_dir( void ) {
  string sndd( get_data_dir() );
  append_path_delim( sndd );
  sndd.append( "sound" );
  append_path_delim( sndd );
  return sndd;
}

////////////////////////////////////////////////////////////////////////
// NAME       : get_levels_dir
// DESCRIPTION: Get the name of the directory containing the map files.
// PARAMETERS : -
// RETURNS    : levels directory name
////////////////////////////////////////////////////////////////////////

string get_levels_dir( void ) {
  string levd( get_data_dir() );
  append_path_delim( levd );
  levd.append( "levels" );
  append_path_delim( levd );
  return levd;
}

////////////////////////////////////////////////////////////////////////
// NAME       : get_home_levels_dir
// DESCRIPTION: Get the name of an optional directory in the user's home
//              directory containing additional map files.
// PARAMETERS : -
// RETURNS    : levels directory name or an empty string if it does not
//              exist
////////////////////////////////////////////////////////////////////////

string get_home_levels_dir( void ) {
  string hld("");

#ifndef WIN32
  string homed( get_home_dir() );
  if ( !homed.empty() ) {
    hld = get_config_dir();
    hld.append( "levels" );
    append_path_delim( hld );
  }
#endif

  return hld;
}

////////////////////////////////////////////////////////////////////////
// DESCRIPTION: Check whether a given filename is already in use.
// PARAMETERS : file - filename (including full path) to check for
// RETURNS    : TRUE if file exists, FALSE otherwise
////////////////////////////////////////////////////////////////////////

bool exists( const char *file ) {
  // try to open the file
  FILE *fd = fopen( file, "r" );
  if ( fd ) fclose( fd );
  return fd != NULL;
}

////////////////////////////////////////////////////////////////////////
// DESCRIPTION: Extract the name of a file from a given path.
// PARAMETERS : path - filename (including full or partial path)
// RETURNS    : file part of the path
////////////////////////////////////////////////////////////////////////

string file_part( string path ) {
  int pathend = path.rfind( PATHDELIM );
  return path.substr( pathend + 1 );
}

////////////////////////////////////////////////////////////////////////
// NAME       : create_files_list
// DESCRIPTION: Read the names of all files with a specified suffix in
//              a directory and put them into a list.
// PARAMETERS : dirname - name of the directory to go through
//              suffix  - suffix for the files to grab (may be NULL)
//              list    - list to append nodes to
// RETURNS    : number of files found or -1 on error; the nodes added
//              to the list are of the TLWNode class.
////////////////////////////////////////////////////////////////////////

short create_files_list( const char *dirname, const char *suffix, TLWList &list ) {
  TLWNode *node;
  string *full;
  short num = 0, suflen = 0;
  if ( suffix ) suflen = strlen( suffix );

  Directory dir( dirname );
  if ( dir.IsValid() ) {
    do {
      int len = dir.GetFileNameLen();

      if ( (suflen <= len) &&
           (strcasecmp( &(dir.GetFileName()[len-suflen]), suffix ) == 0) ) {

        node = new TLWNode( dir.GetFileName() );
        full = new string( dirname );
        if ( node && full ) {
          append_path_delim( *full );        // store full file path
          full->append( dir.GetFileName() );
          node->SetUserData( full );

          list.InsertNodeSorted( node );
          ++num;
        } else {
          delete node;
          delete full;
          num = -1;
          break;
        }
      }
    } while ( dir.NextFile() );
  }
  return num;
}

