/*
 *  methods for LUKS-related key-management for cryptmount
 *  $Revision: 222 $, $Date: 2008-10-03 12:31:16 +0100 (Fri, 03 Oct 2008) $
 *  (C)Copyright 2005-2008, RW Penney
 */

/*
    This file is part of cryptmount

    cryptmount is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    cryptmount is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <config.h>

#include <ctype.h>
#if HAVE_DLFCN && USE_MODULES
#  include <dlfcn.h>
#endif
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "armour.h"
#include "cryptmount.h"
#include "looputils.h"
#include "utils.h"
#ifdef TESTING
#  include "cmtesting.h"
#endif


typedef struct {
    unsigned keyslot;
} luks_overrides_t;


/*
 *  ==== LUKS key-management routines ====
 */

#if USE_LUKSCOMPAT
#  if !USE_MODULES || defined(AS_MODULE)
#    include "luks/luks.h"
#    if HAVE_UUID
#      include <uuid/uuid.h>
#  endif

extern struct setup_backend setup_libdevmapper_backend;

#if defined(TESTING) && defined(AS_MODULE)
cm_testinfo_t *test_ctxtptr;
#endif


static int kmluks_init_algs()
{
    /* nothing needed */
    return 0;
}


static int kmluks_free_algs()
{
    /* nothing needed */
    return 0;
}


void kmluks_mk_default(keyinfo_t *keyinfo)
{
    if (keyinfo == NULL) return;

    if (keyinfo->digestalg == NULL) {
        keyinfo->digestalg = cm_strdup("md5");
    }

    if (keyinfo->cipheralg == NULL) {
        keyinfo->cipheralg = cm_strdup("blowfish");
    }
}


static int kmluks_hdrvalid(FILE *fp_key)
    /* Check whether a valid LUKS header is present */
{   const unsigned char luks_magic[] = LUKS_MAGIC;
    char buff[32];
    int flg = 0;

    if (fp_key == NULL) return 0;

    if (fread((void*)buff, (size_t)LUKS_MAGIC_L, (size_t)1, fp_key) == 1) {
        fseek(fp_key, -((long)LUKS_MAGIC_L), SEEK_CUR);
        flg = (strncmp(buff, (const char*)luks_magic,
                (size_t)LUKS_MAGIC_L) == 0);
    }

    return flg;
}


static int kmluks_is_compat(const keyinfo_t *keyinfo, FILE *fp_key)
{   int flg=0;

    if (keyinfo->format != NULL) {
        flg = (strcmp(keyinfo->format, "luks") == 0);
    } else {
        /* Check header of existing key-file: */
        flg |= kmluks_hdrvalid(fp_key);
    }

    return flg;
}


static unsigned kmluks_get_properties(const keyinfo_t *keyinf)
{   unsigned props;
    FILE *fp;

    props = KM_PROP_HASPASSWD | KM_PROP_FIXEDLOC;

    fp = fopen(keyinf->filename, "rb");
    if (fp != NULL) {
        if (kmluks_hdrvalid(fp)) {
            props |= KM_PROP_FORMATTED;
        }
        fclose(fp);
    }

    return props;
}


static int kmluks_get_key(const char *ident, const keyinfo_t *keyinfo,
            const km_pw_context_t *pw_ctxt,
            unsigned char **key, int *keylen, FILE *fp_key,
            km_overrides_t *overrides)
    /* Extract key from LUKS header file */
{   char *passwd=NULL;
    const char *tgtdev=NULL;
    struct luks_phdr lukshdr;
    struct luks_masterkey *lukskey=NULL;
    int killloop=0, slot, eflag=ERR_NOERROR;
    luks_overrides_t *luksor;
    const size_t namesz = 128;

    km_get_passwd(ident, pw_ctxt, &passwd, 0, 0);

    if (blockify_file(keyinfo->filename, O_RDONLY, NULL, &tgtdev, &killloop) != ERR_NOERROR) {
        fprintf(stderr, _("Failed to create loop device for LUKS keyfile\n"));
        goto bail_out;
    }

    slot = LUKS_open_any_key(tgtdev, passwd, strlen(passwd),
                    &lukshdr, &lukskey, &setup_libdevmapper_backend);
    if (slot < 0) {
        fprintf(stderr, _("Failed to extract LUKS key for \"%s\"\n"), ident);
        eflag = ERR_BADDECRYPT;
        goto bail_out;
    }

    /* Extract cipher-algorithm parameter from LUKS header: */
    overrides->start = lukshdr.payloadOffset;
    overrides->cipher = (char*)malloc(namesz);
    snprintf(overrides->cipher, namesz, "%s-%s",
            lukshdr.cipherName, lukshdr.cipherMode);
    overrides->ivoffset = 0;
    overrides->mask = KM_OR_START | KM_OR_CIPHER | KM_OR_IVOFFSET;
    luksor = (luks_overrides_t*)malloc(sizeof(luks_overrides_t));
    luksor->keyslot = slot;
    overrides->km_data = (void*)luksor;


    /* Take copy of LUKS master-key: */
    *keylen = lukskey->keyLength;
    *key = (unsigned char*)sec_realloc((void*)*key, (size_t)lukskey->keyLength);
    memcpy((void*)*key, (const void*)lukskey->key, (size_t)*keylen);

  bail_out:

    unblockify_file(&tgtdev, killloop);
    if (passwd != NULL) sec_free((void*)passwd);
    if (lukskey != NULL) LUKS_dealloc_masterkey(lukskey);

    return eflag;
}


static int kmluks_put_key(const char *ident, const keyinfo_t *keyinfo,
            const km_pw_context_t *pw_ctxt,
            const unsigned char *key, const int keylen, FILE *fp_key,
            const km_overrides_t *overrides)
    /* Store or create key in LUKS header */
{
#if HAVE_UUID
    char msgbuff[1024], *passwd=NULL;
    const char *tgtdev=NULL;
    struct luks_phdr lukshdr;
    struct luks_masterkey *lukskey=NULL;
    unsigned keyslot=0;
    luks_overrides_t *luksor=NULL;
    int formatting=0, killloop=0, PBKDF2its, eflag=ERR_NOERROR;

    formatting = (overrides == NULL) && !kmluks_hdrvalid(fp_key);

    if (overrides != NULL) {
        luksor = (luks_overrides_t*)overrides->km_data;
    }

    if (formatting) {
#ifndef TESTING
        snprintf(msgbuff, sizeof(msgbuff), _("Formatting \"%s\", will probably destroy all existing data"), keyinfo->filename);
        if (!cm_confirm(msgbuff)) {
            eflag = ERR_ABORT;
            goto bail_out;
        }
#endif  /* !TESTING */
    }

    eflag = km_get_passwd(ident, pw_ctxt, &passwd, 1, 1);
    if (eflag != ERR_NOERROR) goto bail_out;

    eflag = blockify_file(keyinfo->filename, O_RDWR, NULL, &tgtdev, &killloop);
    if (eflag != ERR_NOERROR) {
        fprintf(stderr, _("Failed to create loop device for LUKS keyfile\n"));
        goto bail_out;
    }

    lukskey = LUKS_alloc_masterkey(keylen);
    memcpy((void*)lukskey->key, (const void*)key, (size_t)keylen);

    if (formatting) {
        /* Format new partition */
        keyslot = 0;

        if (LUKS_generate_phdr(&lukshdr, lukskey, "aes", "cbc-plain", LUKS_STRIPES, 0) < 0) {
            fprintf(stderr, _("Failed to create LUKS header for \"%s\"\n"), ident);
            eflag = ERR_BADDEVICE;
            goto bail_out;
        }
    } else {
        /* Changing password on existing partition */
        keyslot = (luksor != NULL ? luksor->keyslot : 0);

        if (LUKS_read_phdr(tgtdev, &lukshdr) < 0) {
            fprintf(stderr, _("Failed to read LUKS header for \"%s\"\n"), ident);
            eflag = ERR_BADDEVICE;
            goto bail_out;
        }

        lukshdr.keyblock[keyslot].active = LUKS_KEY_DISABLED;
    }

    printf(_("Setting password on LUKS keyslot-%u\n"), keyslot);
    PBKDF2its = (int)(1.0 * LUKS_benchmarkt_iterations() + 0.5);
    lukshdr.keyblock[keyslot].passwordIterations = (PBKDF2its >= 1 ? PBKDF2its : 1);

    if (LUKS_set_key(tgtdev, keyslot, passwd, strlen(passwd), &lukshdr, lukskey, &setup_libdevmapper_backend) < 0) {
        fprintf(stderr, _("Failed to create LUKS key for \"%s\"\n"), ident);
        fprintf(stderr, "LUKS error: %s\n", get_error());
        eflag = ERR_BADENCRYPT;
        goto bail_out;
    }

  bail_out:

    unblockify_file(&tgtdev, killloop);
    if (lukskey != NULL) LUKS_dealloc_masterkey(lukskey);
    if (passwd != NULL) sec_free((void*)passwd);

    return eflag;

#else   /* !HAVE_UUID */

    fprintf(stderr, _("Writing LUKS keys is not possible unless cryptmount has been built with the uuid-dev package installed\n"));

    return ERR_BADKEYFORMAT;
#endif  /* HAVE_UUID */
}



static void *kmluks_md_prepare(void)
{   cm_sha1_ctxt_t *mdcontext;

    mdcontext = cm_sha1_init();

    return (void*)mdcontext;
}


static void kmluks_md_block(void *state, unsigned char *buff, size_t len)
{   cm_sha1_ctxt_t *ctxt=(cm_sha1_ctxt_t*)state;

    /* apply built-in SHA1 hash to data-block: */
    cm_sha1_block(ctxt, buff, len);
}


static void kmluks_md_final(void *state, unsigned char **mdval, size_t *mdlen)
{   cm_sha1_ctxt_t *ctxt=(cm_sha1_ctxt_t*)state;

    cm_sha1_final(ctxt, mdval, mdlen);
}


static void kmluks_md_release(void *state)
{   cm_sha1_ctxt_t *ctxt=(cm_sha1_ctxt_t*)state;

    cm_sha1_free(ctxt);
}


keymanager_t keymgr_luks = {
    "luks", 0,   kmluks_init_algs, kmluks_free_algs,
                      kmluks_mk_default, kmluks_is_compat,
                      kmluks_get_properties,
                      kmluks_get_key, kmluks_put_key,
                      kmluks_md_prepare, kmluks_md_block,
                      kmluks_md_final, kmluks_md_release,
    NULL
#ifdef TESTING
    , NULL, NULL, (CM_READONLY)
#endif
};

#  endif    /* !USE_MODULES || defined(AS_MODULE) */
#endif  /* USE_LUKSCOMPAT */


#ifndef AS_MODULE

#if defined(TESTING)
#  define MOD_PATH "./cm-luks.so"
#else
#  define MOD_PATH CM_MODULE_DIR "/cm-luks.so"
#endif

keymanager_t *kmluks_gethandle()
{
#if USE_LUKSCOMPAT
#  if USE_MODULES
    KM_GETHANDLE(MOD_PATH, "keymgr_luks");
#  else
    return &keymgr_luks;
#  endif
#else
    return NULL;
#endif
}

#endif  /* !AS_MODULE */

/* vim: set ts=4 sw=4 et: */

/*
 *  (C)Copyright 2005-2008, RW Penney
 */
