/*
 * The Cryptonit security software suite is developped by IDEALX
 * Cryptonit Team (http://IDEALX.org/ and http://cryptonit.org).
 *
 * Copyright 2003-2006 IDEALX
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version
 * 2 as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301, USA. 
 *
 * In addition, as two special exceptions:
 *
 * 1) IDEALX S.A.S gives permission to:
 *  * link the code of portions of his program with the OpenSSL library under
 *    certain conditions described in each source file
 *  * distribute linked combinations including the two, with respect to the
 *    OpenSSL license and with the GPL
 *
 * You must obey the GNU General Public License in all respects for all of the
 * code used other than OpenSSL. If you modify file(s) with this exception,
 * you may extend this exception to your version of the file(s), but you are
 * not obligated to do so. If you do not wish to do so, delete this exception
 * statement from your version, in all files (this very one along with all
 * source files).

 * 2) IDEALX S.A.S acknowledges that portions of his sourcecode uses (by the
 * way of headers inclusion) some work published by 'RSA Security Inc.'. Those
 * portions are "derived from the RSA Security Inc. PKCS #11Cryptographic
 * Token Interface (Cryptoki)" as described in each individual source file.
 */
#include "AddCrlDlg.hh"
#include "Common.hh"
#include <wx/wx.h>
#include <wx/dir.h>
#include <wx/url.h>
#include <wx/filename.h>
#include <wx/stream.h>
#include <wx/ffile.h>

#ifndef __WXMSW__ // en attendant
#include "pics/folder_mini.xpm"
#endif

BEGIN_EVENT_TABLE (AddCrlDlg, CryptonitDlg)
    EVT_BUTTON(ACD_FILE_BTN_ID , AddCrlDlg::onFileBtn)
    EVT_RADIOBUTTON(ACD_FILE_RADIO_ID , AddCrlDlg::onFile)
    EVT_RADIOBUTTON(ACD_URI_RADIO_ID , AddCrlDlg::onUri)
    EVT_RADIOBUTTON(ACD_LDAP_RADIO_ID , AddCrlDlg::onLDAP)
END_EVENT_TABLE()
    

    
AddCrlDlg::AddCrlDlg(wxWindow *parent, 
		     wxWindowID id, 
		     const wxString& title,
		     const wxString& defaultLDAPDN,
		     const wxString& defaultURI
		     )
    : CryptonitDlg(parent, id, title)
{
    
    fileRadio = new wxRadioButton(this, ACD_FILE_RADIO_ID, _("from file"),wxDefaultPosition,wxDefaultSize, wxRB_GROUP );
    uriRadio = new wxRadioButton(this, ACD_URI_RADIO_ID, _("from an URI") );
    ldapRadio = new wxRadioButton(this, ACD_LDAP_RADIO_ID, _("from a LDAP server") );
    
    fileRadio->SetValue(true);
    uriRadio->SetValue(false);
    ldapRadio->SetValue(false);
    
    wxStaticBox *box = new wxStaticBox(this , -1 , _("Import a new CRL"));
    wxStaticBoxSizer *crlSizer = new wxStaticBoxSizer(box, wxVERTICAL);
    
    crlSizer->Add(fileRadio , 0 , wxEXPAND | wxALL ,5);
    
    
    wxBoxSizer *fileSizer = new wxBoxSizer(wxHORIZONTAL);
    file = new wxTextCtrl(this,-1,_T(""),wxDefaultPosition , wxSize(250,20));
    wxBitmap dirIcon = wxBITMAP(folder_mini);
#ifdef __WXMSW__
    dirIcon.SetMask( new wxMask(dirIcon, wxColour(0xC0,0xC0,0xC0)) );
#endif
    fileBtn = new wxBitmapButton(this,ACD_FILE_BTN_ID,dirIcon);
    fileSizer->Add(file,0,wxCENTER |wxBOTTOM | wxRIGHT | wxTOP, 5);
    fileSizer->Add(fileBtn,0,wxCENTER |wxBOTTOM | wxLEFT | wxTOP,5);

    crlSizer->Add(fileSizer , 0 , wxEXPAND | wxALL , 5);
    
    
    crlSizer->Add(uriRadio, 0, wxEXPAND | wxALL , 5);
    uri = new wxTextCtrl(this,-1, defaultURI ,wxDefaultPosition , wxSize(250,20));
    crlSizer->Add(uri, 0 , wxEXPAND | wxALL , 5);
    

    uri->Disable();


    /* LDAP */
    crlSizer->Add(ldapRadio, 0, wxEXPAND | wxALL , 5);
    wxFlexGridSizer *ldapServerSizer = new wxFlexGridSizer(4, 1);

    /* Server */
    ldapServerSizer->Add( new wxStaticText(this, -1, _("Server:")), 0, wxEXPAND | wxALL | wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL, 5);
    serverName = new wxTextCtrl(this, -1, _T(""), wxDefaultPosition, wxSize(250,20));
    serverName->Disable();
    ldapServerSizer->Add( serverName, 0, wxEXPAND | wxALL| wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL, 5);

    /* Port */
    ldapServerSizer->Add( new wxStaticText(this, -1, _("Port:")), 0, wxEXPAND | wxALL | wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL, 5);
    serverPort = new wxTextCtrl(this, -1, _T("389"), wxDefaultPosition, wxSize(50,20));
    serverPort->Disable();
    ldapServerSizer->Add( serverPort, 0, wxALL| wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL, 5);

    crlSizer->Add( ldapServerSizer, 0, wxEXPAND | wxALL, 5);


    /* Base DN */
    wxFlexGridSizer *ldapDNCNSizer = new wxFlexGridSizer(2, 2);
    ldapDNCNSizer->Add( new wxStaticText(this, -1, _("Base distinguished name:")), 0, wxEXPAND | wxALL | wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL, 5);
    baseDN = new wxTextCtrl(this, -1, defaultLDAPDN, wxDefaultPosition, wxSize(400,20));
    baseDN->Disable();
    ldapDNCNSizer->Add( baseDN, 0, wxEXPAND | wxALL | wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL, 5);


    /* Authorities Filter */
    ldapDNCNSizer->Add( new wxStaticText(this, -1, _("Authority search filter:")), 0, wxEXPAND | wxALL | wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL, 5);
    authorityFilter = new wxTextCtrl(this, -1, _T(""), wxDefaultPosition, wxSize(400,20));
    authorityFilter->Disable();
    ldapDNCNSizer->Add( authorityFilter, 0, wxEXPAND | wxALL | wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL, 5);
    crlSizer->Add( ldapDNCNSizer, 0, wxEXPAND | wxALL, 5);


    mainSizer->Add( crlSizer, 1 ,wxEXPAND | wxCENTER | wxALL , 3 );
    
}


void AddCrlDlg::onFile(wxCommandEvent &WXUNUSED(event)){
    uri->Disable();

    serverName->Disable();
    serverPort->Disable();
    baseDN->Disable();
    authorityFilter->Disable();
    
    file->Enable();
    fileBtn->Enable();
}


void AddCrlDlg::onUri(wxCommandEvent &WXUNUSED(event)){
    file->Disable();
    fileBtn->Disable();

    serverName->Disable();
    serverPort->Disable();
    baseDN->Disable();
    authorityFilter->Disable();

    uri->Enable();
}


void AddCrlDlg::onLDAP(wxCommandEvent &WXUNUSED(event)){
    file->Disable();
    fileBtn->Disable();

    uri->Disable();

    serverName->Enable();
    serverPort->Enable();
    baseDN->Enable();
    authorityFilter->Enable();
}


void AddCrlDlg::onFileBtn(wxCommandEvent &WXUNUSED(event)){
    wxString filename = wxFileSelector(_("Choose a CRL to import"), _T("") , _T("") , _T("crl"), _T("CRL (*.crl)|*.crl|PEM encoded CRL (*.pem)|*.pem|"));
    if ( !filename.empty() )
	file->SetValue(filename);
}


wxString AddCrlDlg::getFile(){
    return file->GetValue();
}


wxString AddCrlDlg::getURI(){
    return uri->GetValue();
}
    

wxArrayString* AddCrlDlg::getLDAP() {
    wxArrayString* array = new wxArrayString();

    array->Add(serverName->GetValue());
    array->Add(serverPort->GetValue());
    array->Add(baseDN->GetValue());
    array->Add(authorityFilter->GetValue());
    
    return array;
}

bool AddCrlDlg::isFile(){
    return fileRadio->GetValue();
}

bool AddCrlDlg::isURI(){
    return uriRadio->GetValue();
}

bool AddCrlDlg::isLDAP(){
    return ldapRadio->GetValue();
}


// fetching crl from uri
wxString AddCrlDlg::fetchCrl( wxString uri ){
    wxURL url(uri);
    wxInputStream *inStream;
    char buffer[1024];
    wxFileName temp = wxFileName::CreateTempFileName(_T(""));
    FILE *target = fopen(wx2std(temp.GetFullPath()).c_str(),"wb");
    
    //fetching crl from uri
    inStream = url.GetInputStream();
    if (inStream != NULL){
	for( ; ; ){
	    inStream->Read(buffer,1024).Eof();
	    if(inStream->LastRead() <=0) break;
	    fwrite(buffer, sizeof(char) , inStream->LastRead() , target);
	}
	fclose(target);
	
	
	Crl *crl = new Crl();
	if(crl->load(wx2std(temp.GetFullPath()).c_str()) != SUCCESS){
	    delete crl;
	    wxRemoveFile(temp.GetFullPath());
	    wxMessageBox(_("Unable to load the CRL!"), _("error"), wxOK | wxICON_ERROR );
	    return _T("");
	}
	delete crl;
	return temp.GetFullPath();
    }
    fclose( target );
    
    wxMessageBox(_("Unable to fetch CRL, network unreachable"), _("error"), wxOK | wxICON_ERROR );
    return _T("");


}

#define DISPLAY_CRL_IN_CA_TAB


bool AddCrlDlg::deleteCrl( std::string *name , Certificate c , User *user ){

#ifdef DISPLAY_CRL_IN_CA_TAB
  std::string crl_issuer_dn =  c.getSubjectName().getValues
    (DN_DISPLAY_LONG | DN_DISPLAY_VALUE, ',');
#else
  std::string crl_issuer_dn =  c.getIssuerName().getValues
    (DN_DISPLAY_LONG | DN_DISPLAY_VALUE, ',');
#endif

 //finding uri index 
  std::vector<std::string> crlValues = user->authorities->getAuthority( crl_issuer_dn )->getAttributeValues( "CRL" );
  std::vector<std::string> uriValues = user->authorities->getAuthority( crl_issuer_dn )->getAttributeValues( "URI" );
  std::vector<std::string>::iterator crlValuesIt =  crlValues.begin();
  std::vector<std::string>::iterator uriValuesIt =  uriValues.begin();
	
  int i=0;
  //try to find the crl in the config  vector
  if( crlValues.size() > 0){
      while(crlValuesIt != crlValues.end() && *crlValuesIt != *name){
	  i++;
	  crlValuesIt++;
	  uriValuesIt++;
      }
      
      //removing crl and uri
      if( *crlValuesIt == *name ){
	  crlValues.erase( crlValuesIt );
	  uriValues.erase( uriValuesIt );
      }
  }
  //updating config : removing crl entry
  if(!user->authorities->setInfos(  crl_issuer_dn, "CRL", crlValues)){
#ifdef DEBUG
      std::cout << "delete CRL: error" << std::endl;
#endif
      return false;
  }	
  
  //removing uri entry
  if(!user->authorities->setInfos( crl_issuer_dn, "URI", uriValues)){
#ifdef DEBUG	  
      std::cout << "delete uri: error" << std::endl;
#endif      
      return false;
  }	    
  
  wxFileName crlFile(std2wx(*name));
  crlFile=wxMyPrepend(crlFile,std2wx(user->getCRLDir()));
  if(!wxRemoveFile(crlFile.GetFullPath())){
#ifdef DEBUG	 
      std::cout << "delete: error"<< std::endl;
#endif      
      return false;
  }
  
  return true;

}


int AddCrlDlg::updateCrl( wxString uri , Certificate c , User *user){
    
    Crl *crl= new Crl();
    wxString tmpFile;
    int ret = SUCCESS;

    if(uri != _T("") ){
	// Is a LDAP URI ?
				if( uri.Left(7).IsSameAs( _T("ldap://"), false ) ) {
	    wxString myUri( uri );
	    // We want to use the ldapcache backend
	    myUri.Replace( _T("ldap://"), _T("ldapcache://") );
#ifdef DEBUG
	    std::cout << "Update from ldap " << myUri << std::endl;
#endif
	
	    DirectoryService *ds;
	    ds = ds->factory( wx2std(myUri) );
	
	    if( ! ds->read( NULL ) ) {
#ifdef DEBUG		    
					std::cout << "Ploof!" << std::endl;
#endif					
	    } else {
#ifdef DEBUG		    
		std::cout << "nbEntry = " << ds->getNbEntry() << std::endl;
		std::cout << "nbAttributes = " << ds->begin().second()->getNbAttribute() << std::endl; //getAttributeFirstValue(
		std::cout << "CRL = " << ds->begin().second()->begin().second()->getFirstValue().size() << std::endl;
#endif		
	    
		// Store the CRL
		wxFileName tmp = wxFileName::CreateTempFileName(_T(""));
		wxFFile target( tmp.GetFullPath(), _T("wb") );
		target.Write( ds->begin().second()->begin().second()->getFirstValue().c_str(),
			      ds->begin().second()->begin().second()->getFirstValue().size() );
	    
		delete ds;
	    
		if( ! target.Close() ) {
#ifdef DEBUG			
		    std::cout << "Cannot write CRL on disk." << std::endl;
#endif		    
		    delete crl;
		    return ERROR_CRL_WRITE_FILE;
		}
	    
		if( crl->load( wx2std(tmp.GetFullPath()).c_str() ) != SUCCESS ) {
		    wxRemoveFile( tmp.GetFullPath() );
		    delete crl;
		    return ERROR_CRL_LOAD;
		}
	    }
	} else {
	    tmpFile = fetchCrl(uri);
	    if(crl->load(wx2std(tmpFile).c_str()) != SUCCESS) {
		wxRemoveFile( tmpFile );
		delete crl;
		return ERROR_CRL_LOAD;
	    }
	}
	
	ret = saveCrl( uri , crl , c , user );
	wxRemoveFile(tmpFile);	    
	delete crl;
    }
    return ret;
}


int AddCrlDlg::saveCrl( wxString uri , Crl *crl , Certificate c , User *user ){

#ifdef DISPLAY_CRL_IN_CA_TAB
    std::string crl_issuer_dn =  c.getSubjectName().getValues
      (DN_DISPLAY_LONG | DN_DISPLAY_VALUE, ',');
#else
    std::string crl_issuer_dn =  c.getIssuerName().getValues
      (DN_DISPLAY_LONG | DN_DISPLAY_VALUE, ',');
#endif

    // getting crl vector (filename)
#ifdef DEBUG
    std::cout << crl_issuer_dn << std::endl;
#endif    
    
    if( ! user->authorities->getAuthority( crl_issuer_dn ) ){
	return ERROR_CRL_NO_CA_ISSUER;
    }

    std::vector<std::string> crlValues = user->authorities->getAuthority
	( crl_issuer_dn )->getAttributeValues( "CRL" );

    //build crl hash
    std::vector<std::string>::iterator valuesIt ;
    std::string hash = crl->getHash();
    hash +=".der";
    bool unique = true;
    
    //check if this hash is already present in the config file
    for(valuesIt = crlValues.begin() ; valuesIt != crlValues.end() ; valuesIt++){
	if(*valuesIt == hash) unique = false;
    }

#ifdef DISPLAY_CRL_IN_CA_TAB
    if(!crl->isIssued(c)) {
      return ERROR_CRL_WRONG_ISSUER;
    }
#else
   
    // loading available CAs
    std::vector<Certificate> *certList = new std::vector<Certificate>;
    std::vector<Certificate> *vect;
    wxDir dir(wxGetCwd());
    wxArrayString filename;
    dir.GetAllFiles(std2wx(user->getCACertificatesDir()), &filename, wxEmptyString, wxDIR_FILES);
    for(unsigned int i = 0 ; i < filename.GetCount() ; i++){
	Certificate cert;
	if(cert.load(wx2std(filename.Item(i)).c_str())==SUCCESS){
#ifdef DEBUG		
	    std::cout<<filename.Item(i)<<std::endl;
#endif	    
	    certList->push_back(cert);
	}
    }
    
    vect = c.getPathToRoot(certList);
    std::vector<Certificate>::iterator certIt;
    
    //check if this crl has been issued for/by the current CA certificate.
    bool issuedOK = false;
    for(  certIt = vect->begin() ; certIt != vect->end() ; certIt++ ) {
	Certificate *ce = new Certificate( *certIt );
	if(crl->isIssued( *ce ) && c.isIssued( *ce ) == X509_V_OK){
	    issuedOK = true ;
	    delete ce ;
	    break;
	}
	delete ce;
    }
    
    if(! issuedOK ) {
	return ERROR_CRL_WRONG_ISSUER;
    }
#endif    
    
    //build crl filename
    wxFileName crlFile(std2wx(hash));
    crlFile.SetExt(_T("der"));
    crlFile=wxMyPrepend(crlFile,std2wx(user->getCRLDir()));

    if(unique) {

	/* try to find if there is a crl with the same url
	 * if we can find such crl, we replace it by the current 
	 * new one 
	 */
	std::vector<std::string> URIValues = user->authorities->getAuthority
	    ( crl_issuer_dn )->getAttributeValues( "URI" );
	std::vector<std::string>::iterator uriIt = URIValues.begin();
	std::vector<std::string>::iterator hashIt = crlValues.begin();

	if( wx2std(uri) != "none" ){
	    int i=0;
	    if( URIValues.size() > 0) {
					while(uriIt != URIValues.end() && *uriIt != wx2std(uri) ){
							i++;
							uriIt++;
							hashIt++;
					}
		
		/* this uri already exist, so we delete it and 
		 * replace it with the new one
		 */
					if( *uriIt == wx2std(uri)) {
							URIValues.erase( uriIt );
							crlValues.erase( hashIt );
					}
	    }
	}
	
	if(crl->save(wx2std(crlFile.GetFullPath()).c_str()) != SUCCESS){
#ifdef DEBUG		
	    std::cout << "Error while saving "<< wx2std(crlFile.GetFullPath()) << std::endl;
#endif	    
	    return ERROR_CRL_SAVE_FILE;
	}
	
	//adding the new crl and its uri.
	crlValues.push_back( wx2std(crlFile.GetFullName()) );
	URIValues.push_back( wx2std(uri) );
	

	//saving config
	if( ! user->authorities->setInfos( crl_issuer_dn, "CRL", crlValues )){
	
	    return ERROR_CRL_CONFIG_UPDATE;
	}
	
	if( ! user->authorities->setInfos( crl_issuer_dn, "URI", URIValues )) {
	    return ERROR_CRL_CONFIG_UPDATE;
	}
	
	return true;    
    } else { // this crl is not unique (same hash found in config file

	return ERROR_CRL_NOT_UNIQUE;
    }
}
