/*
 * C S O U N D
 *
 * L I C E N S E
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#ifndef SCORE_H
#define SCORE_H
#define EIGEN_INITIALIZE_MATRICES_BY_ZERO
#include "Platform.hpp"
#ifdef SWIG
%module CsoundAC
%{
#include <algorithm>
#include <eigen3/Eigen/Dense>
#include "Event.hpp"
#include <iostream>
#include <sstream>
#include <vector>
%}
%include "std_string.i"
%include "std_vector.i"
#else
#include <algorithm>
#include <eigen3/Eigen/Dense>
#include "Event.hpp"
#include <iostream>
#include <sstream>
#include <vector>
#endif

namespace csound
{
/**
This class, part of CsoundAC, implements a geometric approach to some common
operations on chords in neo-Riemannian music theory for use in score
generating procedures:

--  Identifying whether a chord belongs to some equivalence class of music
  theory, or sending a chord to its equivalent within a representative
  fundamental domain of some equivalence class. The equivalence classes are
  octave (O), permutational (P), transpositional, (T), inversional (I), and
  their compounds OP, OPT (set-class or chord type), and OPTI (prime form).

--  Causing chord progressions to move strictly within an orbifold that
  generates some equivalence class.

--  Implementing chord progressions based on the L, P, R, D, K, and Q
  operations of neo-Riemannian theory (thus implementing some aspects of
  "harmony").

--  Implementing chord progressions performed within a more abstract
  equivalence class by means of the best-formed voice-leading within a less
  abstract equivalence class (thus implementing some fundamentals of
  "counterpoint").

The associated ChordSpaceView package can display these chord spaces and
operations for trichords in an interactive 3-dimensional view.

DEFINITIONS

Pitch is the perception of a distinct sound frequency. It is a logarithmic
perception; octaves, which sound 'equivalent' in some sense, represent
doublings or halvings of frequency.

Pitches and intervals are represented as real numbers. Middle C is 60 and the
octave is 12. Our usual system of 12-tone equal temperament, as well as MIDI
key numbers, are completely represented by the whole numbers; any and all
other pitches can be represented simply by using fractions.

A voice is a distinct sound that is heard as having a pitch.

A chord is simply a set of voices heard at the same time, represented here
as a point in a chord space having one dimension of pitch for each voice
in the chord.

For the purposes of algorithmic composition in CsoundAC, a score is considered
as a sequence of more or less fleeting chords.

EQUIVALENCE CLASSES

An equivalence class identifies elements of a set. Operations that send one
equivalent point to another induce quotient spaces or orbifolds, where the
equivalence operation identifies points on one face of the orbifold with
points on an opposing face. The fundamental domain of the equivalence class
is the space "within" the orbifold.

Plain chord space has no equivalence classes. Ordered chords are represented
as vectors in parentheses (p1, ..., pN). Unordered chords are represented as
sorted vectors in braces {p1, ..., pN}. Unordering is itself an equivalence
class.

The following equivalence classes apply to pitches and chords, and exist in
different orbifolds. Equivalence classes can be combined (Callendar, Quinn,
and Tymoczko, "Generalized Voice-Leading Spaces," _Science_ 320, 2008), and
the more equivalence classes are combined, the more abstract is the resulting
orbifold compared to the parent space.

In most cases, a chord space can be divided into a number, possibly
infinite, of geometrically equivalent fundamental domains for the same
equivalence class. Therefore, here we use the notion of 'representative'
fundamental domain. For example, the representative fundamental domain of
unordered sequences, out of all possible orderings, consists of all sequences
in their ordinary sorted order. It is important, in the following, to identify
representative fundamental domains that combine properly, e.g. such that the
representative fundamental domain of OP / the representative fundamental
domain of PI equals the representative fundamental domain of OPI. And this in
turn may require accounting for duplicate elements of the representative
fundamental domain caused by reflections or singularities in the orbifold.

C       Cardinality equivalence, e.g. {1, 1, 2} == {1, 2}. _Not_ assuming
      cardinality equivalence ensures that there is a proto-metric in plain
      chord space that is inherited by all child chord spaces. Cardinality
      equivalence is never assumed here, because we are working in chord
      spaces of fixed dimensionality; e.g. we represent the note middle C
      not as {60}, but as {60, 60, ..., 60}.

O       Octave equivalence. The fundamental domain is defined by the pitches
      in a chord spanning the range of an octave or less, and summing to
      an octave or less.

P       Permutational equivalence. The fundamental domain is defined by a
      "wedge" of plain chord space in which the voices of a chord are always
      sorted by pitch.

T       Transpositional equivalence, e.g. {1, 2} == {7, 8}. The fundamental
      domain is defined as a plane in chord space at right angles to the
      diagonal of unison chords. Represented by the chord always having a
      sum of pitches equal to 0.

I       Inversional equivalence. Care is needed to distinguish the
      mathematician's sense of 'invert', which means 'pitch-space inversion'
      or 'reflect in a point', from the musician's sense of 'invert', which
      varies according to context but in practice often means 'registral
      inversion' or 'revoice by adding an octave to the lowest tone of a
      chord.' Here, we use 'invert' and 'inversion' in the mathematician's
      sense, and we use the terms 'revoice' and 'voicing' for the musician's
      'invert' and 'inversion'. The inversion point for any inversion lies
      on the unison diagonal. A fundamental domain is defined as any half of
      chord space that is bounded by a plane containing the inversion point.
      Represented as the chord having the first interval between voices be
      smaller than or equal to the final interval (recursing for chords of
      more than 3 voices).

PI      Inversional equivalence with permutational equivalence. The
      'inversion flat' of unordered chord space is a hyperplane consisting
      of all those unordered chords that are invariant under inversion. A
      fundamental domain is defined by any half space bounded by a
      hyperplane containing the inversion flat. It is represented as that
      half of the space on or lower than the hyperplane defined by the
      inversion flat and the unison diagonal.

OP      Octave equivalence with permutational equivalence. Tymoczko's orbifold
      for chords; i.e. chords with a fixed number of voices in a harmonic
      context. The fundamental domain is defined as a hyperprism one octave
      long with as many sides as voices and the ends identified by octave
      equivalence and one cyclical permutation of voices, modulo the
      unordering. In OP for trichords in 12TET, the augmented triads run up
      the middle of the prism, the major and minor triads are in 6
      alternating columns around the augmented triads, the two-pitch chords
      form the 3 sides, and the one-pitch chords form the 3 edges that join
      the sides.

OPT     The layer of the OP prism as close as possible to the origin, modulo
      the number of voices. Chord type. Note that CM and Cm are different
      OPT. Because the OP prism is canted down from the origin, at least one
      pitch in each OPT chord (excepting the origin itself) is negative.

OPI     The OP prism modulo inversion, i.e. 1/2 of the OP prism. The
      representative fundamental consits of those chords less than or equal
      to their inversions modulo OP.

OPTI    The OPT layer modulo inversion, i.e. 1/2 of the OPT layer.
      Set-class. Note that CM and Cm are the same OPTI.

OPERATIONS

Each of the above equivalence classes is, of course, an operation that sends
chords outside the fundamental domain to chords inside the fundamental domain.
And we define the following additional operations:

T(p, x)         Translate p by x.

I(p [, x])      Reflect p in x, by default the origin.

P               Send a major triad to the minor triad with the same root,
              or vice versa (Riemann's parallel transformation).

L               Send a major triad to the minor triad one major third higher,
              or vice versa (Riemann's Leittonwechsel or leading-tone
              exchange transformation).

R               Send a major triad to the minor triad one minor third lower,
              or vice versa (Riemann's relative transformation).

D               Send a triad to the next triad a perfect fifth lower
              (dominant transformation).

P, L, and R have been extended as follows, see Fiore and Satyendra,
"Generalized Contextual Groups", _Music Theory Online_ 11, August 2008:

K(c)            Interchange by inversion;
              K(c) := I(c, c[1] + c[2]).
              This is a generalized form of P; for major and minor triads,
              it is exactly the same as P, but it also works with other
              chord types.

Q(c, n, m)      Contexual transposition;
              Q(c, n, m) := T(c, n) if c is a T-form of m,
              or T(c, -n) if c is an I-form of M. Not a generalized form
              of L or R; but, like them, K and Q generate the T-I group.
 */

/**
 * Returns n!
 */
double factorial(double n)
{
    if (n == 0) {
        return 1;
    } else {
        return n * factorial(n = 1.0);
    }
}

double EPSILON()
{
    static double epsilon = 1.0;
    if (epsilon == 1.0) {
        for (;;) {
            epsilon = epsilon / 2.0;
            double nextEpsilon = epsilon / 2.0;
            double onePlusNextEpsilon = 1.0 + nextEpsilon;
            if (onePlusNextEpsilon == 1.0) {
                break;
            }
        }
    }
    return epsilon;
}

double &epsilonFactor()
{
    static double epsilonFactor = 100000.0;
    return epsilonFactor;
}

bool eq_epsilon(double a, double b)
{
    if (std::abs(a - b) < (EPSILON() * epsilonFactor())) {
        return true;
    } else {
        return false;
    }

}

double gt_epsilon(double a, double b)
{
    if (eq_epsilon(a, b)) {
        return false;
    } else {
        return (a > b);
    }
}

double lt_epsilon(double a, double b)
{
    if (eq_epsilon(a, b)) {
        return false;
    } else {
        return (a < b);
    }
}

double ge_epsilon(double a, double b)
{
    if (eq_epsilon(a, b)) {
        return true;
    } else {
        return (a > b);
    }
}

double le_epsilon(double a, double b)
{
    if (eq_epsilon(a, b)) {
        return true;
    } else {
        return (a < b);
    }
}

/**
 * The size of the octave, defined to be consistent with
 * 12 tone equal temperament and MIDI.
 */
double OCTAVE()
{
    return 12.0;
}

double MIDDLE_C()
{
    return 60.0;
}

double C4()
{
    return MIDDLE_C();
}

/**
 * Returns the pitch transposed by semitones, which may be any scalar.
 * NOTE: Does NOT return the result under any equivalence class.
 */
double T(double pitch, double semitones)
{
    return pitch + semitones;
}

/**
 * Returns the pitch reflected in the center, which may be any pitch.
 * NOTE: Does NOT return the result under any equivalence class.
 */
double I(double pitch, double center = 0.0)
{
    return center - pitch;
}

/**
 * Returns the equivalent of the pitch under pitch-class equivalence, i.e.
 * the pitch is in the interval [0, OCTAVE).
 */
double epc(double pitch)
{
    while (ge_epsilon(pitch, OCTAVE())) {
        pitch -= OCTAVE();
    }
    while (lt_epsilon(pitch, 0.0)) {
        pitch += OCTAVE();
    }
    return pitch;
}

// Forward declarations:

class SILENCE_PUBLIC Chord;

double euclidean(const csound::Chord &, const csound::Chord &);

/**
 * Chords represent simultaneously sounding pitches. The pitches are
 * represented as semitones with 0 at the origin and middle C as 60.
 * Each voice also has a duration, velocity, channel, and pan.
 * Eigen matrices are accessed (row, column) and stored as columns
 * vectors, so a Chord is accessed (voice, attribute).
 */
class SILENCE_PUBLIC Chord : public Eigen::MatrixXd
{
public:
    enum {
        PITCH = 0,
        DURATION = 1,
        LOUDNESS = 2,
        INSTRUMENT = 3,
        PAN = 4,
        COUNT = 5
    };
    Chord() {
        resize(3);
    }
    virtual ~Chord() {
    }
    virtual size_t voices() const {
        return rows();
    }
    virtual void resize(size_t voices) {
        Eigen::MatrixXd::resize(voices, COUNT);
    }
    /**
     * Returns a string representation of the chord.
     * Quadratic complexity, but short enough not to matter.
     */
    virtual std::string toString() const {
        char buffer[0x100];
        std::stringstream stream;
        for (size_t i = 0, n = rows(); i < n; ++i) {
            std::snprintf(buffer, 0x100, "%12.7f", getPitch(i));
            if (i > 0) {
                stream << " ";
            }
            stream << buffer;
        }
        return stream.str();
    }
    virtual double getPitch(int voice) const {
        return coeff(voice, PITCH);
    }
    virtual void setPitch(int voice, double value) {
        coeffRef(voice, PITCH) = value;
    }
    virtual double getDuration(int voice = 0) const {
        return coeff(voice, DURATION);
    }
    virtual void setDuration(double value, int voice = -1) {
        if (voice == -1) {
            for (voice = 0; voice < rows(); ++voice) {
                coeffRef(voice, DURATION) = value;
            }
        } else {
            coeffRef(voice, DURATION) = value;
        }
    }
    virtual double getLoudness(int voice = 0) const {
        return coeff(voice, LOUDNESS);
    }
    virtual void setLoudness(double value, int voice = -1) {
        if (voice == -1) {
            for (voice = 0; voice < rows(); ++voice) {
                coeffRef(voice, LOUDNESS) = value;
            }
        } else {
            coeffRef(voice, LOUDNESS) = value;
        }
    }
    virtual double getInstrument(int voice = 0) const {
        return coeff(voice, INSTRUMENT);
    }
    virtual void setInstrument(double value, int voice = -1) {
        if (voice == -1) {
            for (voice = 0; voice < rows(); ++voice) {
                coeffRef(voice, INSTRUMENT) = value;
            }
        } else {
            coeffRef(voice, INSTRUMENT) = value;
        }
    }
    virtual double getPan(int voice = 0) const {
        return coeff(voice, PAN);
    }
    virtual void setPan(double value, int voice = -1) {
        if (voice == -1) {
            for (voice = 0; voice < rows(); ++voice) {
                coeffRef(voice, PAN) = value;
            }
        } else {
            coeffRef(voice, PAN) = value;
        }
    }
    virtual size_t count(double pitch) const {
        size_t n = 0;
        for (size_t voice = 0; voice < voices(); ++voice) {
            if (eq_epsilon(getPitch(voice), pitch)) {
                n++;
            }
        }
        return n;
    }
    virtual bool operator == (const Chord &other) const {
        if (this == &other) {
            return true;
        }
        if (voices() != other.voices()) {
            return false;
        }
        for (size_t voice = 0; voice < voices(); ++voice) {
            if (!eq_epsilon(getPitch(voice), other.getPitch(voice))) {
                return false;
            }
        }
        return true;
    }
    virtual bool operator < (const Chord &other) const {
        size_t n = std::min(voices(), other.voices());
        for (size_t voice = 0; voice < n; voice++) {
            if (lt_epsilon(getPitch(voice), other.getPitch(voice))) {
                return true;
            }
            if (gt_epsilon(getPitch(voice), other.getPitch(voice))) {
                return false;
            }
            if (voices() < other.voices()) {
                return true;
            }
            return false;
        }
    }
    virtual bool operator <= (const Chord &other) const {
        if (*this == other) {
            return true;
        }
        return (*this < other);
    }
    virtual bool operator > (const Chord &other) const {
        size_t n = std::min(voices(), other.voices());
        for (size_t voice = 0; voice < n; voice++) {
            if (gt_epsilon(getPitch(voice), other.getPitch(voice))) {
                return true;
            }
            if (lt_epsilon(getPitch(voice), other.getPitch(voice))) {
                return false;
            }
            if (voices() > other.voices()) {
                return true;
            }
            return false;
        }
    }
    virtual bool operator >= (const Chord &other) const {
        if (*this == other) {
            return true;
        }
        return (*this > other);
    }
    /**
     * Returns whether or not the chord contains the pitch.
     */
    virtual bool contains(double pitch_) const {
        for (size_t voice = 0; voice < voices(); voice++) {
            if (eq_epsilon(getPitch(voice), pitch_)) {
                return true;
            }
        }
        return false;
    }
    /**
    * Returns the lowest pitch in the chord,
    * and also its voice index.
    */
    virtual std::vector<double> min() const {
        std::vector<double> result(2);
        result[0] = getPitch(0);
        result[1] = 0.0;
        for (size_t voice = 1; voice < voices(); voice++) {
            double pitch = getPitch(voice);
            if (lt_epsilon(pitch, result[0])) {
                result[0] = pitch;
                result[1] = double(voice);
            }
        }
        return result;
    }
    /**
    * Returns the highest pitch in the chord,
    * and also its voice index.
    */
    virtual std::vector<double> max() const {
        std::vector<double> result(2);
        size_t voice = 0;
        double pitch = getPitch(voice);
        result[0] = pitch;
        result[1] = double(voice);
        for (voice = 1; voice < voices(); voice++) {
            pitch = getPitch(voice);
            if (gt_epsilon(pitch, result[0])) {
                result[0] = pitch;
                result[1] = double(voice);
            }
        }
        return result;
    }

    virtual double minimumInterval() const {
        double minimumInterval_ = std::abs(getPitch(0) - getPitch(1));
        for (size_t v1 = 0; v1 < voices(); v1++) {
            for (size_t v2 = 0; v2 < voices(); v2++) {
                double interval = std::abs(getPitch(v1) - getPitch(v2));
                if (lt_epsilon(interval, minimumInterval_)) {
                    minimumInterval_ = interval;
                }
            }
        }
        return minimumInterval_;
    }
    virtual double maximumInterval() const {
        double maximumInterval_ = std::abs(getPitch(0) - getPitch(1));
        for (size_t v1 = 0; v1 < voices(); v1++) {
            for (size_t v2 = 0; v2 < voices(); v2++) {
                double interval = std::abs(getPitch(v1) - getPitch(v2));
                if (gt_epsilon(interval, maximumInterval_)) {
                    maximumInterval_ = interval;
                }
            }
        }
        return maximumInterval_;
    }
    /**
     * Returns a new chord whose pitches are the floors of this chord's pitches.
     */
    virtual Chord floor() const {
        Chord clone = *this;
        for (size_t voice = 0; voice  < voices(); voice++) {
            clone.setPitch(voice, std::floor(getPitch(voice)));
        }
        return clone;
    }
    /**
     * Returns a new chord whose pitches are the ceilings of this chord's pitches.
     */
    virtual Chord ceiling() const {
        Chord clone = *this;
        for (size_t voice = 0; voice  < voices(); voice++) {
            clone.setPitch(voice, std::ceil(getPitch(voice)));
        }
        return clone;
    }
    /**
     * Returns the origin of the chord's space.
     */
    virtual Chord origin() const {
        Chord clone;
        clone.resize(voices());
        return clone;
    }
    /**
     * Returns the Euclidean distance of this chord from its space's
     * origin.
     */
    virtual double distanceToOrigin() const {
        Chord origin_ = origin();
        return euclidean(*this, origin_);
    }
    /**
     * Returns the sum of the pitches in the chord.
     */
    virtual double layer() const {
        double sum = 0.0;
        for (size_t voice = 0; voice < voices(); voice++) {
            sum += getPitch(voice);
        }
        return sum;
    }
    /**
     * Returns the distance of this chord from its space's
     * unison diagonal.
     */
    virtual double distanceToUnisonDiagonal() const {
        Chord unison = origin();
        double pitch = layer() / double(voices());
        for (size_t voice = 0; voice < voices(); voice ++) {
            unison.setPitch(voice, pitch);
        }
        return euclidean(*this, unison);
    }
    /**
     * Returns the maximally even chord in the chord's space,
     * e.g. the augmented triad for 3 dimensions.
     */
    virtual Chord maximallyEven() const {
        Chord clone = *this;
        double g = OCTAVE() / double(voices());
        for (size_t voice = 0; voice < voices(); voice++) {
            clone.setPitch(voice,  double(voice) * g);
        }
        return clone;
    }
    /**
     * Transposes the chord by the indicated interval (may be a fraction).
     * NOTE: Does NOT return the result under any equivalence class.
     */
    virtual Chord T(double interval) const {
        Chord clone = *this;
        for (size_t voice = 0; voice < voices(); voice++) {
            clone.setPitch(voice, csound::T(getPitch(voice), interval));
        }
        return clone;
    }
    /**
     * Inverts the chord by another chord that is on the unison diagonal, by
     * default the origin. NOTE: Does NOT return the result under any equivalence
     * class.
     */
    virtual Chord I(double center=0.0) const {
        Chord inverse = *this;
        for (size_t voice = 0; voice < voices(); voice++) {
            inverse.setPitch(voice, csound::I(getPitch(voice), center));
        }
        return inverse;
    }
    /**
     * Returns whether the chord is within the fundamental domain of
     * pitch-class equivalence, i.e. is a pitch-class set.
     */
    virtual bool isepcs() const {
        for (size_t voice = 0; voice < voices(); voice++) {
            if (!eq_epsilon(getPitch(voice), epc(getPitch(voice)))) {
                return false;
            }
        }
        return true;
    }
    /**
     * Returns the equivalent of the chord under pitch-class equivalence,
     * i.e. the pitch-class set of the chord.
     */
    virtual Chord epcs() const {
        Chord chord = *this;
        for (size_t voice = 0; voice < voices(); voice++) {
            chord.setPitch(voice, epc(getPitch(voice)));
        }
        return chord;
    }
    /**
     * Returns whether the chord is within the fundamental domain of
     * transposition to 0.
     */
    virtual bool iset() const {
        Chord et_ = et();
        if (!(*this == et_)) {
            return false;
        }
        return true;
    }
    /**
     * Returns the equivalent of the chord within the fundamental domain of
     * transposition to 0.
     */
    virtual Chord et() const {
        double min_ = min()[0];
        return T(-min_);
    }
    /**
     * Returns whether the chord is within the representative fundamental domain
     * of the indicated range equivalence.
    */
    virtual bool iseR(double range) const {
        // Note that there are several equivalents of certain boundary points
        // in the same domain.
        double max_ = max()[0];
        double min_ = min()[0];
        if (!le_epsilon(max_, (min_ + range))) {
            return false;
        }
        double layer_ = layer();
        if (!(le_epsilon(0.0, layer_) && le_epsilon(layer_, range))) {
            return false;
        }
        return true;
    }
    /**
     * Returns whether the chord is within the representative fundamental domain
     * of octave equivalence.
     */
    virtual bool iseO() const {
        return iseR(OCTAVE());
    }
    /**
     * Returns the equivalent of the chord within the representative
     * fundamental domain of a range equivalence.
     */
    virtual Chord eR(double range) const {
        Chord chord = *this;
        if (chord.iseR(range)) {
            return chord;
        }
        // The clue here is that at least one voice must be >= 0,
        // but no voice can be > range.
        // First, move all pitches inside the interval [0, OCTAVE),
        // which is not the same as the fundamental domain.
        chord = epcs();
        // Then, reflect voices that are outside of the fundamental domain
        // back into it, which will revoice the chord, i.e.
        // the sum of pitches will be in [0, OCTAVE].
        while (gt_epsilon(chord.layer(), range)) {
            std::vector<double> maximum = chord.max();
            // Because no voice has a pitch above the range,
            // any voices that need to be revoiced will now
            // have negative pitch.
            chord.setPitch(maximum[1], maximum[0] - OCTAVE());
        }
        return chord;
    }
    /**
     * Returns the equivalent of the chord within the representative fundamental
     * domain of octave equivalence.
     */
    virtual Chord eO() const {
        return eR(OCTAVE());
    }
    /**
     * Returns whether the chord is within the representative fundamental domain
     * of permutational equivalence.
     */
    virtual bool iseP() const {
        for (size_t voice = 1; voice < voices(); voice++) {
            if (!le_epsilon(getPitch(voice - 1), getPitch(voice))) {
                return false;
            }
        }
        return true;
    }
    /**
     * Returns the equivalent of the chord within the representative
     * fundamental domain of permutational equivalence.	The implementation
     * uses a bubble sort to swap out of order voices in the Eigen matrix.
     */
    virtual Chord eP() const {
        Chord chord = *this;
        bool sorted = false;
        while (!sorted) {
            sorted = true;
            for (size_t voice = 1; voice < voices(); voice++) {
                if (gt_epsilon(chord.getPitch(voice - 1), chord.getPitch(voice))) {
                    sorted = false;
                    chord.row(voice - 1).swap(chord.row(voice));
                }
            }
        }
        return chord;
    }
    /**
     * Returns whether the chord is within the representative fundamental domain
     * of transpositional equivalence.
     */
    virtual bool iseT() const {
        double layer_ = layer();
        if (eq_epsilon(layer_, 0.0)) {
            return true;
        }
        return false;
    }
    /**
     * Returns the equivalent of the chord within the representative fundamental
     * domain of transpositonal equivalence.
     */
    virtual Chord eT() const {
        double layer_ = layer();
        double sumPerVoice = layer_ / double(voices());
        return T(-sumPerVoice);
    }
    /**
     * Returns the equivalent of the chord within the representative fundamental
     * domain of transpositonal equivalence and the equal temperament generated
     * by g. I.e., returns the chord transposed such that its layer is 0 or, under
     * transposition, the positive layer closest to 0. NOTE: Does NOT return the
     * result under any other equivalence class.
     */
    virtual Chord eTT(double g = 1.0) const {
        Chord et = eT();
        double transposition = std::ceil(et.getPitch(0)) - et.getPitch(0);
        Chord ett = et.T(transposition);
        return ett;
    }
    /**
     * Returns whether the chord is within the representative fundamental domain
     * of translational equivalence and the equal temperament generated by g.
     */
    virtual bool iseTT(double g = 1.0) const {
        Chord ep = eP();
        if (ep == ep.eTT(g)) {
            return true;
        }
        return false;
    }
    /**
     * Returns whether the chord is within the representative fundamental domain
     * of inversional equivalence.
     */
    virtual bool iseI(Chord *inverse = 0) const {
        if (inverse) {
            if (!(*this <= *inverse)) {
                return false;
            }
        } else {
            Chord inverse_ = I();
            if (!(*this <= inverse_)) {
                return false;
            }
        }
        return true;
    }
    /**
     * Returns the equivalent of the chord within the representative fundamental
     * domain of inversional equivalence.
     */
    virtual Chord eI() const {
        if (iseI()) {
            return *this;
        }
        return I();
    }
    /**
     * Returns whether the chord is within the representative fundamental domain
     * of range and permutational equivalence.
     */
    virtual bool iseRP(double range) const {
        if (!iseP()) {
            return false;
        }
        if (!iseR(range)) {
            return false;
        }
        return true;
    }
    /**
     * Returns whether the chord is within the representative fundamental domain
     * of octave and permutational equivalence.
     */
    virtual bool iseOP() const {
        return iseRP(OCTAVE());
    }
    /**
     * Returns the equivalent of the chord within the representative fundamental
     * domain of range and permutational equivalence.
     */
    virtual Chord eRP(double range) const {
        return eR(range).eP();
    }
    /**
     * Returns the equivalent of the chord within the representative fundamental
     * domain of octave and permutational equivalence.
     */
    virtual Chord eOP() const {
        return eRP(OCTAVE());
    }
    /**
     * Returns a copy of the chord cyclically permuted by a stride, by default 1.
     * The direction of rotation is by default the same as musicians' first
     * inversion, second inversion, and so on; but negative sign will reverse
     * the direction of rotation.
     * + 1 is pop the front and push it on the back, shifting the middle down.
     * 0 1 2 3 4 => 1 2 3 4 0
     * - 1 is pop the back and push it on the front, shifting the middle up.
     * 0 1 2 3 4 => 4 0 1 2 3
     */
    virtual Chord cycle(int stride = 1) const {
        Chord permuted = *this;
        int voicesToPopAndPush = std::abs(stride) % voices();
        int voicesToShift = voices() - voicesToPopAndPush;
        if (stride < 0) {
            permuted.bottomRows(voicesToShift) = topRows(voicesToShift);
            permuted.topRows(voicesToPopAndPush) = bottomRows(voicesToPopAndPush);
        }
        if (stride > 0) {
            permuted.topRows(voicesToShift) = bottomRows(voicesToShift);
            permuted.bottomRows(voicesToPopAndPush) = topRows(voicesToPopAndPush);
        }
        return permuted;
    }
    /**
     * Returns the permutations of the pitches in a chord. The permutations
     * starting from any particular permutation are always returned in the same order.
     */
    virtual std::vector<Chord> permutations() const {
        std::vector<Chord> permutations_;
        Chord permutation = *this;
        permutations_.push_back(permutation);
        for (size_t i = 1; i < voices(); i++) {
            permutation = permutation.cycle();
            permutations_.push_back(permutation);
        }
        std::sort(permutations_.begin(), permutations_.end());
        return permutations_;
    }
    /**
     * Returns whether the chord is within the representative fundamental domain
     * of voicing equivalence.
     */
    virtual bool iseV() const {
        Chord eV_ = eV();
        if (*this == eV_) {
            return true;
        }
        return false;
    }
    /**
     * Returns the equivalent of the chord within the representative fundamental
     * domain of voicing equivalence.
     */
    virtual Chord eV() const {
        const std::vector<Chord> voicings = permutations();
        for (size_t i = 0; i < voicings.size(); i++) {
            const Chord &voicing = voicings[i];
            double outer = voicing.getPitch(0) + OCTAVE() - voicing.getPitch(voicing.voices() - 1);
            bool iseV_ = true;
            for (size_t voice = 0; voice < voicing.voices() - 1; voice++) {
                double inner = voicing.getPitch(voice + 1) - voicing.getPitch(voice);
                if(!(ge_epsilon(outer, inner))) {
                    iseV_ = false;
                }
            }
            if(iseV_) {
                return voicing;
            }
        }
    }
    /**
     * Returns whether the chord is within the representative fundamental domain
     * of range, permutational, and transpositional equivalence.
     */
    virtual bool iseRPT(double range) const {
        if (!iseR(range)) {
            return false;
        }
        if (!iseP()) {
            return false;
        }
        if (!iseT()) {
            return false;
        }
        // TODO: Should this be here?
        if (!iseV()) {
            return false;
        }
        return true;
    }
    virtual bool iseRPTT(double range, double g = 1.0) const {
        if (!iseP()) {
            return false;
        }
        if (!iseR(range)) {
            return false;
        }
        // TODO: Should this be here?
        if (!iseTT(g)) {
            return false;
        }
        if (!iseV()) {
            return false;
        }
        return true;
    }
    /**
     * Returns whether the chord is within the representative fundamental domain
     * of octave, permutational, and transpositional equivalence.
     */
    virtual bool iseOPT() const {
        return iseRPT(OCTAVE());
    }
    virtual bool iseOPTT(double g = 1.0) const {
        return iseRPTT(OCTAVE(), g);
    }
    /**
     * Returns a copy of the chord 'inverted' in the musician's sense,
     * i.e. revoiced by cyclically permuting the chord and
     * adding (or subtracting) an octave to the highest (or lowest) voice.
     * The revoicing will move the chord up or down in pitch.
     * A positive direction is the same as a musician's first inversion,
     * second inversion, etc.
     */
    virtual Chord v(int direction = 1) const {
        Chord chord = *this;
        int head = voices() - 1;
        while (direction > 0) {
            chord = chord.cycle(1);
            chord.setPitch(head, chord.getPitch(head) + OCTAVE());
            direction--;
        }
        while (direction < 0) {
            chord = chord.cycle(-1);
            chord.setPitch(0, chord.getPitch(0) + OCTAVE());
            direction++;
        }
        return chord;
    }
    /**
     * Returns all the 'inversions' (in the musician's sense)
     * or octavewise revoicings of the chord.
     */
    virtual std::vector<Chord> voicings() const {
        Chord chord = *this;
        std::vector<Chord> voicings;
        voicings.push_back(chord);
        for (size_t voicing = 1; voicing < voices(); voicing++) {
            chord = chord.v();
            voicings.push_back(chord);
        }
        return voicings;
    }
    /**
     * Returns the equivalent of the chord within the representative fundamental
     * domain of range, permutational, and transpositional equivalence; the same
     * as set-class type, or chord type.
     */
    virtual Chord eRPT(double range) const {
        Chord erp = eRP(range);
        std::vector<Chord> voicings_ = erp.voicings();
        for (size_t voice = 0; voice < erp.voices(); voice++) {
            const Chord &voicing = voicings_[voice];
            if (voicing.iseV()) {
                return voicing.eT();
            }
        }
    }
    virtual Chord eRPTT(double range, double g = 1.0) const {
        Chord erp = eRP(range);
        std::vector<Chord> voicings_ = erp.voicings();
        for (size_t voice = 0; voice < erp.voices(); voice++) {
            const Chord &voicing = voicings_[voice].eTT(g);
            if (voicing.iseV()) {
                return voicing;
            }
        }
    }
    /**
     * Returns the equivalent of the chord within the representative fundamental
     * domain of octave, permutational, and transpositional equivalence.
     */
    virtual Chord eOPT() const {
        return eRPT(OCTAVE());
    }
    virtual Chord eOPTT(double g = 1.0) const {
        return eRPTT(OCTAVE(), g);
    }
    /**
     * Returns whether the chord is within the representative fundamental domain
     * of range, permutational, and inversional equivalence.
     */
    virtual bool iseRPI(double range) const {
        if (!iseRP(range)) {
            return false;
        }
        Chord inverse = I().eRP(range);
        if (!iseI(inverse)) {
            return false;
        }
        return true;
    }
    /**
     * Returns whether the chord is within the representative fundamental domain
     * of octave, permutational, and inversional equivalence.
     */
    virtual bool iseOPI() const {
        return iseRPI(OCTAVE());
    }
    /**
     * Returns the equivalent of the chord within the representative fundamental
     * domain of range, permutational, and inversional equivalence.
     */
    virtual Chord eRPI(double range) const {
        Chord erp = eRP(range);
        if (erp.iseRPI(range)) {
            return erp;
        }
        return erp.I().eRP(range);
    }
    /**
     * Returns the equivalent of the chord within the representative fundamental
     * domain of octave, permutational, and inversional equivalence.
     */
    virtual Chord eOPI() const {
        return eRPI(OCTAVE());
    }
};

/**
 * Returns the Euclidean distance between chords a and b,
 * which must have the same number of voices.
 */
double euclidean(const Chord &a, const Chord &b)
{
    double sumOfSquaredDifferences = 0.0;
    for (size_t voice = 0, voices = a.voices(); voice < voice; ++voice) {
        sumOfSquaredDifferences += std::pow((a.getPitch(voice) - b.getPitch(voice)), 2.0);
    }
    return std::sqrt(sumOfSquaredDifferences);
}

}
#endif
/*
-- Returns the equivalent of the chord within the representative fundamental
-- domain of range, permutational, and inversional equivalence.

function Chord:eRPI(range)
    local erp = self:eRP(range)
    if erp:iseRPI(range) then
        return erp
    end
    return erp:I():eRP(range)
end

-- Returns the equivalent of the chord within the representative fundamental
-- domain of octave, permutational, and inversional equivalence.

function Chord:eOPI()
    return self:eRPI(ChordSpace.OCTAVE)
end

-- Returns whether the chord is within the representative fundamental domain
-- of range, permutational, transpositional, and inversional equivalence.

function Chord:iseRPTI(range)
    --[[ GVLS:
    -- GVLS: if not (self[#self] <= self[1] + ChordSpace.OCTAVE) then
    if not ChordSpace.le_epsilon(self[#self], (self[1] + range)) then
        return false
    end
    local layer_ = self:layer()
    -- GVLS: if not (layer_ == 0) then
    if not ChordSpace.eq_epsilon(layer_, 0) then
        return false
    end
    if #self <= 2 then
        return true
    end
    local wraparound = self[1] + range - self[#self]
    for voice = 1, #self - 1  do
        local inner = self[voice + 1] - self[voice]
        if not ChordSpace.le_epsilon(wraparound, inner) then
            return false
        end
    end
    if not self:iseI() then
        return false
    end
    return true
    --]]
    ----[[ MKG:
    if not self:iseRPT(range) then
        return false
    end
    if not self:iseI() then
        return false
    end
    return true
    --]]
end

function Chord:iseRPTTI(range)
    if not self:iseRPTT(range) then
        return false
    end
    if not self:iseI() then
        return false
    end
    return true
end

-- Returns whether the chord is within the representative fundamental domain
-- of octave, permutational, transpositional, and inversional equivalence.

function Chord:iseOPTI()
    return self:iseRPTI(ChordSpace.OCTAVE)
end

function Chord:iseOPTTI()
    return self:iseRPTTI(ChordSpace.OCTAVE)
end

-- Returns the equivalent of the chord within the representative fundamental
-- domain of range, permutational, transpositional, and inversional
-- equivalence.

function Chord:eRPTI(range)
    local rpt = self:eRPT(range)
    if rpt:iseI() then
        return rpt
    end
    return rpt:I():eRPT(range)
end

function Chord:eRPTTI(range)
    local rpt = self:eRPTT(range)
    if rpt:iseRPTTI(range) then
        return rpt
    end
    return rpt:I():eRPTT(range)
end

-- Returns the equivalent of the chord within the representative fundamental
-- domain of range, permutational, transpositional, and inversional
-- equivalence.

function Chord:eOPTI()
    return self:eRPTI(ChordSpace.OCTAVE)
end

function Chord:eOPTTI()
    return self:eRPTTI(ChordSpace.OCTAVE)
end

function Chord:name()
    local chordName = ChordSpace.namesForChords[self:__hash()]
    if chordName == nil then
        chordName = ''
    end
    return chordName
end

-- Returns a formatted string with information about the chord.

function Chord:information()
    local et = self:eT():et()
    local evt = self:eV():et()
    local eopt = self:eOPT():et()
    local epcs = self:epcs():eP()
    local eopti = self:eOPTI():et()
    local eOP = self:eOP()
    chordName = eOP:name()
    return string.format([[pitches:  %s  %s
I:        %s
eO:       %s  iseO:    %s
eP:       %s  iseP:    %s
eT:       %s  iseT:    %s
          %s
eI:       %s  iseI:    %s
eV:       %s  iseV:    %s
          %s
eOP:      %s  iseOP:   %s
pcs:      %s
eOPT:     %s  iseOPT:  %s
eOPTT:    %s
          %s
eOPI:     %s  iseOPI:  %s
eOPTI:    %s  iseOPTI: %s
eOPTTI:   %s
          %s
layer:      %6.2f]],
tostring(self), chordName,
tostring(self:I()),
tostring(self:eO()),    tostring(self:iseO()),
tostring(self:eP()),    tostring(self:iseP()),
tostring(self:eT()),    tostring(self:iseT()),
tostring(et),
tostring(self:eI()),    tostring(self:iseI()),
tostring(self:eV()),    tostring(self:iseV()),
tostring(evt),
tostring(self:eOP()),   tostring(self:iseOP()),
tostring(epcs),
tostring(self:eOPT()),  tostring(self:iseOPT()),
tostring(self:eOPTT()),
tostring(eopt),
tostring(self:eOPI()),  tostring(self:iseOPI()),
tostring(self:eOPTI()), tostring(self:iseOPTI()),
tostring(self:eOPTTI()),
tostring(eopti),
self:layer())
end

function ChordSpace.set(collection)
    local set_ = {}
    for key, value in pairs(collection) do
        set_[value:__hash()] = value
    end
    return set_
end

function ChordSpace.sortedSet(collection)
    local set_ = ChordSpace.set(collection)
    local sortedSet_ = {}
    for key, value in pairs(set_) do
        table.insert(sortedSet_, value)
    end
    table.sort(sortedSet_)
    return sortedSet_
end

function ChordSpace.zeroBasedSet(sortedSet)
    local zeroBasedSet = {}
    for index, value in pairs(sortedSet) do
        zeroBasedSet[index - 1] = value
    end
    return zeroBasedSet
end

function ChordSpace.setContains(setA, x)
    if setA[x:__hash()] == x then
        return true
    end
    return false
end

function ChordSpace.setInsert(setA, x)
    if not ChordSpace.setContains(setA, x) then
        setA[x:__hash()] = x
    end
end

function ChordSpace.sortedEquals(sortedA, sortedB)
    if not (#sortedA == #sortedB) then
        return false
    end
    for i = 1, #sortedA do
        if not (sortedA[i] == sortedB[i]) then
            return false
        end
    end
    return true
end

function ChordSpace.setIntersection(A, setB)
    local result = {}
    for index, value in pairs(A) do
        if ChordSpace.setContains(setB, value) then
            ChordSpace.setInsert(result, value)
        end
    end
    return result
end

function ChordSpace.union(A, B)
    local result = {}
    for index, value in pairs(A) do
        ChordSpace.setInsert(result, value)
    end
    for index, value in pairs(B) do
        ChordSpace.setInsert(result, value)
    end
    return result
end

ChordSpace.pitchClassesForNames = {}

ChordSpace.pitchClassesForNames["C" ] =  0
ChordSpace.pitchClassesForNames["C#"] =  1
ChordSpace.pitchClassesForNames["Db"] =  1
ChordSpace.pitchClassesForNames["D" ] =  2
ChordSpace.pitchClassesForNames["D#"] =  3
ChordSpace.pitchClassesForNames["Eb"] =  3
ChordSpace.pitchClassesForNames["E" ] =  4
ChordSpace.pitchClassesForNames["F" ] =  5
ChordSpace.pitchClassesForNames["F#"] =  6
ChordSpace.pitchClassesForNames["Gb"] =  6
ChordSpace.pitchClassesForNames["G" ] =  7
ChordSpace.pitchClassesForNames["G#"] =  8
ChordSpace.pitchClassesForNames["Ab"] =  8
ChordSpace.pitchClassesForNames["A" ] =  9
ChordSpace.pitchClassesForNames["A#"] = 10
ChordSpace.pitchClassesForNames["Bb"] = 10
ChordSpace.pitchClassesForNames["B" ] = 11

ChordSpace.chordsForNames = {}
ChordSpace.namesForChords = {}

local function fill(rootName, rootPitch, typeName, typePitches)
    local chordName = rootName .. typeName
    local chord = Chord:new()
    local splitPitches = Silencio.split(typePitches)
    chord:resize(#splitPitches)
    for voice, pitchName in ipairs(splitPitches) do
        local pitch = ChordSpace.pitchClassesForNames[pitchName]
        chord[voice] = rootPitch + pitch
    end
    chord = chord:eOP()
    ChordSpace.chordsForNames[chordName] = chord
    ChordSpace.namesForChords[chord:__hash()] = chordName
end

for rootName, rootPitch in pairs(ChordSpace.pitchClassesForNames) do
    fill(rootName, rootPitch, " minor second",     "C  C#                             ")
    fill(rootName, rootPitch, " major second",     "C     D                           ")
    fill(rootName, rootPitch, " minor third",      "C        Eb                       ")
    fill(rootName, rootPitch, " major third",      "C           E                     ")
    fill(rootName, rootPitch, " perfect fourth",   "C              F                  ")
    fill(rootName, rootPitch, " tritone",          "C                 F#              ")
    fill(rootName, rootPitch, " perfect fifth",    "C                    G            ")
    fill(rootName, rootPitch, " augmented fifth",  "C                       G#        ")
    fill(rootName, rootPitch, " sixth",            "C                          A      ")
    fill(rootName, rootPitch, " minor seventh  ",  "C                             Bb  ")
    fill(rootName, rootPitch, " major seventh",    "C                                B")
    -- Scales.
    fill(rootName, rootPitch, " major",            "C     D     E  F     G     A     B")
    fill(rootName, rootPitch, " minor",            "C     D  Eb    F     G  Ab    Bb  ")
    fill(rootName, rootPitch, " natural minor",    "C     D  Eb    F     G  Ab    Bb  ")
    fill(rootName, rootPitch, " harmonic minor",   "C     D  Eb    F     G  Ab       B")
    fill(rootName, rootPitch, " chromatic",        "C  C# D  D# E  F  F# G  G# A  A# B")
    fill(rootName, rootPitch, " whole tone",       "C     D     E     F#    G#    A#  ")
    fill(rootName, rootPitch, " diminished",       "C     D  D#    F  F#    G# A     B")
    fill(rootName, rootPitch, " pentatonic",       "C     D     E        G     A      ")
    fill(rootName, rootPitch, " pentatonic major", "C     D     E        G     A      ")
    fill(rootName, rootPitch, " pentatonic minor", "C        Eb    F     G        Bb  ")
    fill(rootName, rootPitch, " augmented",        "C        Eb E        G  Ab    Bb  ")
    fill(rootName, rootPitch, " Lydian dominant",  "C     D     E     Gb G     A  Bb  ")
    fill(rootName, rootPitch, " 3 semitone",       "C        D#       F#       A      ")
    fill(rootName, rootPitch, " 4 semitone",       "C           E           G#        ")
    fill(rootName, rootPitch, " blues",            "C     D  Eb    F  Gb G        Bb  ")
    fill(rootName, rootPitch, " bebop",            "C     D     E  F     G     A  Bb B")
    -- Major chords.
    fill(rootName, rootPitch, "M",                 "C           E        G            ")
    fill(rootName, rootPitch, "6",                 "C           E        G     A      ")
    fill(rootName, rootPitch, "69",                "C     D     E        G     A      ")
    fill(rootName, rootPitch, "69b5",              "C     D     E     Gb       A      ")
    fill(rootName, rootPitch, "M7",                "C           E        G           B")
    fill(rootName, rootPitch, "M9",                "C     D     E        G           B")
    fill(rootName, rootPitch, "M11",               "C     D     E  F     G           B")
    fill(rootName, rootPitch, "M#11",              "C     D     E  F#    G           B")
    fill(rootName, rootPitch, "M13",               "C     D     E  F     G     A     B")
    -- Minor chords.
    fill(rootName, rootPitch, "m",                 "C        Eb          G            ")
    fill(rootName, rootPitch, "m6",                "C        Eb          G     A      ")
    fill(rootName, rootPitch, "m69",               "C     D  Eb          G     A      ")
    fill(rootName, rootPitch, "m7",                "C        Eb          G        Bb  ")
    fill(rootName, rootPitch, "m#7",               "C        Eb          G           B")
    fill(rootName, rootPitch, "m7b5",              "C        Eb       Gb          Bb  ")
    fill(rootName, rootPitch, "m9",                "C     D  Eb          G        Bb  ")
    fill(rootName, rootPitch, "m9#7",              "C     D  Eb          G           B")
    fill(rootName, rootPitch, "m11",               "C     D  Eb    F     G        Bb  ")
    fill(rootName, rootPitch, "m13",               "C     D  Eb    F     G     A  Bb  ")
    -- Augmented chords.
    fill(rootName, rootPitch, "+",                 "C            E         G#         ")
    fill(rootName, rootPitch, "7#5",               "C            E         G#     Bb  ")
    fill(rootName, rootPitch, "7b9#5",             "C  Db        E         G#     Bb  ")
    fill(rootName, rootPitch, "9#5",               "C     D      E         G#     Bb  ")
    -- Diminished chords.
    fill(rootName, rootPitch, "o",                 "C        Eb       Gb              ")
    fill(rootName, rootPitch, "o7",                "C        Eb       Gb       A      ")
    -- Suspended chords.
    fill(rootName, rootPitch, "6sus",              "C              F     G     A      ")
    fill(rootName, rootPitch, "69sus",             "C     D        F     G     A      ")
    fill(rootName, rootPitch, "7sus",              "C              F     G        Bb  ")
    fill(rootName, rootPitch, "9sus",              "C     D        F     G        Bb  ")
    fill(rootName, rootPitch, "M7sus",             "C              F     G           B")
    fill(rootName, rootPitch, "M9sus",             "C     D        F     G           B")
    -- Dominant chords.
    fill(rootName, rootPitch, "7",                 "C            E       G        Bb  ")
    fill(rootName, rootPitch, "7b5",               "C            E    Gb          Bb  ")
    fill(rootName, rootPitch, "7b9",               "C  Db        E       G        Bb  ")
    fill(rootName, rootPitch, "7b9b5",             "C  Db        E    Gb          Bb  ")
    fill(rootName, rootPitch, "9",                 "C     D      E       G        Bb  ")
    fill(rootName, rootPitch, "9#11",              "C     D      E F#    G        Bb  ")
    fill(rootName, rootPitch, "13",                "C     D      E F     G     A  Bb  ")
    fill(rootName, rootPitch, "13#11",             "C     D      E F#    G     A  Bb  ")
end

table.sort(ChordSpace.chordsForNames)
table.sort(ChordSpace.namesForChords)

-- Increment a chord voicewise through chord space,
-- from a low point on the unison diagonal through a high point
-- on the unison diagonal. g is the generator of transposition.
-- It may be necessary to set the chord to the low point to start.

function ChordSpace.next(odometer, low, high, g)
    local voices = #odometer
    odometer[voices] = odometer[voices] + g
     -- "Carry."
    for voice = voices, 2, -1 do
        if odometer[voice] > high then
            odometer[voice] = low
            odometer[voice - 1] = odometer[voice - 1] + g
        end
    end
    if odometer[1] > high then
        return false
    end
    return true
end

function ChordSpace.allOfEquivalenceClass(voices, equivalence, g)
    g = g or 1
    local equivalenceMapper = nil
    if equivalence == 'OP' then
        equivalenceMapper = Chord.iseOP
    end
    if equivalence == 'OPT' then
        equivalenceMapper = Chord.iseOPT
    end
    if equivalence == 'OPTT' then
        equivalenceMapper = Chord.iseOPT
    end
    if equivalence == 'OPI' then
        equivalenceMapper = Chord.iseOPI
    end
    if equivalence == 'OPTI' then
        equivalenceMapper = Chord.iseOPTI
    end
    if equivalence == 'OPTTI' then
        equivalenceMapper = Chord.iseOPTTI
    end
    local equivalentChords = {}
    -- Enumerate all chords in [-O, O].
    local iterator = ChordSpace.iterator(voices, -13)
    -- print('iterator:', tostring(iterator))
    while ChordSpace.next(iterator, -13, 13, g) == true do
        if iterator:iseP() == true then
            local eP = iterator:clone()
            if equivalenceMapper(eP) then
                ChordSpace.setInsert(equivalentChords, eP)
            end
        end
    end
    local equivalentChords = ChordSpace.sortedSet(equivalentChords)
    local zeroBasedChords = ChordSpace.zeroBasedSet(equivalentChords)
    return zeroBasedChords, equivalentChords
end

-- Returns a chord with the specified number of voices all set to a first
-- pitch, useful as an iterator.

function ChordSpace.iterator(voices, first)
    local odometer = Chord:new()
    odometer:resize(voices)
    for voice = 1, voices do
        odometer[voice] = first
    end
    return odometer
end

-- Returns a collection of all chords for the specified number of voices in a
-- range, by default the octave. g is the generator of transposition, by default the
-- semitone.

function ChordSpace.allChordsInRange(voices, first, last, g)
    first = first or 0
    last = last or ChordSpace.OCTAVE
    g = g or 1
    -- Enumerate all chords in the range.
    local chordset = {}
    local iterator = ChordSpace.iterator(voices, first)
    while ChordSpace.next(iterator, first, last, g) do
        local chord = iterator:clone()
        chordset[chord:__hash()] = chord
    end
    return ChordSpace.sortedSet(chordset)
end

-- Move 1 voice of the chord,
-- optionally under range equivalence
-- NOTE: Does NOT return the result under any equivalence class.

function Chord:move(voice, interval)
    local chord = self:clone()
    chord[voice] = T(chord[voice], interval)
    return chord
end

-- Performs the neo-Riemannian Lettonwechsel transformation.
-- NOTE: Does NOT return the result under any equivalence class.

function Chord:nrL()
    local cv = self:eV()
    local cvt = self:eV():et()
    if cvt[2] == 4 then
        cv[1] = cv[1] - 1
    else
        if cvt[2] == 3 then
            cv[3] = cv[3] + 1
        end
    end
    return cv
end

-- Performs the neo-Riemannian parallel transformation.
-- NOTE: Does NOT return the result under any equivalence class.

function Chord:nrP()
    local cv = self:eV()
    local cvt = self:eV():et()
    if cvt[2] == 4 then
        cv[2] = cv[2] - 1
    else
        if cvt[2] == 3 then
            cv[2] = cv[2] + 1
        end
    end
    return cv
end

-- Performs the neo-Riemannian relative transformation.
-- NOTE: Does NOT return the result under any equivalence class.

function Chord:nrR()
    local cv = self:eV()
    local cvt = self:eV():et()
    if cvt[2] == 4 then
        cv[3] = cv[3] + 2
    else
        if cvt[2] == 3 then
            cv[1] = cv[1] - 2
        end
    end
    return cv
end

-- Performs the neo-Riemannian dominant transformation.
-- NOTE: Does NOT return the result under any equivalence class.

function Chord:nrD()
    return self:eep():T(-7)
end

-- Returns the chord inverted by the sum of its first two voices.
-- NOTE: Does NOT return the result under any equivalence class.

function Chord:K(range)
    range = range or ChordSpace.OCTAVE
    local chord = self:clone()
    if #chord < 2 then
        return chord
    end
    local ep = chord:eP()
    local x = ep[1] + ep[2]
    return self:I(x)
end

-- Returns whether the chord is a transpositional form of Y with interval size g.
-- Only works in equal temperament.

function Chord:Tform(Y, g)
    local eopx = self:epcs()
    local i = 0
    while i < ChordSpace.OCTAVE do
        local ty = Y:T(i)
        local eopty = ty:epcs()
        if eopx == eopty then
            return true
        end
        i = i + g
    end
    return false
end

-- Returns whether the chord is an inversional form of Y with interval size g.
-- Only works in equal temperament.

function Chord:Iform(Y, g)
    local eopx = self:epcs()
    local i = 0
    while i < ChordSpace.OCTAVE do
        local iy = Y:I(i)
        local eopiy = iy:epcs()
        if eopx == eopiy then
            return true
        end
        i = i + g
    end
    return false
end

-- Returns the contextual transposition of the chord by x with respect to m
-- with minimum interval size g.
-- NOTE: Does NOT return the result under any equivalence class.

function Chord:Q(x, m, g)
    g = g or 1
    if self:Tform(m, g) then
        return self:T(x)
    end
    if self:Iform(m, g) then
        return self:T(-x)
    end
    return self:clone()
end

-- Returns the voice-leading between chords a and b,
-- i.e. what you have to add to a to get b, as a
-- chord of directed intervals.

function ChordSpace.voiceleading(a, b)
    local voiceleading = a:clone()
    for voice = 1, #voiceleading do
        voiceleading[voice] = b[voice] - a[voice]
    end
    return voiceleading
end

-- Returns whether the voiceleading
-- between chords a and b contains a parallel fifth.

function ChordSpace.parallelFifth(a, b)
    local v = ChordSpace.voiceleading(a, b)
    if v:count(7) > 1 then
        return true
    else
        return false
    end
end

-- Returns the smoothness of the voiceleading between
-- chords a and b by L1 norm.

function ChordSpace.voiceleadingSmoothness(a, b)
    local L1 = 0
    for voice = 1, #a do
        L1 = L1 + math.abs(b[voice] - a[voice])
    end
    return L1
end

-- Returns which of the voiceleadings (source to d1, source to d2)
-- is the smoother (shortest moves), optionally avoiding parallel fifths.

function ChordSpace.voiceleadingSmoother(source, d1, d2, avoidParallels, range)
    range = range or ChordSpace.OCTAVE
    if avoidParallels then
        if ChordSpace.parallelFifth(source, d1) then
            return d2
        end
        if ChordSpace.parallelFifth(source, d2) then
            return d1
        end
    end
    local s1 = ChordSpace.voiceleadingSmoothness(source, d1)
    local s2 = ChordSpace.voiceleadingSmoothness(source, d2)
    if s1 <= s2 then
        return d1
    else
        return d2
    end
end

-- Returns which of the voiceleadings (source to d1, source to d2)
-- is the simpler (fewest moves), optionally avoiding parallel fifths.

function ChordSpace.voiceleadingSimpler(source, d1, d2, avoidParallels)
    avoidParallels = avoidParallels or false
    if avoidParallels then
        if ChordSpace.parallelFifth(source, d1) then
            return d2
        end
        if ChordSpace.parallelFifth(source, d2) then
            return d1
        end
    end
    local v1 = ChordSpace.voiceleading(source, d1):eP()
    local v2 = ChordSpace.voiceleading(source, d2):eP()
    for voice = #v1, 1, -1 do
        if v1[voice] < v2[voice] then
            return d1
        end
        if v2[voice] < v1[voice] then
            return d2
        end
    end
    return d1
end

-- Returns which of the voiceleadings (source to d1, source to d2)
-- is the closer (first smoother, then simpler), optionally avoiding parallel fifths.

function ChordSpace.voiceleadingCloser(source, d1, d2, avoidParallels)
    avoidParallels = avoidParallels or false
    if avoidParallels then
        if ChordSpace.parallelFifth(source, d1) then
            return d2
        end
        if ChordSpace.parallelFifth(source, d2) then
            return d1
        end
    end
    local s1 = ChordSpace.voiceleadingSmoothness(source, d1)
    local s2 = ChordSpace.voiceleadingSmoothness(source, d2)
    if s1 < s2 then
        return d1
    end
    if s1 > s2 then
        return d2
    end
    return ChordSpace.voiceleadingSimpler(source, d1, d2, avoidParallels)
end

-- Returns the voicing of the destination which has the closest voice-leading
-- from the source within the range, optionally avoiding parallel fifths.

function ChordSpace.voiceleadingClosestRange(source, destination, range, avoidParallels)
    local destinationeOP = destination:eOP()
    local d = destinationeOP:clone()
    local odometer = source:origin()
    while ChordSpace.next(odometer, 0, range, ChordSpace.OCTAVE) == true do
        local revoicing = odometer:clone()
        for voice = 1, #revoicing do
            revoicing[voice] = revoicing[voice] + destinationeOP[voice]
        end
        d = ChordSpace.voiceleadingCloser(source, d, revoicing, avoidParallels)
    end
    return d
end

-- Creates a complete Silencio "note on" event for the
-- indicated voice of the chord. The other parameters are used
-- if the internal duration, channel, velocity, and pan of the
-- chord are nil.

function Chord:note(voice_, time_, duration_, channel_, velocity_, pan_)
    local note_ = Event:new()
    note_[TIME] = time_
    note_[DURATION] = duration_ or self.duration[voice_]
    note_[CHANNEL] = channel_ or self.channel[voice_]
    note_[KEY] = self[voice_]
    note_[VELOCITY] = velocity_ or self.velocity[voice_]
    note_[PAN] = pan_ or self.pan[voice_]
    return note_
end

-- Returns an individual note for each voice of the chord.
-- The chord's duration, instrument, and loudness are used if present,
-- if not the specified values are used.

function Chord:notes(time_, duration_, channel_, velocity_, pan_)
    local notes_ = Score:new()
    for voice, key in ipairs(self) do
        table.insert(notes_, self:note(voice, time_, duration_, channel_, velocity_, pan_))
    end
    return notes_
end

function Chord:toScore(score, time_, duration_, channel_, velocity_, pan_)
    for voice, key in ipairs(self) do
        score:append(self:note(voice, time_, duration_, channel_, velocity_, pan_))
    end
    return score
end

-- If the event is a note, moves its pitch
-- to the closest pitch of the chord.
-- If octaveEquivalence is true (the default),
-- the pitch-class of the note is moved to the closest pitch-class
-- of the chord; otherwise, the pitch of the note is moved to the closest
-- absolute pitch of the chord.

function conformToChord(event, chord, octaveEquivalence)
    octaveEquivalence = octaveEquivalence or true
    if event[STATUS] ~= 144 then
        return
    else
        local pitch = event[KEY]
        if octaveEquivalence then
            local pitchClass = pitch % ChordSpace.OCTAVE
            local octave = pitch - pitchClass
            local chordPitchClass = chord[1] % ChordSpace.OCTAVE
            local distance = math.abs(chordPitchClass - pitchClass)
            local closestPitchClass = chordPitchClass
            local minimumDistance = distance
            for voice = 2, #chord do
                chordPitchClass = chord[voice] % ChordSpace.OCTAVE
                distance = math.abs(chordPitchClass - pitchClass)
                if distance < minimumDistance then
                    minimumDistance = distance
                    closestPitchClass = chordPitchClass
                end
            end
            event[KEY] = octave + closestPitchClass
        else
            local chordPitch = chord[1]
            local distance = math.abs(chordPitch - pitch)
            local closestPitch = chordPitch
            local minimumDistance = distance
            for voice = 2, #chord do
                chordPitch = chord[voice]
                distance = math.abs(chordPitch - pitch)
                if distance < minimumDistance then
                    minimumDistance = distance
                    closestPitch = chordPitch
                end
            end
            event[KEY] = closestPitch
        end
    end
end

-- Inserts the notes of the chord into the score at the specified time.
-- The internal duration, instrument, and loudness are used if present,
-- if not the specified values are used.

function ChordSpace.insert(score, chord, time_, duration, channel, velocity, pan)
    -- print(score, chord, time_, duration, channel, velocity, pan)
    for voice = 1, #chord do
        local event = chord:note(voice, time_, duration, channel, velocity, pan)
        table.insert(score, event)
    end
end

-- For all the notes in the score
-- beginning at or later than the start time,
-- and up to but not including the end time,
-- moves the pitch of the note to belong to the chord, using the
-- conformToChord function.

function ChordSpace.apply(score, chord, start, end_, octaveEquivalence)
    octaveEquivalence = octaveEquivalence or true
    local slice = score:slice(start, end_)
    for index, event in ipairs(slice) do
        conformToChord(event, chord, octaveEquivalence)
    end
end

-- Returns a chord containing all the pitches of the score
-- beginning at or later than the start time,
-- and up to but not including the end time.

function gather(score, start, end_)
    local chord = Chord:new()
    local slice = score:slice(start, end_)
    for index, event in ipairs(slice) do
        local pitch = event[KEY]
        if not chord:contains(pitch) then
            table.insert(chord, pitch)
        end
    end
    return chord
end

-- Orthogonal additive groups for unordered chords of given arity under range
-- equivalence (RP): prime form or P, inversion or I, transposition or T, and
-- voicing or V. P x I x T = OP, P x I x T x V = RP. Therefore, an
-- operation on P, I, T, or V may be used to independently transform the
-- respective symmetry of any chord. Some of these operations will reflect
-- in RP.

ChordSpaceGroup = {}

-- N is the number of voices in the chord space, g is the generator of
-- transposition, range is the size of chord space,
-- optis is an ordered table of all OPTI chords for g,
-- voicings is an ordered table of all octavewise permutations in RP.

function ChordSpaceGroup:new(o)
    local o = o or {optisForIndexes = {}, indexesForOptis = {}, voicingsForIndexes = {}, indexesForVoicings = {}}
    setmetatable(o, self)
    self.__index = self
    return o
end

function ChordSpace.octavewiseRevoicings(chord, range)
    range = range or ChordSpace.OCTAVE
    local voices = #chord
    local odometer = chord:origin()
    -- Enumerate the permutations.
    -- iterator[1] is the most significant voice, and
    -- iterator[N] is the least significant voice.
    local voicings = 0
    while ChordSpace.next(odometer, 0, range, ChordSpace.OCTAVE) == true do
        voicings = voicings + 1
    end
    return voicings
end

function ChordSpace.octavewiseRevoicing(chord, index, range)
    local voices = #chord
    local odometer = chord:origin()
    local eop = chord:eOP()
    -- Enumerate the permutations.
    -- iterator[1] is the most significant voice, and
    -- iterator[N] is the least significant voice.
    local voicings = 0
    for v = 1, index do
        ChordSpace.next(odometer, 0, range, ChordSpace.OCTAVE)
        -- Wrap around?
        if odometer[1] > range then
            odometer = chord:origin()
        end
        voicings = voicings + 1
    end
    for voice = 1, #chord do
        odometer[voice] = odometer[voice] + eop[voice]
    end
    return odometer
end

-- Returns the ith arpeggiation, current voice, and corresponding revoicing
-- of the chord. Positive arpeggiations start with the lowest voice of the
-- chord and revoice up; negative arpeggiations start with the highest voice
-- of the chord and revoice down.

function Chord:a(arpeggiation)
    local chord = self:v(arpeggiation)
    if arpeggiation < 0 then
        return chord[#chord], #chord, chord
    end
    return chord[1], 1, chord
end

function ChordSpaceGroup:initialize(voices, range, g)
    self.voices = voices or 3
    self.range = range or 60
    self.g = g or 1
    self.countP = 0
    self.countI = 2
    self.countT = ChordSpace.OCTAVE / self.g
    local chord = Chord:new()
    chord:resize(voices)
    self.countV = ChordSpace.octavewiseRevoicings(chord, self.range)
    self.indexesForOptis = {}
    self.optisForIndexes = ChordSpace.allOfEquivalenceClass(voices, 'OPTTI')
    for index, optti in pairs(self.optisForIndexes) do
        self.indexesForOptis[optti:__hash()] = index
        self.countP = self.countP + 1
    end
    self:list()
end

function ChordSpace.createFilename(voices, range, g, extension)
    extension = extension or '.lua'
    local gstring = string.format('g%.6f', g)
    gstring = string.gsub(gstring, '%.', '_')
    local filename = string.format('ChordSpaceGroup_V%d_R%d_%s%s', voices, range, gstring, extension)
    return filename
end

-- Loads the group if found, creates and saves it otherwise.

function ChordSpace.createChordSpaceGroup(voices, range, g)
    local filename = ChordSpace.createFilename(voices, range, 1)
    local file, message, error = io.open(filename, 'r')
    if file == nil then
        print(string.format('File "%s" not found, creating...', filename))
        chordSpaceGroup = ChordSpaceGroup:new()
        chordSpaceGroup:initialize(voices, range, g)
        chordSpaceGroup:save()
        return chordSpaceGroup
    else
        print(string.format('Loading ChordSpaceGroup from file "%s"...', filename))
        return ChordSpace.load(voices, range, g)
    end
end

function ChordSpace.load(voices, range, g)
    local filename = ChordSpace.createFilename(voices, range, 1)
    print('Loading:', filename)
    local deserialized = ChordSpaceGroup.load(filename)
    return deserialized
end

function ChordSpaceGroup:save(filename)
    filename = filename or ChordSpace.createFilename(self.voices, self.range, self.g, '.lua')
    local text = serialize(self)
    local writer = io.open(filename, 'w+')
    writer:write(text)
    writer:close()
end

function ChordSpaceGroup.load(filename)
    local reader = io.open(filename)
    local text = reader:read("*all")
    -- What's deserialized is a function, which needs to be called.
    local object = loadstring(text)()
    -- Fix up metatable.
    local chordSpaceGroup = ChordSpaceGroup:new(object)
    -- Fix up metatables of chords too.
    for index, opti in pairs(chordSpaceGroup.optisForIndexes) do
        chordSpaceGroup.optisForIndexes[index] = Chord:new(opti)
    end
    return chordSpaceGroup
end

-- Returns the chord for the indices of prime form, inversion,
-- transposition, and voicing. The chord is not in RP; rather, each voice of
-- the chord's OP may have zero or more octaves added to it.

function ChordSpaceGroup:toChord(P, I, T, V, printme)
    printme = printme or false
    P = P % self.countP
    I = I % 2
    T = T % ChordSpace.OCTAVE
    V = V % self.countV
    if printme then
        print('toChord:             ', P, I, T, V)
    end
    local optti = self.optisForIndexes[P]
    if printme then
        print('toChord:   optti:    ', optti, optti:__hash())
    end
    local optt = nil
    if I == 0 then
        optt = optti
    else
        optt = optti:I():eOPTT()
    end
    if printme then
        print('toChord:   optt:     ', optt)
    end
    local optt_t = optt:T(T)
    if printme then
        print('toChord:   optt_t:   ', optt_t)
    end
    local op = optt_t:eOP()
    if printme then
        print('toChord:   op:       ', op)
    end
    V = V % self.countV
    local revoicing = ChordSpace.octavewiseRevoicing(op, V, self.range)
    if printme then
        print('toChord:   revoicing:', revoicing)
    end
    return revoicing, opti, op
end

-- Returns the indices of prime form, inversion, transposition,
-- and voicing for a chord.

function ChordSpaceGroup:fromChord(chord, printme)
    printme = printme or false
    if printme then
        print('fromChord: chord:    ', chord, chord:iseOP())
    end
    local op = nil
    if chord:iseOP() then
        op = chord:clone()
    else
        op = chord:eOP()
    end
    if printme then
        print('fromChord: op:       ', op)
    end
    local optt = chord:eOPTT()
    if printme then
        print('fromChord: optt:     ', optt)
    end
    local T = 0
    for t = 0, ChordSpace.OCTAVE - 1, self.g do
        local optt_t = optt:T(t):eOP()
        if printme then
            print('fromChord: optt_t:   ', optt_t, t)
        end
        if optt_t:__eq_epsilon(op) == true then
            if printme then
                print('equals')
            end
            T = t
            break
        end
    end
    local optti = chord:eOPTTI()
    if printme then
        print('fromChord: optti:    ', optti, optti:__hash())
    end
    local P = self.indexesForOptis[optti:__hash()]
    local I = 0
    if optti ~= optt then
        I = 1
        local optt_i_optt = optt:I():eOPTT()
        if optt_i_optt ~= optti then
            print("Error: OPTT(I(OPTT)) must equal OPTTI.")
            print('optt_i_optt:', optt_i_optt:information())
            print('optti:      ', optti:information())
            os.exit()
        end
    end
    local voicing = ChordSpace.voiceleading(op, chord)
    V = self.indexesForVoicings[voicing:__hash()]
    if printme then
        print('fromChord: voicing:  ', voicing, V)
        print('fromChord:           ', P, I, T, V)
    end
    return P, I, T, V
end

function ChordSpaceGroup:list(listheader, listopttis, listvoicings)
    listheader = listheader or true
    listopttis = listopttis or false
    listvoicings = listvoicings or false
    if listheader then
        print(string.format('ChordSpaceGroup.voices: %8d', self.voices))
        print(string.format('ChordSpaceGroup.range : %8d', self.range))
        print(string.format('ChordSpaceGroup.g     : %13.4f', self.g))
        print(string.format('ChordSpaceGroup.countP: %8d', self.countP))
        print(string.format('ChordSpaceGroup.countI: %8d', self.countI))
        print(string.format('ChordSpaceGroup.countT: %8d', self.countT))
        print(string.format('ChordSpaceGroup.countV: %8d', self.countV))
    end
    if listopttis then
        for index, opti in pairs(self.optisForIndexes) do
            print(string.format('index: %5d  opti: %s  index from opti: %s', index, tostring(opti), self.indexesForOptis[opti:__hash()]))
        end
        for index = 0, #self.optisForIndexes - 1 do
            print(string.format('opti from index: %s  index:  %5d', tostring(self.optisForIndexes[index]), index))
        end
    end
    if listvoicings then
        for index, voicing in pairs(self.voicingsForIndexes) do
            print(string.format('voicing index: %5d  voicing: %s  index from voicing: %5d', index, tostring(voicing), self.indexesForVoicings[voicing:__hash()]))
        end
    end
end

function ChordSpaceGroup:printChords()
    for index, opti in pairs(self.optisForIndexes) do
        print(string.format('index: %5d  opti: %s %s', index, tostring(opti), opti:name()))
    end
end

function ChordSpaceGroup:printNamedChords()
    for index, opti in pairs(self.optisForIndexes) do
        local name = opti:name()
        if name ~= '' then
            print(string.format('index: %5d  opti: %s %s', index, tostring(opti), opti:name()))
        end
    end
end

return ChordSpace
*/
