/*  cssed (c) Iago Rubio 2003, 2004 - A tiny CSS editor.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtk/gtk.h>

#include "cssedwindow.h"
#include "configparser.h"
#include "interface.h"
#include "document.h"
#include "support.h"
#include "utils.h"
#include "debug.h"

/* some debugging macros, check debug.h */
#if !defined(DEBUG)
void do_nothing(char *s, ...) {return;}
#endif

#ifdef WIN32
# include <windows.h>
#endif

#ifdef WITH_HELP_MENUS
# include <stdlib.h>
#endif

#ifdef WITH_IPC_QUEUE
#ifdef DARWIN
# undef WITH_IPC_QUEUE
#else
# include <sys/types.h>
# include <sys/ipc.h>
# include <sys/msg.h>
# include <signal.h>
# include <limits.h>

# define CSSED_IPC_MSG_QUEUE 5423623
# define MAX_PATH 1024

struct ipcmsg
{
	long type;
	char file[MAX_PATH];
};

struct ipcproc_data
{
	CssedWindow *window;
	gint ident;
};

gpointer listen_queue_proc (gpointer * ident);
#endif
#endif

#ifdef WIN32
#ifdef NDEBUG
static void nulloutput( const gchar* s ){};
static void null_log( const gchar* dom, GLogLevelFlags l, const gchar* m, gpointer d ){};
#endif
#endif

int
main (int argc, char *argv[])
{
	GtkWidget *window;				// the main window
	CssedWindow *cssedwin;		// the main window object
	gint i;							// to parse arguments
	GList* plugins_to_load = NULL;	// list wilth all plugins to be loaded when the window is created.
	gchar* full_path;
	gchar* curdir;
#ifdef WITH_IPC_QUEUE
	gint identifier;
	key_t clave;
	gint size;
	struct ipcmsg buff_msg;
	struct ipcproc_data pdata;
	gboolean isserv;
	GThread *listener;

	if (!g_thread_supported ())
	{
		g_thread_init (NULL);
		gdk_threads_init ();
	}
#endif

	cssedwin = cssed_window_new();

#ifdef ENABLE_NLS
	bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
#endif

#ifdef WITH_IPC_QUEUE
	isserv = FALSE;
	clave = CSSED_IPC_MSG_QUEUE + getuid ();		// one queue for each user
	DBGMSG ("Will start ipc test %d\n", getpid ());
	// try to open the queue
	if ((identifier = msgget (clave, 0)) < 0)
	{
		DBGMSG ("Queue doesn't exists\n");
		if ((identifier =
			 msgget (clave, IPC_CREAT | 0666 | IPC_EXCL)) < 0)
		{
			DBGMSG ("Can't create queue %d\n",	 getpid ());
		}
		else
		{
			DBGMSG ("Queue created, this instance is the server %d\n",
																getpid ());
			isserv = TRUE;
			pdata.window = cssedwin;
			pdata.ident = identifier;
			listener =	g_thread_create ((GThreadFunc) listen_queue_proc,
						 &pdata, FALSE, NULL);
		}
	}
	else
	{
		DBGMSG ("Queue exists\n");
		//*
		if( argc > 1 ){ // avoid to open or listen if no args
			// the queue exists
			// check if dead (timeout)
			struct msqid_ds msg_stat;
			gint timediff;
			msgctl ( identifier, IPC_STAT, &msg_stat );
			if ( msg_stat.msg_qnum > 0 )
			{
				timediff = time (NULL) - msg_stat.msg_ctime;
				if ( timediff > 3 )
				{
					DBGMSG (
						"more than 3 seconds no reads diff=%d, deleting queue\n",
						timediff);
					msgctl (identifier, IPC_RMID, NULL);
					// must check here
					if (
						(identifier =
							msgget (clave, IPC_CREAT | 0666 | IPC_EXCL)
						) < 0)
					{
						DBGMSG ("Can't create queue %d\n", getpid ());
					}
					else
					{
						DBGMSG (
							"Queue created, this instance is the server %d\n",
							getpid ()
							);
						isserv = TRUE;
						listener =
							g_thread_create ((GThreadFunc) listen_queue_proc,
																&identifier,
																FALSE, NULL);
					}
				}
			}
			else
			{	// no messages send message
				buff_msg.type = getpid ();
				memset(buff_msg.file, 0, MAX_PATH);
				if( g_path_is_absolute (argv[1]) ){
					memcpy(buff_msg.file , argv[1], MAX_PATH);
				}else{
					curdir = g_get_current_dir ();
					full_path = g_build_filename( curdir, argv[1], NULL );
					memcpy(buff_msg.file , full_path, MAX_PATH);
					g_free(	full_path );
					g_free( curdir );
				}
				DBGMSG ("Queue exists %d\n", getpid ());

				if ((msgsnd (identifier, &buff_msg, MAX_PATH, 0)) < 0)
				{
					DBGMSG (
						"Can't send message, will start but not a server %d\n",
						getpid ()
						);
				}
				else
				{
					for (i = 2; i < argc; i++)
					{
						memset(buff_msg.file, 0, MAX_PATH);
						if( g_path_is_absolute (argv[i]) ){
							memcpy(buff_msg.file , argv[i], MAX_PATH);
						}else{
							curdir = g_get_current_dir ();
							full_path = g_build_filename( curdir, argv[i], NULL );
							memcpy(buff_msg.file , full_path, MAX_PATH);
							g_free(	full_path );
							g_free( curdir );
						}
						msgsnd (identifier, &buff_msg, MAX_PATH, 0);
					}
					DBGMSG ("Msgs sent bail out %d\n", getpid ());
					return 0;
				}
			}
		}// end args
	}
#endif

	gtk_set_locale ();
	gtk_init (&argc, &argv);
	gdk_rgb_init ();

#ifdef WIN32
#ifdef NDEBUG
	// no output console
	g_set_print_handler( nulloutput );
	g_set_printerr_handler( nulloutput );
	g_log_set_handler("Gdk",
				G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION,
				null_log, NULL);
	g_log_set_handler("Gtk",
				G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION,
				null_log, NULL);
				
	  g_log_set_handler("GLib",
				G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION,
				null_log, NULL);
				
	  g_log_set_handler("GLib-GObject",
				G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION,
				null_log, NULL);
				
	  g_log_set_handler("GModule",
				G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION,
				null_log, NULL);
				
	  g_log_set_handler("GThread",
				G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION,
				null_log, NULL);
 
#endif
#endif

#ifdef WIN32
	add_pixmap_directory ("pixmaps");
#else
	add_pixmap_directory (PACKAGE_DATA_DIR "/" PACKAGE "/pixmaps");
#endif

	init_default_config ( cssedwin );
	// this will return a GList with the filenames of plugins to be loaded
	plugins_to_load = parse_configuration_file ( cssedwin );
	// load all files in plugins list
	window = create_window (cssedwin);
	if( plugins_to_load != NULL ){
		load_all_plugins( cssedwin, plugins_to_load );
		g_list_free( plugins_to_load );
	}
	gtk_widget_show (window);

	/* args processing everything passed in taken as a file name */
	for ( i = 1; i < argc; i++ )
	{
		if( g_path_is_absolute (argv[i]) ){
			create_and_attach_new_named_doc ( cssedwin, argv[i] );
		}else{
			curdir = g_get_current_dir ();
			full_path = g_build_filename( curdir, argv[i], NULL );
			create_and_attach_new_named_doc ( cssedwin, full_path );
			g_free(	full_path );
			g_free( curdir );
		}
	}

#ifdef WITH_IPC_QUEUE
	gdk_threads_enter ();
	gtk_main ();
	gdk_threads_leave ();
#else
	gtk_main ();
#endif

#ifdef WITH_IPC_QUEUE
	  if( isserv ){
		DBGMSG("It's a server let's kill the queue\n");
		if( (msgctl( identifier, IPC_RMID, NULL )) < 0){
		  DBGMSG("Can't delete queue\n");
		}else{
		  DBGMSG("Queue deleted\n");
		}
		pthread_cancel(listener);
	  }
#endif

	return 0;
}

#ifdef WIN32
int APIENTRY WinMain(struct HINSTANCE__ *hInstance,
					 struct HINSTANCE__ *hPrevInstance,
					 LPSTR cmd,
					 int show)
	{
		return main( __argc, __argv );
	}
#endif

#ifdef WITH_IPC_QUEUE
gpointer
listen_queue_proc (gpointer * data)
{
	struct ipcmsg buff_msg;
	CssedWindow* cssedwin;
	int identifier;
	int size;
	GtkWidget *dialog;

	struct ipcproc_data *pdata;

	pdata = (struct ipcproc_data*) data;
	identifier = pdata->ident;
	cssedwin = pdata->window;

	gdk_threads_enter ();
	DBGMSG ("Checking queue %d\n", getpid ());
	gdk_threads_leave ();

	while (TRUE)
	{
		size = msgrcv (identifier, &buff_msg, MAX_PATH, 0, 0);
		while (size > 0)
		{
			gdk_threads_enter ();
			create_and_attach_new_named_doc (cssedwin, buff_msg.file);
			gdk_threads_leave ();
			size = msgrcv (identifier, &buff_msg, MAX_PATH, 0, 0);
		}
		sleep (2);
	}
}
#endif

