# elements.rb, copyright (c) 2006 by Vincent Fourmond: 
# The class describing a Tioga element
  
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
  
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details (in the COPYING file).

require 'Dobjects/Dvector'
require 'CTioga/debug'
require 'CTioga/log'

module CTioga

  # The base class for every single object that has to appear at one
  # point as a command in the tioga plot. It provides only one function:
  # do(f), which will be used by subclasses to do what they need.


  class TiogaElement
    include Debug
    include Log

    # this function must be called with a FigureMaker object to
    # draw the contents of the TiogaElement onto it. It calls
    # the function real_do, which should be redefined by the
    # children. You can redefine do too if you need another
    # debugging output.
    def do(f)
      debug "plotting #{self.inspect}"
      real_do(f)
    end

    # this function returns true if the object needs a CurveStyle parameter.
    def need_style?
      return false
    end
  end 
  

  # A class to represent a small function call to FigureMaker. Can represent
  # basically anything.
  class TiogaFuncall < TiogaElement

    # _symbol_ is the symbol to be called, and the remainder will be used
    # as arguments for the call.
    def initialize(symbol, *args)
      @symbol = symbol
      @args = args
    end
    
    def real_do(f)
      f.send(@symbol, *@args)
    end
  end

  # A class that can hold several 'child' elements, such as
  # the SubPlot class and the Region class. A missed funcall will
  # result in a try to call the parent, if it exists.
  class Container < TiogaElement
    # The various properties defining the class
    attr_reader :parent, :elements, :funcalls
    
    def initialize(p = nil)
      @parent = p

      # elements to be given to tioga
      @elements = []

      # Used to add initial function calls.
      @funcalls = []
    end

    def add_elem(elem)
      @elements << elem
    end

    def add_funcall(funcall)
      @funcalls << funcall
    end

    # Do the funcalls
    def make_funcalls(t)
      for f in @funcalls
        f.do(t)
      end
    end

    def has_plots?
      @elements.each {|e| 
        return true if e.is_a?(Curve2D)
        if e.respond_to? :has_plots? and e.has_plots?
          return true
        end
      }
      return false
    end


    # this function returns the boundaries for all subobjects
    def get_boundaries
      bounds = []
      @elements.each do |e|
        bounds.push(e.get_boundaries) if e.respond_to? :get_boundaries
      end
      return Curve2D.compute_boundaries(bounds)
    end
    

    # We redirect missed funcall to the @parent
    def method_missing(meth, *args)
      debug "Redirecting call #{meth} to parent"
      @parent.send(meth,*args)
    end

  end

  # This class represent a subplot, a subfigure or the main plot.

  class SubPlot < Container
    
    # these are the attributes that will be set 
    attr_writer :show_legend, :title,:x_label, :y_label, :legend

    # various important things:
    attr_accessor :legend_specs
    attr_reader :parent
    
    # The frame margins...
    attr_accessor :frame_margins

    # The amount of space that should be left between the given curve and
    # the box. It is an array [left,right,top,bottom].
    attr_accessor :plot_margins

    def number
      return @elements.length
    end
    
    def initialize(type = :subplot, parent = nil)
      super(parent)
      # @type is the type of the element, :subfigure, :subplot or nil
      # to indicate that it is the main plot ;-) !
      @type = type

      # the direct parent of that element. If nil, it means that it's
      # directly a subchild of the main plot.
      @parent = parent

      # Defaults to no margins (looks somehow less nice in many cases)
      @plot_margins = [0.0,0.0,0.0,0.0]
      
      # The margins of the subplot, if that happens to be a subplot
      # or a subfigure
      @frame_margins = {}


      # wether we need to show a legend or not
      @show_legend = if type == :subplot || type == nil
                       true
                     else
                       false
                     end
      
      @title = "A nice plot"
      
      @x_label = "$x$"
      @y_label = "$y$"

      # the hash used by show_plot_with_legend: we inherit it from the parent
      # if it exists
      @legend_specs = if parent 
                        parent.legend_specs.dup
                      else
                        {}
                      end


      # User-specified boundaries
      @bounds = [nil,nil,nil,nil]
    end

    def bound_left=(a)
      @bounds[0] = a
    end

    def bound_right=(a)
      @bounds[1] = a
    end

    def bound_top=(a)
      @bounds[2] = a
    end

    def bound_bottom=(a)
      @bounds[3] = a
    end


    def set_margin(which, what)
      @margins[which.to_s] = what.to_f
    end

    # this function computes the boundaries for the given plot...
    def compute_boundaries
      bounds = get_boundaries

      # Increase the boundary with the margins
      width = bounds[1] - bounds[0]
      bounds[0] -= width * @plot_margins[0]
      bounds[1] += width * @plot_margins[1]
      height = bounds[2] - bounds[3]
      bounds[2] += height * @plot_margins[2]
      bounds[3] -= height * @plot_margins[3]

      # overriding with user-specified values if applicable
      4.times do |i|
        if @bounds[i] 
          bounds[i] = @bounds[i] 
        end
      end
      return bounds
            
    end
    # this function returns true if at least one of the elements is a Curve


    def has_legends?
      @elements.each {|e| 
        if e.is_a?(Curve2D)
          return true if e.has_legend?
        end
      }
      return false
    end


    # this function takes a FigureMaker object and returns the block of
    # instructions corresponding to the current contents of the SubPlot.
    # To actually plot it correctly, it just needs a further wrapping
    # in subplot or subfigure.

    def make_main_block(t)
      if has_plots?
        block = proc {
          t.show_plot(compute_boundaries) {
            t.do_box_labels(@title, @x_label, @y_label)
            @elements.each do |e|
              e.do(t)
            end
          } 
        }
      else
        block = proc {
          @elements.each { |e|
            e.do(t)
          }
        }
      end

      # Wrap the call in a show_plot_with_legends if necessary
      if @show_legend && has_plots?  && has_legends?
        blk =  proc {
          t.show_plot_with_legend(@legend_specs,&block)
        }
      else
        blk = block
      end
      return proc {
        make_funcalls(t)
        blk.call
      }

    end

    # this function sets up the appropriate environment and calls the block...
    def real_do(t)
      if @type
        t.send(@type, @frame_margins) { make_main_block(t).call }
      else
        make_main_block(t).call
      end
    end
  end

  # The Region class handles the cases where the user wants some parts
  # between curves to be coloured. In this case, all the curves which
  # are included between the --region and --end options will
  class Region < Container
    
    attr_accessor :region_color
    
    def initialize(parent = nil)
      super
      @region_color = [0.9,0.9,0.9] # Very light gray.
    end
    
    def real_do(t)
      debug "Region: plotting elements"
      warn "The --region scheme is still very experimental and fragile"
      warn "Don't depend on current behavior, as it can change without notice"
      make_funcalls(t)
      # We fill first, as it looks way better
      t.context do 
        for el in @elements
          if el.respond_to?(:make_path)
            el.make_path(t)
            t.close_path 
            t.clip
          end
        end
        
        t.fill_color = @region_color
        bound = get_boundaries
        t.fill_rect(bound[0], bound[3], bound[1] - bound[0], 
                    bound[2] - bound[3])
      end
      # Then, we plot the elements.
      @elements.each do |e|
        e.do(t)
      end

    end

  end

end
