# themes.rb, copyright (c) 2007 by Vincent Fourmond: 
# The main support for themes in ctioga
  
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
  
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details (in the COPYING file).

require 'CTioga/log'


module CTioga

  # The namespace for themes. Supposedly, all themes should end up
  # here.
  module Themes

    class BaseTheme

      THEMES = {}
      
      # Basic initialization. Cannot take any compulsory parameter
      def initialize
      end

      # Returns an object appropriate to describe the
      # style of the next curve. It must return all information
      # as appropriate, in the form of a CurveStyle object.
      # It takes the _name_ of the next set.
      def next_curve_style(name)
      end

      # A hook to be run at the beginning of the drawing.
      # _t_ is the FigureMaker object.
      def bod_hook(t)
        # Does nothing by default.
      end

      # returns extra arguments to be pushed onto the
      # cmdline. Can come in useful if you don't want to
      # play with ctiog's internals
      def cmdline_extra_args
        return []
      end

      # A function to be reimplemented by children to handle color_sets.
      def choose_set(type,set)
        warn "Theme #{self.class.name} does not support sets"
      end

      # A callback to register themes.
      def self.inherited(cls)
        # THEMES[cls.name] = cls # Probably too long to be useful.
        THEMES[cls.name.split(/::/).last.gsub(/Theme/,"")] = cls
        THEMES[cls.name.split(/::/).last.gsub(/Theme/,"").downcase] = cls
      end

    end

    # Some constants that could be of interest to themes:
    module Data
      include Tioga::FigureConstants

      COLORS = { 
        "standard" => 
        [Red, Green, Blue, Cyan, Magenta, Orange],
        "colorblind" => 
        [BrightBlue, Goldenrod, Coral, Lilac, FireBrick, RoyalPurple],
        "black-only" => 
        [Black,]
      }
      
      # The sets of markers. Well, sets, that's a big word...
      MARKERS = { 
        "standard" => 
        [Bullet, TriangleUp, Square, Plus, Times]
      }

      Line_Type_Dash_Dot_Dot = [[5,2,1,2,1,2],0]
      
      # Linestyles.
      LINES = {
        "solid-only" => [Line_Type_Solid], # A single style, solid line.
        "standard" => 
        [Line_Type_Solid, 
         Line_Type_Dots, 
         Line_Type_Dashes, 
         Line_Type_Dot_Long_Dash,
         Line_Type_Dash_Dot_Dot
        ],
      }
      
    end

    # Now, some code to be used as an inclusion directly.

    attr_accessor :override_style

    def initialize_themes
      @override_style = CurveStyle.new

      # To reproduce old ctioga behavior, we manually set
      # :marker to false
      @override_style[:marker] = false

      # For now, manually require themes. This *is* ugly ;-) !
#      require 'CTioga/themes/classical'
      for f in Dir.glob(File.dirname(__FILE__) + "/themes/*") +
          Dir.glob("#{ENV['HOME']}/.ctioga/themes/*")
        begin
          require f
          info "Successfully loaded theme file #{f}"
        rescue Exception => e
          warn "Failed to load theme file #{f}, ignoring"
          debug "Failed to load theme #{f} with #{e.inspect}"
        end
      end

      # The current theme:
      choose_theme(BaseTheme::THEMES.keys.first) unless
        choose_theme('Classical')
    end

    # Selects the current theme:
    def choose_theme(theme_name)
      if BaseTheme::THEMES.key?(theme_name)
        @theme = BaseTheme::THEMES[theme_name].new
        info "Selecting theme #{theme_name}"
        args = @theme.cmdline_extra_args
        debug "Theme #{theme_name} pushes #{args.join ' '} onto the "+
          "command line"
        unshift_cmdline_args(*args)
        return true
      else
        warn "Theme #{theme_name} doesn't exist, ignoring"
        return false
      end
    end

    # Processes a style argument.
    def style_argument(style_element, value, namespace)
      if value =~ /auto/i
        override_style.delete(style_element)
        debug "Deleting override #{style_element}"
        return
      elsif value =~ /no|off/i or not value
        override_style[style_element] = false
      else
        override_style[style_element] = if block_given?
                                          yield(value)
                                        else
                                          namespace.const_get(value)
                                        end
      end
      debug "Setting override #{style_element} to " +
        "#{override_style[style_element].inspect}"
    end


    def theme_prepare_parser(parser)
      parser.separator "\nStyle and themes options"
      parser.on("-c", "--[no-]color COLOR",
                "Sets the color for drawing the curves.", 
                "Use 'auto' to leave the decision to ",
                "the themes, and 'no' to get no lines.") do |val|
        style_argument(:color, val, nil) do |v|
          CTioga.get_tioga_color(v)
        end
      end

      parser.on("-m", "--[no-]marker [MARKER]",
                 "Sets the markers for drawing data points", 
                 "Use 'auto' to get automatic markers,",
                 "and 'no' to get no marker (the default)") do |val|
        style_argument(:marker, val, Tioga::FigureConstants)
      end
      
      parser.on("--marker-color COLOR",
                 "Sets the markers' color. See also --color") do |val|
        style_argument(:marker_color, val, nil) do |v|
          CTioga.get_tioga_color(v)
        end
      end

      parser.on("--line-style STYLE",
                "Sets the line style") do |val|
        style_argument(:line_style, val, Data)
      end

      parser.on("--line-width WIDTH",
                "Sets the line width") do |w|
        style_argument(:linewidth, w, nil) do |val|
          Float(val)
        end
      end

      parser.on("--[no-]interpolate",
                "If set, the points will be joined", "by a nice "+
                "interpolated curve") do |w|
        override_style.interpolate = w
      end

      parser.on("--marker-scale SCALE",
                "The scale of the markers used for curves.", 
                "Defaults to 0.5"
                 ) do |w|
        style_argument(:marker_scale, w, nil) do |v|
          Float(v)
        end
      end

      parser.on("--error-bar-color COLOR",
                "Sets the error bars' color. See also --color") do |val|
        style_argument(:error_bar_color, val, nil) do |v|
          CTioga.get_tioga_color(v)
        end
      end

      parser.on("--drawing-order ORDER",
                "Sets the order for drawing curve elements") do |val|
        begin
          style_argument(:drawing_order, val, nil) do |v|
            i = Integer(v)
            CurveStyle::DrawingOrder.fetch(i) # To raise an
            # exception in case i is not valid
            i 
          end
        rescue
          puts "Invalid drawing order #{val}. Valid ones are the " +
            "following integers"
          CurveStyle::DrawingOrder.each_with_index do |vals, i|
            puts "#{i} -> #{vals.map {|s| s.to_s}.join(', ')}"
          end
        end
      end

      parser.separator ''
      parser.on("--theme THEME",
                "Chooses the current theme. See --theme-list for ",
                "a list of current valid themes"
                 ) do |w|
        choose_theme(w)
      end

      parser.on("--theme-list",
                "Lists available themes."
                 ) do 
        puts
        puts "Currently available themes are"
        puts
        puts BaseTheme::THEMES.keys.map {|i| i.downcase}.uniq.join(" ")
      end


      @parser.on("--mono",
                 "Compatibility option for --theme mono") do |w|
        choose_theme('mono')
      end

      # Sets:
      # Note that, due to scoping side-effects, one has to use the
      # block form of the iteration, else type gets overwritten
      # and we end up writing only to the last possibility.
      {
        :colors => "color",
        :markers =>  "marker",
        :markers_colors => "marker-color",
        :linestyle => "line-style"
      }.each do |type, name|
        parser.on("--#{name}-set SET", 
                  "Choose the current set for #{name} ",
                  "(#{@theme.sets[type].available_sets.join(' ')})") do |set|
          debug "Using set #{set} for #{type} on theme " +
            "#{current_theme.class}"
          current_theme.choose_set(type, set)
        end
      end
        
    end

    # The current theme:
    def current_theme
      return @theme
    end


    # Returns the current style and resets all pending elements.
    def get_current_style(set)
      debug "Legend: #{override_style.legend}"
      style = @theme.next_curve_style(tex_quote_text(set))

      style.legend = false unless @autolegends
      debug "Override is #{override_style.inspect}"
      style.override!(override_style)

      # We remove the legend information, that it doesn't get used
      # again.
      override_style.delete(:legend)
      debug "Style: #{style.inspect}"
      return style
    end

    # Run the hook at beginning of the drawing:
    def theme_bod_hook(t)
      current_theme.bod_hook(t)
    end

  end

end
