#ifndef _SERVER_H
#define _SERVER_H 1

#include <stddef.h>
#include <stdint.h>
#include <sys/epoll.h>
#include <sys/types.h>
#include <time.h>
#include <memory>
#include <mutex>
#include <queue>
#include <string>
#include <unordered_map>
#include <vector>

#include "tlse.h"

#include "client.h"
#include "stream.h"
#include "thread.h"

class Acceptor;
class ClientProto;
struct Stream;

#define EPOLL_MAX_EVENTS 8192
#define EPOLL_TIMEOUT_MS 20
#define MAX_CLIENT_REQUEST 16384
#define REQUEST_READ_TIMEOUT_SEC 60

class CubemapStateProto;
class StreamProto;
class HLSZombieProto;

// See Server::hls_zombies, below.
struct HLSZombie {
	std::string remote_addr;
	std::string url;
	std::string referer;
	std::string user_agent;
	timespec expires;
};

class Server : public Thread {
public:
	Server();
	~Server();

	// Get the list of all currently connected clients.	
	std::vector<ClientStats> get_client_stats() const;

	// See hls_zombies, below.
	std::vector<HLSZombie> get_hls_zombies();

	// Set header (both HTTP header and any stream headers) for the given stream.
	void set_header(int stream_index,
	                const std::string &http_header,
	                const std::string &stream_header);

	// Set that the given stream should use the given max pacing rate from now on.
	// NOTE: This should be set before any clients are connected!
	void set_pacing_rate(int stream_index, uint32_t pacing_rate);

	// These will be deferred until the next time an iteration in do_work() happens,
	// and the order between them are undefined.
	// XXX: header should ideally be ordered with respect to data.
	void add_client_deferred(int sock, Acceptor *acceptor);
	void add_data_deferred(int stream_index, const char *data, size_t bytes, uint16_t metacube_flags, const RationalPTS &pts);

	// These should not be called while running, since that would violate
	// threading assumptions (ie., that epoll is only called from one thread
	// at the same time).
	CubemapStateProto serialize(std::unordered_map<const std::string *, size_t> *short_response_pool);
	void add_client_from_serialized(const ClientProto &client, const std::vector<std::shared_ptr<const std::string>> &short_responses);
	int add_stream(const std::string &url,
	               const std::string &hls_url,
	               size_t bytes_received,
	               size_t prebuffering_bytes,
	               Stream::Encoding encoding,
	               Stream::Encoding src_encoding,
	               unsigned hls_frag_duration,
	               size_t hls_backlog_margin,
	               const std::string &allow_origin);
	int add_stream_from_serialized(const StreamProto &stream, int data_fd);
	void add_hls_zombie_from_serialized(const HLSZombieProto &hls_zombie);
	int lookup_stream_by_url(const std::string &url) const;
	void set_backlog_size(int stream_index, size_t new_size);
	void set_prebuffering_bytes(int stream_index, size_t new_amount);
	void set_encoding(int stream_index, Stream::Encoding encoding);
	void set_src_encoding(int stream_index, Stream::Encoding encoding);
	void set_hls_frag_duration(int stream_index, unsigned hls_frag_duration);
	void set_hls_backlog_margin(int stream_index, size_t hls_backlog_margin);
	void set_allow_origin(int stream_index, const std::string &allow_origin);
	void register_hls_url(int stream_index, const std::string &hls_url);
	void add_gen204(const std::string &url, const std::string &allow_origin);
	void create_tls_context_for_acceptor(const Acceptor *acceptor);

private:
	// Mutex protecting queued_add_clients.
	// Note that if you want to hold both this and <mu> below,
	// you will need to take <mu> before this one.
	mutable std::mutex queued_clients_mutex;

	// Deferred commands that should be run from the do_work() thread as soon as possible.
	// We defer these for two reasons:
	//
	//  - We only want to fiddle with epoll from one thread at any given time,
	//    and doing add_client() from the acceptor thread would violate that.
	//  - We don't want the input thread(s) hanging on <mu> when doing
	//    add_data(), since they want to do add_data() rather often, and <mu>
	//    can be taken a lot of the time.
	//	
	// Protected by <queued_clients_mutex>.
	std::vector<std::pair<int, Acceptor *>> queued_add_clients;

	// All variables below this line are protected by the mutex.
	mutable std::mutex mu;

	// All streams.
	std::vector<std::unique_ptr<Stream>> streams;

	// Map from URL to index into <streams>.
	std::unordered_map<std::string, int> stream_url_map, stream_hls_url_map;

	// Map from URL to CORS Allow-Origin header (or empty string).
	std::unordered_map<std::string, std::string> ping_url_map;

	// Map from file descriptor to client.
	std::unordered_map<int, Client> clients;

	// A list of all clients, ordered by the time they connected (first element),
	// and their file descriptor (second element). It is ordered by connection time
	// (and thus also by read timeout time) so that we can read clients from the
	// start and stop processing once we get to one that isn't ready to be
	// timed out yet (which means we only have to look at each client exactly once,
	// save for the first element of the queue, which is always checked).
	//
	// Note that when we delete a client, we don't update this queue.
	// This means that when reading it, we need to check if the client it
	// describes is still exists (ie., that the fd still exists, and that
	// the timespec matches).
	std::queue<std::pair<timespec, int>> clients_ordered_by_connect_time;

	// HLS is harder to keep viewer statistics for than regular streams,
	// since there's no 1:1 mapping between ongoing HTTP connections and
	// actual viewers. After a HLS fragment has been successfully sent,
	// we keep a note of that in this structure, so that we can add some
	// fake entries in the .stats file for clients that we believe are still
	// watching, but are not downloading anything right now. We clean this
	// out whenever we write statistics centrally.
	//
	// The structure is keyed by X-Playback-Session-Id if it exists
	// (typically iOS clients) or IP address otherwise; we can't use the socket,
	// since clients can (and do) keep open multiple HTTP connections for
	// the same video playack session, and may also close the socket
	// between downloading fragments. This means multiple clients between
	// the same NAT may be undercounted, but that's how it is.
	std::unordered_map<std::string, HLSZombie> hls_zombies;

	// Used for epoll implementation (obviously).
	int epoll_fd;
	epoll_event events[EPOLL_MAX_EVENTS];

	// For each TLS-enabled acceptor, our private server context for its key pair.
	std::unordered_map<const Acceptor *, TLSContext *> tls_server_contexts;

	// The actual worker thread.
	virtual void do_work();

	// Process a client; read and write data as far as we can.
	// After this call, one of these four is true:
	//
	//  1. The socket is closed, and the client deleted.
	//  2. We are still waiting for more data from the client.
	//  3. We've sent all the data we have to the client,
	//     and put it in <sleeping_clients>.
	//  4. The socket buffer is full (which means we still have
	//     data outstanding).
	//
	// For #2, we listen for EPOLLIN events. For #3 and #4, we listen
	// for EPOLLOUT in edge-triggered mode; it will never fire for #3,
	// but it's cheaper than taking it in and out all the time.
	void process_client(Client *client);

	// If the TLS library wants to write anything to this client,
	// output it. Returns true if the processing should go to sleep
	// (an error, or lack of outgoing buffer space).
	bool send_pending_tls_data(Client *client);

	// Reads regular data fro ma socket. Returns -1 if the processing
	// should go to sleep (an error, or no data available yet), otherwise
	// the number of bytes read.
	int read_plain_data(Client *client, char *buf, size_t max_size);

	// Reads (decrypted) data from a TLS socket. Returns -1 if the processing
	// should go to sleep (an error, or no data available yet), otherwise
	// the number of bytes read. The buffer will be used as scratch space
	// for TLS data, so it can be overwritten by more bytes than what is returned.
	int read_tls_data(Client *client, char *buf, size_t max_size);

	// Close a given client socket, and clean up after it.
	void close_client(Client *client);

	// Listen for a different set of epoll events.
	void change_epoll_events(Client *client, uint32_t events);

	// If we're supposed to listen for more requests (persistent HTTP connections),
	// puts the client back into READING_REQUEST, changes its epoll flags and returns
	// true.
	bool more_requests(Client *client);

	// Parse the HTTP request. Returns a HTTP status code (200/204/400/404).
	int parse_request(Client *client);

	// Construct the HTTP header for a regular stream, and set the client into
	// the SENDING_HEADER state.
	void construct_stream_header(Client *client);

	// Construct a HLS playlist (or get it from the cache), and set the client into
	// the SENDING_HEADER state.
	void construct_hls_playlist(Client *client);

	// Construct a generic error with the given line, and set the client into
	// the SENDING_SHORT_RESPONSE state.
	void construct_error(Client *client, int error_code);

	// Construct a 204, and set the client into the SENDING_SHORT_RESPONSE state.
	void construct_204(Client *client);

	void process_queued_data();
	void skip_lost_data(Client *client);

	void add_client(int sock, Acceptor *acceptor);

	// Mark that a client just went into READING_REQUEST state, so we should
	// note the current time of day and then put it into <clients_ordered_by_connect_time>.
	void start_client_timeout_timer(Client *client);
};

#endif  // !defined(_SERVER_H)
