#!/usr/bin/ruby -w
# -*- ruby -*-

require './env'
require './log'


class CVSDeltaOptions
  attr_accessor :verbose, :compression, :adds, :changes, :deletes, :execute
  attr_accessor :confirm, :quiet, :process_unknown_dirs
  attr_accessor :fromdate, :fromrev, :todate, :torev, :tmpdir
  attr_accessor :package, :version

  def initialize(package = "undef", version = "1.2.3.4")
    @package = package
    @version = version

    @verbose     = false        # whether to spew debugging output

    @quiet       = false        # whether to suppress warnings
    @confirm     = false        # whether to confirm remove commands
    @compression = 3            # compression for net transfer

    @changes     = true         # whether to show files that changed
    @adds        = true         # whether to show files that were added
    @deletes     = true         # whether to show files that were deleted
    @execute     = false        # whether to execute

    @fromdate    = nil          # starting date of comparison ("cvs -D @fromrev")
    @todate      = nil          # ending date of comparison ("cvs -. @from... -D @todate")
    @fromrev     = nil          # starting revision of comparison ("cvs -r @fromrev")
    @torev       = nil          # ending revision of comparison ("cvs -... @from.... -r @torev")
    @process_unknown_dirs = true    # whether to skip directories that are not in CVS
    @tmpdir      = ENV["TMPDIR"] || "/tmp"
  end

  def run
    read_global_rcfile
    read_local_rcfile
    read_environment_variable
    read_options
  end

  def read_environment_variable
    # process the environment variable
    if ENV["CVSDELTAOPTS"]
      options = ENV["CVSDELTAOPTS"].split(/\s+/)
      while options.length > 0
        opt = options.shift
        Log.log "processing opt " + opt
        arg = options.shift
        process_option(opt, options)
      end
    end
  end

  def read_options
    while ARGV.length > 0
      # can't modify ARGV, as of Ruby 1.8.1:
      arg = ARGV.shift.dup
      break if process_option(arg, ARGV)
    end
  end

  # Returns whether the value matches a true value, such as "yes", "true", or
  # "on".

  def to_boolean(value)
    [ "yes", "true", "on" ].include?(value.to_s.downcase)
  end

  def process_option(opt, args = nil)
    opt.gsub!(/^\-+/, "")

    case opt
      
    when "a", "adds"
      @adds = true
    when "A", "no-adds"
      @adds = false

    when "c", "changes"
      @changes = true
    when "C", "no-changes", "nodiff"
      @changes = false
      @changes = false

    when "d", "deletes"
      @deletes = true
    when "D", "no-deletes"
      @deletes = false

    when "e", "execute"
      @execute = true

    when "h", "help"
      show_help
      exit

    when "i", "confirm"
      @confirm = true
      
    when "q", "quiet"
      @quiet = true

    when "s", "skip-unknown-directories"
      @process_unknown_dirs = false

    when "V", "verbose"
      @verbose = true

    when "v", "version"
      print @package, ", version ", @version, "\n"
      print "Written by Jeff Pace (jpace@incava.org).\n"
      print "Released under the Lesser GNU Public License.\n"
      exit 1

    when "z", "compression"
      # no need to convert it from an integer, since it'll be written back out as
      # a string:
      @compression = args.shift

    when "f", "from-date"
      @fromdate = args.shift

    when "t", "to-date"
      @todate = args.shift

    when "F", "from-revision"
      @fromrev = args.shift
      
    when "T", "to-revision"
      @torev = args.shift

    else
      return true
    end
    return false
  end

  def read_rc_file(rc)
    IO.readlines(rc).each do |line|
      line.sub!(/\s*#.*/, "")
      line.chomp!
      name, value = line.split(/\s*[=:]\s*/)
      next unless name && value

      case name
      when "adds"
        @adds = to_boolean(value)
      when "skip-unknown-directories"
        @process_unknown_dirs = !to_boolean(value)
      when "changes"
        @changes = to_boolean(value)
      when "compression"
        # no need to convert it from an integer, since it'll be written back out
        # as a string:
        @compression = value
      when "confirm"
        @confirm = to_boolean(value)
      when "deletes"
        @deletes = to_boolean(value)
      when "diff"
        @changes = !to_boolean(value)
      when "execute"
        @execute = to_boolean(value)
      when "quiet"
        @quiet = to_boolean(value)
      when "verbose"
        @verbose = to_boolean(value)
      end
    end
  end

  def read_global_rcfile
    # process the rc file
    if hd = home_directory
      rc = hd + "/.cvsdeltarc"
      Log.log "reading RC file: #{rc}"
      if File.exists?(rc)
        read_rc_file(rc)
      end
    end
  end

  def read_local_rcfile
    # Use the topmost resource file in this project. We may refine this
    # functionality so that multiple rc files can be used within a project

    topdir = find_top_of_project

    if topdir && File.exists?(topdir + "/.cvsdeltarc")
      read_rc_file(topdir + "/.cvsdeltarc")
    end
  end

  def find_top_of_project(dir = File.expand_path("."))
    repfile = dir + "/CVS/Repository"
    if File.exists?(repfile)
      IO.readlines(repfile).each do |line|
        if line.index("/")
          # keep going up the directory structure
          if dir == "/"
            return nil
          else
            return find_top_of_project(File.dirname(dir))
          end
        else
          return dir
        end
      end
    end
    return nil
  end

  def show_help

    puts "USAGE"
    puts "    cvsdelta [options] directory..."
    puts ""
    puts "OPTIONS"
    puts "    -a, --adds"
    puts "        Display the files that were added."
    puts ""
    puts "    -A, --no-adds"
    puts "        Do not display the files that were added."
    puts ""
    puts "    -c, --changes"
    puts "        Display the files that were changed."
    puts ""
    puts "    -C, --no-changes, --nodiff"
    puts "        Do not compare files that exist locally and in CVS."
    puts ""
    puts "    -d, --deletes"
    puts "        Display the files that were deleted."
    puts ""
    puts "    -D, --no-deletes"
    puts "        Do not display the files that were deleted."
    puts ""
    puts "    -e, --execute"
    puts "        Execute the associated CVS commands (\"add\" and \"remove\") for "
    puts "        the added and deleted files."
    puts ""
    puts "    -f, --from-date"
    puts "        Compare the files to their version as of the given date, rather"
    puts "        than their current version in CVS."
    puts ""
    puts "    -F, --from-revision"
    puts "        Compare the files to their version as of the given revision, rather"
    puts "        than their current version in CVS."
    puts ""
    puts "    -h, --help"
    puts "        Display a help message."
    puts ""
    puts "    -i, --confirm"
    puts "        Interactively confirm deleted files with the user before removing them"
    puts "        from CVS."
    puts ""
    puts "    -q, --quiet"
    puts "        Run with minimum output."
    puts ""
    puts "    -s, --skip-unknown-directories"
    puts "        Skip directories that are not in CVS."
    puts ""
    puts "    -t, --to-date"
    puts "        Compare the files to their version as of the given date, rather"
    puts "        than to the local files. This is valid only with the --from-date"
    puts "        or --from-revision options."
    puts ""
    puts "    -T, --to-revision"
    puts "        Compare the files to their version as of the given revision, rather"
    puts "        than to the local files. This is valid only with the --from-date"
    puts "        or --from-revision options."
    puts ""
    puts "    -v, --version"
    puts "        Display the version and exit."
    puts ""
    puts "    -V, --verbose"
    puts "        Run with maximum output."
    puts ""
    puts "    -z [LEVEL], ---compression [LEVEL]"
    puts "        Set the compression to the given level for net traffic."

  end

end


if __FILE__ == $0
  dopt = CVSDeltaOptions.new
  dopt.run
end
