#!/usr/bin/ruby -w
# -*- ruby -*-

# Processes output from "cvs diff".

require './log'
require './cvsstats'

class DiffOutput
  include Loggable

  def initialize(total)
    fmt = '(\d+)(?:,(\d+))?'
    @total  = total
    @regexp = Regexp.new("^" + fmt + char + fmt)
    @md     = nil
  end

  def match(line)
    @md = @regexp.match(line)
  end

  def update(record)
    nlines = number_of_lines
    update_record(record, nlines)
    update_record(@total, nlines)
  end

  def to_s
    self.class.to_s + " " + @regexp.source
  end

  # Returns the amount of lines that changed, based on the MatchData object
  # which is from standard diff output

  def number_of_lines
    from = diff_difference(1, 2)
    to   = diff_difference(3, 4)
    1 + [from, to].max
  end

  # Returns the difference between the two match data objects, which represent
  # diff output (3,4c4).

  def diff_difference(from, to)
    if @md[to] then @md[to].to_i - @md[from].to_i else 0 end
  end

end


class DiffOutputAdd < DiffOutput

  def char; 'a'; end

  def update_record(rec, nlines)
    rec.adds += nlines
  end

end


class DiffOutputChange < DiffOutput

  def char; 'c'; end

  def update_record(rec, nlines)
    rec.changes += nlines
  end

end


class DiffOutputDelete < DiffOutput

  def char; 'd'; end

  def update_record(rec, nlines)
    rec.deletes += nlines
  end

end


class CVSDiff
  include Loggable

  attr_accessor :tmpdir, :fromdate, :fromrev, :todate, :torev, :compression, :progress, :total

  attr_reader :missing, :deleted, :changed

  def initialize(args, verbose = nil)
    name = self.class

    @total = DeltaFile.new("total")

    @addre = DiffOutputAdd.new(@total)
    @delre = DiffOutputDelete.new(@total)
    @chgre = DiffOutputChange.new(@total)

    @tmpdir = "/tmp"
    @fromdate = nil
    @fromrev = nil
    @todate = nil
    @torev = nil
    @compression = 3
    @progress = nil
    @verbose = verbose
    @args = args
  end

  def run
    # Ignore the .cvsrc file; handle only normal diff output.
    
    # Tweaking compression (via -z[0 .. 9]) makes diff less likely to hang
    # after producing output. Both -z0 and -z9 work best on my system (against
    # the doctorj CVS repository at SourceForge.net).

    curfile = nil

    diffopts  = ""
    if @fromrev || @fromdate
      if @fromrev
        diffopts += " -r #{@fromrev} "
      else
        diffopts += " -D \"#{@fromdate}\" "
      end
      if @torev || @todate
        if @torev
          diffopts += " -r #{@torev} "
        else
          diffopts += " -D \"#{@todate}\" "
        end
      end
    elsif @torev || @todate
      $stderr.puts "ERROR: --to-... option requires --from-..."
      exit
    end
    
    diffopts += @args.join(" ")

    @missing = FileArray.new
    @deleted = FileArray.new
    @changed = FileHash.new

    # backticks seem to work more consistenty than IO.popen, which was losing
    # lines from the CVS diff output.

    cmd = "cvs -fq -z" + @compression.to_s + " diff " + diffopts + " 2>&1"
    log "executing command " + cmd

    # For whatever reason, between CVS and Ruby, the best combination is to
    # write CVS's output to a temporary file. Trying to tie to the output
    # streams was resulting in some output being lost, especially that written
    # to stderr.

    pid = Process.pid
    outfile = @tmpdir + "/cvsdelta." + pid.to_s

    trap("INT") do 
      # If we get interrupted, make sure we delete the outfile:
      File.unlink(outfile) if File.exists?(outfile)

      # This bypasses the stack trace on exit.
      abort
    end

    cmd = "(" + cmd + ") > " + outfile
    `#{cmd}`

    lines = IO.readlines(outfile)
    if lines.size > 0 && lines[0].index(/cvs \[diff aborted\]: Can\'t parse date\/time: (.+)/)
      $stderr.puts "ERROR: invalid date/time: '#{$1}'"
      exit
    end
    
    lines.each do |line|
      @progress.tick(line) if @progress

      #log "line: " + line

      if line.index(/^\?\s*(\S*)/)
        # some CVS servers seem to write new files as "? foo/bar.x", but we'll
        # figure out the new files for ourselves anyway
      elsif line.index(/^cvs server:\s*(\S*)was removed/) ||
          line.index(/^cvs (?:diff|server): *cannot find\s*(\S*)/)
        # various ways that CVS servers tell us what was removed, but we'll
        # figure it out for ourself
        file = $1
        # add_deleted_file(file)
        log "deleted file: " + file
      elsif line.index(/^Index:\s+(\S+)/)
        curfile = $1
        log "new current file: #{curfile}"
      elsif line.index(/^\s*cvs server: no revision for .+ in file (.*)/)
        fname = $1
        log "missing file line: #{line}"
        log "adding missing file: #{fname}"
        @missing.push(fname)
      elsif line.index(/^\s*cvs server: tag .*? is not in file (.*)/)
        fname = $1
        log "missing file line: #{line}"
        log "adding missing file: #{fname}"
        @missing.push(fname)
      elsif line.index(/^\s*cvs server: (.*) no longer exists/)
        fname = $1
        log "deleted file #{fname} from line: #{line}"
        @deleted.push(fname)
      elsif line.index(/^Binary files .*? and .*? differ/)
        log "binary files differ"
        rec = get_record(curfile)
        rec.changes = DeltaFile::BINARY
      else
        [ @addre, @chgre, @delre ].each do |re|
          if re.match(line)
            rec = get_record(curfile)
            re.update(rec)
            break
          else
            # log re.to_s + ": not a match line: " + line.chomp
          end
        end
      end
    end

    File.unlink(outfile)
  end

  def dump
    puts "missing:"
    @missing.each do |f|
      puts "    #{f}"
    end
    
    puts "deleted:"
    @deleted.each do |f|
      puts "    #{f}"
    end

    puts "changed:"
    @changed.each do |fname, record|
      puts "    #{record}"
    end
  end

  def get_record(file)
    unless @changed.include?(file)
      @changed[file] = ExistingFile.new(file)
    end
    @changed[file]
  end

end

if __FILE__ == $0
  Log.verbose = true
  dp = CVSDiff.new(ARGV, true)
  # dp.fromrev = "1.4"
  dp.run
  dp.dump
end

