/*
 * Copyright 2004-2007 J. Dahl and L. Vandenberghe.
 *
 * This file is part of CVXOPT version 0.9.
 *
 * CVXOPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * CVXOPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include "cvxopt.h"
#include "misc.h"
#include "mosek.h"

PyDoc_STRVAR(mosek__doc__,
"Interface to the LP and QP solvers in MOSEK.\n\n" 
"The MOSEK control parameters have the default values listed in \n"
"the MOSEK documentation at\n"
"  http://www.mosek.com/products/3/tools/doc/html/tools/node22.html.\n"
"They can be modified by making an entry in the dictionary\n"
"mosek.options.  For example, the command\n"
"  mosek.options['MSK_IPAR_LOG']=0\n"
"turns off the printed output during execution of mosek.solvelp().");

static PyObject *mosek_module;

static void MSKAPI printstr(void *handle, char str[])
{
  printf("%s",str);
} /* printstr */

static char doc_mosek_solvelp[] = 
    "Solves a linear program using MOSEK.\n\n"
    "(prosta, solsta, x, z, y) = solvelp(c, G, h, A, b)\n"
    "(prosta, solsta, x, z) = solvelp(c, G, h)\n\n"
    "PURPOSE\n"
    "(prosta, solsta, x, z, y) = solvelp(c, G, h, A, b) solves the pair\n"
    "of primal and dual LPs\n\n"
    "    minimize    c'*x            maximize    -h'*z + b'*y\n"
    "    subject to  G*x <= h        subject to  G'*z + A'*y + c = 0\n"
    "                A*x = b                     z >= 0.\n\n"
    "(prosta, solsta, x, z) = solvelp(c, G, h) solves the pair of\n"
    "primal and dual LPs\n\n"
    "    minimize    c'*x            maximize    -h'*z \n"
    "    subject to  G*x <= h        subject to  G'*z + c = 0\n"
    "                                            z >= 0.\n\n"
    "ARGUMENTS\n"
    "c         nx1 'd' matrix with n>=1\n\n" 
    "G         mxn 'd' or nonsymmetric sparse matrix with m>=1\n\n"
    "h         mx1 'd' matrix\n\n"
    "A         pxn 'd' or nonsymmetric sparse matrix with p>=0\n\n"
    "b         px1 'd' matrix\n\n"
    "prosta    the problem status string returned by MOSEK\n\n"
    "solsta    the solution status string returned by MOSEK\n\n"
    "x         the primal solution\n\n"
    "z,y       the dual solution.";

static PyObject *
mosek_solvelp(PyObject *self, PyObject *args, PyObject *kwrds)
{
    matrix *c, *h, *b=NULL, *x=NULL, *z=NULL, *yu=NULL, *yl=NULL;
    PyObject *G, *A=NULL, *t=NULL;
    int m, n, p, i, j, k, nnz=0, nnzmax;
    char *kwlist[] = {"c", "G", "h", "A", "b", NULL};

    if (!PyArg_ParseTupleAndKeywords(args, kwrds, "OOO|OO", kwlist, &c,
        &G, &h, &A, &b)) return NULL;

    if (!((Matrix_Check(G) && MAT_ID(G) == DOUBLE) || 
	    (SpMatrix_Check(G) && SP_ID(G) == DOUBLE))) {
      PY_ERR(PyExc_ValueError, "G must be a dense or sparse matrix"
	  " with typecode 'd'");
        return NULL;
    }
    if ((m = Matrix_Check(G) ? MAT_NROWS(G) : SP_NROWS(G)) <= 0) 
        err_p_int("m");
    if ((n = Matrix_Check(G) ? MAT_NCOLS(G) : SP_NCOLS(G)) <= 0) 
        err_p_int("n");

    if (!Matrix_Check(h) || h->id != DOUBLE) 
        err_dbl_mtrx("h");
    if (h->nrows != m || h->ncols != 1){
        PyErr_SetString(PyExc_ValueError, "incompatible dimensions");
        return NULL;
    }

    if (A){
      if (!((Matrix_Check(A) && MAT_ID(A) == DOUBLE) || 
	      (SpMatrix_Check(A) && SP_ID(A) == DOUBLE))) {
	PyErr_SetString(PyExc_ValueError, "A must be a dense "
                    "or sparse matrix with typecode 'd'");
                return NULL;
	}
        if ((p = Matrix_Check(A) ? MAT_NROWS(A) : SP_NROWS(A)) < 0)
            err_p_int("p");
        if ((Matrix_Check(A) ? MAT_NCOLS(A) : SP_NCOLS(A)) != n){
            PyErr_SetString(PyExc_ValueError, "incompatible "
                "dimensions");
            return NULL;
	}
    }
    else p = 0;
    if (b && (!Matrix_Check(b) || b->id != DOUBLE)) 
        err_dbl_mtrx("b");
    if ((b && (b->nrows != p || b->ncols != 1)) || (!b && p !=0 )){
        PyErr_SetString(PyExc_ValueError, "incompatible dimensions");
        return NULL;
    }


    /* Setting up MOSEK environment */
    nnzmax = (SpMatrix_Check(G) ? SP_NNZ(G) : m*n ) + 
        ((A && SpMatrix_Check(A)) ? SP_NNZ(A) : p*n);

    int *bkc, *bkx, *ptrb, *ptre, *sub; 
    bkc  = malloc((m+p)*sizeof(int));
    bkx  = malloc(n*sizeof(int));
    ptrb = malloc(n*sizeof(int));
    ptre = malloc(n*sizeof(int));
    sub  = malloc(nnzmax*sizeof(int));

    double *blc, *buc, *blx, *bux, *val;
    blc = malloc((m+p)*sizeof(double));
    buc = malloc((m+p)*sizeof(double));
    blx = malloc(n*sizeof(double));
    bux = malloc(n*sizeof(double));
    val = malloc(nnzmax*sizeof(double));

    int r;
    MSKenv_t  env;
    MSKtask_t task;
    
    /* Make mosek environment. */
    r = MSK_makeenv(&env,NULL,NULL,NULL,NULL); 
    if (!bkc || !blc || !buc || !bkx || !blx || !buc ||
	!ptrb || !ptre || !sub || !val || r!=MSK_RES_OK) {
      free(bkc); free(bkx); free(ptrb); free(ptre); free(sub); 
      free(blc); free(buc); free(blx); free(bux); free(val);
      return PyErr_NoMemory();
    }

    /* Directs the env log stream to the user
       specified procedure 'printstr'. */
    MSK_linkfunctoenvstream(env,MSK_STREAM_LOG,NULL,printstr);
    
    /* Initialize the environment. */   
    r = MSK_initenv(env);
    if (r!=MSK_RES_OK) {
      free(bkc); free(bkx); free(ptrb); free(ptre); free(sub); 
      free(blc); free(buc); free(blx); free(bux); free(val);
      return PyErr_NoMemory();
    }

    /* Make the optimization task. */
    r = MSK_maketask(env, (m+p), n, &task);
    if (r!=MSK_RES_OK) {
      free(bkc); free(bkx); free(ptrb); free(ptre); free(sub); 
      free(blc); free(buc); free(blx); free(bux); free(val);
      MSK_deleteenv(&env);
      return PyErr_NoMemory();
    }
      
    /* Directs the log task stream to the user
       specified procedure 'printstr'. */    
    MSK_linkfunctotaskstream(task,MSK_STREAM_LOG,NULL,printstr);
        
    /* Define bounds for the constraints
       
      -infty  <=  G*x  <=   h
         b    <=  A*x  <=   b
    */
    
    for (i=0; i<m; i++) {
      bkc[i] =  MSK_BK_UP;  
      blc[i] = -MSK_INFINITY;
      buc[i] =  MAT_BUFD(h)[i];
    }
    
    for (i=0; i<p; i++) {
      bkc[m+i] = MSK_BK_FX;  
      blc[m+i] = MAT_BUFD(b)[i];
      buc[m+i] = MAT_BUFD(b)[i];
    }

    /* bounds on the variables (none) */
    for (i=0; i<n; i++) {
      bkx[i] =  MSK_BK_FR;		  
      blx[i] = -MSK_INFINITY;
      bux[i] =  MSK_INFINITY;
    }
    
    /* constraint matrix */       
    for (j=0; j<n; j++) {
      
      if (SpMatrix_Check(G)) {
	for (k=SP_COL(G)[j]; k<SP_COL(G)[j+1]; k++) {
	  val[nnz]   = SP_VALD(G)[k];
	  sub[nnz++] = SP_ROW(G)[k];
	}
      }
      else {
	for (k=0; k<m; k++) {
	  if (MAT_BUFD(G)[j*m+k] != 0.0) {
	    val[nnz]   = MAT_BUFD(G)[j*m+k];
	    sub[nnz++] = k;
	  }
	}
      }

      if (A) {
	if (SpMatrix_Check(A)) {
	  for (k=SP_COL(A)[j]; k<SP_COL(A)[j+1]; k++) {
	    val[nnz]   = SP_VALD(A)[k];
	    sub[nnz++] = m+SP_ROW(A)[k];
	  }
	}
	else {
	  for (k=0; k<p; k++) {
	    if (MAT_BUFD(A)[j*p+k] != 0.0) {
	      val[nnz]   = MAT_BUFD(A)[j*p+k];
	      sub[nnz++] = m+k;
	    }
	  }
	}	
      }
      
      ptrb[j] = (j>0 ? ptre[j-1] : 0);
      ptre[j] = nnz;
    }
    
    r = MSK_inputdata(task,
	(m+p),n,
	(m+p),n,
	MAT_BUFD(c),0.0,
	ptrb, ptre,
	sub, val,
	bkc, blc, buc,
	bkx, blx, bux);
    
    free(bkc); free(blc); free(buc); 
    free(bkx); free(blx); free(bux); 
    free(ptrb); free(ptre);
    free(sub); free(val);
    
    MSK_putintparam(task,MSK_IPAR_OBJECTIVE_SENSE,MSK_OBJECTIVE_SENSE_MIN);
    
    PyObject *param;
    if (!(param = PyObject_GetAttrString(mosek_module, "options")) ||
	!PyDict_Check(param)) {
      MSK_deletetask(&task);
      MSK_deleteenv(&env);
      PY_ERR(PyExc_AttributeError,err_msk_noparam);
    }
    
    PyObject *key, *value;
    int param_id;
    int_compat pos = 0;    

    while (PyDict_Next(param, &pos, &key, &value)) {
      MSK_isintparname(task, PyString_AS_STRING(key), &param_id);
      if (param_id < MSK_IPAR_SOLUTION_CALLBACK) {
	if (!PyInt_Check(value)) 
	{
	  char err_str[100] = "invalid integer parameter: ";
	  strncat(err_str, PyString_AS_STRING(key), 100);
	  MSK_deletetask(&task);
	  MSK_deleteenv(&env);
	  Py_DECREF(param);
	  PY_ERR(PyExc_ValueError,err_str);
	}
	  
	MSK_putintparam(task, param_id, PyInt_AsLong(value)); 
	continue;
      } 
      MSK_isdouparname(task, PyString_AS_STRING(key), &param_id);
      if (param_id < MSK_DPAR_NONCONVEX_TOL_OPT) {
	if (!PyInt_Check(value) && !PyFloat_Check(value)) 
	{
	  char err_str[100] = "invalid floating point parameter: ";
	  strncat(err_str, PyString_AS_STRING(key), 100);
	  MSK_deletetask(&task);
	  MSK_deleteenv(&env);
	  Py_DECREF(param);
	  PY_ERR(PyExc_ValueError,err_str);
	}

	MSK_putdouparam(task, param_id, PyFloat_AsDouble(value)); 	
	continue;
      }      
    }
    Py_DECREF(param);

    r = MSK_optimize(task);    
    if (r==MSK_RES_OK) {

      if (!(t = PyTuple_New(A ? 5 : 4))) {
	MSK_deletetask(&task);
	MSK_deleteenv(&env);
	return PyErr_NoMemory();
      }
    
      x = (matrix *) Matrix_New(n,1,DOUBLE);
      z = (matrix *) Matrix_New(m,1,DOUBLE);
      if (A) {
	yu = (matrix *) Matrix_New(p,1,DOUBLE);
	yl = (matrix *) Matrix_New(p,1,DOUBLE);
      }
      if (!x || !z || (A && (!yu || !yl))){
	Py_XDECREF(x);
	Py_XDECREF(z);
	Py_XDECREF(yu);
	Py_XDECREF(yl);
	Py_XDECREF(t);
	MSK_deletetask(&task);
	MSK_deleteenv(&env);
	return PyErr_NoMemory();
      }
      
      char statstr[50];
      int solsta, prosta;
      double tmp;

      MSK_getsolutioninf(task, 0, &prosta, &solsta, 
	  &tmp, &tmp, &tmp, &tmp, &tmp, &tmp, &tmp, &tmp, &tmp);
      
      MSK_prostatostr (task, prosta, statstr);
      PyTuple_SET_ITEM(t, 0, PyString_FromString(statstr));

      MSK_solstatostr (task, solsta, statstr);
      PyTuple_SET_ITEM(t, 1, PyString_FromString(statstr));

      MSK_getsolutionslice(task, 0, MSK_SOL_ITEM_XX, 0, n, MAT_BUFD(x));
      PyTuple_SET_ITEM(t, 2, (PyObject *) x);
	
      MSK_getsolutionslice(task, 0, MSK_SOL_ITEM_SUC, 0, m, MAT_BUFD(z));
      PyTuple_SET_ITEM(t, 3, (PyObject *) z);
	
      if (A) {	  
	MSK_getsolutionslice(task,0,MSK_SOL_ITEM_SUC,m,m+p,MAT_BUFD(yu));
	MSK_getsolutionslice(task,0,MSK_SOL_ITEM_SLC,m,m+p,MAT_BUFD(yl));

	for (i=0; i<yu->nrows; i++) MAT_BUFD(yu)[i] -= MAT_BUFD(yl)[i];

	Py_DECREF(yl);
	PyTuple_SET_ITEM(t, 4, (PyObject *) yu);
      }

      MSK_deletetask(&task);
      MSK_deleteenv(&env);     

      return (PyObject *) t;
    } 
      
    MSK_deletetask(&task);
    MSK_deleteenv(&env);
    t = PyString_FromString("internal mosek error");        
    return (PyObject *) t;    
}

static char doc_mosek_solveqp[] = 
    "Solves a convex quadratic program using MOSEK.\n\n"
    "(prosta, solsta, x, z, y) = solveqp(P, q, G, h, A, b)\n"
    "(prosta, solsta, x, z) = solveqp(P, q, G, h)\n\n"
    "PURPOSE\n"
    "(prosta, solsta, x, z, y) = solveqp(P, q, G, h, A, b) solves the QP\n\n"
    "    minimize    (1/2)*x'*P*x + q'*x\n"
    "    subject to  G*x <= h\n"
    "                A*x = b\n\n"
    "and its dual\n\n" 
    "    maximize    (1/2)*(q+G'*z+A'*y)'*Pinv*(q+G'*z+A'*y) - h'*z - b'*y\n"
    "    subject to  G*x <= h\n"
    "                A*x = b\n"
    "                z >= 0\n"
    "                (I-P*Pinv)*(q+G'*z+A'*y) = 0\n\n"
    "where Pinv is the Moore-Penrose pseudoinverse of P.\n\n"
    "(prosta, solsta, x, z) = solveqp(P, q, G, h) solves the LP\n\n"
    "    minimize    (1/2)*x'*P*q + q'*x\n"
    "    subject to  G*x <= h\n\n"
    "and its dual\n\n"
    "    maximize    (1/2)*(q+G'*z)'*Pinv*(q+G'*z) - h'*z\n"
    "    subject to  G*x <= h\n"
    "                z >= 0\n"
    "                (I-P*Pinv)*(q+G'*z) = 0\n\n"
    "where Pinv is the Moore-Penrose pseudoinverse of P.\n\n"
    "ARGUMENTS\n"
    "P         nxn 'd' matrix with \n\n" 
    "q         nx1 'd' matrix with n>=1\n\n" 
    "G         mxn 'd' or nonsymmetric sparse matrix with m>=0\n\n"
    "h         mx1 'd' matrix\n\n"
    "A         pxn 'd' or nonsymmetric sparse matrix with p>=0\n\n"
    "b         px1 'd' matrix\n\n"
    "prosta    the problem status string returned by MOSEK\n\n"
    "solsta    the solution status string returned by MOSEK\n\n"
    "x         the primal solution\n\n"
    "z,y       the dual solution.";

static PyObject *
mosek_solveqp(PyObject *self, PyObject *args, PyObject *kwrds)
{
    matrix *q, *h=NULL, *b=NULL, *x=NULL, *z=NULL, *yu=NULL, *yl=NULL;
    PyObject *P, *G=NULL, *A=NULL, *t=NULL;
    int m, n, p, i, j, k, nnz=0, nnzmax, numqonz;
    char *kwlist[] = {"P", "q", "G", "h", "A", "b", NULL};

    if (!PyArg_ParseTupleAndKeywords(args, kwrds, "OO|OOOO", kwlist, 
	    &P, &q, &G, &h, &A, &b)) return NULL;

    if (!(Matrix_Check(P) && MAT_ID(P) == DOUBLE && 
	    MAT_NCOLS(P) == MAT_NROWS(P)) &&
        !(SpMatrix_Check(P) && SP_ID(P) == DOUBLE)) {
      PyErr_SetString(PyExc_ValueError, "P must be a square symmetric"
	  " sparse or dense matrix with typecode 'd'");
      return NULL;
    }
    
    if (G && ((Matrix_Check(G) && MAT_ID(G) != DOUBLE) || 
	    (SpMatrix_Check(G) && SP_ID(G) != DOUBLE))) {
      PyErr_SetString(PyExc_ValueError, "G must be a sparse or dense"
	  " matrix with typecode 'd'");
      return NULL;
    }

    if (!(SpMatrix_Check(P) && ((G && SpMatrix_Check(G)) || !G) && 
	    ((A && SpMatrix_Check(A)) || !A)) &&
	!(Matrix_Check(P) && ((G && Matrix_Check(G)) || !G) && 
	    ((A && Matrix_Check(A)) || !A))) 
      PY_ERR(PyExc_ValueError, "P, G and A must all be either sparse"
	  " or dense matrices");      

    int nP = (Matrix_Check(P) ? MAT_NROWS(P) : SP_NROWS(P));    
    if (!Matrix_Check(q) || MAT_ID(q) != DOUBLE) 
      PY_ERR(PyExc_ValueError, "q must be a 'd' matrix");

    if (G) {
      if ((m = Matrix_Check(G) ? MAT_NROWS(G) : SP_NROWS(G)) < 0)
        err_nn_int("m");
      if ((n = Matrix_Check(G) ? MAT_NCOLS(G) : SP_NCOLS(G)) <= 0)
        err_p_int("n");

      if (!h || !Matrix_Check(h) || h->id != DOUBLE) 
        err_dbl_mtrx("h");
      if (h->nrows != m || h->ncols != 1 || nP != n || 
	  q->nrows != n || q->ncols != 1) {
        PyErr_SetString(PyExc_ValueError, "incompatible dimensions");
        return NULL;
      }
    } 
    else {
      m = 0; n = nP;
    }
      
    if (A) {
        if ((Matrix_Check(A) && MAT_ID(A) != DOUBLE) || 
            (SpMatrix_Check(A) && SP_ID(A) != DOUBLE)) {
                PyErr_SetString(PyExc_ValueError, "A must be a sparse "
                    "or dense matrix with typecode 'd'");
                return NULL;
	}
        if ((p = Matrix_Check(A) ? MAT_NROWS(A) : SP_NROWS(A)) < 0)
            err_nn_int("p");
        if ((Matrix_Check(A) ? MAT_NCOLS(A) : SP_NCOLS(A)) != n){
            PyErr_SetString(PyExc_ValueError, "incompatible "
                "dimensions");
            return NULL;
	}
    }
    else p = 0;
    if (b && (!Matrix_Check(b) || b->id != DOUBLE)) 
        err_dbl_mtrx("b");
    if ((b && (b->nrows != p || b->ncols != 1)) || (!b && p !=0)) {
        PyErr_SetString(PyExc_ValueError, "incompatible dimensions");
        return NULL;
    }

    /* Setting up MOSEK environment */
    numqonz = (SpMatrix_Check(P) ? SP_NNZ(P) : n*(n+1)/2);
    nnzmax = ((G && SpMatrix_Check(G)) ? SP_NNZ(G) : m*n ) + 
      ((A && SpMatrix_Check(A)) ? SP_NNZ(A) : p*n);

    int *bkc, *bkx, *ptrb, *ptre, *sub, *qosubi, *qosubj; 
    bkc    = malloc((m+p)*sizeof(int));
    bkx    = malloc(n*sizeof(int));
    ptrb   = malloc(n*sizeof(int));
    ptre   = malloc(n*sizeof(int));
    sub    = malloc(nnzmax*sizeof(int));
    qosubi = malloc(numqonz*sizeof(int));
    qosubj = malloc(numqonz*sizeof(int));

    double *blc, *buc, *blx, *bux, *val, *qoval;
    blc = malloc((m+p)*sizeof(double));
    buc = malloc((m+p)*sizeof(double));
    blx = malloc(n*sizeof(double));
    bux = malloc(n*sizeof(double));
    val = malloc(nnzmax*sizeof(double));
    qoval = malloc(numqonz*sizeof(double));

    int r;
    MSKenv_t  env;
    MSKtask_t task;

    /* Make mosek environment. */
    r = MSK_makeenv(&env,NULL,NULL,NULL,NULL); 
    if (!bkc || !blc || !buc || !bkx || !blx || !buc ||
	!ptrb || !ptre || !sub || !val || 
	!qosubi || !qosubj || !qoval || r!=MSK_RES_OK) {
      free(bkc); free(bkx); free(ptrb); free(ptre); free(sub); 
      free(blc); free(buc); free(blx); free(bux); free(val);
      free(qosubi); free(qosubj); free(qoval);
      return PyErr_NoMemory();
    }

    /* Directs the env log stream to the user
       specified procedure 'printstr'. */
    MSK_linkfunctoenvstream(env,MSK_STREAM_LOG,NULL,printstr);
    
    /* Initialize the environment. */   
    r = MSK_initenv(env);
    if (r!=MSK_RES_OK) {
      free(bkc); free(bkx); free(ptrb); free(ptre); free(sub); 
      free(blc); free(buc); free(blx); free(bux); free(val);
      free(qosubi); free(qosubj); free(qoval);
      return PyErr_NoMemory();
    }

    /* Make the optimization task. */
    r = MSK_maketask(env, (m+p), n, &task);
    if (r!=MSK_RES_OK) {
      free(bkc); free(bkx); free(ptrb); free(ptre); free(sub); 
      free(blc); free(buc); free(blx); free(bux); free(val);
      free(qosubi); free(qosubj); free(qoval);
      MSK_deleteenv(&env);
      return PyErr_NoMemory();
    }
    
    /* Directs the log task stream to the user
       specified procedure 'printstr'. */    
    MSK_linkfunctotaskstream(task,MSK_STREAM_LOG,NULL,printstr);

    /* Setup quadratic term of objective */
    if (Matrix_Check(P)) {
      for (j=0, k=0; j<n; j++) 
	for (i=j; i<n; i++, k++) {
	  qosubi[k] = i;
	  qosubj[k] = j;
	  qoval[k]  = MAT_BUFD(P)[j*n+i];
	}
    } else {
      for (j=0, k=0; j<n; j++) {
	for (i=SP_COL(P)[j]; i<SP_COL(P)[j+1]; i++, k++) {
	  qosubi[k] = SP_ROW(P)[k];
	  qosubj[k] = j;
	  qoval[k]  = SP_VALD(P)[k];	  
	}
      }
    }

    /* Define bounds for the constraints
       
      -infty  <=  G*x  <=   h
         b    <=  A*x  <=   b
    */
    
    for (i=0; i<m; i++) {
      bkc[i] =  MSK_BK_UP;  
      blc[i] = -MSK_INFINITY;
      buc[i] =  MAT_BUFD(h)[i];
    }
    
    for (i=0; i<p; i++) {
      bkc[m+i] = MSK_BK_FX;  
      blc[m+i] = MAT_BUFD(b)[i];
      buc[m+i] = MAT_BUFD(b)[i];
    }

    /* bounds on the variables (none) */
    for (i=0; i<n; i++) {
      bkx[i] =  MSK_BK_FR;		  
      blx[i] = -MSK_INFINITY;
      bux[i] =  MSK_INFINITY;
    }
    
    /* constraint matrix */       
    for (j=0; j<n; j++) {
      
      if (G && SpMatrix_Check(G)) {
	for (k=SP_COL(G)[j]; k<SP_COL(G)[j+1]; k++) {
	  val[nnz]   = SP_VALD(G)[k];
	  sub[nnz++] = SP_ROW(G)[k];
	}
      }
      else {
	for (k=0; k<m; k++) {
	  if (MAT_BUFD(G)[j*m+k] != 0.0) {
	    val[nnz]   = MAT_BUFD(G)[j*m+k];
	    sub[nnz++] = k;
	  }
	}
      }

      if (A) {
	if (SpMatrix_Check(A)) {
	  for (k=SP_COL(A)[j]; k<SP_COL(A)[j+1]; k++) {
	    val[nnz]   = SP_VALD(A)[k];
	    sub[nnz++] = m+SP_ROW(A)[k];
	  }
	}
	else {
	  for (k=0; k<p; k++) {
	    if (MAT_BUFD(A)[j*p+k] != 0.0) {
	      val[nnz]   = MAT_BUFD(A)[j*p+k];
	      sub[nnz++] = m+k;
	    }
	  }
	}	
      }
      
      ptrb[j] = (j>0 ? ptre[j-1] : 0);
      ptre[j] = nnz;
    }
    
    r = MSK_inputdata(task,
	(m+p),n,
	(m+p),n,
	MAT_BUFD(q),0.0,
	ptrb, ptre,
	sub, val,
	bkc, blc, buc,
	bkx, blx, bux);

    r = MSK_putqobj (task, numqonz, qosubi, qosubj, qoval); 

    free(bkc); free(blc); free(buc); 
    free(bkx); free(blx); free(bux); 
    free(ptrb); free(ptre);
    free(sub); free(val);
    free(qosubi); free(qosubj); free(qoval);

    MSK_putintparam(task,MSK_IPAR_OBJECTIVE_SENSE,MSK_OBJECTIVE_SENSE_MIN);
          
    PyObject *param;
    if (!(param = PyObject_GetAttrString(mosek_module, "options")) ||
	!PyDict_Check(param)) {
      MSK_deletetask(&task);
      MSK_deleteenv(&env);
      PY_ERR(PyExc_AttributeError,err_msk_noparam);
    }
    
    PyObject *key, *value;
    int param_id;
    int_compat pos = 0;    

    while (PyDict_Next(param, &pos, &key, &value)) {
      MSK_isintparname(task, PyString_AS_STRING(key), &param_id);
      if (param_id < MSK_IPAR_SOLUTION_CALLBACK) {
	if (!PyInt_Check(value)) 
	{
	  char err_str[100] = "invalid integer parameter: ";
	  strncat(err_str, PyString_AS_STRING(key), 100);
	  MSK_deletetask(&task);
	  MSK_deleteenv(&env);
	  Py_DECREF(param);
	  PY_ERR(PyExc_ValueError,err_str);
	}
	  
	MSK_putintparam(task, param_id, PyInt_AsLong(value)); 
	continue;
      } 
      MSK_isdouparname(task, PyString_AS_STRING(key), &param_id);
      if (param_id < MSK_DPAR_NONCONVEX_TOL_OPT) {
	if (!PyInt_Check(value) && !PyFloat_Check(value)) 
	{
	  char err_str[100] = "invalid floating point parameter: ";
	  strncat(err_str, PyString_AS_STRING(key), 100);
	  MSK_deletetask(&task);
	  MSK_deleteenv(&env);
	  Py_DECREF(param);
	  PY_ERR(PyExc_ValueError,err_str);
	}

	MSK_putdouparam(task, param_id, PyFloat_AsDouble(value)); 	
	continue;
      }      
    }
    Py_DECREF(param);

    r = MSK_optimize(task);    
    if (r==MSK_RES_OK) {

      if (!(t = PyTuple_New(A ? 5 : 4))) {
	MSK_deletetask(&task);
	MSK_deleteenv(&env);
	return PyErr_NoMemory();
      }
    
      x = (matrix *) Matrix_New(n,1,DOUBLE);
      z = (matrix *) Matrix_New(m,1,DOUBLE);
      if (A) {
	yu = (matrix *) Matrix_New(p,1,DOUBLE);
	yl = (matrix *) Matrix_New(p,1,DOUBLE);
      }
      if (!x || !z || (A && (!yu || !yl))){
	Py_XDECREF(x);
	Py_XDECREF(z);
	Py_XDECREF(yu);
	Py_XDECREF(yl);
	Py_XDECREF(t);
	MSK_deletetask(&task);
	MSK_deleteenv(&env);
	return PyErr_NoMemory();
      }
      
      char statstr[50];
      int solsta, prosta;
      double tmp;

      MSK_getsolutioninf(task, 0, &prosta, &solsta, 
	  &tmp, &tmp, &tmp, &tmp, &tmp, &tmp, &tmp, &tmp, &tmp);
      
      MSK_prostatostr (task, prosta, statstr);
      PyTuple_SET_ITEM(t, 0, PyString_FromString(statstr));

      MSK_solstatostr (task, solsta, statstr);
      PyTuple_SET_ITEM(t, 1, PyString_FromString(statstr));

      MSK_getsolutionslice(task, 0, MSK_SOL_ITEM_XX, 0, n, MAT_BUFD(x));
      PyTuple_SET_ITEM(t, 2, (PyObject *) x);
	
      MSK_getsolutionslice(task, 0, MSK_SOL_ITEM_SUC, 0, m, MAT_BUFD(z));
      PyTuple_SET_ITEM(t, 3, (PyObject *) z);
	
      if (A) {	  
	MSK_getsolutionslice(task,0,MSK_SOL_ITEM_SUC,m,m+p,MAT_BUFD(yu));
	MSK_getsolutionslice(task,0,MSK_SOL_ITEM_SLC,m,m+p,MAT_BUFD(yl));

	for (i=0; i<yu->nrows; i++) MAT_BUFD(yu)[i] -= MAT_BUFD(yl)[i];

	Py_DECREF(yl);
	PyTuple_SET_ITEM(t, 4, (PyObject *) yu);
      }

      MSK_deletetask(&task);
      MSK_deleteenv(&env);     

      return (PyObject *) t;
    } 

    MSK_deletetask(&task);
    MSK_deleteenv(&env);
 
    char err_str[50];

    switch (r) {
    case MSK_RES_ERR_INV_PROBLEM:
      PY_ERR(PyExc_ValueError,"MOSEK error: probably a non-convex "
	  "problem was specified");
      break;
    default:      
      sprintf(err_str, "Internal MOSEK error. Error code %i",r);
      PY_ERR(PyExc_Exception,err_str);
    }
   
}


static PyMethodDef mosek_functions[] = {
  {"solvelp", (PyCFunction) mosek_solvelp, METH_VARARGS|METH_KEYWORDS, 
   doc_mosek_solvelp},
 {"solveqp", (PyCFunction) mosek_solveqp, METH_VARARGS|METH_KEYWORDS, 
   doc_mosek_solveqp}, 
  {NULL}  /* Sentinel */
};

PyMODINIT_FUNC
initmosek(void)
{
  mosek_module = Py_InitModule3("cvxopt.mosek", mosek_functions, mosek__doc__);

  PyModule_AddObject(mosek_module, "options", PyDict_New());
  
  if (import_cvxopt() < 0)
    return;
}
