/*
 * Copyright 2004-2007 J. Dahl and L. Vandenberghe.
 *
 * This file is part of CVXOPT version 0.9.
 *
 * CVXOPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * CVXOPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "cvxopt.h"

#include <complex.h>

#include "misc.h"
#include "rngs/rngs.h"
#include "rngs/rvgs.h"

PyDoc_STRVAR(random__doc__,"Random Module.");

static char doc_getseed[] =
    "Returns the seed value for the random number generator.\n\n"
    "getseed()";
static PyObject *
getseed(PyObject *self) 
{
  long seed;
  GetSeed(&seed);

  return Py_BuildValue("l",seed);
}

static char doc_setseed[] =
    "Sets the seed value for the random number generator.\n\n"
    "setseed(value = 0.0)\n\n"
    "ARGUMENTS\n"
    "value     a nonnegative number that specifies the seed value.\n"
    "          If value is 0, the system clock is used.";

static PyObject *
setseed(PyObject *self, PyObject *args) 
{
  long seed = 0;

  if (!PyArg_ParseTuple(args, "|l", &seed))
    return NULL;

  if (seed < 0) PY_ERR(PyExc_ValueError, "seed value must be non-negative");

  PutSeed( (seed>0 ? seed : -1));
  return Py_BuildValue("");
}

static char doc_normal[] =
    "Randomly generates a matrix with normally distributed entries.\n\n"
    "normal(nrows, ncols=1, mean=0, std=1)\n\n"
    "PURPOSE\n"
    "Returns a matrix with typecode 'd' and size nrows by ncols, with\n"
    "its entries randomly generated from a normal distribution with mean\n"
    "m and standard deviation std.\n\n"
    "ARGUMENTS\n"
    "nrows     number of rows\n\n"
    "ncols     number of columns\n\n"
    "mean      approximate mean of the distribution\n\n"
    "std       standard deviation of the distribution";
static PyObject *
normal(PyObject *self, PyObject *args, PyObject *kwrds) 
{
  matrix *obj;
  int i, nrows, ncols = 1;
  double m = 0, s = 1;
  char *kwlist[] = {"nrows", "ncols", "mean", "std", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, kwrds, "i|idd", kwlist, 
	  &nrows, &ncols, &m, &s)) return NULL;

  if (s < 0.0) PY_ERR(PyExc_ValueError, "std must be non-negative");

  if ((nrows<0) || (ncols<0)) {
    PyErr_SetString(PyExc_TypeError, "dimensions must be non-negative");
    return NULL;
  }

  if (!(obj = Matrix_New(nrows, ncols, DOUBLE)))
    return PyErr_NoMemory();
  
  for (i= 0; i < nrows*ncols; i++)
    MAT_BUFD(obj)[i] = Normal(m,s);
  
  return (PyObject *)obj;
}

static char doc_uniform[] =
    "Randomly generates a matrix with uniformly distributed entries.\n\n"
    "uniform(nrows, ncols=1, a=0, b=1)\n\n"
    "PURPOSE\n"
    "Returns a matrix with typecode 'd' and size nrows by ncols, with\n"
    "its entries randomly generated from a uniform distribution on the\n"
    "interval (a,b).\n\n"
    "ARGUMENTS\n"
    "nrows     number of rows\n\n"
    "ncols     number of columns\n\n"
    "a         lower bound\n\n"
    "b         upper bound";

static PyObject *
uniform(PyObject *self, PyObject *args, PyObject *kwrds) 
{
  matrix *obj;
  int i, nrows, ncols = 1;
  double a = 0, b = 1;

  char *kwlist[] = {"nrows", "ncols", "a", "b", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, kwrds, "i|idd", kwlist, 
	  &nrows, &ncols, &a, &b)) return NULL;
  
  if (a>b) PY_ERR(PyExc_ValueError, "a must be less than b");

  if ((nrows<0) || (ncols<0)) 
    PY_ERR_TYPE("dimensions must be non-negative");
  
  if (!(obj = (matrix *)Matrix_New(nrows, ncols, DOUBLE)))
    return PyErr_NoMemory();
  
  for (i= 0; i < nrows*ncols; i++)
    MAT_BUFD(obj)[i] = Uniform(a,b);
  
  return (PyObject *)obj;
}

static PyMethodDef random_functions[] = {
  {"getseed",(PyCFunction)getseed,METH_NOARGS, doc_getseed},  
  {"setseed",(PyCFunction)setseed,METH_VARARGS, doc_setseed},  
  {"normal", (PyCFunction)normal, METH_VARARGS|METH_KEYWORDS, doc_normal},  
  {"uniform",(PyCFunction)uniform,METH_VARARGS|METH_KEYWORDS, doc_uniform},  
  {NULL}  /* Sentinel */
};

PyMODINIT_FUNC
initrandom(void)
{
  PyObject *m;

  m = Py_InitModule3("cvxopt.random", random_functions, random__doc__);
  
  if (import_cvxopt() < 0)
    return;
}
