"""
Convex programming solver.

A primal-dual interior-point solver written in Python and interfaces
for quadratic and geometric programming.  Also includes an interface
to the quadratic programming solver from MOSEK.
"""

# Copyright 2004-2007 J. Dahl and L. Vandenberghe.
# 
# This file is part of CVXOPT version 0.9.
#
# CVXOPT is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# CVXOPT is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import math 
from cvxopt import base, blas, lapack, misc
from cvxopt.base import matrix, spmatrix, sqrt, exp, mul, div

__all__ = []
options = {}

def nlclp(c, kktsolver, F=None, G=None, h=None, A=None, b=None, 
    xnewcopy=matrix, xdot=blas.dot, xaxpy=blas.axpy, xscal=blas.scal,
    ynewcopy=matrix, ydot=blas.dot, yaxpy=blas.axpy, yscal=blas.scal):

    """
    Solves a nonlinearly constrained convex optimization problem with a
    linear objective

        minimize    c'*x 
        subject to  f(x) <= 0
                    G(x) <= h
                    A(x) = b.                      

    f: V -> R^mnl is convex and twice differentiable,  G: V -> R^ml and
    A: V -> W are linear mappings, where V and W are real vector spaces.
    The default choices for V and W are Euclidean vector spaces V=R^n 
    and W=R^p, with elements represented as 'd' matrices of size (n,1) 
    and (p,1).


    Input arguments 

        c is an element in V (stored as a matrix, list, dictionary,...).

        kktsolver is a function for the factorization of a KKT system

            [ H   *   *    *  ] [ ux   ]   [ bx   ]
            [ A   0   *    *  ] [ uy   ] = [ by   ]
            [ Df  0  -Dnl  *  ] [ uznl ]   [ bznl ]
            [ G   0   0   -Dl ] [ uzl  ]   [ bzl  ] 
 
        where H = sum_k z[k]*Hk, Hk is the Hessian of fk, Df is the 
        derivative matrix of f, and Dnl and Dl are positive diagonal 
        matrices.
        Called as g = kktsolver(x, z, dnl, dl), it returns a function g 
        that can be used to solve the system with Df and H evaluated 
        at x and z, Dnl = diag(dnl)^-2, and Dl = diag(dl)^-2.  It can 
        be assumed that x is in the domain of f and that z is positive.
        The function call g(bx, by, bznl, bzl) replaces its arguments 
        with the solution ux, uy, uznl, uzl.


        F is a function that handles the following arguments.

            F() returns a tuple (mnl, x0).  mnl is the number of 
            nonlinear inequality constraints.  x0 is a point in the
            domain of f.

            F(x), with x an element in V, returns a tuple (f, Df).

                f is  a dense 'd' matrix of size (mnl,1) with the 
                function values f(x). 

                Df is a function or a dense or sparse 'd' matrix.   
                When Df is a function, Df(u, v) where u is a 'd' matrix 
                of size (mnl,1) and v an element in V, updates v as 

                    v := sum_k u[k] * grad fk(x) + v.

                When V=R^n, Df can also be passed as 'd' matrix of 
                size (mnl,n) that contains the derivative matrix of f 
                at x, i.e., Df[k,:] is the transpose of the gradient of 
                fk at x.  

                If x is not in dom f, F(x) should return (None, None) 
                or None.

            If F is None, it is assumed that mnl = 0.


        G is None, a function, or a sparse or dense 'd' matrix. 

            If G is None, it is assumed that ml = 0.  

            If G is a function, then

                G(u, v, alpha=1.0, beta=0.0, trans='N')

            evaluates

                v := alpha*G(u) + beta*v if trans is 'N'
                v := alpha*G'(u) + beta*v if trans is 'T'.

            
        h is None, or a dense 'd' matrix with one column.  If h is None,
        it is interpreted as a 0x1 matrix.


        A is None, a function, or a dense or sparse 'd' matrix.  

            If A is None, it is assumed that dim W  = 0.  

            If A is a function then

                A(u, v, alpha=1.0, beta=0.0, trans='N')

            evaluates

                v := alpha*A(u) + beta*v if trans is 'N'
                v := alpha*A'(u) + beta*v if trans is 'T'.


        b is None, or an element in W.  If b is None, it is assumed 
        that dim W = 0.


    Returns a dictionary with keys 'status', 'x', 'snl', 'sl', 'y', 
    'znl', 'zl'.  

        status is 'optimal' or 'unknown'.

        x, s, snl, sl, y, znl, zl are None if status is 'unknown'.  
        Otherwise x is the optimal solution;  snl, sl are the optimal 
        slacks for the nonlinear and linear inequalities;  y is the 
        optimal dual variable associated with the equality constraints; 
        znl, zl are the optimal dual variables associated with the 
        nonlinear and linear inequalities.
    """

    STEP = 0.99
    BETA = 0.5
    ALPHA = 0.01
    GAMMA1 = 1e5
    GAMMA2 = 1e-5
    GAMMA3 = 0.5
    GAMMA4 = 1.0

    try: MAXITERS = options['maxiters']
    except KeyError: MAXITERS = 100
    else: 
        if type(MAXITERS) is not int or MAXITERS < 1: 
            raise ValueError, "options['maxiters'] must be a positive "\
                "integer"

    try: ABSTOL = options['abstol']
    except KeyError: ABSTOL = 1e-7
    else: 
        if type(ABSTOL) is not float and type(ABSTOL) is not int: 
            raise ValueError, "options['abstol'] must be a scalar"

    try: RELTOL = options['reltol']
    except KeyError: RELTOL = 1e-7
    else: 
        if type(RELTOL) is not float and type(RELTOL) is not int: 
            raise ValueError, "options['reltol'] must be a scalar"

    try: FEASTOL = options['feastol']
    except KeyError: FEASTOL = 1e-7
    else: 
        if type(FEASTOL) is not float and type(FEASTOL) is not int: 
            raise ValueError, "options['feastol'] must be a scalar"

    try: show_progress = options['show_progress']
    except KeyError: show_progress = True

    if F is None:  # There are no nonlinear inequality constraints.
        def F(x=None, z=None):
            if x is None: 
                x0 = xnewcopy(c)
                xscal(0.0, x0)
                return 0, x0
            def Df(u, v): pass
            return matrix(0.0, (0,1)), Df

    mnl, x0 = F()

    def fzero(x, y, alpha=1.0, beta=0.0, trans='N'):
        if trans == 'T':  xscal(beta, y)

    if G is None: G = fzero
    if h is None: h = matrix(0.0, (0,1))
    ml = h.size[0]
    if type(G) in (matrix, spmatrix):
        if G.typecode != 'd' or G.size[0] != ml:
            raise TypeError, "'G' must be a 'd' matrix with %d rows" %ml
        def fG(u, v, alpha=1.0, beta=0.0, trans='N'):
            base.gemv(G, u, v, alpha=alpha, beta=beta, trans=trans)
    else:    
        fG = G 

    if (mnl+ml == 0):
        raise ValueError, "problem must have at least one inequality "\
            "constraint"

    if A is None: A = fzero 
    if b is None: b = matrix(0.0, (0,1))
    if type(A) in (matrix,spmatrix):
        if A.typecode != 'd' or A.size[0] != b.size[0]:
            raise TypeError, "'A' must be a 'd' matrix with %d rows" \
            %b.size[0]
        def fA(u, v, alpha=1.0, beta=0.0, trans='N'):
            base.gemv(A, u, v, alpha=alpha, beta=beta, trans=trans)
    else:
        fA = A

    def xcopy(x, y):  xscal(0.0, y);  xaxpy(x, y)
    def ycopy(x, y):  yscal(0.0, y);  yaxpy(x, y)

    x = xnewcopy(x0)
    y = ynewcopy(b);  yscal(0.0, y)
    znl, zl = matrix(1.0, (mnl,1)), matrix(1.0, (ml,1)) 
    snl, sl = matrix(1.0, (mnl,1)), matrix(1.0, (ml,1))
    rx, rx2 = xnewcopy(x0), xnewcopy(x0)
    ry = ynewcopy(b)
    rznl, rznl2 = matrix(0.0, (mnl,1)), matrix(0.0, (mnl,1))
    rzl = matrix(0.0, (ml,1))
    dx, dy = xnewcopy(x), ynewcopy(y)   
    newx, newy = xnewcopy(x),  ynewcopy(y)

    if show_progress: 
        print "% 10s% 12s% 10s% 8s% 7s" %("pcost", "dcost", "gap", 
            "pres", "dres")

    for iters in xrange(MAXITERS):  

        f, Df = F(x)
        f = matrix(f, tc='d')
        if f.typecode != 'd' or f.size != (mnl,1):
            raise TypeError, "first output argument of F() must be a "\
                "'d' matrix of size (%d,%d)" %(mnl,1)
        if type(Df) in (matrix, spmatrix):
            if Df.typecode != 'd' or Df.size[0] != mnl:
                raise TypeError, "second output argument of F() must "\
                    "be a 'd' matrix with %d rows" %mnl
            def fDf(u, v): 
                base.gemv(Df, u, v, beta=1.0, trans='T')
        else: 
            fDf = Df

        gap = blas.dot(snl, znl) + blas.dot(sl, zl) 
        mu = gap / (mnl + ml)

        # rx = c + A'*y + Df'*znl + G'*zl
        xcopy(c, rx) 
        fA(y, rx, beta=1.0, trans='T')
        fDf(znl, rx)
        fG(zl, rx, beta=1.0, trans='T')
        resx = math.sqrt(xdot(rx, rx))
           
        # ry = A*x - b
        ycopy(b, ry)
        fA(x, ry, alpha=1.0, beta=-1.0)
        resy = math.sqrt(ydot(ry, ry))

        # rznl = snl + f 
        blas.copy(snl, rznl)
        blas.axpy(f, rznl)
        resznl = blas.nrm2(rznl)

        # rzl = sl + G*x - h
        blas.copy(sl, rzl)
        blas.axpy(h, rzl, alpha=-1.0)
        fG(x, rzl, beta=1.0)
        reszl = blas.nrm2(rzl)

        # pcost = c'*x
        # dcost = c'*x + y'*(Ax-b) + znl'*f(x) + zl'*(G*x-h)
        #       = c'*x + y'*(Ax-b) + znl'*(f(x)+snl) + zl'*(G*x-h+sl) 
        #         - znl'*snl - zl'*sl
        pcost = xdot(c,x)
        dcost = pcost + ydot(y, ry) + blas.dot(znl, rznl) + \
            blas.dot(zl, rzl) - gap

        pres = math.sqrt( resy**2 + resznl**2 + reszl**2 )
        dres = resx
        if iters == 0: 
            pres0 = max(1.0, pres)
            pres0_nl = max(1.0, resznl)
            dres0 = max(1.0, dres)

        if show_progress:
            print "%2d: % 8.4e % 8.4e % 4.0e% 7.0e% 7.0e" \
                %(iters, pcost, dcost, gap, pres/pres0, dres/dres0) 

        # Stopping criteria.    
        if pres/pres0 <= FEASTOL and dres/dres0 <= FEASTOL and \
            (gap <= ABSTOL or (dcost > 0 and gap/dcost <= RELTOL) or
            (pcost < 0 and gap/(-pcost) <= RELTOL)):
            return {'status': 'optimal', 'x': x,  'y': y, 'znl': znl,  
                'zl': zl, 'snl': snl, 'sl': sl}

        dnl, dl = sqrt(div(znl, snl)), sqrt(div(zl, sl)), 
        g = kktsolver(x, znl, dnl, dl)

        sigma = 0.0    
        for i in [0,1]:

            # Solve
            #
            # [0 ]   [ hessian  A^T  [Df;G]' ]   [dx]
            # [0 ] + [ A        0    0       ] * [dy] = -r  
            # [ds]   [ [Df;G]   0    0       ]   [dz]
            #
            # s.*dz + z.*ds = -s.*z + sigma*mu  
            #
            # by solving
            #
            # [ hessian   A^T   [Df;G]^T    ]   [dx]      
            # [ A         0     0           ] * [dy] = rhs 
            # [ [Df;G]    0     -diag(s./z) ]   [dz]     
            #         
            # with rhs = -r + (0,0,0, (s*z - sigma*mu)./z).
            # Take  ds = -(s.*z - sigma*mu + s.*dz) ./ z.
       
            rcpnl = mul(snl, znl) - sigma*mu
            rcpl = mul(sl, zl) - sigma*mu
            xscal(0.0, dx);  xaxpy(rx, dx, alpha=-1.0)
            yscal(0.0, dy);  yaxpy(ry, dy, alpha=-1.0)
            dznl = -rznl + div(rcpnl, znl)
            dzl = -rzl + div(rcpl, zl)
            g(dx, dy, dznl, dzl)
            dsnl = -div( rcpnl + mul(snl,dznl), znl ) 
            dsl = -div( rcpl + mul(sl,dzl), zl ) 

            # line search 
            step = min( 1.0, STEP / max( list(-div(dsnl,snl)) + 
                list(-div(dsl, sl)) + list(-div(dznl,znl)) 
                + list(-div(dzl,zl))) )
            rescpnl, rescpl = blas.nrm2(rcpnl), blas.nrm2(rcpl) 
            aa, bb = max(rescpnl, rescpl), min(rescpnl, rescpl)
            if aa: rescp = aa * math.sqrt( 1.0 + (bb/aa)**2 )
            else: rescp = 0.0
            phi = resx  + resznl + rescp

            for lsiters in xrange(50):
                
                xcopy(x, newx);  xaxpy(dx, newx, alpha=step)
                ycopy(y, newy);  yaxpy(dy, newy, alpha=step)
                newznl, newzl = znl + step*dznl, zl+step*dzl
                newsnl, newsl = snl + step*dsnl, sl+step*dsl
                newgap = blas.dot(newsnl, newznl) + \
                    blas.dot(newsl, newzl)
                t = F(newx)
                if t is None: newf = None
                else: newf, newDf = t[0], t[1]

                if newf is not None:
                    newf = matrix(newf, tc='d')
                    if type(newDf) in (matrix, spmatrix):
                        def fDf(u, v):
                            base.gemv(newDf, u, v, beta=1.0, trans='T')
                    else: 
                        fDf = newDf

                    # rx2 = c + A'*newy + newDf'*newznl + G'*newzl
                    xcopy(c, rx2) 
                    fA(newy, rx2, beta=1.0, trans='T')
                    fDf(newznl, rx2)
                    fG(newzl, rx2, beta=1.0, trans='T')
                    resx2 = math.sqrt(xdot(rx2, rx2))

                    # rznl2 = newsnl + newf
                    blas.copy(newsnl, rznl2)
                    blas.axpy(newf, rznl2)
                    resznl2 = blas.nrm2(rznl2)

                    # rcpnl = newsnl .* newznl
                    # rcpl = newsl .* newzl
                    rcpnl = mul(newsnl, newznl)
                    rcpl = mul(newsl, newzl)
                    rescpnl = blas.nrm2(rcpnl - sigma*mu)
                    rescpl = blas.nrm2(rcpl - sigma*mu)
                    aa, bb = max(rescpnl, rescpl), min(rescpnl, rescpl)
                    if aa: rescp = aa * math.sqrt( 1.0 + (bb/aa)**2 )
                    else: rescp = 0.0

                    newphi = resx2  + resznl2 + rescp
                    newmu = newgap / (mnl+ml)

                    if resx2 <= GAMMA1 * dres0 * newgap and \
                        resznl2 <= GAMMA1 * pres0_nl * newgap and \
                        (not rcpnl or min(rcpnl) >= GAMMA2*newmu) \
                        and (not rcpl or min(rcpl) >= GAMMA2*newmu) \
                        and newphi <= (1.0 - ALPHA*step) * phi: 
                        break
                
                step *= BETA

            else: raise StandardError, "line search did not terminate"

            if i == 0:  sigma = min(GAMMA3*newmu/mu, (newmu/mu)**GAMMA4)

        xaxpy(dx, x, alpha=step)
        yaxpy(dy, y, alpha=step)
        blas.axpy(dznl, znl, alpha=step)
        blas.axpy(dzl, zl, alpha=step)
        blas.axpy(dsnl, snl, alpha=step)
        blas.axpy(dsl, sl, alpha=step)

    return {'status': 'unknown', 'x': None,  'y': None, 'znl': None, 
        'zl': None, 'snl': None, 'sl': None}



def nlcp(kktsolver, F, G=None, h=None, A=None, b=None,
    xnewcopy=matrix, xdot=blas.dot, xaxpy=blas.axpy, xscal=blas.scal,
    ynewcopy=matrix, ydot=blas.dot, yaxpy=blas.axpy, yscal=blas.scal):

    """
    Solves a convex optimization problem
    
        minimize    f0(x)
        subject to  fk(x) <= 0, k=1,...,mnl
                    G(x) <= h
                    A(x) = b.                      

    fk: V -> R is convex and twice differentiable.  
    G: V -> R^ml and A: V -> W are linear mappings, where V and W are 
    real vector spaces.  The default choices for V and W are Euclidean 
    vector spaces V=R^n and W=R^p, with elements stored as 'd' matrices 
    of size (n,1) and (p,1).

    Input arguments

        kktsolver is a function for the factorization of a KKT system

            [ H       *   *    *  ] [ ux   ]    [ bx   ]
            [ A       0   *    *  ] [ uy   ] =  [ by   ]
            [ Df[1:]  0  -Dnl  *  ] [ uznl ]    [ bznl ]
            [ G       0   0   -Dl ] [ uzl  ]    [ bzl  ] 
 
        where H = sum_{k=0}^mnl z[k]*Hk and H[k] is the Hessian 
        of fk,  Df is the derivative of f and Dnl and Dl are positive 
        diagonal matrices.
        g = kktsolver(x, z, dnl, dl) returns a function g that can be 
        used to solve the system with H and Df evaluated at x and z, 
        Dnl = diag(dnl)^-2, Dl = diag(dl)^-2.
        It can be assumed that x is in the domain of f0 and f.
        The function calll g(bx, by, bznl, bzl) replaces the
        arguments with the solution ux, uy, uznl, uzl.

        F: a function that handles the following arguments.

            F() returns a tuple (mnl, x0).  

                mnl is the number of nonlinear inequality constraints.
                x0 is a point in dom f.

            F(x), with x a point in V, returns (f, Df).

                f is  a dense 'd' matrix of size (mnl+1,1) with the 
                function values f0(x), ...., fmnl(x). 
                Df is a function or a sparse or dense 'd' matrix.  
                If Df is a function, Df(u, v) evaluates 
 
                    v := sum_k=0^mnl u[k]*gradfk + v.

                If V = Rn, Df can also be a 'd' matrix or sparse matrix
                of size (mnl, n).  In this case Df[k,:] contains the 
                transpose of gradfk.  
 
                If x is not in dom f, F(x) returns None or (None, None).
 

        G is None, a function, or a sparse or dense 'd' matrix. 

            If G is a function then

                G(u, v, alpha=1.0, beta=0.0, trans='N')

            evaluates

                v := alpha * G(u)  + beta * v if trans is 'N'
                v := alpha * G'(u) + beta * v if trans is 'T'.

            If G is None, it is assumed that ml = 0.

            
        h is None, or a dense 'd' matrix with one column.  If h is None,
        it is interpreted as a 0x1 matrix.


        A is None, a function, or a dense or sparse 'd' matrix. 

            If A is a function then

                A(u, v, alpha=1.0, beta=0.0, trans='N')

            evaluates

                v := alpha * A(u)  + beta * v if trans is 'N'
                v := alpha * A'(u) + beta * v if trans is 'T'.

            If A is None, it is assumed that dim W  = 0.


        b is None, or an element in W.  If b is None, it is assumed 
        that dim W = 0.


    Returns a dictionary with keys 'status', 'x', 'snl', 'sl', 'y', 
    'znl', 'zl'.  

        status is 'optimal' or 'unknown'.

        x, s, snl, sl, y, znl, zl are None if status is 'unknown'.  
        Otherwise x is the optimal solution;  snl, sl are the optimal 
        slacks for the nonlinear and linear inequalities;  y is the 
        optimal dual variable associated with the equality constraints; 
        znl, zl are the optimal dual variables associated with the 
        nonlinear and linear inequalities.
    """

    mnl, x0 = F()
    if type(mnl) is not int or mnl < 0:
        raise TypeError, "invalid return value for 'mnl, x = F()'"

    def fzero(x, y, alpha=1.0, beta=0.0, trans='N'):
        if trans == 'T':  xscal(beta, y)

    if G is None: G = fzero
    if h is None: h = matrix(0.0, (0,1))
    ml = h.size[0]
    if type(G) in (matrix, spmatrix):
        if G.typecode != 'd' or G.size[0] != ml:
            raise TypeError, "'G' must be a 'd' matrix with %d rows" %ml
        def fG(u, v, alpha=1.0, beta=0.0, trans='N'):
            base.gemv(G, u, v, alpha=alpha, beta=beta, trans=trans)
    else:    
        fG = G 

    if A is None: A = fzero
    if b is None: b = matrix(0.0, (0,1))
    if type(A) in (matrix,spmatrix):
        if A.typecode != 'd' or A.size[0] != b.size[0]:
            raise TypeError, "'A' must be a 'd' matrix with %d rows" \
            %b.size[0]
        def fA(u, v, alpha=1.0, beta=0.0, trans='N'):
            base.gemv(A, u, v, alpha=alpha, beta=beta, trans=trans)
    else:
        fA = A

    def xcopy(x, y):  xscal(0.0, y);  xaxpy(x, y)
    def ycopy(x, y):  yscal(0.0, y);  yaxpy(x, y)


    # Apply nlclp() to problem in epigraph form:
    #    
    #     minimize    t
    #     subject to  f0(x) - t <= 0
    #                 fk(x) <= 0,  k=1,...,mnl
    #                 G(x) <= h
    #                 A(x) = b.
    #
    # Store the variable as a list [ x, t ].

    # c = [0, 1.0]
    c = [ xnewcopy(x0), 1.0 ]
    xscal(0.0, c[0])

    ux, uznl = xnewcopy(x0), matrix(0.0, (mnl,1))

    def kktsolver_epi(x, z, dnl, dl):
             
        # g(x, y, znl, zl) is a solver for the smaller KKT system
        #
        #     [ H         *   *     *  ] [ x   ]   [ bx   ]
        #     [ A         0   *     *  ] [ y   ] = [ by   ]
        #     [ Df[1:,:]  0  -Dnl   *  ] [ znl ]   [ bznl ]
        #     [ G         0   0    -Dl ] [ zl  ]   [ bzl  ]
        #
        # where H = sum_k=0^mnl zk*Hk, Df is the derivative matrix of
        # (f0, ..., fmnl), Dnl = diag(dnl[1:])^-2, and Dl = diag(dl)^-2.
        # The variables are x (in V), y (in W), znl (in R^mnl), 
        # zl (in R^ml).

        g = kktsolver(x[0], z, dnl[1:], dl)
 
        f, Df = F(x[0])
        if type(Df) in (matrix, spmatrix):
            gradf0 = Df[0,:].T
        else: 
            gradf0 = xnewcopy(x[0])
            xscal(0.0, gradf0)
            e0 = matrix(0.0, (mnl+1,1))
            e0[0] = 1.0
            Df(e0, gradf0)
 
        def g_epi(x, y, znl, zl):

            # g_epi(x, y, znl, zl) solves the augmented KKT system
            #
            #    [ [H,    0;               ] [ x   ]    [ bx   ]
            #    [  0,    0]  *   *    *   ] [     ]    [      ]
            #    [ [A,    0]  0   *    *   ] [ y   ] =  [ by   ]
            #    [ [Df, -e0]  0  -Dnl  *   ] [ znl ]    [ bznl ]
            #    [ [G,    0]  0   0    -Dl ] [ zl  ]    [ bzl  ]
            #
            # The variables are x (a tuple (x[0], x[1]) with x[0] in
            # V and x[1] in R), y (in W), znl (in R^(mnl+1)) and
            # zl (in R^ml).
            #
            # The system is solved as follows:
            #
            # x[0], y, znl[1:], zl solve the smaller KKT system
            # with rhs (bx[0] + bx[1]*Df[0,:]', by, bnl[1:], bl).
            #
            # Take znl[0] = -bx[1] and 
            # x[1] = Df[0,:]*x[0] + bx[1]/dnl[0]**2 - bnl[0].

            a = znl[0]
            xcopy(x[0], ux)
            xaxpy(gradf0, ux, alpha=x[1])
            uznl[:] = znl[1:] 
            g(ux, y, uznl, zl)
            znl[1:] = uznl
            znl[0] = -x[1]
            xcopy(ux , x[0])
            x[1] = xdot(gradf0, x[0]) - znl[0] / dnl[0]**2 - a

        return g_epi

    
    # fA_epi evaluates [A, 0] * [x[0]; x[1]] and its adjoint.
    #
    #      v := alpha * A(u[0]) + beta*v if trans is 'N'
    #      v := alpha * ( A'(u), 0 ) + beta*v if trans is 'N'
 
    def fA_epi(u, v, alpha=1.0, beta=0.0, trans='N'):
        if trans == 'N':
            fA(u[0], v, alpha=alpha, beta=beta) 
        else:
            fA(u, v[0], alpha=alpha, beta=beta, trans='T') 
            v[1] *= beta


    # fG_epi evaluates  [G, 0] * [x[0]; x[1]] and its adjoint
    #
    #     v := alpha * G(u[0]) + beta*v  if trans is 'N' 
    #     v := alpha * ( G'(u), 0 )  + beta*v  if trans is 'T'.

    def fG_epi(u, v, alpha=1.0, beta=0.0, trans='N'):
        if trans == 'N':
            fG(u[0], v, alpha=alpha, beta=beta) 
        else:
            fG(u, v[0], alpha=alpha, beta=beta, trans='T') 
            v[1] *= beta


    # F_epi evaluates nonlinear constraint functions and derivatives.
    
    def F_epi(x=None, z=None):
        if x is None: 
            x0_epi = (x0, 0.0)
            return mnl+1, x0_epi 
        else:
            t = F(x[0])
            if t is None or t[0] is None:  return None, None
            f_epi, Df = matrix(t[0]), t[1]
            f_epi[0] -= x[1]
            def Df_epi(u, v):  
                # Evaluates v := [Df, -e0]' * u + v
                if type(Df) in (matrix, spmatrix):
                    base.gemv(Df, u, v[0], beta=1.0, trans='T')
                else:
                    Df(u, v[0])
                v[1] -= u[0]
            return f_epi, Df_epi

    def xnewcopy_epi(x):
        return [ xnewcopy(x[0]), 0.0 ]

    def xdot_epi(x, y):
        return xdot(x[0], y[0]) + x[1]*y[1]

    def xaxpy_epi(x, y, alpha=1.0):
        xaxpy(x[0], y[0], alpha=alpha)
        y[1] += alpha*x[1]

    def xscal_epi(alpha, x):
        xscal(alpha, x[0])
        x[1] *= alpha

    sol = nlclp(c, kktsolver_epi, F_epi, fG_epi, h, fA_epi, b, 
        xnewcopy_epi, xdot_epi, xaxpy_epi, xscal_epi, ynewcopy, ydot, 
        yaxpy, yscal) 
    if sol['status'] == 'optimal':
        sol['x'] = sol['x'][0]
        sol['znl'], sol['snl'] = sol['znl'][1:], sol['snl'][1:]
    else:
        sol['x'] = None
        sol['znl'], sol['snl'] = None, None 
    return sol


def cp(F, G=None, h=None, A=None, b=None):

    """
    Solves a convex optimization problem
    
        minimize    f0(x)
        subject to  fk(x) <= 0, k=1,...,m
                    G*x <= h
                    A*x = b.                      

    f: R^n -> R^(m+1) is convex and twice differentiable.  

    Input arguments

        F is a function that handles the following arguments.

            F() returns a tuple (m, x0).  

                m is the number of nonlinear inequality constraints.
                x0 is an nx1 dense 'd' matrix specifying a point in 
                dom f.

            F(x), with x a 'd' matrix of size (n,1), returns (f, Df).

                f is  a dense 'd' matrix of size (m+1,1) with the 
                function values f(x). 

                Df is a dense or sparse 'd' matrix of size (m+1,n). 
                Its kth row contains the transpose of the gradient
                of fk at x.

                If x is not in dom f, F(x) returns None or (None, None).
 
            F(x, z), with x a 'd' matrix of size (n,1) and z a positive 
            'd' matrix of size (m+1,1), returns (f, Df, H).
            
                f and Df are defined as above.
                
                H is a dense or sparse 'd' matrix of size (n,n).  The 
                lower triangular part of H contains the lower triangular
                part of sum_{k=0}^m z[k]*Hk where Hk is the Hessian of 
                fk at x.  

                If F is called with two arguments, it can be assumed
                that x is dom f. 

        G is None or a sparse or dense 'd' matrix.  If G is None, it 
        is interpreted as a 0xn matrix.  

        h is None or a dense 'd' matrix with one column.  If h is None, 
        it is interpreted as a 0x1 matrix.

        A is None, or a dense or sparse 'd' matrix.  If A is None, it 
        is interpreted as a 0xn matrix.  

        b is a dense 'd' matrix with one column or None.  If b is None,
        it is interpreted as a 0x1 matrix.

    Returns a dictionary with keys 'status', 'x', 'snl', 'sl', 'y', 
    'znl', 'zl'.  

        status is 'optimal' or 'unknown'.

        x, s, snl, sl, y, znl, zl are None if status is 'unknown'.  
        Otherwise x is the optimal solution;  snl, sl are the optimal 
        slacks for the nonlinear and linear inequalities;  y is the 
        optimal dual variable associated with the equality constraints; 
        znl, zl are the optimal dual variables associated with the 
        nonlinear and linear inequalities.
    """

    try: kktmethod = options['kktmethod']
    except KeyError: kktmethod = 3
    else: 
        if type(kktmethod) is not int or kktmethod not in [1,2,3]:
            raise ValueError, "options['kktmethod'] must be 1, 2 or 3"

    mnl, x0 = F()
    if type(mnl) is not int or mnl < 0 or type(x0) is not matrix or \
        x0.typecode != 'd' or x0.size[1] != 1:
        raise TypeError, "invalid return values for '(m, x) = F()'"
    n = x0.size[0]

    if G is None: G = spmatrix([], [], [], (0,n))
    if h is None: h = matrix(0.0, (0,1))
    if type(G) not in (matrix, spmatrix) or G.typecode != 'd' or \
        G.size[1] != n:
        raise TypeError, "'G' must be a dense or sparse 'd' matrix "\
            "with %d columns" %n 
    ml = G.size[0]
    if type(h) is not matrix or h.typecode != 'd' or h.size != (ml,1):
        raise TypeError, "'h' must be a dense 'd' matrix of size "\
            "(%d,1)" %ml

    if A is None: A = spmatrix([], [], [], (0,n))
    if b is None: b = matrix(0.0, (0,1))
    if type(A) not in (matrix,  spmatrix) or A.typecode != 'd' or \
        A.size[1] != n: 
        raise TypeError, "'A' must be a dense or sparse 'd' matrix "\
            "with %d columns" %n
    p = A.size[0]
    if type(b) is not matrix or b.typecode != 'd' or b.size != (p,1): 
        raise TypeError, "'b' must be a dense 'd' matrix of "\
            "size (%d,1)" %p

    status = {'kkt_num': None}  # 'pointer' to numeric factorization
    def kkt(x, z, dnl, dl):
        f, Df, H = F(x, z)
        if mnl==0:  
            Df2 = spmatrix([],[],[],(0,n))
        else:  
            Df2 = Df[1:,:]
        if status['kkt_num'] is None:
            status['kkt_num'] = misc.kkt(H, A, Df2, G, kktmethod)
        return status['kkt_num'](H, Df2, dnl, dl)
    
    return nlcp(kkt, F, G, h, A, b) 


def qp(P, q, G=None, h=None, A=None, b=None, solver=None):

    """
    Solves a quadratic program

        minimize    (1/2)*x'*P*x + q'*x 
        subject to  G*x <= h      
                    A*x = b.


    Input arguments 

        P is a nxn dense or sparse 'd' matrix with the lower triangular 
        part of P stored in the lower triangle.  Must be positive 
        semidefinite.

        q is an nx1 dense 'd' matrix.

        G is an mxn dense or sparse 'd' matrix.

        h is an mx1 dense 'd' matrix.

        A is a pxn dense or sparse 'd' matrix.

        b is a px1 dense 'd' matrix or None.

        solver is None or 'mosek'.

        The default values for G, h, A and b are empty matrices with 
        zero rows.


    Returns a dictionary with keys 'status', 'x', 's', 'y', 'z'.

        The default solver returns with status 'optimal' or 'unknown'.
        The MOSEK solver can also return with status 'primal infeasible'
        or 'dual infeasible'.

        If status is 'optimal', x, s, y, z are the primal and dual 
        optimal solutions.

        If status is 'primal infeasible', x = s = None and z, y are 
        a proof of primal infeasibility:

            G'*z + A'*y = 0,  h'*z + b'*y = -1,  z >= 0.

        If status is 'dual infeasible', z = y = None, and x, s are 
        a proof of dual infeasibility:

            P*x = 0,  q'*x = -1,  G*x + s = 0,  A*x = 0,  s >=0

        If status is 'unknown', x, y, s, z are None.  
    """

    if solver == 'mosek':
        try: from cvxopt import mosek
        except ImportError: raise ValueError, "invalid option "\
            "(solver='mosek'): cvxopt.mosek is not installed" 
        mosek.options = options
        prosta, solsta, x, z, y = mosek.solveqp(P, q, G, h, A, b)
        m = G.size[0]
        if solsta == 'OPTIMAL':
            s = matrix(0.0, (m,1))
            blas.copy(h, s)    
            base.gemv(G, x, s, alpha=-1.0, beta=1.0)
            status = 'optimal'
        elif solsta == 'PRIMAL_INFEASIBLE_CER':
            status = 'primal infeasible'
            ducost = -blas.dot(h,z) - blas.dot(b,y)
            blas.scal(1.0/ducost, y);
            blas.scal(1.0/ducost, z);
            x, s = None, None
        elif solsta == 'DUAL_INFEASIBLE_CER':
            status = 'dual infeasible'
            qx = blas.dot(q,x)
            if qx:  x /= (-qx)
            s = matrix(0.0, (m,1))
            base.gemv(G, x, s, alpha=-1.0)
            z, y = None, None
        else: 
            status = 'unknown'
            x, s, y, z = None, None, None, None
        return {'status': status, 'x': x, 's': s, 'y': y, 'z': z}

    if type(P) not in (matrix, spmatrix) or P.typecode != 'd' or \
        P.size[0] != P.size[1]:
        raise TypeError, "'P' must be a square dense or sparse 'd' "\
            "matrix"
    n = P.size[0]
    if type(q) is not matrix or q.typecode != 'd' or q.size != (n,1): 
        raise TypeError, "'q' must be a dense 'd' matrix of "\
            "size (%d,1)" %n

    def F(x=None, z=None):
        if x is None: return 0, matrix(0.0, (n,1))
        grad = matrix(0.0, (1,n))
        base.symv(P, x, grad) 
        f = .5 * blas.dot(grad, x) + blas.dot(q, x) 
        blas.axpy(q, grad)
        if z is None: return f, grad
        else: return f, grad, z[0]*P

    sol =  cp(F, G, h, A, b)
    return {'status': sol['status'], 'x': sol['x'], 's': sol['sl'],
        'y': sol['y'], 'z': sol['zl']}



def gp(K, F, g, G=None, h=None, A=None, b=None):

    """
    Solves a geometric program

        minimize    log sum exp (F0*x+g0)
        subject to  log sum exp (Fi*x+gi) <= 0,  i=1,...,m
                    G*x <= h      
                    A*x = b

    Input arguments

        K is a list of positive integers [K0, K1, K2, ..., Km].

        F is a sum(K)xn dense or sparse 'd' matrix with block rows F0, 
        F1, ..., Fm.  Each Fi is Kixn.

        g is a sum(K)x1 dense or sparse 'd' matrix with blocks g0, g1, 
        g2, ..., gm.  Each gi is Kix1.

        G is an mxn dense or sparse 'd' matrix.

        h is an mx1 dense 'd' matrix.

        A is a pxn dense or sparse 'd' matrix.

        b is a px1 dense 'd' matrix.

        The default values for G, h, A and b are empty matrices with 
        zero rows.


    Returns a dictionary with keys 'status', 'x', 'snl', 'sl, 'y', 
        'znl', 'zl'.

        If status is 'optimal', x is the optimal solution, snl and sl
        are the optimal slacks for the nonlinear and linear 
        inequalities, znl and zl are the optimal dual veriables 
        associated with the nonlinear and linear inequalities, and 
        y are the optimal variables associated with the equality 
        constraints.

        If status is 'unknown', x, snl, sl, y, znl and zl are None.
    """

    if type(K) is not list or [ k for k in K if type(k) is not int 
        or k <= 0 ]:
        raise TypeError, "'K' must be a list of positive integers" 
    mnl = len(K)-1
    l = sum(K)

    if type(F) not in (matrix, spmatrix) or F.typecode != 'd' or \
        F.size[0] != l:
        raise TypeError, "'F' must be a dense or sparse 'd' matrix "\
            "with %d rows" %l
    if type(g) is not matrix or g.typecode != 'd' or g.size != (l,1): 
        raise TypeError, "'g' must be a dene 'd' matrix of "\
            "size (%d,1)" %l
    n = F.size[1]

    if G is None: G = spmatrix([], [], [], (0,n))
    if h is None: h = matrix(0.0, (0,1))
    if type(G) not in (matrix, spmatrix) or G.typecode != 'd' or \
        G.size[1] != n:
        raise TypeError, "'G' must be a dense or sparse 'd' matrix "\
            "with %d columns" %n 
    ml = G.size[0]
    if type(h) is not matrix or h.typecode != 'd' or h.size != (ml,1):
        raise TypeError, "'h' must be a dense 'd' matrix of "\
            "size (%d,1)" %ml

    if A is None: A = spmatrix([], [], [], (0,n))
    if b is None: b = matrix(0.0, (0,1))
    if type(A) not in (matrix, spmatrix) or A.typecode != 'd' or \
        A.size[1] != n:
        raise TypeError, "'A' must be a dense or sparse 'd' matrix "\
            "with %d columns" %n
    p = A.size[0]
    if type(b) is not matrix or b.typecode != 'd' or b.size != (p,1): 
        raise TypeError, "'b' must be a dense 'd' matrix of "\
            "size (%d,1)" %p

    y = matrix(0.0, (l,1))
    u = matrix(0.0, (max(K),1))
    Fsc = matrix(0.0, (max(K),n))

    cs1 = [ sum(K[:i]) for i in xrange(mnl+1) ] 
    cs2 = [ cs1[i] + K[i] for i in xrange(mnl+1) ]
    ind = zip(range(mnl+1), cs1, cs2)

    def Fgp(x=None, z=None):

        if x is None: return mnl, matrix(0.0, (n,1))
	
        f = matrix(0.0, (mnl+1,1))
        Df = matrix(0.0, (mnl+1,n))

        # y = F*x+g
        blas.copy(g, y)
        base.gemv(F, x, y, beta=1.0)

        if z is not None: H = matrix(0.0, (n,n))

        for i, start, stop in ind:

            # yi := exp(yi) = exp(Fi*x+gi) 
            ymax = max(y[start:stop])
            y[start:stop] = exp(y[start:stop] - ymax)

            # fi = log sum yi = log sum exp(Fi*x+gi)
            ysum = blas.asum(y, n=stop-start, offset=start)
            f[i] = ymax + math.log(ysum)

            # yi := yi / sum(yi) = exp(Fi*x+gi) / sum(exp(Fi*x+gi))
            blas.scal(1.0/ysum, y, n=stop-start, offset=start)

            # gradfi := Fi' * yi 
            #        = Fi' * exp(Fi*x+gi) / sum(exp(Fi*x+gi))
            base.gemv(F, y, Df, trans='T', m=stop-start, incy=mnl+1,
                offsetA=start, offsetx=start, offsety=i)

            if z is not None:

                # Hi = Fi' * (diag(yi) - yi*yi') * Fi 
                #    = Fisc' * Fisc
                # where 
                # Fisc = diag(yi)^1/2 * (I - 1*yi') * Fi
                #      = diag(yi)^1/2 * (Fi - 1*gradfi')

                Fsc[:K[i], :] = F[start:stop, :] 
                for k in xrange(start,stop):
                   blas.axpy(Df, Fsc, n=n, alpha=-1.0, incx=mnl+1,
                       incy=Fsc.size[0], offsetx=i, offsety=k-start)
                   blas.scal(math.sqrt(y[k]), Fsc, inc=Fsc.size[0],
                       offset=k-start)

                # H += z[i]*Hi = z[i] * Fisc' * Fisc
                blas.syrk(Fsc, H, trans='T', k=stop-start, alpha=z[i],
                    beta=1.0)

        if z is None: return f, Df
        else: return f, Df, H

    return cp(Fgp, G, h, A, b)
