/*
 * cwdaemon - morse sounding daemon for the parallel or serial port
 * Copyright (C) 2002 -2004 Joop Stakenborg <pg4i@amsat.org>
 *                       and many authors, see the AUTHORS file.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

# if HAVE_STDIO_H
# include <stdio.h>
#endif
#if HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#if HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif
#if STDC_HEADERS
# include <stdlib.h>
# include <stddef.h>
#else
# if HAVE_STDLIB_H
#  include <stdlib.h>
# endif
#endif
#if HAVE_STRING_H
# if !STDC_HEADERS && HAVE_MEMORY_H
#  include <memory.h>
# endif
# include <string.h>
#endif
#if HAVE_UNISTD_H
# include <unistd.h>
#endif
#if DHAVE_ARPA_INET
# include <arpa/inet.h>
#endif
#if HAVE_FCNTL_H
# include <fcntl.h>
#endif
#if HAVE_NETDB
# include <netdb.h>
#endif
#if HAVE_NETINET_IN_H
# include <netinet/in.h>
#endif
#if HAVE_SYS_SOCKET_H
# include <sys/socket.h>
#endif
#if HAVE_SYSLOG_H
# include <syslog.h>
#endif
#if HAVE_ERRNO_H
# include <errno.h>
#endif
#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif
#if HAVE_SIGNAL_H
# include <signal.h>
#endif
#if HAVE_STDARG_H
# include <stdarg.h>
#endif
#if HAVE_SYS_RESOURCE_H
# include <sys/resource.h>
#endif
#if (defined(__unix__) || defined(unix)) && !defined(USG)
# include <sys/param.h>
#endif

#include "cwlib.h"
#include "cwdaemon.h"

#ifdef HAVE_DEV_PPBUS_PPI_H
static char *actual_device(const char *in);
#endif

/* network vars */
int sin_len, reply_socklen;
int socket_descriptor;
struct sockaddr_in k_sin, reply_sin;
int port = 6789;		/* default UDP port we listen on */
char reply_data[256];

/* morse defaults */
int morse_speed = 24;		/* speed (wpm) */
int morse_tone = 800;		/* tone (Hz) */
int morse_sound = 1;		/* speaker on */
int morse_volume = 70;		/* initial volume */
int wordmode = 0;		/* start in character mode */
int ptt_delay = 0;		/* default = 0 ms */
int console_sound = 1;		/* speaker on */
int soundcard_sound = 0;	/* soundcard off */

/* various variables */
int forking = 1; 		/* we fork by default */
int bandswitch;
int priority = 0;

/* flags for different states */
int ptt_timer_running = 0;	/* flag for PTT state */
int aborting = 0;
int sendingmorse = 0;

struct timeval now, end, left;	/* PTT timers */
struct timespec sleeptime, time_remainder; /* delay timers */

#define MAXMORSE 4000
char morsetext[MAXMORSE];

cwdevice cwdevice_ttys = {
	init:ttys_init,
	free:ttys_free,
	reset:ttys_reset,
	cw:ttys_cw,
	ptt:ttys_ptt,
	ssbway:NULL,
	switchband:NULL,
	footswitch:NULL
};

#if defined (HAVE_LINUX_PPDEV_H) || defined (HAVE_DEV_PPBUS_PPI_H)
cwdevice cwdevice_lp = {
	init:lp_init,
	free:lp_free,
	reset:lp_reset,
	cw:lp_cw,
	ptt:lp_ptt,
	ssbway:lp_ssbway,
	switchband:lp_switchband,
	footswitch:lp_footswitch
};

cwdevice *cwdev = &cwdevice_lp;
#else
cwdevice *cwdev = &cwdevice_ttys;
#endif


/* catch ^C when running in foreground */
static RETSIGTYPE
catchint (int signal)
{
	cwdev->free (cwdev);
	printf ("%s: Exiting\n", PACKAGE);
	exit (0);
}

/* print error message to the console or syslog if we are forked */
void
errmsg (char *info, ...)
{
	va_list ap;
	char s[1025];

	va_start (ap, info);
	vsnprintf (s, 1024, info, ap);
	va_end (ap);

	if (forking)
		syslog (LOG_ERR, "%s\n", s);
	else
		printf ("%s: %s failed: %s\n", PACKAGE, s, strerror (errno));
}

/* print only debug message to the console */
void
debug (char *info, ...)
{
	va_list ap;
	char s[1025];

	if (!forking)
	{
		va_start (ap, info);
		vsnprintf (s, 1024, info, ap);
		va_end (ap);
		printf ("%s: %s \n", PACKAGE, s);
	}
}

/* delay in microseconds */
static void
udelay (unsigned long us)
{
	sleeptime.tv_sec = 0;
	sleeptime.tv_nsec = us * 1000;

	if (nanosleep (&sleeptime, &time_remainder) == -1)
	{
		if (errno == EINTR)
			nanosleep (&time_remainder, NULL);
		else
			errmsg ("Nanosleep");
	}
}


/* some simple timing utilities, see 
 * http://www.erlenstar.demon.co.uk/unix/faq_8.html */
static void
timer_add (struct timeval *tv, long secs, long usecs)
{
	tv->tv_sec += secs;
	if ((tv->tv_usec += usecs) >= 1000000)
	{
		tv->tv_sec += tv->tv_usec / 1000000;
		tv->tv_usec %= 1000000;
	}
}

/* Set *RES = *A - *B, returning the sign of the result */
static int
timer_sub (struct timeval *res, const struct timeval *a,
	   const struct timeval *b)
{
	long sec = a->tv_sec - b->tv_sec;
	long usec = a->tv_usec - b->tv_usec;

	if (usec < 0)
		usec += 1000000, --sec;

	res->tv_sec = sec;
	res->tv_usec = usec;

	return (sec < 0) ? (-1) : ((sec == 0 && usec == 0) ? 0 : 1);
}

/* band switch function,  pin 2, 7, 8, 9 */
static void
set_switch (unsigned int bandswitch)
{
	unsigned int lp_switchbyte = 0;

	lp_switchbyte = (bandswitch & 0x01) | ((bandswitch & 0x0e) * 16);
	if (cwdev->switchband)
	{
		cwdev->switchband (cwdev, lp_switchbyte);
		debug ("Set bandswitch to %x", bandswitch);
	}
	else
		debug ("Bandswitch output not implemented");
}

/* tune a number of seconds */
static void
tune (int seconds)
{
	int us;

	if (seconds > 0)
	{
		cw_flush_tone_queue ();
		if (ptt_delay)
		{
			cwdev->ptt (cwdev, ON);
			debug ("PTT (TUNE) on");
			/* TOD */
			udelay (ptt_delay);
		}
		us = seconds * 1000000;
		debug ("CW (TUNE) on");
		cw_queue_tone (us, morse_tone);
		
		if (ptt_delay)
		{
			udelay (us + ptt_delay);
			cwdev->ptt (cwdev, OFF);
			debug ("PTT (TUNE) off");
		}
	}
}

/* initialize unixcw */
static void
initmorse (void)
{
	cw_set_frequency (morse_tone);	
	cw_set_send_speed (morse_speed);
	cw_set_soundcard_sound (soundcard_sound);
	cw_set_console_sound (console_sound);
	cw_set_volume (morse_volume);
	cw_set_gap (0);
}

/* watch the socket and if there is an escape character check what it is,
   otherwise play morse. Return 0 with escape characters and empty messages.*/
static int
recv_code (void)
{
	char message[257];
	ssize_t recv_rc;
	int speed = 24;
	int weight = 0;
	int f = 800;
	int vol = 70;
	int valid_sdevice = 0;
	int tunesec;

	recv_rc = recvfrom (socket_descriptor, message, sizeof (message) - 1,
		0, (struct sockaddr *) &k_sin, &sin_len);

	if (recv_rc == -1 && errno != EAGAIN)
	{
		errmsg ("Recvfrom");
		exit (1);
	}

	if (recv_rc > 0)
	{
		if (message[0] != 27)
		{	/* no ESCAPE */
			message[recv_rc] = '\0';
			if ((strlen (message) + strlen (morsetext)) <= MAXMORSE - 1)
				strcat (morsetext, message);
			return 1;
		}
		else
		{	/* check ESCAPE characters */
			switch ((int) message[1])
			{
			case '0':	/* reset  all values */
				morsetext[0] = '\0';
				morse_speed = 24;
				morse_tone = 800;
				morse_volume = 70;
				console_sound = 1;
				soundcard_sound = 0;
				initmorse ();
				wordmode = 0;
				cwdev->reset (cwdev);
				debug ("Reset all values");
				break;
			case '2':	/*speed */
				speed = atoi (message + 2);
				if (speed > 4 && speed < 61)
				{
					morse_speed = speed;
					cw_set_send_speed (morse_speed);
					debug ("Speed: %d wpm", morse_speed);
				}
				break;
			case '3':	/* tone */
				f = atoi (message + 2);
				if (f > 0 && f < 4001)
				{
					morse_tone = f;
					cw_set_frequency (morse_tone);
					cw_set_volume (70);
					morse_sound = 1;
					debug ("Tone: %s Hz, volume 70%", message + 2);
				}
				else if (f == 0)	/* sidetone off */
				{
					morse_tone = 0;
					cw_set_volume (0);
					morse_sound = 0;
					debug ("Volume off");
				}
				break;
			case '4':	/* message abort */
				debug ("Message abort");
				if (sendingmorse == 1)
				{
					cw_flush_tone_queue ();
					aborting = 1;
				}
				break;
			case '5':	/* exit */
				cwdev->free (cwdev);
				errno = 0;
				errmsg ("Sender has told me to end the connection");
				exit (0);
				break;
			case '6':	/* set uninterruptable */
				message[0] = '\0';
				morsetext[0] = '\0';
				wordmode = 1;
				debug ("Wordmode set");
				break;
			case '7':	/* set weighting */
				weight = atoi (message + 2);
				if ((weight > -51) && (weight < 51))	/* only allowed range */
				{	
					cw_set_weighting (weight);
					debug ("Weight: %d", weight);
				}
				break;
			case '8':	/* device type */
				if (!strncmp (message + 2, "tty", 3))
				{
					cwdev->free (cwdev);
					cwdev = &cwdevice_ttys;
#ifdef BSD
					if (!strncmp (message + 2, "ttyS", 4))
						message[3] = 'd';
#endif
					cwdev->desc = message + 2;
					cwdev->init (cwdev);
				}
#ifdef HAVE_LINUX_PPDEV_H
				else if (!strncmp (message + 2, "parport", 7))
				{	
					cwdev->free (cwdev);
					cwdev = &cwdevice_lp;
					cwdev->desc = message + 2;
					cwdev->init (cwdev);
				}
#endif
#ifdef HAVE_DEV_PPBUS_PPI_H
				else if (strncmp (message + 2, "parport", 7) == 0)
				{
					cwdev->free (cwdev);
					cwdev = &cwdevice_lp;
					cwdev->desc = actual_device(message + 2);
					cwdev->init (cwdev);
				}
				else if (strncmp (message + 2, "ppi", 3) == 0)
				{
					cwdev->free (cwdev);
					cwdev = &cwdevice_lp;
					cwdev->desc = message + 2;
					cwdev->init (cwdev);
				}
#endif
				else debug ("Unknown device");
				break;
			case '9':	/* base port number */
				debug ("Obsolete");
				break;
			case 'a':	/* PTT keying on or off */
				if (atoi (message + 2))
				{
					cwdev->ptt (cwdev, ON);
					debug ("PTT on");
				}
				else
				{
					cwdev->ptt (cwdev, OFF);
					debug ("PTT off");
				}
				break;
			case 'b':	/* SSB way */
				if (atoi (message + 2))
				{
					if (cwdev->ssbway)
					{
						cwdev->ssbway (cwdev, SOUNDCARD);
						debug ("SSB way set to SOUNDCARD", PACKAGE);
					}
					else
						debug ("SSB way to SOUNDCARD unimplemented");
				}
				else
				{
					if (cwdev->ssbway)
					{
						cwdev->ssbway (cwdev, MICROPHONE);
						debug ("SSB way set to MIC");
					}
					else
						debug ("SSB way to MICROPHONE unimplemented");
				}
				break;
			case 'c':	/* Tune for a number of seconds */
				tunesec = atoi (message + 2);
				if (tunesec <= 10)
					tune (tunesec);
				break;
			case 'd':	/* set ptt delay (TOD, Turn On Delay) */
				ptt_delay = atoi (message + 2);
				if ((ptt_delay >= 0) && (ptt_delay < 51))	/* only allowed range 0..50 ms */
					ptt_delay *= 1000;	/* 0 = ptt delay off */
				else
					ptt_delay = 50000;	/* longest delay (secure) if parameter wrong */
				debug ("PTT delay(TOD): %d ms", ptt_delay / 1000);
				if (ptt_delay)
					;
				else
				{
					cwdev->ptt (cwdev, OFF);
					debug ("PTT off");
				}
				break;
			case 'e':	/* set bandswitch output on parport bits 2(lsb),7,8,9(msb) */
				bandswitch = atoi (message + 2);
				if (bandswitch <= 15 && bandswitch >= 0)
					set_switch (bandswitch);
				break;
			case 'f':	/* switch console/soundcard */
				if (!strncmp (message + 2, "c", 1))
				{
					console_sound = 1;
					soundcard_sound = 0;
					valid_sdevice = 1;
				}
				else if (!strncmp (message + 2, "s", 1))
				{
					console_sound = 0;
					soundcard_sound = 1;
					valid_sdevice = 1;
				}
				else if (!strncmp (message + 2, "b", 1))
				{
					console_sound = 1;
					soundcard_sound = 1;
					valid_sdevice = 1;
				}
				if (valid_sdevice == 1)
				{
					debug ("Sound device: %s", message + 2);
					cw_set_soundcard_sound (soundcard_sound);
					cw_set_console_sound (console_sound);
				}
				break;
			case 'g':	/* volume */
				vol = atoi (message + 2);
				if (vol <= 100 && vol >= 0)
				{
					morse_volume = vol;
					cw_set_volume (morse_volume);
				}
				break;
            case 'h':   /* send echo to main program when CW playing is done */
                memcpy(&reply_sin, &k_sin, sizeof(reply_sin)); /* remember sender */
                reply_socklen = sin_len;
                strncpy(reply_data+1, message, sizeof(reply_data) - 2);
                reply_data[sizeof(reply_data) - 1] = '\0';
                reply_data[0]='h';
                if (strlen (message) + 1 <= MAXMORSE - 1) strcat (morsetext, "^");
                break;
			}
			return 0;
		}
	}
	return 0;
}

/* check every character for speed increase or decrease, and play others */
static void
playmorsestring (char *x)
{
	int i = 0, validchar = 0;
	char c;

	sendingmorse = 1;
	while (*x)
	{
		if (aborting == 1)
		{
			aborting = 0;
			break;
		}
		c = *x;
		if ((c == '+') || (c == '-'))
		{		/* speed in- & decrease */
			if ((c == '+') && (morse_speed <= 58))
				morse_speed += 2;
			if ((c == '-') && (morse_speed >= 10))
				morse_speed -= 2;
			cw_set_send_speed (morse_speed);
		}
		else if (c == '~')
			cw_set_gap (2); /* 2 dots time additional for the next char */
        else if (c == '^'){
            debug("Echo '%s'", reply_data);
            if (strlen(reply_data)==0) return;
            sendto(socket_descriptor, reply_data, strlen(reply_data), 0, 
                (struct sockaddr *)&reply_sin, reply_socklen);
            strcpy(reply_data,"");
            break;
  
        }
		else
		{
			validchar = 1;
			if (ptt_delay)
			{
				if (ptt_timer_running == 1)
				{
					gettimeofday (&end, NULL);
					timer_add (&end, 0, ptt_delay);
				}
				else
				{
					cwdev->ptt (cwdev, ON);
					debug ("PTT on");
					/* TOD */
					udelay (ptt_delay);
				}
			}
			debug ("Morse = %c", c);
			cw_send_character (c);
			cw_tone_queue_wait_critical (1);
			if (cw_get_gap () == 2) cw_set_gap (0);
		}
		x++;
		i++;
		if (i >= strlen (morsetext))
		{
			i = 0;
			break;
		}
		if (wordmode == 0)
			recv_code ();
	}
	morsetext[0] = '\0';

	/* start ptt off timer */
	if (ptt_delay && validchar)
	{
		gettimeofday (&end, NULL);
		timer_add (&end, 0, ptt_delay);
		ptt_timer_running = 1;
	}
	sendingmorse = 0;
}

static void keyingevent (int keystate)
{
	if (keystate == 1)
		cwdev->cw (cwdev, ON);
	else
		cwdev->cw (cwdev, OFF);
}

static void pttevent (int pttstate)
{
	if (pttstate == 1)
		cwdev->ptt(cwdev,ON);
	else
		cwdev->ptt(cwdev,OFF);
}

/* parse the command line and check for options, do some error checking */
static void
parsecommandline (int argc, char *argv[])
{
	int p, w;

	while ((p = getopt (argc, argv, "d:hnp:P:s:t:v:Vw:x:")) != -1)
	{
		switch (p)
		{
		case ':':
		case '?':
		case 'h':
			printf ("Usage: %s [option]...\n", PACKAGE);
			printf ("       -d <device>   ");
			printf ("Use a different device\n                     ");
#if defined (HAVE_LINUX_PPDEV_H)
			printf ("(e.g. ttyS0,1,2, parport0,1, etc. default = parport0)\n");
#elif defined (HAVE_DEV_PPBUS_PPI_H)
			printf ("(e.g. ttyd0,1,2, ppi0,1, etc. default = ppi0)\n");
#else
#ifdef BSD
			printf ("(e.g. ttyd0,1,2, etc. default = ttyd0)\n");
#else
			printf ("(e.g. ttyS0,1,2, etc. default = ttyS0)\n");
#endif
#endif
			printf ("       -h            ");
			printf ("Display this help and exit\n");
			printf ("       -n            ");
			printf ("Do not fork and print debug information to stdout\n");
			printf ("       -p <port>     ");
			printf ("Use a different UDP port number (> 1023, default = 6789)\n");
#if defined(HAVE_SETPRIORITY) && defined(PRIO_PROCESS)
			printf ("       -P <priority> ");
			printf ("Set cwdaemon priority (-20 ... 20, default = 0)\n");
#endif
			printf ("       -s <speed>    ");
			printf ("Set morse speed (4 ... 60 wpm, default = 24)\n");
			printf ("       -t <time>     ");
			printf ("Set PTT delay (0 ... 50 ms, default = 0)\n");
			printf ("       -v <volume>   ");
			printf ("Set volume for soundcard output\n");
			printf ("       -V            ");
			printf ("Output version information and exit\n");
			printf ("       -w <weight>   ");
			printf ("Set weighting (-50 ... 50, default = 0)\n");
			printf ("       -x <sdevice>  ");
			printf ("Use a different sound device\n                     ");
			printf ("(c = console (default), s = soundcard, b = both)\n");
			exit (0);
		case 'd':
			if (!strncmp (optarg, "tty", 3)
			    && strlen (optarg) < 9)
				cwdev = &cwdevice_ttys;
#ifdef HAVE_LINUX_PPDEV_H
			else if (!strncmp (optarg, "parport", 7)
				 && strlen (optarg) < 9)
				cwdev = &cwdevice_lp;
#endif
#ifdef HAVE_DEV_PPBUS_PPI_H
			else if ((strncmp (optarg, "parport", 7) == 0)
				 && strlen (optarg) < 9)
				cwdev = &cwdevice_lp;
			else if ((strncmp (optarg, "ppi", 3) == 0)
				 && strlen (optarg) < 5)
				cwdev = &cwdevice_lp;
#endif
			else
			{
				printf ("%s: Wrong device %s, ", PACKAGE,
					optarg);
#if defined (HAVE_LINUX_PPDEV_H)
				printf ("use one of ttyS0,1,2,3 or parport0,1 etc.\n");
#elif defined (HAVE_DEV_PPBUS_PPI_H)
				printf ("use one of ttyd0,1,2,3 or ppi0,1 etc.\n");
#else
#ifdef BSD
				printf ("use one of ttyd0,1,2,3 etc.\n");
#else
				printf ("use one of ttyS0,1,2,3 etc.\n");
#endif
#endif
				exit (1);
			}
#ifdef HAVE_DEV_PPBUS_PPI_H
			cwdev->desc = actual_device(optarg);
#else
			cwdev->desc = optarg;
#endif
			break;
		case 'n':
			forking = 0;
			printf ("%s: Not forking...\n", PACKAGE);
			break;
		case 'p':
			if ((port = atoi (optarg)) <= 0)
			{
				printf ("%s: Invalid port number - %s\n",
					PACKAGE, optarg);
				exit (1);
			}
			if (port < 1024)
			{
				printf ("%s: Port should be larger than 1023\n", PACKAGE);
				exit (1);
			}
			break;
#if defined(HAVE_SETPRIORITY) && defined(PRIO_PROCESS)
		case 'P':
			priority = atoi (optarg);
			if (priority < -20 || priority > 20)
			{
				printf ("%s: Wrong priority, use a value between -20 and 20\n",
					PACKAGE);
				exit (1);
			}
		break;
#endif
		case 's':
			morse_speed = atoi (optarg);
			if (morse_speed < 5 || morse_speed > 60)
			{
				printf ("Wrong speed, use a value between 4 and 60\n");
				exit (1);
			}
			break;
		case 't':
			ptt_delay = atoi (optarg);
			if (ptt_delay < 0 || ptt_delay > 50)
			{
				printf ("Wrong PTT delay value, must be between 0 and 50 ms\n");
				exit (1);
			}
			else
				ptt_delay *= 1000;
			break;
		case 'v':
			morse_volume = atoi (optarg);
			if ((morse_volume < 0) || (morse_volume > 100))
			{
				printf ("Wrong volume value, use 0 ... 100\n");
				exit (1);
			}
			break;
		case 'V':
			printf ("%s version %s\n", PACKAGE, VERSION);
			exit (0);
		case 'w':
			w = atoi (optarg);
			if ((w > 50) || (w < -50))
			{
				printf ("Wrong weighting value, use -50 ... 50\n");
				exit (1);
			}
			cw_set_weighting (w);
			break;
		case 'x':
			if (!strncmp(optarg, "c", 1))
			{
				console_sound = 1;
				soundcard_sound = 0;
			}
			else if (!strncmp(optarg, "s", 1))
			{
				console_sound = 0;
				soundcard_sound = 1;
			}
			else if (!strncmp(optarg, "b", 1))
			{
				console_sound = 1;
				soundcard_sound = 1;
			}
			else
			{
				printf ("Wrong sound device, use c(onsole), s(soundcard) or b(oth)\n");
				exit (1);
			}
			break;
		}
	}
}

/* main program: fork, open network connection and go into an endless loop
   waiting for something to happen on the UDP port */
int
main (int argc, char *argv[])
{
	pid_t pid, sid;
	int bind_rc, close_rc;
	long save_file_flags;

#if defined HAVE_LINUX_PPDEV_H
	cwdev->desc = "parport0";
#elif defined HAVE_DEV_PPBUS_PPI_H
	cwdev->desc = "ppi0";
#else
 #ifdef BSD
	cwdev->desc = "ttyd0";
 #else
	cwdev->desc = "ttyS0";
 #endif
#endif

	parsecommandline (argc, argv);
	if (geteuid () != 0)
	{
		printf ("You must run this program as root\n");
		exit (1);
	}
	initmorse ();
	cw_keying_callback (keyingevent);
	cw_ptt_callback (pttevent);

	debug ("Device used: %s", cwdev->desc);

	if (forking)
	{
		pid = fork ();
		if (pid < 0)
		{
			printf ("%s: Fork failed: %s\n", PACKAGE,
				strerror (errno));
			exit (1);
		}

		if (pid > 0)
			exit (0);

		openlog ("netkeyer", LOG_PID, LOG_DAEMON);
		if ((sid = setsid ()) < 0)
		{
			syslog (LOG_ERR, "%s\n", "setsid");
			exit (1);
		}
		if ((chdir ("/")) < 0)
		{
			syslog (LOG_ERR, "%s\n", "chdir");
			exit (1);
		}
		umask (0);
		close (STDIN_FILENO);
		close (STDOUT_FILENO);
		close (STDERR_FILENO);
	}
	else
	{
		debug ("Press ^C to quit");
		signal (SIGINT, catchint);
	}

	cwdev->init (cwdev);

	bzero (&k_sin, sizeof (k_sin));
	k_sin.sin_family = AF_INET;
	k_sin.sin_addr.s_addr = htonl (INADDR_ANY);
	k_sin.sin_port = htons (port);
	sin_len = sizeof (k_sin);

	socket_descriptor = socket (AF_INET, SOCK_DGRAM, 0);
	if (socket_descriptor == -1)
	{
		errmsg ("Socket open");
		exit (1);
	}

	bind_rc =
		bind (socket_descriptor, (struct sockaddr *) &k_sin,
		      sizeof (k_sin));
	if (bind_rc == -1)
	{
		errmsg ("Bind");
		exit (1);
	}

	save_file_flags = fcntl (socket_descriptor, F_GETFL);
	save_file_flags |= O_NONBLOCK;

	if (fcntl (socket_descriptor, F_SETFL, save_file_flags) == -1)
	{
		errmsg ("Trying non-blocking");
		exit (1);
	}

#if defined(HAVE_SETPRIORITY) && defined(PRIO_PROCESS)
	if (priority != 0)
	{
		if (setpriority (PRIO_PROCESS, getpid (), priority) < 0) 
		{
			errmsg ("Setting priority");
			exit (1);
		}
	}
#endif

	morsetext[0] = '\0';
	while (1)
	{
		udelay (500);	/*prevent 100% CPU */
		if (cwdev->footswitch)
			cw_ptt_control (!((cwdev->footswitch(cwdev))));
		if (recv_code ())
			playmorsestring (morsetext);
		/* check for ptt off timer */
		if (1 == ptt_timer_running)
		{
			gettimeofday (&now, NULL);
			if (timer_sub (&left, &end, &now) <= 0)
			{
				cwdev->ptt (cwdev, OFF);
				debug ("PTT off");
				ptt_timer_running = 0;
			}
		}
	}

	cwdev->free (cwdev);
	close_rc = close (socket_descriptor);
	if (close_rc == -1)
	{
		errmsg ("Close socket");
		exit (1);
	}
	exit (0);
}

#ifdef HAVE_DEV_PPBUS_PPI_H
/*
 * inputs	- parport linux name
 * output	- pointer to *static* char with BSD equivalent name
 * side effects - none
 */

static char *
actual_device(const char *in)
{
	static char actual_name[MAX_DEVICE];
	/*
	 * parportN
	 * 01234567
	 */
	if (strncmp(in, "parport", 7) == 0)
	{
		/* Note no possible overflow here, snprintf not needed. */
		sprintf(actual_name, "ppi%c", in[7]);
	}
	else
	{
		strncpy(actual_name, in, MAX_DEVICE-2);
		actual_name[MAX_DEVICE-1] = '\0';
	}
	return(actual_name);
}
#endif
