/* Esound.m - this file is part of Cynthiune
 *
 * Copyright (C) 2003, 2004 Wolfgang Sourdeau
 *
 * Author: Wolfgang Sourdeau <wolfgang@contre.com>
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef _REENTRANT
#define _REENTRANT 1
#endif

#import <AppKit/AppKit.h>
#import <unistd.h>
#import <esd.h>

#import <CynthiuneBundle.h>
#import <Output.h>
#import <Preference.h>
#import <utils.h>

#import "Esound.h"
#import "EsoundPreference.h"

#define LOCALIZED(X) _b ([Esound class], X)

static NSNotificationCenter *nc = nil;
static NSArray *loopModes;

@implementation Esound : NSObject

+ (void) initialize
{
  nc = [NSNotificationCenter defaultCenter];
  loopModes = [NSArray arrayWithObjects: NSDefaultRunLoopMode,
                       NSEventTrackingRunLoopMode, nil];
  [loopModes retain];
}

+ (void) unload
{
  [loopModes release];
}

- (void) setParentPlayer: (id) aPlayer;
{
  parentPlayer = aPlayer;
}

+ (NSArray *) bundleClasses
{
  return [NSArray arrayWithObjects:
                    [self class],
                  [EsoundPreference class],
                  nil];
}

- (id) init
{
  if ((self = [super init]))
    {
      esd = nil;
    }

  return self;
}

- (int) _openSocket
{
  int esdSocket;
  NSString *hostString;
  EsoundPreference *esoundPreference;

  esoundPreference = [EsoundPreference instance];

  if ([esoundPreference socketIsTCP])
    {
      hostString = [esoundPreference tcpHostConnectString];
      esdSocket = esd_play_stream ((ESD_BITS16 | ESD_STREAM
                                    | ESD_PLAY | channels),
                                   rate, [hostString cString],
                                   "Cynthiune stream");
    }
  else
    esdSocket = esd_play_stream_fallback ((ESD_BITS16 | ESD_STREAM
                                           | ESD_PLAY | channels),
                                          rate, NULL,
                                          "Cynthiune stream");

  return esdSocket;
}

- (BOOL) openDevice
{
  int esdSocket;

  esdSocket = [self _openSocket];
  if (esdSocket > -1)
    {
      esd = [[NSFileHandle alloc] initWithFileDescriptor: esdSocket];
      if (esd)
        [nc addObserver: self
            selector: @selector (_writeCompleteNotification:)
            name: GSFileHandleWriteCompletionNotification
            object: esd];
      else
        close (esdSocket);
    }

  return (esd != nil);
}

- (void) closeDevice
{
  [esd closeFile];
  [esd release];
  esd = nil;
}

- (BOOL) prepareDeviceWithChannels: (unsigned int) numberOfChannels
                           andRate: (unsigned long) sampleRate
{
  BOOL result;

  result = YES;
  rate = sampleRate;

  switch (numberOfChannels)
    {
    case 1:
      channels = ESD_MONO;
      break;
    case 2:
      channels = ESD_STEREO;
      break;
    default:
      result = NO;
    }

  if (result && esd)
    {
      [self closeDevice];
      result = [self openDevice];
    }

  return result;
}

- (void) _writeCompleteNotification: (NSNotification *) aNotification
{
  [parentPlayer chunkFinishedPlaying];
}

- (void) playChunk: (NSData *) chunk
{
  bytes = 0;
  [esd writeInBackgroundAndNotify: chunk forModes: loopModes];
  bytes = [chunk length];
}

- (unsigned int) outputBytes
{
  return bytes;
}

@end
