// This file may be redistributed and modified only under the terms of
// the GNU General Public License (See COPYING for details).
// Copyright (C) 2000,2001 Alistair Riddoch

#include "Thing.h"
#include "Script.h"

#include <common/log.h>
#include <common/const.h>
#include <common/debug.h>
#include <common/inheritance.h>
#include <common/BaseWorld.h>

#include <common/Setup.h>
#include <common/Tick.h>
#include <common/Nourish.h>
#include <common/Burn.h>

#include <Atlas/Objects/Operation/Create.h>
#include <Atlas/Objects/Operation/Sight.h>
#include <Atlas/Objects/Operation/Set.h>
#include <Atlas/Objects/Operation/Delete.h>
#include <Atlas/Objects/Operation/Move.h>
#include <Atlas/Objects/Operation/Look.h>
#include <Atlas/Objects/Operation/Appearance.h>
#include <Atlas/Objects/Operation/Disappearance.h>

static const bool debug_flag = false;

Thing::Thing()
{
    subscribe("setup", OP_SETUP);
    subscribe("action", OP_ACTION);
    subscribe("create", OP_CREATE);
    subscribe("delete", OP_DELETE);
    subscribe("burn", OP_BURN);
    subscribe("move", OP_MOVE);
    subscribe("set", OP_SET);
    subscribe("look", OP_LOOK);
}

Thing::~Thing() { }

void Thing::scriptSubscribe(const std::string & op)
{
    OpNo n = Inheritance::instance().opEnumerate(op);
    if (n != OP_INVALID) {
        debug(std::cout << "SCRIPT requesting subscription to " << op
                        << std::endl << std::flush;);
        subscribe(op, n);
    } else {
        std::string msg = std::string("SCRIPT requesting subscription to ")
                        + op + " but inheritance could not give me a reference";
        log(ERROR, msg.c_str());
    }
}

OpVector Thing::SetupOperation(const Setup & op)
{

    // This is a bit of a hack that I am not entirely happy with.
    // We broadcast a sight of create of ourselves so that everything
    // nearby can see us. This is to get round the fact that the sight
    // of create broadcast by the entity that created us may have
    // been elsewhere on the map.
    RootOperation * sight = new Sight(Sight::Instantiate());
    Create c(Create::Instantiate());
    c.SetArgs(Fragment::ListType(1,asObject()));
    c.SetTo(getId());
    c.SetFrom(getId());
    sight->SetArgs(Fragment::ListType(1, c.AsObject()));

    OpVector sres;
    if (script->Operation("setup", op, sres) != 0) {
        sres.push_back(sight);
        return sres;
    }

    OpVector res(2);

    res[0] = sight;

    RootOperation * tick = new Tick(Tick::Instantiate());
    tick->SetTo(getId());

    res[1] = tick;

    return res;
}

OpVector Thing::ActionOperation(const Action & op)
{
    OpVector res;
    if (script->Operation("action", op, res) != 0) {
        return res;
    }
    RootOperation * s = new Sight(Sight::Instantiate());
    s->SetArgs(Fragment::ListType(1,op.AsObject()));
    return OpVector(1,s);
}

OpVector Thing::CreateOperation(const Create & op)
{
    OpVector res;
    if (script->Operation("create", op, res) != 0) {
        return res;
    }
    const Fragment::ListType & args = op.GetArgs();
    if (args.empty()) {
       return OpVector();
    }
    try {
        Fragment::MapType ent = args.front().AsMap();
        Fragment::MapType::const_iterator I = ent.find("parents");
        if ((I == ent.end()) || !I->second.IsList()) {
            return error(op, "Entity to be created has no type");
        }
        const Fragment::ListType & parents = I->second.AsList();
        I = ent.find("loc");
        if ((I == ent.end()) && (location.ref != NULL)) {
            ent["loc"] = location.ref->getId();
            if (ent.find("pos") == ent.end()) {
                ent["pos"] = location.coords.asObject();
            }
        }
        std::string type;
        if (parents.empty()) {
            type = "thing";
        } else {
            type = parents.front().AsString();
        }
        debug( std::cout << getId() << " creating " << type;);
        Entity * obj = world->addObject(type,ent);
        Create c(op);
        c.SetArgs(Fragment::ListType(1,obj->asObject()));
        RootOperation * s = new Sight(Sight::Instantiate());
        s->SetArgs(Fragment::ListType(1,c.AsObject()));
        // This should no longer be required as it is now handled centrally
        // s->SetRefno(op.GetSerialno());
        return OpVector(1,s);
    }
    catch (Atlas::Message::WrongTypeException) {
        log(ERROR, "EXCEPTION: Malformed object to be created");
        return error(op, "Malformed object to be created\n");
    }
    return OpVector();
}

OpVector Thing::DeleteOperation(const Delete & op)
{
    OpVector res;
    if (script->Operation("delete", op, res) != 0) {
        return res;
    }
    // The actual destruction and removal of this entity will be handled
    // by the WorldRouter
    RootOperation * s = new Sight(Sight::Instantiate());
    s->SetArgs(Fragment::ListType(1,op.AsObject()));
    return OpVector(1,s);
}

OpVector Thing::BurnOperation(const Burn & op)
{
    OpVector res;
    if (script->Operation("burn", op, res) != 0) {
        return res;
    }
    if (op.GetArgs().empty() || !op.GetArgs().front().IsMap()) {
	return error(op, "Fire op has no argument");
    }
    Fragment::MapType::const_iterator I = attributes.find("burn_speed");
    if ((I == attributes.end()) || !I->second.IsNum()) {
        return res;
    }
    double bspeed = I->second.AsNum();
    const Fragment::MapType & fire_ent = op.GetArgs().front().AsMap();
    double consumed = bspeed * fire_ent.find("status")->second.AsNum();
    Fragment::MapType self_ent;
    self_ent["id"] = getId();
    self_ent["status"] = status - (consumed / mass);

    const std::string & to = fire_ent.find("id")->second.AsString();
    Fragment::MapType nour_ent;
    nour_ent["id"] = to;
    nour_ent["mass"] = consumed;

    Set * s = new Set(Set::Instantiate());
    s->SetTo(getId());
    s->SetArgs(Fragment::ListType(1,self_ent));

    Nourish * n = new Nourish(Nourish::Instantiate());
    n->SetTo(to);
    n->SetArgs(Fragment::ListType(1,nour_ent));

    OpVector res2(2);
    res2[0] = s;
    res2[1] = n;
    return res2;
}

OpVector Thing::MoveOperation(const Move & op)
{
    debug( std::cout << "Thing::move_operation" << std::endl << std::flush;);
    seq++;
    OpVector res;
    if (script->Operation("move", op, res) != 0) {
        return res;
    }
    const Fragment::ListType & args = op.GetArgs();
    if (args.empty()) {
        debug( std::cout << "ERROR: move op has no argument" << std::endl << std::flush;);
        return OpVector();
    }
    try {
        Vector3D oldpos = location.coords;
        const Fragment::MapType & ent = args.front().AsMap();
        Fragment::MapType::const_iterator I = ent.find("loc");
        if ((I == ent.end()) || !I->second.IsString()) {
            return error(op, "Move op has no loc");
        }
        const std::string & ref = I->second.AsString();
        EntityDict::const_iterator J = world->getObjects().find(ref);
        if (J == world->getObjects().end()) {
            return error(op, "Move op loc invalid");
        }
        debug(std::cout << "{" << ref << "}" << std::endl << std::flush;);
        Entity * newref = J->second;
        if (newref == this) {
            return error(op, "Attempt by entity to move into itself");
        }
        I = ent.find("pos");
        if ((I == ent.end()) || !I->second.IsList()) {
            return error(op, "Move op has no pos");
        }

        // Up until this point nothing should have changed, but the changes
        // have all now been checked for validity.

        if (location.ref != newref) {
            location.ref->contains.erase(this);
            newref->contains.insert(this);
            location.ref = newref;
        }

        location.coords = Vector3D(I->second.AsList());
        I = ent.find("velocity");
        if (I != ent.end()) {
            location.velocity = Vector3D(I->second.AsList());
        }
        I = ent.find("orientation");
        if (I != ent.end()) {
            location.orientation = Quaternion(I->second.AsList());
        }

        RootOperation * s = new Sight(Sight::Instantiate());
        s->SetArgs(Fragment::ListType(1,op.AsObject()));
        OpVector res2(1,s);
        // I think it might be wise to send a set indicating we have changed
        // modes, but this would probably be wasteful

        // This code handles sending Appearance and Disappearance operations
        // to this entity and others to indicate if one has gained or lost
        // sight of the other because of this movement
        if (consts::enable_ranges && isPerceptive()) {
            debug(std::cout << "testing range" << std::endl;);
            EntitySet::const_iterator I = location.ref->contains.begin();
            Fragment::ListType appear, disappear;
            Fragment::MapType this_ent;
            this_ent["id"] = getId();
            this_ent["stamp"] = (double)seq;
            Fragment::ListType this_as_args(1,this_ent);
            for(;I != location.ref->contains.end(); I++) {
                const bool wasInRange = (*I)->location.inRange(oldpos, consts::sight_range);
                const bool isInRange = (*I)->location.inRange(location.coords, consts::sight_range);
                // Build appear and disappear lists, and send operations
                // Also so operations to (dis)appearing perceptive
                // entities saying that we are (dis)appearing
                if (wasInRange ^ isInRange) {
                    Fragment::MapType that_ent;
                    that_ent["id"] = (*I)->getId();
                    that_ent["stamp"] = (double)(*I)->getSeq();
                    if (wasInRange) {
                        // We are losing sight of that object
                        disappear.push_back(that_ent);
                        debug(std::cout << getId() << ": losing site of " <<(*I)->getId() << std::endl;);
                        if ((*I)->isPerceptive()) {
                            // Send operation to the entity in question so it
                            // knows it is losing sight of us.
                            Disappearance * d = new Disappearance(Disappearance::Instantiate());
                            d->SetArgs(this_as_args);
                            d->SetTo((*I)->getId());
                            res2.push_back(d);
                        }
                    } else /*if (isInRange)*/ {
                        // We are gaining sight of that object
                        appear.push_back(that_ent);
                        debug(std::cout << getId() << ": gaining site of " <<(*I)->getId() << std::endl;);
                        if ((*I)->isPerceptive()) {
                            // Send operation to the entity in question so it
                            // knows it is gaining sight of us.
                            Appearance * a = new Appearance(Appearance::Instantiate());
                            a->SetArgs(this_as_args);
                            a->SetTo((*I)->getId());
                            res2.push_back(a);
                        }
                    }
                }
            }
            if (!appear.empty()) {
                // Send an operation to ourselves with a list of entities
                // we are losing sight of
                Appearance * a = new Appearance(Appearance::Instantiate());
                a->SetArgs(appear);
                a->SetTo(getId());
                res2.push_back(a);
            }
            if (!disappear.empty()) {
                // Send an operation to ourselves with a list of entities
                // we are gaining sight of
                Disappearance * d = new Disappearance(Disappearance::Instantiate());
                d->SetArgs(disappear);
                d->SetTo(getId());
                res2.push_back(d);
            }
        }
        return res2;
    }
    catch (Atlas::Message::WrongTypeException) {
        log(ERROR, "EXCEPTION: Malformed object to be moved");
        return error(op, "Malformed object to be moved\n");
    }
    return OpVector();
}

OpVector Thing::SetOperation(const Set & op)
{
    seq++;
    OpVector res;
    if (script->Operation("set", op, res) != 0) {
        return res;
    }
    const Fragment::ListType & args = op.GetArgs();
    if (args.empty()) {
       return OpVector();
    }
    try {
        const Fragment::MapType & ent = args.front().AsMap();
        Fragment::MapType::const_iterator I;
        for (I = ent.begin(); I != ent.end(); I++) {
            set(I->first, I->second);
        }
        RootOperation * s = new Sight(Sight::Instantiate());
        s->SetArgs(Fragment::ListType(1,op.AsObject()));
        OpVector res2(1,s);
        if (status < 0) {
            RootOperation * d = new Delete(Delete::Instantiate());
            d->SetArgs(Fragment::ListType(1,this->asObject()));
            d->SetTo(getId());
            res2.push_back(d);
        }
        return res2;
    }
    catch (Atlas::Message::WrongTypeException) {
        log(ERROR, "EXCEPTION: Malformed set operation");
        return error(op, "Malformed set operation\n");
    }
    return OpVector();
}

OpVector Thing::LookOperation(const Look & op)
{
    OpVector res;
    if (script->Operation("look", op, res) != 0) {
        return res;
    }
    return BaseEntity::LookOperation(op);
}
