/* Copyright 2008 Simon Richter <Simon.Richter@hogyros.de>
 *
 * Released under the GNU General Public Licence version 3.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "execute.hpp"

#include "parallel_node.hpp"
#include "project_node.hpp"
#include "configuration_node.hpp"
#include "input_node.hpp"
#include "temporary_node.hpp"
#include "output_node.hpp"
#include "tool_node.hpp"
#include "environment_node.hpp"
#include "action_node.hpp"

#include "exceptions.hpp"

#include <iostream>
#include <sstream>

namespace dammit {

execute::execute(cache &c) : c(c)
{
	return;
}

void execute::visit(parallel_node const &n)
{
	bool failed = false;
	for(parallel_node::node_const_iterator i = n.nodes.begin();
			i != n.nodes.end(); ++i)
	{
		try
		{
			(**i).apply(*this);
		}
		catch(...)
		{
			failed = true;
		}
	}
	if(failed)
		throw build_failed();
}

void execute::visit(project_node const &n)
{
	for(project_node::node_const_iterator i = n.nodes.begin();
			i != n.nodes.end(); ++i)
		(**i).apply(*this);
}

void execute::visit(configuration_node const &n)
{
	for(configuration_node::node_const_iterator i = n.nodes.begin();
			i != n.nodes.end(); ++i)
		(**i).apply(*this);
}

void execute::visit(input_node const &)
{
	return;
}

bool check_deps(cache &c, path const &target)
{
	if(!exists(target))
		return false;

	bool uptodate = true;

	time_t target_time = last_write_time(target);
	for(cache::dependency_const_iterator i = c.dependencies.lower_bound(target);
			i != c.dependencies.upper_bound(target); ++i)
	{
		if(last_write_time(i->second) >= target_time)
		{
			uptodate = false;
			break;
		}
	}

	return uptodate;
}

void execute::visit(temporary_node const &n)
{
	if(check_deps(c, n.directory / n.filename))
		return;
	n.action->apply(*this);
}

void execute::visit(output_node const &n)
{
	if(check_deps(c, n.directory / n.filename))
		return;
	n.action->apply(*this);
}

void execute::visit(tool_node const &n)
{
	n.tool_project->apply(*this);
}

void execute::visit(environment_node const &)
{
	return;
}

void execute::visit(action_node const &n)
{
	for(action_node::input_const_iterator i = n.inputs.begin();
			i != n.inputs.end(); ++i)
		(**i).apply(*this);
	std::ostringstream cmd;

	if(!exists(n.cmd.workingdir))
		create_directories(n.cmd.workingdir);
	
	cmd << "cd " << n.cmd.workingdir.string() << " && " << n.cmd;

	std::cerr << cmd.str() << std::endl;
	if(system(cmd.str().c_str()) != 0)
		throw build_failed();
}

}
