/*
 * Danpei -- a GTK+ based Image Viewer
 * Copyright (C) 2001-2003 Shinji Moiino
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/* option_menu.c */

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <gdk/gdkx.h>
#include <gdk/gdkkeysyms.h>

#include "config.h"
#include "intl.h"
#include "main.h"
#include "option_menu.h"
#include "utils.h"
#include "version.h"

/* Constant definitions. */
#define CONFIG_READ_ERROR  99

/* Static variables declarations. */
static const gint config_file_lines = 12;
static gchar *viewer_entry_save;
static gchar *editor_entry_save;
static gchar *printer_entry_save;
static gchar *rotate_left_entry_save;
static gchar *rotate_right_entry_save;

/* Static function declarations. */
static void     option_menu_set_option_default   (AppOption*    );

static void     option_menu_create_config_dialog (AppOption*  ,
                                                  GtkWindow*    );

static void     option_menu_cb_config_ok         (GtkWidget*  ,
                                                  gpointer      );

static void     option_menu_cb_config_cancel     (GtkWidget*  ,
                                                  gpointer      );

static gboolean option_menu_cb_entry_key_pressed (GtkWidget*  ,
                                                  GdkEventKey*,
                                                  gpointer      );

static gboolean option_menu_cb_entry_focus_in    (GtkWidget*  , 
                                                  GdkEvent*   , 
                                                  gpointer      );

/* Function definitions. */
/*
 * @option_menu_init_app_option_structure
 *
 *  Initialize the AppOption structure object.
 *
 */
void option_menu_init_app_option_structure(AppOption *option) {
  option->filename = (gchar*)malloc(sizeof(gchar) * 
                                    (strlen(getenv("HOME"))     +
                                     strlen("/.danpei/.danpei") + 1));
  if (option->filename == NULL) {
    /* Out of memory. */
    fprintf(stderr, "danpei: Out of memory.\n");
    fprintf(stderr, "        option_menu.c: error -- 01.\n");
    gtk_exit(-1);
  }
  else {
    sprintf(option->filename, "%s%s", getenv("HOME"), "/.danpei/.danpei");
  }
  option->format.jpg_on           = TRUE;
  option->format.bmp_on           = TRUE;
  option->format.png_on           = TRUE;
  option->format.xpm_on           = TRUE;
  option->format.xbm_on           = TRUE;
  option->format.tif_on           = TRUE;
  option->format.gif_on           = TRUE;
  option->format.pcx_on           = TRUE;
  option->image_cache.cache_on    = TRUE;
  option->image_cache.cache_size  = OPTION_DEFAULT_CACHE_SIZE;
  option->image_cache.cache_dir   = NULL;
  option->image_cache.remain_on   = TRUE;
  option->path.viewer             = NULL;
  option->path.editor             = NULL;
  option->path.printer            = NULL;
  option->path.rotate_left        = NULL;
  option->path.rotate_right       = NULL;
  option->dialog.dir_remove       = TRUE;
  option->dialog.remove           = TRUE;
  option->dialog.overwrite        = TRUE;
  option->max_num                 = OPTION_DEFAULT_MAX_NUM;
  option->icon_size               = ICON_MIDDLE;
  option->sort                    = SORT_FILE_NAME_A;
  option->dot_file                = FALSE;
  option->auto_refresh            = TRUE;
  option->viewer.default_external = TRUE;
  option->viewer.width            = OPTION_VIEWER_WIDTH;
  option->viewer.height           = OPTION_VIEWER_HEIGHT;
  option->is_viewer_multi_process = TRUE;
  option->is_editor_multi_process = FALSE;

  /* Set command strings. */
  option_menu_set_option_default(option);

  return;
}

/*
 * @option_menu_destroy_app_option_structure
 *
 *  Destroy the AppOption structure object.
 *
 */
void option_menu_destroy_app_option_structure(AppOption *option) {
  if (option->filename != NULL) {
    free(option->filename); 
    option->filename = NULL;
  }
  if (option->image_cache.cache_dir != NULL) {
    free(option->image_cache.cache_dir);
    option->image_cache.cache_dir = NULL;
  }

  if (option->path.viewer != NULL) {
    free(option->path.viewer);
    option->path.viewer = NULL;
  }

  if (option->path.editor != NULL) {
    free(option->path.editor);
    option->path.editor = NULL;
  }

  if(option->path.printer != NULL) {
    free(option->path.printer);
    option->path.printer = NULL;
  }

  if(option->path.rotate_left != NULL) {
    free(option->path.rotate_left);
    option->path.rotate_left = NULL;
  }

  if(option->path.rotate_right != NULL) {
    free(option->path.rotate_right);
    option->path.rotate_right = NULL;
  }

  return;
}

/*
 * @option_menu_read_config_file
 *
 *  Read the configuration file (~/.danpei/.danpei) and set the options.
 *  When the ~/.danpei/.danpei does not exist, a new file will be created.
 *
 */
void option_menu_read_config_file(AppOption *option) {
  FILE      *fp;
  char      buf[1024], num[5];
  char      *ret, *cr_pos;
  gint      wkgint;
  AppOption option_bkup;
  gboolean  need_write_config;

  /* Initialize local variables. */
  need_write_config = FALSE;

  /* Back up the original AppOption structure object. */
  option_bkup       = *option;

  /* Open the configurarition file and read it. */
  if (access(option->filename, F_OK) == 0) {
    fp = fopen(option->filename, "r");
    if (fp == NULL) {
      /* Open failed. */
      *option = option_bkup;
    }
    else {
      for (wkgint = 0; wkgint < config_file_lines; wkgint++) {
        ret = fgets(buf, 1023, fp);
        if (ferror(fp) != 0) {
          /* Read error. */
          *option = option_bkup;
          wkgint = CONFIG_READ_ERROR;
          continue;
        }

        if ((ret == NULL) && (feof(fp) != 0)) {
          /* Write out .danpei since it's format would be too old.
           * CONFIG_READ_ERROR must be larger than config_file_lines.
           */
          need_write_config = TRUE;
          wkgint = CONFIG_READ_ERROR;
          continue;
        }
        else {
          switch (wkgint) {
            case 0:
              if (buf[0] == '0') {
                option->format.jpg_on = FALSE;
              }
              else {
                option->format.jpg_on = TRUE;
              }
              if (buf[1] == '0') {
                option->format.bmp_on = FALSE;
              }
              else {
                option->format.bmp_on = TRUE;
              }
              if (buf[2] == '0') {
                option->format.png_on = FALSE;
              }
              else {
                option->format.png_on = TRUE;
              }
              if (buf[3] == '0') {
                option->format.xpm_on = FALSE;
              }
              else {
                option->format.xpm_on = TRUE;
              }
              if (buf[4] == '0') {
                option->format.xbm_on = FALSE;
              }
              else {
                option->format.xbm_on = TRUE;
              }
              if (buf[5] == '0') {
                option->format.tif_on = FALSE;
              }
              else {
                option->format.tif_on = TRUE;
              }
              if (buf[6] == '0') {
                option->format.gif_on = FALSE;
              }
              else {
                option->format.gif_on = TRUE;
              }
              if (buf[7] == '0') {
                option->format.pcx_on = FALSE;
              }
              else {
                option->format.pcx_on = TRUE;
              }
              if (((buf[0] != '0') && (buf[0] != '1')) ||
                  ((buf[1] != '0') && (buf[1] != '1')) ||
                  ((buf[2] != '0') && (buf[2] != '1')) ||
                  ((buf[3] != '0') && (buf[3] != '1')) ||
                  ((buf[4] != '0') && (buf[4] != '1')) ||
                  ((buf[5] != '0') && (buf[5] != '1')) ||
                  ((buf[6] != '0') && (buf[6] != '1')) ||
                  ((buf[7] != '0') && (buf[7] != '1'))    ) {

                need_write_config = TRUE;
              }
              break;

            case 1:
              buf[1023] = '\0';
              if (buf[0] == '0') {
                option->image_cache.cache_on = FALSE;
              }
              else {
                option->image_cache.cache_on = TRUE;
              }

              num[0] = buf[1];
              num[1] = buf[2];
              num[2] = buf[3];
              num[3] = buf[4];
              num[4] = '\0';
              option->image_cache.cache_size = atoi(num);

              option->image_cache.cache_dir =
                (gchar*)malloc(sizeof(gchar) * strlen(&buf[5]) + 1);
              if (option->image_cache.cache_dir == NULL) {
                /* Out of memory. */
                fprintf(stderr, "danpei: Out of memory.\n");
                fprintf(stderr, "        option_menu.c: error -- 02.\n");
                gtk_exit(-1);
              }
              else {
                buf[1023] = '\0';
                cr_pos = strchr(&buf[5], '\n');
                if (cr_pos != NULL) { *cr_pos = '\0'; }
                strcpy(option->image_cache.cache_dir, &buf[5]);
              }
              break;

            case 2:
              buf[1023] = '\0';
              cr_pos = strchr(buf, '\n');
              if (cr_pos != NULL) { *cr_pos = '\0'; }

              if (option->path.viewer != NULL) { free(option->path.viewer); }
              option->path.viewer = strdup(buf);
              if (option->path.viewer == NULL) {
                /* Out of memory. */
                fprintf(stderr, "danpei: Out of memory.\n");
                fprintf(stderr, "        option_menu.c: error -- 03.\n");
                gtk_exit(-1);
              }
              break;
            case 3:
              buf[1023] = '\0';
              cr_pos = strchr(buf, '\n');
              if (cr_pos != NULL) { *cr_pos = '\0'; }

              if (option->path.editor != NULL) { free(option->path.editor); }
              option->path.editor = strdup(buf);
              if (option->path.editor == NULL) {
                /* Out of memory. */
                fprintf(stderr, "danpei: Out of memory.\n");
                fprintf(stderr, "        option_menu.c: error -- 04.\n");
                gtk_exit(-1);
              }
              break;
            case 4:
              buf[1023] = '\0';
              cr_pos = strchr(buf, '\n');
              if (cr_pos != NULL) { *cr_pos = '\0'; }

              if (option->path.printer != NULL) { free(option->path.printer); }
              option->path.printer = strdup(buf);
              if (option->path.printer == NULL) {
                /* Out of memory. */
                fprintf(stderr, "danpei: Out of memory.\n");
                fprintf(stderr, "        option_menu.c: error -- 05.\n");
                gtk_exit(-1);
              }
              break;
            case 5:
              if (buf[0] == '0') {
                option->dialog.dir_remove = FALSE;
              }
              else {
                option->dialog.dir_remove = TRUE;
              }
              if (buf[1] == '0') {
                option->dialog.remove = FALSE;
              }
              else {
                option->dialog.remove = TRUE;
              }
              if (buf[2] == '0') {
                option->dialog.overwrite = FALSE;
              }
              else {
                option->dialog.overwrite = TRUE;
              }
              if (((buf[0] != '0') && (buf[0] != '1')) ||
                  ((buf[1] != '0') && (buf[1] != '1')) ||
                  ((buf[2] != '0') && (buf[2] != '1'))    ) {
                need_write_config = TRUE;
              }
              break;
            case 6:
              num[0] = buf[0];
              num[1] = buf[1];
              num[2] = buf[2];
              num[3] = buf[3];
              num[4] = '\0';
              option->max_num = atoi(num);
              break;
            case 7:
              num[0] = buf[0];
              num[1] = buf[1];
              num[2] = buf[2];
              num[3] = '\0';
              option->icon_size = atoi(num);
              break;
            case 8:
              switch (buf[0]) {
                case '0' + SORT_FILE_NAME_A:
                  option->sort = SORT_FILE_NAME_A;
                  break;
                case '0' + SORT_CTIME_A:
                  option->sort = SORT_CTIME_A;
                  break;
                case '0' + SORT_ATIME_A:
                  option->sort = SORT_ATIME_A;
                  break;
                case '0' + SORT_MTIME_A:
                  option->sort = SORT_MTIME_A;
                  break;
                case '0' + SORT_FILE_NAME_D:
                  option->sort = SORT_FILE_NAME_D;
                  break;
                case '0' + SORT_CTIME_D:
                  option->sort = SORT_CTIME_D;
                  break;
                case '0' + SORT_ATIME_D:
                  option->sort = SORT_ATIME_D;
                  break;
                case '0' + SORT_MTIME_D:
                  option->sort = SORT_MTIME_D;
                  break;
                default:
                  option->sort = SORT_FILE_NAME_A;
                  break;
              }

              if (buf[1] == '0') {
                option->dot_file = FALSE;
              }
              else {
                option->dot_file = TRUE;
              }

              if (buf[2] == '0') {
                option->auto_refresh = FALSE;
              }
              else {
                option->auto_refresh = TRUE;
              }

              if (buf[3] == '0') {
                option->image_cache.remain_on = FALSE;
              }
              else {
                option->image_cache.remain_on = TRUE;
              }

              if (buf[4] == '0') {
                option->is_viewer_multi_process = FALSE;
              }
              else {
                option->is_viewer_multi_process = TRUE;
                if (buf[4] != '1') { need_write_config = TRUE; }
              }

              if (buf[5] == '1') {
                option->is_editor_multi_process = TRUE;
              }
              else {
                if (buf[5] != '0') { need_write_config = TRUE; }
                option->is_editor_multi_process = FALSE;
              }
              break;
            case 9:
              buf[1023] = '\0';
              cr_pos = strchr(buf, '\n');
              if (cr_pos != NULL) { *cr_pos = '\0'; }

              if (option->path.rotate_left != NULL) {
                free(option->path.rotate_left);
              }
              option->path.rotate_left = strdup(buf);
              if (option->path.rotate_left == NULL) {
                /* Out of memory. */
                fprintf(stderr, "danpei: Out of memory.\n");
                fprintf(stderr, "        option_menu.c: error -- 15.\n");
                gtk_exit(-1);
              }
              break;
            case 10:
              buf[1023] = '\0';
              cr_pos = strchr(buf, '\n');
              if (cr_pos != NULL) { *cr_pos = '\0'; }

              if (option->path.rotate_right != NULL) {
                free(option->path.rotate_right);
              }
              option->path.rotate_right = strdup(buf);
              if (option->path.rotate_right == NULL) {
                /* Out of memory. */
                fprintf(stderr, "danpei: Out of memory.\n");
                fprintf(stderr, "        option_menu.c: error -- 16.\n");
                gtk_exit(-1);
              }
              break;

            case 11:
              buf[1023] = '\0';
              if (buf[0] == '0') {
                option->viewer.default_external = FALSE;
              }
              else {
                option->viewer.default_external = TRUE;
                if (buf[0] != '1') { need_write_config = TRUE; }
              }
              num[0] = buf[1];
              num[1] = buf[2];
              num[2] = buf[3];
              num[3] = buf[4];
              num[4] = '\0';
              option->viewer.width = atoi(num);
              num[0] = buf[5];
              num[1] = buf[6];
              num[2] = buf[7];
              num[3] = buf[8];
              num[4] = '\0';
              option->viewer.height = atoi(num);

            default:
              break;
          }
        }
      }
    }
    fclose(fp);
    if (need_write_config == TRUE) {
      /* Write new formated configuration file. */
      option_menu_write_config_file(option);
    }
  }
  else {
    sprintf(buf, "%s%s", getenv("HOME"), "/.danpei");
    mkdir (buf, 0700);
    fp = fopen(option->filename, "w");
    if (fp == NULL) {
      /* Cretation the file failed. */
      fprintf(stderr, "danpei: fopen(%s) failed.\n", option->filename);
      fprintf(stderr, "        option_menu.c: error -- 06.\n");
      fprintf(stderr, "        <errno.h> - errno = %d\n", errno);
      gtk_exit(-1);
    }
    else {
      /* Create a new configuration file. */
      fclose(fp);
      option_menu_set_option_default(option);
      option_menu_write_config_file (option);
    }
  }

  return;
}

/*
 * @option_menu_write_config_file
 *
 *  Write the configuration file (~/.danpei/.danpei).
 *
 */
void option_menu_write_config_file(AppOption *option) {
  FILE  *fp;
  char  buf[config_file_lines][1024];
  gint  wkgint;
  gchar *wknum;

  /* Initialize the local variables. */
  wkgint = 0;
  wknum  = NULL;

  fp = fopen(option->filename, "w");
  if (fp == NULL) {
    /* Open error. */
    fprintf(stderr, "danpei: fopen(%s) failed.\n", option->filename);
    fprintf(stderr, "        option_menu.c: error -- 07.\n");
    fprintf(stderr, "        <errno.h> - errno = %d\n", errno);
    gtk_exit(-1);
  }
  else {
    /* Line #1. */
    strcpy(&buf[0][0], "11111111\n");
    if (option->format.jpg_on == FALSE) { buf[0][0] = '0'; }
    if (option->format.bmp_on == FALSE) { buf[0][1] = '0'; }
    if (option->format.png_on == FALSE) { buf[0][2] = '0'; }
    if (option->format.xpm_on == FALSE) { buf[0][3] = '0'; }
    if (option->format.xbm_on == FALSE) { buf[0][4] = '0'; }
    if (option->format.tif_on == FALSE) { buf[0][5] = '0'; }
    if (option->format.gif_on == FALSE) { buf[0][6] = '0'; }
    if (option->format.pcx_on == FALSE) { buf[0][7] = '0'; }

    /* Line #2. */
    if (option->image_cache.cache_on == FALSE) {
      buf[1][0] = '0';
    }
    else {
      buf[1][0] = '1';
    }

    utils_gint_to_gchar(option->image_cache.cache_size, 4, FALSE, &wknum);
    if (wknum == NULL) {
      /* Can't convert the size to chars. */
      fprintf(stderr, "danpei: Can't get the size.\n");
      fprintf(stderr, "        option_menu.c: error -- 08.\n");
      fprintf(stderr, "        size = %d\n", option->image_cache.cache_size);
      gtk_exit(-1);
    }
    else {
      strcpy(&buf[1][1], wknum);
      free(wknum);
    }

    if (strlen(option->image_cache.cache_dir) < 1018) {
      sprintf(&buf[1][5], "%s\n", option->image_cache.cache_dir);
    }
    else {
      strncpy(&buf[1][5], option->image_cache.cache_dir, 1017);
      buf[1][1022] = '\n';
      buf[1][1023] = '\0';
    }

    /* Line #3. */
    if (strlen(option->path.viewer) < 1023) {
      sprintf(&buf[2][0], "%s\n", option->path.viewer);
    }
    else {
      strncpy(&buf[2][0], option->path.viewer, 1022);
      buf[2][1022] = '\n';
      buf[2][1023] = '\0';
    }

    /* Line #4. */
    if (strlen(option->path.editor) < 1023) {
      sprintf(&buf[3][0], "%s\n", option->path.editor);
    }
    else {
      strncpy(&buf[3][0], option->path.editor, 1022);
      buf[3][1022] = '\n';
      buf[3][1023] = '\0';
    }

    /* Line #5. */
    if (strlen(option->path.printer) < 1023) {
      sprintf(&buf[4][0], "%s\n", option->path.printer);
    }
    else {
      strncpy(&buf[4][0], option->path.printer, 1022);
      buf[4][1022] = '\n';
      buf[4][1023] = '\0';
    }

    /* Line #6. */
    strcpy(&buf[5][0], "111\n");
    if (option->dialog.dir_remove == FALSE) { buf[5][0] = '0'; }
    if (option->dialog.remove     == FALSE) { buf[5][1] = '0'; }
    if (option->dialog.overwrite  == FALSE) { buf[5][2] = '0'; }

    /* Line #7. */
    utils_gint_to_gchar(option->max_num, 4, FALSE, &wknum);
    if (wknum == NULL) {
      /* Can't convert the size to chars. */
      fprintf(stderr, "danpei: Can't get the size.\n");
      fprintf(stderr, "        option_menu.c: error -- 09.\n");
      fprintf(stderr, "        size = %d\n", option->max_num);
      gtk_exit(-1);
    }
    else {
      sprintf(&buf[6][0], "%s\n", wknum);
      free(wknum);
    }

    /* Line #8. */
    utils_gint_to_gchar(option->icon_size, 3, FALSE, &wknum);
    if (wknum == NULL) {
      /* Can't convert the size to chars. */
      fprintf(stderr, "danpei: Can't get the size.\n");
      fprintf(stderr, "        option_menu.c: error -- 10.\n");
      fprintf(stderr, "        size = %d\n", option->icon_size);
      gtk_exit(-1);
    }
    else {
      sprintf(&buf[7][0], "%s\n", wknum);
      free(wknum);
    }

    /* Line #9. */
    strcpy(&buf[8][0], "011110\n");
    buf[8][0] = '0' + option->sort;
    if (option->dot_file                == FALSE) { buf[8][1] = '0'; }
    if (option->auto_refresh            == FALSE) { buf[8][2] = '0'; }
    if (option->image_cache.remain_on   == FALSE) { buf[8][3] = '0'; }
    if (option->is_viewer_multi_process == FALSE) { buf[8][4] = '0'; }
    if (option->is_editor_multi_process == TRUE ) { buf[8][5] = '1'; }

    /* Line #10. */
    if (strlen(option->path.rotate_left) < 1023) {
      sprintf(&buf[9][0], "%s\n", option->path.rotate_left);
    }
    else {
      strncpy(&buf[9][0], option->path.rotate_left, 1022);
      buf[9][1022] = '\n';
      buf[9][1023] = '\0';
    }

    /* Line #11. */
    if (strlen(option->path.rotate_right) < 1023) {
      sprintf(&buf[10][0], "%s\n", option->path.rotate_right);
    }
    else {
      strncpy(&buf[10][0], option->path.rotate_right, 1022);
      buf[10][1022] = '\n';
      buf[10][1023] = '\0';
    }

    /* Line #12. */
    if (option->viewer.default_external == FALSE) {
      buf[11][0] = '0';
    }
    else {
      buf[11][0] = '1';
    }

    utils_gint_to_gchar(option->viewer.width, 4, FALSE, &wknum);
    if (wknum == NULL) {
      /* Can't convert the size to chars. */
      fprintf(stderr, "danpei: Can't get the size.\n");
      fprintf(stderr, "        option_menu.c: error -- 08.\n");
      fprintf(stderr, "        size = %d\n", option->image_cache.cache_size);
      gtk_exit(-1);
    }
    else {
      strcpy(&buf[11][1], wknum);
      free(wknum);
    }

    utils_gint_to_gchar(option->viewer.height, 4, FALSE, &wknum);
    if (wknum == NULL) {
      /* Can't convert the size to chars. */
      fprintf(stderr, "danpei: Can't get the size.\n");
      fprintf(stderr, "        option_menu.c: error -- 08.\n");
      fprintf(stderr, "        size = %d\n", option->image_cache.cache_size);
      gtk_exit(-1);
    }
    else {
      strcpy(&buf[11][5], wknum);
      free(wknum);
    }

    /* Write file. */
    for (wkgint = 0; wkgint < config_file_lines; wkgint++) {
      if((fputs(&buf[wkgint][0], fp)) == EOF) {
        /* Write error. */
        fclose(fp);
        fprintf(stderr, "danpei: fputs() failed.\n");
        fprintf(stderr, "        option_menu.c: error -- 11.\n");
        fprintf(stderr, "        <errno.h> - errno = %d.\n", errno);
        fprintf(stderr, "        buf = %s\n", &buf[wkgint][0]);
        gtk_exit(-1);
      }
    }
    fclose(fp);
  }

  fprintf(stderr, "Danpei: Wrote configuration file => ~/.danpei.\n");

  return;
}

/* Static function definitions. */
static void option_menu_set_option_default(AppOption *option) {
  option->format.jpg_on          = TRUE;
  option->format.bmp_on          = TRUE;
  option->format.png_on          = TRUE;
  option->format.xpm_on          = TRUE;
  option->format.xbm_on          = TRUE;
  option->format.tif_on          = TRUE;
  option->format.gif_on          = TRUE;
  option->format.pcx_on          = TRUE;
  option->image_cache.cache_on   = TRUE;
  option->image_cache.cache_size = OPTION_DEFAULT_CACHE_SIZE;
  option->image_cache.cache_dir  = 
       (gchar*)malloc(sizeof(gchar) *
                      (strlen(getenv("HOME")) + strlen("/.danpei") + 1));
  if (option->image_cache.cache_dir == NULL) {
    /* Out of memory. */
    fprintf(stderr, "danpei: Out of memory.\n");
    fprintf(stderr, "        option_menu.c: error -- 12.\n");
    gtk_exit(-1);
  }
  else {
    sprintf(option->image_cache.cache_dir, "%s%s",
                                           getenv("HOME"), "/.danpei");
  }
  option->image_cache.remain_on  = TRUE;

  /* Free the memory of command path. */
  if (option->path.viewer != NULL) {
    free(option->path.viewer);
    option->path.viewer = NULL;
  }
  if (option->path.editor != NULL) {
    free(option->path.editor);
    option->path.editor = NULL;
  }
  if(option->path.printer != NULL) {
    free(option->path.printer);
    option->path.printer = NULL;
  }
  if(option->path.rotate_left != NULL) {
    free(option->path.rotate_left);
    option->path.rotate_left = NULL;
  }
  if(option->path.rotate_right != NULL) {
    free(option->path.rotate_right);
    option->path.rotate_right = NULL;
  }
  option->path.viewer       = strdup(OPTION_DEFAULT_VIEWER);
  option->path.editor       = strdup(OPTION_DEFAULT_EDITOR);
  option->path.printer      = strdup(OPTION_DEFAULT_PRINTER);
  option->path.rotate_left  = strdup(OPTION_DEFAULT_ROTATE_LEFT);
  option->path.rotate_right = strdup(OPTION_DEFAULT_ROTATE_RIGHT);
  if ((option->path.viewer       == NULL) ||
      (option->path.editor       == NULL) ||
      (option->path.printer      == NULL) ||
      (option->path.rotate_left  == NULL) ||
      (option->path.rotate_right == NULL)    ) {
    /* Out of memory. */
    free(option->image_cache.cache_dir);
    if (option->path.viewer       != NULL) { free(option->path.viewer      ); }
    if (option->path.editor       != NULL) { free(option->path.editor      ); }
    if (option->path.printer      != NULL) { free(option->path.printer     ); }
    if (option->path.rotate_left  != NULL) { free(option->path.rotate_left ); }
    if (option->path.rotate_right != NULL) { free(option->path.rotate_right); }
    fprintf(stderr, "danpei: Out of memory.\n");
    fprintf(stderr, "        option_menu.c: error -- 13.\n");
    gtk_exit(-1);
  }
  option->dialog.dir_remove       = TRUE;
  option->dialog.remove           = TRUE;
  option->dialog.overwrite        = TRUE;
  option->max_num                 = OPTION_DEFAULT_MAX_NUM;
  option->icon_size               = ICON_MIDDLE;
  option->sort                    = SORT_FILE_NAME_A;
  option->dot_file                = FALSE;
  option->auto_refresh            = TRUE;
  option->viewer.default_external = TRUE;
  option->viewer.width            = OPTION_VIEWER_WIDTH;
  option->viewer.height           = OPTION_VIEWER_HEIGHT;
  option->is_viewer_multi_process = TRUE;
  option->is_editor_multi_process = FALSE;

  return;
}

/*
 * @option_menu_cb_configuration
 *
 *  Display the configuration dialog and write ~/.danpei/.danpei.
 *
 */
void option_menu_cb_configuration(GtkWidget *widget,
                                  gpointer  data   ,
                                  guint     action   ) {
  TopLevel *tp;

  tp = (TopLevel*)data;

  option_menu_create_config_dialog(&(tp->app_option), 
                                   GTK_WINDOW(tp->window));

  return;
}


/* Static function definitions. */
/*
 * @option_menu_create_config_dialog
 *
 *  Display the configuration dialog and write ~/.danpei/.danpei.
 *  -- called by option_menu_cb_configuration().
 *
 */
static void option_menu_create_config_dialog(AppOption *option,
                                             GtkWindow *window  ) {
  GtkWidget *dialog;
  GtkWidget *notebook;
  GtkWidget *label;
  GtkWidget *frame;
  GtkWidget *vbox, *vbox1, *vbox2;
  GtkWidget *hbox;
  GSList    *group1, *group2, *group3, *group4;

  GtkWidget *format_jpg_btn;
  GtkWidget *format_bmp_btn;
  GtkWidget *format_png_btn;
  GtkWidget *format_xpm_btn;
  GtkWidget *format_xbm_btn;
  GtkWidget *format_tif_btn;
  GtkWidget *format_gif_btn;
  GtkWidget *format_pcx_btn;

  GtkWidget *use_cache_btn;
  GtkWidget *unuse_cache_btn;
  GtkWidget *remain_cache_btn;

  GtkWidget *icon_small_btn;
  GtkWidget *icon_middle_btn;
  GtkWidget *icon_large_btn;

  GtkWidget *sort_ascend_btn;
  GtkWidget *sort_descend_btn;
  GtkWidget *sort_file_btn;
  GtkWidget *sort_ctime_btn;
  GtkWidget *sort_atime_btn;
  GtkWidget *sort_mtime_btn;

  GtkObject *adj1,*adj2, *adj3;
  GtkWidget *spinner1, *spinner2, *spinner3;

  GtkWidget *external_btn;
  GtkWidget *multi_viewer_btn;
  GtkWidget *multi_editor_btn;
  GtkWidget *viewer_entry;
  GtkWidget *editor_entry;
  GtkWidget *printer_entry;
  GtkWidget *rotate_left_entry;
  GtkWidget *rotate_right_entry;

  GtkWidget *dot_file_btn;
  GtkWidget *dir_remove_dialog_btn;
  GtkWidget *remove_dialog_btn;
  GtkWidget *overwrite_dialog_btn;
  GtkWidget *auto_refresh_btn;

  GtkWidget *ok_button;
  GtkWidget *cancel_button;

  gboolean  ret;

  /* Initialize the local variables. */
   group1 = group2 = group3 = group4 = NULL;

  /* Create the configuration dialog. */
  dialog = gtk_dialog_new();
  gtk_window_set_title(GTK_WINDOW(dialog), _("Configuration"));
  gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, FALSE, FALSE);
  gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);
  gtk_window_set_transient_for(GTK_WINDOW(dialog), window);
  gtk_signal_connect(GTK_OBJECT(dialog), "delete_event",
                     GTK_SIGNAL_FUNC(option_menu_cb_config_cancel), &ret);

  hbox = gtk_hbox_new(TRUE, 0);
  gtk_widget_show(hbox);
  gtk_container_border_width(GTK_CONTAINER(hbox), 8);
  notebook = gtk_notebook_new();
  gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
  gtk_widget_show(notebook);
  gtk_box_pack_start(GTK_BOX(hbox), notebook, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox, TRUE, TRUE, 0);

  /* Create "Thumbnail" page. */
  vbox1 = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(vbox1), 4);
  gtk_widget_show(vbox1);

  /* Create "Thumbnail::Format" frame. */
  frame = gtk_frame_new(_("Format"));
  gtk_container_border_width(GTK_CONTAINER(frame), 4);
  gtk_widget_show(frame);
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(vbox);

  format_jpg_btn = gtk_check_button_new_with_label("JPEG  ");
  if (option->format.jpg_on == TRUE) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(format_jpg_btn), TRUE);
  }
  else {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(format_jpg_btn), FALSE);
  }
  gtk_widget_show(format_jpg_btn);

  format_bmp_btn = gtk_check_button_new_with_label("BMP   ");
  if (option->format.bmp_on == TRUE) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(format_bmp_btn), TRUE);
  }
  else {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(format_bmp_btn), FALSE);
  }
  gtk_widget_show(format_bmp_btn);

  format_png_btn = gtk_check_button_new_with_label("PNG   ");
  if (option->format.png_on == TRUE) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(format_png_btn), TRUE);
  }
  else {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(format_png_btn), FALSE);
  }
  gtk_widget_show(format_png_btn);

  format_xpm_btn = gtk_check_button_new_with_label("XPM   ");
  if (option->format.xpm_on == TRUE) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(format_xpm_btn), TRUE);
  }
  else {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(format_xpm_btn), FALSE);
  }
  gtk_widget_show(format_xpm_btn);

  format_xbm_btn = gtk_check_button_new_with_label("XBM   ");
  if (option->format.xbm_on == TRUE) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(format_xbm_btn), TRUE);
  }
  else {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(format_xbm_btn), FALSE);
  }
  gtk_widget_show(format_xbm_btn);

  format_tif_btn = gtk_check_button_new_with_label("TIFF  ");
  if (option->format.tif_on == TRUE) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(format_tif_btn), TRUE);
  }
  else {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(format_tif_btn), FALSE);
  }
  gtk_widget_show(format_tif_btn);

  format_gif_btn = gtk_check_button_new_with_label("GIF   ");
  if (option->format.gif_on == TRUE) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(format_gif_btn), TRUE);
  }
  else {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(format_gif_btn), FALSE);
  }
  gtk_widget_show(format_gif_btn);

  format_pcx_btn = gtk_check_button_new_with_label("PCX   ");
  if (option->format.pcx_on == TRUE) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(format_pcx_btn), TRUE);
  }
  else {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(format_pcx_btn), FALSE);
  }
  gtk_widget_show(format_pcx_btn);

  hbox = gtk_hbox_new(FALSE, 2);
  gtk_widget_show(hbox);
  gtk_box_pack_start(GTK_BOX(hbox), format_jpg_btn, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), format_bmp_btn, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), format_tif_btn, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), format_gif_btn, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

  hbox = gtk_hbox_new(FALSE, 2);
  gtk_widget_show(hbox);
  gtk_box_pack_start(GTK_BOX(hbox), format_png_btn, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), format_xpm_btn, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), format_xbm_btn, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), format_pcx_btn, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

  gtk_container_add(GTK_CONTAINER(frame), vbox);
  gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);

  /* Create 'Thumbnail::Cache' frame. */
  frame = gtk_frame_new(_("Cache"));
  gtk_container_border_width(GTK_CONTAINER(frame), 4);
  gtk_widget_show(frame);
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(vbox);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  gtk_widget_show(hbox);

  use_cache_btn = gtk_radio_button_new_with_label(NULL, _("Use    "));
  group1 = gtk_radio_button_group(GTK_RADIO_BUTTON(use_cache_btn));
  if (option->image_cache.cache_on == TRUE) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(use_cache_btn), TRUE);
  }
  gtk_widget_show(use_cache_btn);
  gtk_box_pack_start(GTK_BOX(hbox), use_cache_btn, FALSE, FALSE, 0);

  unuse_cache_btn = gtk_radio_button_new_with_label(group1, _("Not use"));
  group1 = gtk_radio_button_group(GTK_RADIO_BUTTON(unuse_cache_btn));
  if (option->image_cache.cache_on != TRUE) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(unuse_cache_btn), TRUE);
  }
  gtk_widget_show(unuse_cache_btn);
  gtk_box_pack_start(GTK_BOX(hbox), unuse_cache_btn, FALSE, FALSE, 0);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  gtk_widget_show(hbox);
  remain_cache_btn =
      gtk_check_button_new_with_label(
               _("Re-use cache files next time."));
  if (option->image_cache.remain_on == TRUE) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(remain_cache_btn), TRUE);
  }
  else {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(remain_cache_btn), FALSE);
  }
  gtk_box_pack_start(GTK_BOX(hbox), remain_cache_btn, FALSE, FALSE, 0);
  gtk_widget_show(remain_cache_btn);

  hbox = gtk_hbox_new(FALSE, 2);
  gtk_widget_show(hbox);
  gtk_container_border_width(GTK_CONTAINER(hbox), 4);
  label = gtk_label_new(_("Size: "));
  gtk_widget_show(label);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

  adj1 = gtk_adjustment_new(option->image_cache.cache_size,
                            OPTION_SYSTEM_MIN_CACHE,
                            OPTION_SYSTEM_MAX_CACHE, 1.0, 10.0, 10.0);
  spinner1 = gtk_spin_button_new(GTK_ADJUSTMENT(adj1), 5, 0);
  gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spinner1), FALSE);
  gtk_widget_set_usize(spinner1, 60, 20);
  gtk_widget_show(spinner1);
  gtk_box_pack_start(GTK_BOX(hbox), spinner1, FALSE, FALSE, 0);

  label = gtk_label_new("x 100KB");
  gtk_widget_show(label);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox , FALSE, FALSE, 0);

  gtk_container_add(GTK_CONTAINER(frame), vbox);
  gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);

  /* Create 'Thumbnail::Icon Size' frame. */
  frame = gtk_frame_new(_("Icon Size"));
  gtk_container_border_width(GTK_CONTAINER(frame), 4);
  gtk_widget_show(frame);
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(vbox);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  gtk_widget_show(hbox);

  icon_small_btn = gtk_radio_button_new_with_label(NULL, _("Small "));
  group2 = gtk_radio_button_group(GTK_RADIO_BUTTON(icon_small_btn));
  if (option->icon_size == ICON_SMALL) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(icon_small_btn), TRUE);
  }
  gtk_widget_show(icon_small_btn);
  gtk_box_pack_start(GTK_BOX(hbox), icon_small_btn, FALSE, FALSE, 0);

  icon_middle_btn = gtk_radio_button_new_with_label(group2, _("Middle"));
  group2 = gtk_radio_button_group(GTK_RADIO_BUTTON(icon_middle_btn));
  if (option->icon_size == ICON_MIDDLE) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(icon_middle_btn), TRUE);
  }
  gtk_widget_show(icon_middle_btn);
  gtk_box_pack_start(GTK_BOX(hbox), icon_middle_btn, FALSE, FALSE, 0);

  icon_large_btn = gtk_radio_button_new_with_label(group2, _("Large "));
  group2 = gtk_radio_button_group(GTK_RADIO_BUTTON(icon_large_btn));
  if (option->icon_size == ICON_LARGE) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(icon_large_btn), TRUE);
  }
  gtk_widget_show(icon_large_btn);
  gtk_box_pack_start(GTK_BOX(hbox), icon_large_btn, FALSE, FALSE, 0);

  gtk_container_add(GTK_CONTAINER(frame), vbox);
  gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);

  /* Create 'Thumbnail::Sort order' frame. */
  frame = gtk_frame_new(_("Sort order"));
  gtk_container_border_width(GTK_CONTAINER(frame), 4);
  gtk_widget_show(frame);
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(vbox);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  gtk_widget_show(hbox);

  sort_ascend_btn = gtk_radio_button_new_with_label(NULL, _("Ascend"));
  group3 = gtk_radio_button_group(GTK_RADIO_BUTTON(sort_ascend_btn));
  if (option->sort < SORT_FILE_NAME_D) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(sort_ascend_btn), TRUE);
  }
  gtk_widget_show(sort_ascend_btn);
  gtk_box_pack_start(GTK_BOX(hbox), sort_ascend_btn, FALSE, FALSE, 0);

  sort_descend_btn = gtk_radio_button_new_with_label(group3, _("Descend"));
  group3 = gtk_radio_button_group(GTK_RADIO_BUTTON(sort_descend_btn));
  if (option->sort > SORT_MTIME_A) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(sort_descend_btn), TRUE);
  }
  gtk_widget_show(sort_descend_btn);
  gtk_box_pack_start(GTK_BOX(hbox), sort_descend_btn, FALSE, FALSE, 0);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  gtk_widget_show(hbox);

  sort_file_btn = gtk_radio_button_new_with_label(NULL, _("File name"));
  group4 = gtk_radio_button_group(GTK_RADIO_BUTTON(sort_file_btn));
  if ((option->sort == SORT_FILE_NAME_A) ||
      (option->sort == SORT_FILE_NAME_D)    ){
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(sort_file_btn), TRUE);
  }
  gtk_widget_show(sort_file_btn);
  gtk_box_pack_start(GTK_BOX(hbox), sort_file_btn, FALSE, FALSE, 0);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  gtk_widget_show(hbox);

  sort_ctime_btn = gtk_radio_button_new_with_label(group4, 
                                                   _("Created time"));
  group4 = gtk_radio_button_group(GTK_RADIO_BUTTON(sort_ctime_btn));
  if ((option->sort == SORT_CTIME_A) ||
      (option->sort == SORT_CTIME_D)    ) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(sort_ctime_btn), TRUE);
  }
  gtk_widget_show(sort_ctime_btn);
  gtk_box_pack_start(GTK_BOX(hbox), sort_ctime_btn, FALSE, FALSE, 0);

  sort_atime_btn = gtk_radio_button_new_with_label(group4, 
                                                   _("Access time"));
  group4 = gtk_radio_button_group(GTK_RADIO_BUTTON(sort_atime_btn));
  if ((option->sort == SORT_ATIME_A) ||
      (option->sort == SORT_ATIME_D)    ) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(sort_atime_btn), TRUE);
  }
  gtk_widget_show(sort_atime_btn);
  gtk_box_pack_start(GTK_BOX(hbox), sort_atime_btn, FALSE, FALSE, 0);

  sort_mtime_btn = gtk_radio_button_new_with_label(group4, 
                                                   _("Modified time"));
  group4 = gtk_radio_button_group(GTK_RADIO_BUTTON(sort_mtime_btn));
  if ((option->sort == SORT_MTIME_A) ||
      (option->sort == SORT_MTIME_D)    ) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(sort_mtime_btn), TRUE);
  }
  gtk_widget_show(sort_mtime_btn);
  gtk_box_pack_start(GTK_BOX(hbox), sort_mtime_btn, FALSE, FALSE, 0);

  gtk_container_add(GTK_CONTAINER(frame), vbox);
  gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);

  /* Create 'Thumbnail::Maximum' frame. */
  frame = gtk_frame_new(_("Maximum"));
  gtk_container_border_width(GTK_CONTAINER(frame), 4);
  gtk_widget_show(frame);
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(vbox);

  hbox = gtk_hbox_new(FALSE, 2);
  gtk_widget_show(hbox);
  gtk_container_border_width(GTK_CONTAINER(hbox), 4);
  label = gtk_label_new(_("Thumbnails per page: "));
  gtk_widget_show(label);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

  adj2 = gtk_adjustment_new(option->max_num,
                            OPTION_SYSTEM_MIN_NUM,
                            OPTION_SYSTEM_MAX_NUM, 1.0, 10.0, 10.0);
  spinner2 = gtk_spin_button_new(GTK_ADJUSTMENT(adj2), 5, 0);
  gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spinner2), FALSE);
  gtk_widget_set_usize(spinner2, 60, 20);
  gtk_widget_show(spinner2);
  gtk_box_pack_start(GTK_BOX(hbox), spinner2, FALSE, FALSE, 0);

  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  gtk_container_add(GTK_CONTAINER(frame), vbox);
  gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);

  label = gtk_label_new(_("Thumbnail "));
  gtk_notebook_append_page(GTK_NOTEBOOK (notebook), vbox1, label);

  /* Create 'Command' page. */
  vbox1 = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(vbox1), 4);
  gtk_widget_show(vbox1);

  /* Create 'Command::Editor' frame. */
  frame = gtk_frame_new(_("External editor"));
  gtk_container_border_width(GTK_CONTAINER(frame), 4);
  gtk_widget_show(frame);
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(vbox), 4);
  gtk_widget_show(vbox);

  vbox2 = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(vbox2), 2);
  external_btn  = 
      gtk_check_button_new_with_label(
               _("Start external viewer when double clicked."));
  if (option->viewer.default_external == TRUE) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(external_btn), TRUE);
  }
  else {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(external_btn), FALSE);
  }
  multi_viewer_btn  = 
      gtk_check_button_new_with_label(
               _("Raise viewer process as many as selected files."));
  if (option->is_viewer_multi_process == TRUE) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(multi_viewer_btn), TRUE);
  }
  else {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(multi_viewer_btn), FALSE);
  }
  multi_editor_btn  = 
      gtk_check_button_new_with_label(
               _("Raise editor process as many as selected files."));
  if (option->is_editor_multi_process == TRUE) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(multi_editor_btn), TRUE);
  }
  else {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(multi_editor_btn), FALSE);
  }
  gtk_box_pack_start(GTK_BOX(vbox2), external_btn    , FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox2), multi_viewer_btn, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox2), multi_editor_btn, FALSE, FALSE, 0);
  gtk_widget_show(external_btn);
  gtk_widget_show(multi_viewer_btn);
  gtk_widget_show(multi_editor_btn);
  gtk_widget_show(vbox2);
  gtk_box_pack_start(GTK_BOX(vbox), vbox2, TRUE, TRUE, 0);

  hbox = gtk_hbox_new(FALSE, 2);
  label = gtk_label_new(_("Viewer "));
  gtk_widget_show(label);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

  viewer_entry = gtk_entry_new();
  gtk_signal_connect(GTK_OBJECT(viewer_entry), "key_press_event",
                     GTK_SIGNAL_FUNC(option_menu_cb_entry_key_pressed),
                     VIEWER_ENTRY);
  gtk_signal_connect(GTK_OBJECT(viewer_entry), "focus_in_event",
                     GTK_SIGNAL_FUNC(option_menu_cb_entry_focus_in),
                     VIEWER_ENTRY);
  gtk_widget_set_usize(viewer_entry, 220, 20);
  gtk_entry_set_text(GTK_ENTRY(viewer_entry), option->path.viewer);
  gtk_editable_set_position(GTK_EDITABLE(viewer_entry), 0);
  gtk_widget_show(viewer_entry);
  gtk_box_pack_start(GTK_BOX(hbox), viewer_entry, TRUE, TRUE, 0);
  gtk_widget_show(hbox);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);

  hbox = gtk_hbox_new(FALSE, 2);
  label = gtk_label_new(_("Editor "));
  gtk_widget_show(label);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

  editor_entry = gtk_entry_new();
  gtk_signal_connect(GTK_OBJECT(editor_entry), "key_press_event",
                     GTK_SIGNAL_FUNC(option_menu_cb_entry_key_pressed), 
                     (gpointer)EDITOR_ENTRY);
  gtk_signal_connect(GTK_OBJECT(editor_entry), "focus_in_event",
                     GTK_SIGNAL_FUNC(option_menu_cb_entry_focus_in),
                     (gpointer)EDITOR_ENTRY);
  gtk_entry_set_text(GTK_ENTRY(editor_entry), option->path.editor);
  gtk_editable_set_position(GTK_EDITABLE(editor_entry), 0);
  gtk_widget_set_usize(editor_entry, 220, 20);
  gtk_widget_show(editor_entry);
  gtk_box_pack_start(GTK_BOX(hbox), editor_entry, TRUE, TRUE, 0);
  gtk_widget_show(hbox);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 2);

  gtk_container_add(GTK_CONTAINER(frame), vbox);
  gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);

  /* Create 'Command::Print' frame. */
  frame = gtk_frame_new(_("Print"));
  gtk_container_border_width(GTK_CONTAINER(frame), 4);
  gtk_widget_show(frame);
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(vbox), 4);
  gtk_widget_show(vbox);

  hbox = gtk_hbox_new(FALSE, 2);
  label = gtk_label_new(_("Command "));
  gtk_widget_show(label);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

  printer_entry = gtk_entry_new();
  gtk_signal_connect(GTK_OBJECT(printer_entry), "key_press_event",
                     GTK_SIGNAL_FUNC(option_menu_cb_entry_key_pressed), 
                     (gpointer)PRINTER_ENTRY);
  gtk_signal_connect(GTK_OBJECT(printer_entry), "focus_in_event",
                     GTK_SIGNAL_FUNC(option_menu_cb_entry_focus_in),
                     (gpointer)PRINTER_ENTRY);
  gtk_widget_set_usize(printer_entry, 220, 20);
  gtk_entry_set_text(GTK_ENTRY(printer_entry), option->path.printer);
  gtk_editable_set_position(GTK_EDITABLE(printer_entry), 0);
  gtk_widget_show(printer_entry);
  gtk_box_pack_start(GTK_BOX(hbox), printer_entry, TRUE, TRUE, 0);
  gtk_widget_show(hbox);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

  gtk_container_add(GTK_CONTAINER(frame), vbox);
  gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 2);

  /* Create 'Command::Rotate' frame. */
  frame = gtk_frame_new(_("Rotate"));
  gtk_container_border_width(GTK_CONTAINER(frame), 4);
  gtk_widget_show(frame);
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(vbox), 4);
  gtk_widget_show(vbox);

  hbox = gtk_hbox_new(FALSE, 2);
  label = gtk_label_new(_("Left   "));
  gtk_widget_show(label);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  rotate_left_entry = gtk_entry_new();
  gtk_signal_connect(GTK_OBJECT(rotate_left_entry), "key_press_event",
                     GTK_SIGNAL_FUNC(option_menu_cb_entry_key_pressed), 
                     (gpointer)ROTATE_LEFT_ENTRY);
  gtk_signal_connect(GTK_OBJECT(rotate_left_entry), "focus_in_event",
                     GTK_SIGNAL_FUNC(option_menu_cb_entry_focus_in),
                     (gpointer)ROTATE_LEFT_ENTRY);
  gtk_widget_set_usize(rotate_left_entry, 220, 20);
  gtk_entry_set_text(GTK_ENTRY(rotate_left_entry), option->path.rotate_left);
  gtk_editable_set_position(GTK_EDITABLE(rotate_left_entry), 0);
  gtk_widget_show(rotate_left_entry);
  gtk_box_pack_start(GTK_BOX(hbox), rotate_left_entry, TRUE, TRUE, 0);
  gtk_widget_show(hbox);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

  hbox = gtk_hbox_new(FALSE, 2);
  label = gtk_label_new(_("Right "));
  gtk_widget_show(label);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  rotate_right_entry = gtk_entry_new();
  gtk_signal_connect(GTK_OBJECT(rotate_right_entry), "key_press_event",
                     GTK_SIGNAL_FUNC(option_menu_cb_entry_key_pressed), 
                     (gpointer)ROTATE_RIGHT_ENTRY);
  gtk_signal_connect(GTK_OBJECT(rotate_right_entry), "focus_in_event",
                     GTK_SIGNAL_FUNC(option_menu_cb_entry_focus_in),
                     (gpointer)ROTATE_RIGHT_ENTRY);
  gtk_widget_set_usize(rotate_right_entry, 220, 20);
  gtk_entry_set_text(GTK_ENTRY(rotate_right_entry), option->path.rotate_right);
  gtk_editable_set_position(GTK_EDITABLE(rotate_right_entry), 0);
  gtk_widget_show(rotate_right_entry);
  gtk_box_pack_start(GTK_BOX(hbox), rotate_right_entry, TRUE, TRUE, 0);
  gtk_widget_show(hbox);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

  gtk_container_add(GTK_CONTAINER(frame), vbox);
  gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 2);

  label = gtk_label_new(_("Command "));
  gtk_notebook_append_page(GTK_NOTEBOOK (notebook), vbox1, label);

  /* Create 'Misc' page. */
  vbox1 = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(vbox1), 4);
  gtk_widget_show(vbox1);

  /* Create 'Misc::DotFile' frame. */
  frame = gtk_frame_new(_("Dot File"));
  gtk_container_border_width(GTK_CONTAINER(frame), 4);
  gtk_widget_show(frame);
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(vbox);
  dot_file_btn = 
      gtk_check_button_new_with_label(
               _("Show the directory which name starts a dot."));
  if (option->dot_file == TRUE) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dot_file_btn), TRUE);
  }
  else {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dot_file_btn), FALSE);
  }
  gtk_box_pack_start(GTK_BOX(vbox), dot_file_btn, FALSE, FALSE, 0);
  gtk_widget_show(dot_file_btn);
  gtk_container_add(GTK_CONTAINER(frame), vbox);
  gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 2);

  /* Create 'Misc::Dialog' frame. */
  frame = gtk_frame_new(_("Dialog"));
  gtk_container_border_width(GTK_CONTAINER(frame), 4);
  gtk_widget_show(frame);
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(vbox);

  dir_remove_dialog_btn = 
               gtk_check_button_new_with_label(
                        _("Show a dialog before the directory removed."));
  if (option->dialog.dir_remove == TRUE) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dir_remove_dialog_btn), 
                                 TRUE);
  }
  else {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dir_remove_dialog_btn), 
                                 FALSE);
  }
  gtk_box_pack_start(GTK_BOX(vbox), dir_remove_dialog_btn, FALSE, FALSE, 0);
  gtk_widget_show(dir_remove_dialog_btn);

  remove_dialog_btn = 
               gtk_check_button_new_with_label(
                        _("Show a dialog before the file removed."));
  if (option->dialog.remove == TRUE) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(remove_dialog_btn), 
                                 TRUE);
  }
  else {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(remove_dialog_btn), 
                                 FALSE);
  }
  gtk_box_pack_start(GTK_BOX(vbox), remove_dialog_btn, FALSE, FALSE, 0);
  gtk_widget_show(remove_dialog_btn);

  overwrite_dialog_btn =
            gtk_check_button_new_with_label(
                     _("Show a dialog before the file overwritten."));
  if (option->dialog.overwrite == TRUE) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(overwrite_dialog_btn),
                                 TRUE);
  }
  else {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(overwrite_dialog_btn),
                                 FALSE);
  }
  gtk_box_pack_start(GTK_BOX(vbox), overwrite_dialog_btn, FALSE, FALSE, 0);
  gtk_widget_show(overwrite_dialog_btn);

  gtk_container_add(GTK_CONTAINER(frame), vbox);
  gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 2);

  label = gtk_label_new(_("Misc "));
  gtk_notebook_append_page(GTK_NOTEBOOK (notebook), vbox1, label);

  /* Create 'Misc::Refresh' frame. */
  frame = gtk_frame_new(_("Refresh"));
  gtk_container_border_width(GTK_CONTAINER(frame), 4);
  gtk_widget_show(frame);
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(vbox);
  auto_refresh_btn = 
       gtk_check_button_new_with_label(
                           _("Refresh thumbnails automatically, if needed."));
  if (option->auto_refresh == TRUE) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(auto_refresh_btn), TRUE);
  }
  else {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(auto_refresh_btn), FALSE);
  }
  gtk_box_pack_start(GTK_BOX(vbox), auto_refresh_btn, FALSE, FALSE, 0);
  gtk_widget_show(auto_refresh_btn);
  gtk_container_add(GTK_CONTAINER(frame), vbox);
  gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 2);

  /* Create 'Misc::Simple editor' frame. */
  frame = gtk_frame_new(_("Simple viewer"));
  gtk_container_border_width(GTK_CONTAINER(frame), 4);
  gtk_widget_show(frame);
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(vbox);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 2);
  gtk_widget_show(hbox);
  
  label = gtk_label_new(_("Width: "));
  gtk_widget_show(label);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2);
  
  adj3 = gtk_adjustment_new((option->viewer).width,
                            OPTION_SIMPLE_EDITOR_MIN,
                            OPTION_SIMPLE_EDITOR_MAX, 1.0, 10.0, 10.0);
  spinner3 = gtk_spin_button_new(GTK_ADJUSTMENT(adj3), 5, 0);
  gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spinner3), FALSE);
  gtk_widget_set_usize(spinner3, 60, 20);
  gtk_box_pack_start(GTK_BOX(hbox), spinner3, FALSE, FALSE, 0);
  gtk_widget_show(spinner3);
  gtk_container_add(GTK_CONTAINER(frame), vbox);
  gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 2);

  /* Create 'OK' button and 'Cancel' button. */
  hbox = gtk_hbutton_box_new();
  gtk_hbutton_box_set_layout_default(GTK_BUTTONBOX_END);
  gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbox), 5);
  gtk_widget_show(hbox);

  ok_button = gtk_button_new_with_label(_("  OK  "));
  gtk_widget_show(ok_button);
  gtk_signal_connect(GTK_OBJECT(ok_button), "clicked",
                     GTK_SIGNAL_FUNC(option_menu_cb_config_ok), &ret);

  cancel_button = gtk_button_new_with_label(_("  Cancel  "));
  gtk_widget_show(cancel_button);
  gtk_signal_connect(GTK_OBJECT(cancel_button), "clicked",
                     GTK_SIGNAL_FUNC(option_menu_cb_config_cancel), &ret);

  gtk_container_add(GTK_CONTAINER(hbox), ok_button);
  gtk_container_add(GTK_CONTAINER(hbox), cancel_button);

  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area), 
                     hbox, FALSE, FALSE, 0);

  GTK_WIDGET_SET_FLAGS(ok_button    , GTK_CAN_DEFAULT);
  GTK_WIDGET_SET_FLAGS(cancel_button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default(ok_button);
  gtk_widget_grab_focus  (ok_button);

  gtk_widget_show(dialog);

  /* Make a modal dialog and waits to push a button. */ 
  ret = FALSE;
  viewer_entry_save = editor_entry_save = printer_entry_save = NULL;
  rotate_left_entry_save = rotate_right_entry_save = NULL;
  gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);
  gtk_main();

  if (ret == TRUE) {
    /* Format option. */
    if (GTK_TOGGLE_BUTTON(format_jpg_btn)->active) {
      option->format.jpg_on = TRUE;
    }
    else {
      option->format.jpg_on = FALSE;
    }
    if (GTK_TOGGLE_BUTTON(format_bmp_btn)->active) {
      option->format.bmp_on = TRUE;
    }
    else {
      option->format.bmp_on = FALSE;
    }
    if (GTK_TOGGLE_BUTTON(format_png_btn)->active) {
      option->format.png_on = TRUE;
    }
    else {
      option->format.png_on = FALSE;
    }
    if (GTK_TOGGLE_BUTTON(format_xpm_btn)->active) {
      option->format.xpm_on = TRUE;
    }
    else {
      option->format.xpm_on = FALSE;
    }
    if (GTK_TOGGLE_BUTTON(format_xbm_btn)->active) {
      option->format.xbm_on = TRUE;
    }
    else {
      option->format.xbm_on = FALSE;
    }
    if (GTK_TOGGLE_BUTTON(format_tif_btn)->active) {
      option->format.tif_on = TRUE;
    }
    else {
      option->format.tif_on = FALSE;
    }
    if (GTK_TOGGLE_BUTTON(format_gif_btn)->active) {
      option->format.gif_on = TRUE;
    }
    else {
      option->format.gif_on = FALSE;
    }
    if (GTK_TOGGLE_BUTTON(format_pcx_btn)->active) {
      option->format.pcx_on = TRUE;
    }
    else {
      option->format.pcx_on = FALSE;
    }

    /* ImageCache option. */
    if (GTK_TOGGLE_BUTTON(use_cache_btn)->active) {
      option->image_cache.cache_on = TRUE;
    }
    else {
      option->image_cache.cache_on = FALSE;
    }

    option->image_cache.cache_size =
           gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spinner1));

    if (GTK_TOGGLE_BUTTON(remain_cache_btn)->active) {
      option->image_cache.remain_on = TRUE;
    }
    else {
      option->image_cache.remain_on = FALSE;
    }

    /* Default icon size option. */
    if (GTK_TOGGLE_BUTTON(icon_small_btn)->active) {
      option->icon_size = ICON_SMALL;
    }
    else {
      if (GTK_TOGGLE_BUTTON(icon_middle_btn)->active) {
        option->icon_size = ICON_MIDDLE;
      }
      else {
        option->icon_size = ICON_LARGE;
      }
    }

    /* Default sort order option. */
    if (GTK_TOGGLE_BUTTON(sort_file_btn)->active) {
      if (GTK_TOGGLE_BUTTON(sort_ascend_btn)->active) {
        option->sort = SORT_FILE_NAME_A;
      }
      else {
        option->sort = SORT_FILE_NAME_D;
      }
    }
    else {
      if (GTK_TOGGLE_BUTTON(sort_ctime_btn)->active) {
        if (GTK_TOGGLE_BUTTON(sort_ascend_btn)->active) {
          option->sort = SORT_CTIME_A;
        }
        else {
          option->sort = SORT_CTIME_D;
        }
      }
      else{
        if (GTK_TOGGLE_BUTTON(sort_atime_btn)->active) {
          if (GTK_TOGGLE_BUTTON(sort_ascend_btn)->active) {
            option->sort = SORT_ATIME_A;
          }
          else {
            option->sort = SORT_ATIME_D;
          }
        }
        else {
          if (GTK_TOGGLE_BUTTON(sort_ascend_btn)->active) {
            option->sort = SORT_MTIME_A;
          }
          else {
            option->sort = SORT_MTIME_D;
          }
        }
      }
    }

    /* Command options. */
    if (GTK_TOGGLE_BUTTON(external_btn)->active) {
      option->viewer.default_external = TRUE;
    }
    else {
      option->viewer.default_external = FALSE;
    }
    if (option->path.viewer != NULL) {
      free(option->path.viewer);
      option->path.viewer = NULL;
    }
    if (option->path.editor != NULL) {
      free(option->path.editor);
      option->path.editor = NULL;
    }
    if(option->path.printer != NULL) {
      free(option->path.printer);
      option->path.printer = NULL;
    }
    if(option->path.rotate_left != NULL) {
      free(option->path.rotate_left);
      option->path.rotate_left = NULL;
    }
    if(option->path.rotate_right != NULL) {
      free(option->path.rotate_right);
      option->path.rotate_right = NULL;
    }
    option->path.viewer  =
                 strdup(gtk_entry_get_text(GTK_ENTRY(viewer_entry)));
    option->path.editor  =
                 strdup(gtk_entry_get_text(GTK_ENTRY(editor_entry)));
    option->path.printer =
                 strdup(gtk_entry_get_text(GTK_ENTRY(printer_entry)));
    option->path.rotate_left  =
                 strdup(gtk_entry_get_text(GTK_ENTRY(rotate_left_entry)));
    option->path.rotate_right =
                 strdup(gtk_entry_get_text(GTK_ENTRY(rotate_right_entry)));
    if ((option->path.viewer       == NULL) ||
        (option->path.editor       == NULL) ||
        (option->path.printer      == NULL) ||
        (option->path.rotate_left  == NULL) ||
        (option->path.rotate_right == NULL)    ) {
      if (option->path.viewer       != NULL) { free(option->path.viewer);  }
      if (option->path.editor       != NULL) { free(option->path.editor);  }
      if (option->path.printer      != NULL) { free(option->path.printer); }
      if (option->path.rotate_left  != NULL) { free(option->path.rotate_left);}
      if (option->path.rotate_right != NULL) { free(option->path.rotate_right);}

      /* Out of memory. */
      fprintf(stderr, "danpei: Out of memory.\n");
      fprintf(stderr, "        option_menu.c: error -- 14.\n");
      gtk_exit(-1);
    }

    /* Max files option. */
    option->max_num =
           gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spinner2));

    /* Dialog options. */
    if (GTK_TOGGLE_BUTTON(dir_remove_dialog_btn)->active) {
      option->dialog.dir_remove = TRUE;
    }
    else {
      option->dialog.dir_remove = FALSE;
    }
    if (GTK_TOGGLE_BUTTON(remove_dialog_btn)->active) {
      option->dialog.remove = TRUE;
    }
    else {
      option->dialog.remove = FALSE;
    }
    if (GTK_TOGGLE_BUTTON(overwrite_dialog_btn)->active) {
      option->dialog.overwrite = TRUE;
    }
    else {
      option->dialog.overwrite = FALSE;
    }
    if (GTK_TOGGLE_BUTTON(dot_file_btn)->active) {
      option->dot_file = TRUE;
    }
    else {
      option->dot_file = FALSE;
    }
    if (GTK_TOGGLE_BUTTON(auto_refresh_btn)->active) {
      option->auto_refresh = TRUE;
    }
    else {
      option->auto_refresh = FALSE;
    }
    if (GTK_TOGGLE_BUTTON(multi_viewer_btn)->active) {
      option->is_viewer_multi_process = TRUE;
    }
    else {
      option->is_viewer_multi_process = FALSE;
    }
    if (GTK_TOGGLE_BUTTON(multi_editor_btn)->active) {
      option->is_editor_multi_process = TRUE;
    }
    else {
      option->is_editor_multi_process = FALSE;
    }

    /* Simple viewer options. */
    option->viewer.width = option->viewer.height =
           gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spinner3));

    option_menu_write_config_file(option);
  }

  gtk_widget_destroy(dialog);

  /* Free the work area. */
  if (viewer_entry_save  != NULL) { 
    free(viewer_entry_save );
    viewer_entry_save = NULL;
  }
  if (editor_entry_save  != NULL) { 
    free(editor_entry_save );
    editor_entry_save = NULL;
  }
  if (printer_entry_save != NULL) { 
    free(printer_entry_save);
    printer_entry_save = NULL;
  }
  if (rotate_left_entry_save != NULL) { 
    free(rotate_left_entry_save);
    rotate_left_entry_save = NULL;
  }
  if (rotate_right_entry_save != NULL) { 
    free(rotate_right_entry_save);
    rotate_right_entry_save = NULL;
  }

  return;
}

/* Callback function definitions. */
/*
 * @option_menu_cb_config_ok
 *
 *
 *
 */
static void option_menu_cb_config_ok(GtkWidget *widget,
                                     gpointer  data     ) {
  gboolean *ret;

  ret = (gboolean*)data;

  *ret = TRUE;

  gtk_main_quit();

  return;
}

/*
 * @option_menu_cb_config_cancel
 *
 *
 *
 */
static void option_menu_cb_config_cancel(GtkWidget *widget,
                                         gpointer  data     ) {
  gboolean *ret;

  ret = (gboolean*)data;

  *ret = FALSE;

  gtk_main_quit();

  return;
}

/*
 * @option_menu_cb_viewer_entry_key_pressed
 *
 *  Recover the text in the entry, when ESC key pressed.
 *
 */
static gboolean option_menu_cb_entry_key_pressed(GtkWidget   *widget,
                                                 GdkEventKey *ev    ,
                                                 gpointer    data     ) {
  gint entry_type;

  entry_type = (gint)data;

  /* When ESC key pressed, recover the file name before changing. */
  if ((widget     != NULL) && 
      (ev         != NULL) && 
      (ev->keyval == GDK_Escape)) {
    switch (entry_type) {
      case VIEWER_ENTRY:
        if (viewer_entry_save != NULL) {
          gtk_entry_set_text(GTK_ENTRY(widget), viewer_entry_save);
          gtk_editable_set_position(GTK_EDITABLE(widget), 0);
        }
        break;
      case EDITOR_ENTRY:
        if (editor_entry_save != NULL) {
          gtk_entry_set_text(GTK_ENTRY(widget), editor_entry_save);
          gtk_editable_set_position(GTK_EDITABLE(widget), 0);
        }
        break;
      case PRINTER_ENTRY:
        if (printer_entry_save != NULL) {
          gtk_entry_set_text(GTK_ENTRY(widget), printer_entry_save);
          gtk_editable_set_position(GTK_EDITABLE(widget), 0);
        }
        break;
      case ROTATE_LEFT_ENTRY:
        if (rotate_left_entry_save != NULL) {
          gtk_entry_set_text(GTK_ENTRY(widget), rotate_left_entry_save);
          gtk_editable_set_position(GTK_EDITABLE(widget), 0);
        }
        break;
      case ROTATE_RIGHT_ENTRY:
        if (rotate_right_entry_save != NULL) {
          gtk_entry_set_text(GTK_ENTRY(widget), rotate_right_entry_save);
          gtk_editable_set_position(GTK_EDITABLE(widget), 0);
        }
        break;
    }
  }

  return FALSE;
}

/*
 * @option_menu_cb_entry_focus_in
 *
 *  Save the text in the entry for recovery. 
 *
 */
static gboolean option_menu_cb_entry_focus_in(GtkWidget *widget, 
                                              GdkEvent  *ev    , 
                                              gpointer  data     ) {
  gint entry_type;

  entry_type = (gint)data;

  if ((widget != NULL) && (ev != NULL)) {  
    switch (entry_type) {
      case VIEWER_ENTRY:
        if (viewer_entry_save != NULL) { free(viewer_entry_save); }
        viewer_entry_save = strdup(gtk_entry_get_text(GTK_ENTRY(widget)));
        break;
      case EDITOR_ENTRY:
        if (editor_entry_save != NULL) { free(editor_entry_save); }
        editor_entry_save = strdup(gtk_entry_get_text(GTK_ENTRY(widget)));
        break;
      case PRINTER_ENTRY:
        if (printer_entry_save != NULL) { free(printer_entry_save); }
        printer_entry_save = strdup(gtk_entry_get_text(GTK_ENTRY(widget)));
        break;
      case ROTATE_LEFT_ENTRY:
        if (rotate_left_entry_save != NULL) { free(rotate_left_entry_save); }
        rotate_left_entry_save = strdup(gtk_entry_get_text(GTK_ENTRY(widget)));
        break;
      case ROTATE_RIGHT_ENTRY:
        if (rotate_right_entry_save != NULL) { free(rotate_right_entry_save); }
        rotate_right_entry_save = 
                           strdup(gtk_entry_get_text(GTK_ENTRY(widget)));
        break;
    }
  }
  return FALSE;
}


