%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\subsection{darcs initialize}
\begin{code}
module Init ( initialize ) where
import Directory ( createDirectory )
import IO ( isAlreadyExistsError )

import DarcsCommands ( DarcsCommand(..), nodefaults )
import DarcsArguments ( DarcsFlag, pristine_tree )
import RepoPrefs ( write_default_prefs )
import Pristine ( createPristine, flagsToPristine )
import DarcsRepo ( am_not_in_repo, write_inventory )
\end{code}

\options{initialize}

\haskell{initialize_description}

\begin{code}
initialize_description :: String
initialize_description = "Initialize a new source tree as a darcs repository."
\end{code}
Generally you will only call initialize once for each project you work on,
and calling it is just about the first thing you do.  Just make sure you
are in the main directory of the project, and initialize will set up all the
directories and files darcs needs in order to start keeping track of
revisions for your project.

\begin{code}
initialize_help :: String
initialize_help =
 "Generally you will only call initialize once for each project you work on,   \n"++
 "and calling it is just about the first thing you do.  Just make sure   \n"++
 "you are in the main directory of the project, and initialize will set up all \n"++
 "the directories and files darcs needs in order to start keeping track of   \n"++
 "revisions for your project.\n"
\end{code}

\begin{code}
initialize :: DarcsCommand
initialize = DarcsCommand {command_name = "initialize",
                         command_help = initialize_help,
                         command_description = initialize_description,
                         command_extra_args = 0,
                         command_extra_arg_help = [],
                         command_prereq = am_not_in_repo,
                         command_command = initialize_cmd,
                         command_get_arg_possibilities = return [],
                         command_argdefaults = nodefaults,
                         command_darcsoptions = [pristine_tree]}
\end{code}

The \verb|initialize| command actually follows a very simple procedure.
It creates the directories \verb|_darcs|, \verb|_darcs/current| (or
\verb|_darcs/pristine|) and \verb|_darcs/patches|, and then creates an
empty file, \verb|_darcs/inventory|.  However, it is strongly
recommended that you use \verb|darcs initialize| to do this, as this
procedure may change in a future version of darcs.

\begin{options}
--no-pristine-tree
\end{options}
In order to save disk space, you can use \verb|initialize| with the
\verb|--no-pristine-tree| flag to create a repository with no pristine
tree.  Please see Section~\ref{disk-usage} for more information.

\begin{code}
initialize_cmd :: [DarcsFlag] -> [String] -> IO ()
initialize_cmd opts _ = do
    createDirectory "_darcs" `catch`
        (\e-> if isAlreadyExistsError e
              then fail "Tree has already been initialized!"
              else fail "Error creating directory `_darcs'.")
    createPristine $ flagsToPristine opts
    createDirectory "_darcs/patches"
    createDirectory "_darcs/prefs"
    write_default_prefs
    write_inventory "." [[]]
\end{code}

