#pragma once

#include <string>
#include <functional>
#include <sigc++/trackable.h>

#include "imodel.h"
#include "imap.h"

class Entity;

namespace map
{

/**
 * Helper class dealing with any models that still have an unmodified
 * scale when they're being passed to the IMapWriters.
 * During a regular save operation, all models with modified scale are
 * processed and have their model files generated with the desired scale.
 * In auto-save and prefab export scenarios, no new models are generated
 * (in order not to mess up the user's map project folder with models 
 * generated by auto-saves). To still preserve the model scale as it has 
 * been set by the user at the time of the auto-save, this class will
 * write the modified scale to an editor_* spawnarg, such that it can be
 * reconstructed when loading an auto-save or a partial export (#5220).
 */
class ModelScalePreserver :
    public sigc::trackable
{
private:
    const std::string _modelScaleKey;

public:
    ModelScalePreserver();

    // Noncopyable
    ModelScalePreserver(const ModelScalePreserver&) = delete;
    ModelScalePreserver& operator=(const ModelScalePreserver&) = delete;

private:
    void forEachScaledModel(const scene::IMapRootNodePtr& root, 
        const std::function<void(Entity&, model::ModelNode&)>& func);

    void onResourceExporting(const scene::IMapRootNodePtr& root);
    void onResourceExported(const scene::IMapRootNodePtr& root);

    void restoreModelScale(const scene::IMapRootNodePtr& root);

    void onMapEvent(IMap::MapEvent ev);
};

}
