/*
 * Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either 
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>. */

#include "DarkroomView.h"

#include <png.h>

#include <QGraphicsPixmapItem>
#include <QLabel>
#include <QListView>
#include <QWheelEvent>

#include <KDebug>
#include <klocale.h>
#include <KFileDialog>
#include <KMessageBox>

#include <threadweaver/ThreadWeaver.h>

#include <libkdcraw/kdcraw.h>
#include <libkdcraw/rawfiles.h>
#include <libkdcraw/dcrawsettingswidget.h>

#include <Darkroom.h>
#include <RawImageInfo.h>
#include <JobExport.h>
#include <JobPreview.h>
#include <PostProcessor.h>

using namespace KDcrawIface;

struct DarkroomView::Private {
  Darkroom* darkRoom;
  RawImageInfo* currentRawFile;
  QGraphicsPixmapItem* pixmapItem;
  QGraphicsScene* scene;
  float scale;
};

DarkroomView::DarkroomView(Darkroom* _parent) : QGraphicsView(_parent), d(new Private)
{
  d->darkRoom = _parent;
  d->currentRawFile = 0;
  // Init the scene
  d->scene = new QGraphicsScene;
  setScene( d->scene );
  d->pixmapItem = new QGraphicsPixmapItem;
  d->scene->addItem( d->pixmapItem );
  
}

DarkroomView::~DarkroomView()
{
}

void DarkroomView::setRawImageInfo( RawImageInfo* v)
{
  d->currentRawFile = v;
  updatePreview();
}

void DarkroomView::fileSaveAs()
{
  if( not d->currentRawFile ) return ;
  QFileInfo fi = d->currentRawFile->fileInfo();
  
  QString file = KFileDialog::getSaveFileName( KUrl(fi.baseName() + ".png"), "*.png" );
  if( not file.isEmpty() and not file.isNull() )
  {
    if( QFile( file).exists())
    {
      if( KMessageBox::warningContinueCancel( this, i18n("File %1 allready exists, overwrite ?", file), i18n("Overwrite ?")) != KMessageBox::Continue )
      {
          return;
      }
    }
    JobExport::weaver()->enqueue( new JobExport( d->currentRawFile, file, d->darkRoom->rawDecodingSettings(), d->darkRoom->postProcessor() ) );
  }
}

void DarkroomView::zoomIn()
{
  scale( 1.5, 1.5);
}

void DarkroomView::zoomOut()
{
  scale( 1.0 / 1.5, 1.0 / 1.5);
}

void DarkroomView::actualSize()
{
  setMatrix( QMatrix() );
}

void DarkroomView::fitToPage()
{
  fitInView( d->pixmapItem, Qt::KeepAspectRatio );
}

void DarkroomView::updatePreview()
{
  kDebug() << "updatePreview " << d->currentRawFile;
  if( not d->currentRawFile) return;
  JobPreview* job = new JobPreview(d->currentRawFile, d->darkRoom->rawDecodingSettings(), d->darkRoom->postProcessor() );
  connect( job, SIGNAL(imageFinished(const QImage&)), this, SLOT(setImage( const QImage& ) ) );
  JobPreview::weaver()->enqueue( job );
}

void DarkroomView::setImage( const QImage& _image )
{
  d->pixmapItem->setPixmap( QPixmap::fromImage( _image ) );
}

void DarkroomView::wheelEvent ( QWheelEvent * event )
{
  if( event->modifiers() == Qt::ControlModifier )
  {
    if( event->delta() > 0 )
    {
      zoomIn();
    } else {
      zoomOut();
    }
  } else {
    QGraphicsView::wheelEvent( event );
  }
}

#include "DarkroomView.moc"
