/*
 * Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either 
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>. */

#include "JobPreview.h"

#include <QFileInfo>
#include <qendian.h>

#include <threadweaver/ThreadWeaver.h>

#include <libkdcraw/kdcraw.h>

#include "RawImageInfo.h"
#include "PostProcessor.h"

using namespace KDcrawIface;

struct JobPreview::Private {
  Private(const PostProcessor& _postProcessor) : processor(_postProcessor)
  {
  }
  RawImageInfo* rawImageInfo;
  RawDecodingSettings settings;
  PostProcessor processor;
  static ThreadWeaver::Weaver* weaver;
};

ThreadWeaver::Weaver* JobPreview::Private::weaver = 0;

JobPreview::JobPreview( RawImageInfo* _rawImageInfo, const RawDecodingSettings& _settings, const PostProcessor& _processor  ) : d(new Private(_processor))
{
  d->rawImageInfo = _rawImageInfo;
  d->settings = _settings;
  d->processor.setConvertToSRGB( true );
}

JobPreview::~JobPreview()
{
  delete d;
}

ThreadWeaver::Weaver* JobPreview::weaver()
{
  if(not Private::weaver)
  {
    Private::weaver = new ThreadWeaver::Weaver;
  }
  return Private::weaver;
}

void JobPreview::run()
{
  QByteArray imageData;
  int width, height, rgbmax;
  KDcraw dcraw;
  dcraw.decodeHalfRAWImage( d->rawImageInfo->fileInfo().absoluteFilePath(), d->settings, imageData, width, height, rgbmax);
  QImage image( width, height, QImage::Format_RGB32);
  quint8 buffer[3];
  for( int y = 0; y < height; ++y)
  {
    for(int x = 0; x < width; ++x )
    {
      if( d->settings.sixteenBitsImage )
      {
        quint16* ptr = (quint16*)(imageData.data() + (y * width + x )* 3 *sizeof(quint16));
        for( int i = 0; i < 3; ++i)
        {
          ptr[i] = qFromBigEndian(ptr[i]);
        }
        d->processor.apply16( ptr );        
        for( int i = 0; i < 3; ++i)
        {
          buffer[i] = ptr[i] >> 8;
        }
        image.setPixel( x,y, qRgb( buffer[0], buffer[1], buffer[2] ));
      } else {
        quint8* ptr = (quint8*)imageData.data() + (y * width + x )* 3;
        d->processor.apply8( ptr );
        image.setPixel( x,y, qRgb( ptr[0], ptr[1], ptr[2] ));
      }
    }
  }
  emit( imageFinished( image ) );
}

#include "JobPreview.moc"
