/*
 * Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either 
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>. */

#include "JobThumbnail.h"

#include <QFileInfo>
#include <QImage>
#include <QMatrix>
#include <QMutexLocker>

#include <threadweaver/ThreadWeaver.h>

#include <libkdcraw/kdcraw.h>
#include <libkexiv2/kexiv2.h>

#include "RawImageInfo.h"
#include "ExportCodec.h"

struct JobThumbnail::Private
{
  RawImageInfoSP rawImageInfo;
  static ThreadWeaver::Weaver* weaver;
};

ThreadWeaver::Weaver* JobThumbnail::Private::weaver = 0;

JobThumbnail::JobThumbnail( RawImageInfoSP _rawImageInfo) : d(new Private)
{
  Q_ASSERT( _rawImageInfo );
  d->rawImageInfo = _rawImageInfo;
}

JobThumbnail::~JobThumbnail()
{
  delete d;
}

ThreadWeaver::Weaver* JobThumbnail::weaver()
{
  if(not Private::weaver)
  {
    Private::weaver = new ThreadWeaver::Weaver;
  }
  return Private::weaver;
}

void JobThumbnail::run()
{
  QImage img;
  KDcrawIface::KDcraw::loadDcrawPreview( img, d->rawImageInfo->fileInfo().absoluteFilePath() );
  QMutexLocker locker(ExportCodec::exiv2Mutex());
  KExiv2Iface::KExiv2 exiv2;
  if(exiv2.load( d->rawImageInfo->fileInfo().absoluteFilePath()))
  {
    if(exiv2.hasExif()) {
    	KExiv2Iface::KExiv2::ImageOrientation orient = exiv2.getImageOrientation ();
    	if (orient == KExiv2Iface::KExiv2::ORIENTATION_ROT_90)
    	{
    	  QMatrix m_90;
    	  m_90.rotate (90.0);
    	  img = img.transformed (m_90);
    	} else if (orient == KExiv2Iface::KExiv2::ORIENTATION_ROT_270)
    	{
    	  QMatrix m_270;	
    	  m_270.rotate (270.0);
    	  img = img.transformed (m_270);
	}
    }
  }
  img = img.scaled ( 100, 100, Qt::KeepAspectRatio);
  d->rawImageInfo->setThumbnail(img );
}
