/*
 * Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either 
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>. */

#include "ProcessingOptions.h"

#include <QDomDocument>
#include <QLocale>
#include <QMap>
#include <QString>

#include <libkdcraw/dcrawsettingswidget.h>

struct ProcessingOptions::Private {
  QMap<QString, QVariant> values;
};

ProcessingOptions::ProcessingOptions() : d(new Private)
{
  
}

ProcessingOptions::ProcessingOptions(const ProcessingOptions& _rhs) : d(new Private( *_rhs.d))
{
}

ProcessingOptions& ProcessingOptions::operator=(const ProcessingOptions& _rhs )
{
  *d = *_rhs.d;
  return *this;
}

ProcessingOptions::~ProcessingOptions()
{
  delete d;
}

KDcrawIface::RawDecodingSettings ProcessingOptions::rawDecodingSettings() const
{
  KDcrawIface::RawDecodingSettings settings;
  settings.sixteenBitsImage = true; // asBool("SixteenBits");
  // RAW Options
  settings.whiteBalance = (KDcrawIface::RawDecodingSettings::WhiteBalance) asInteger("WhiteBalance");
  settings.customWhiteBalance = asInteger("Temperature");
  settings.customWhiteBalanceGreen = asInteger("Tint");
  settings.RAWQuality = (KDcrawIface::RawDecodingSettings::DecodingQuality)asInteger("DemosaicQuality");
  
  // Light Options
  int highlight = asInteger("HighlightsType");
  if( highlight <= 2)
  {
    settings.unclipColors = highlight;
  } else {
    settings.unclipColors = asInteger("LevelValue");
  }
  
  // Color Options
  settings.enableCACorrection = asBool("ChromaticAberrationCorrection");
  settings.caMultiplier[0] = asDouble("RedMultiplier");
  settings.caMultiplier[1] = asDouble("GreenMultiplier");
  
  // Noise Reduction Options
  settings.enableNoiseReduction = asBool("EnableNoiseReduction");
  settings.NRThreshold = asInteger("ThresholdNoiseReduction");
  
  return settings;
}

void ProcessingOptions::setOption( const QString& _key, const QVariant& _value )
{
  d->values[ _key ] = _value;
}

bool ProcessingOptions::asBool( const QString& _key ) const
{
  return (bool)option( _key ).toInt();
}

qint32 ProcessingOptions::asInteger( const QString& _key ) const
{
  return option( _key ).toInt();
}

double ProcessingOptions::asDouble( const QString& _key ) const
{
  return option( _key ).toDouble();
}

QString ProcessingOptions::asString( const QString& _key, const QString& _defaultValue  ) const
{
  if( d->values.contains( _key ) )
  {
    return option( _key ).toString();
  } else {
    return _defaultValue;
  }
}

QList<QPointF> ProcessingOptions::asPointFList( const QString& _key ) const
{
  QList<QVariant> list = option( _key ).toList();
  QList<QPointF> plist;
  foreach( QVariant variant, list )
  {
    plist.push_back( variant.toPointF() );
  }
  return plist;
}

const QVariant& ProcessingOptions::option( const QString& _key ) const
{
  return d->values[ _key ];
}

QString ProcessingOptions::toXml() const
{
  QDomDocument doc = QDomDocument("params");
  QDomElement root = doc.createElement( "params" );
  doc.appendChild( root );
  toXml(doc, root);
  return doc.toString();
}

void ProcessingOptions::toXml(QDomDocument& doc, QDomElement& root) const
{
  QLocale locale (QLocale::C);
  QMap<QString, QVariant>::Iterator it;
  for ( it = d->values.begin(); it != d->values.end(); ++it ) {
    QDomElement e = doc.createElement( "param" );
    e.setAttribute( "name", QString(it.key().toLatin1()) );
    QVariant v = it.value();
    QString vs;
    if( v.type () == QVariant::List )
    {
      vs = "listpointf ";
      QList<QVariant> points = v.toList();
      for(int i = 0; i < points.size(); ++i)
      {
        QPointF f = points[i].toPointF();
        vs += locale.toString( f.x() ) + " , " + locale.toString( f.y() );
        if( i != points.size() - 1 )
        {
          vs += ";";
        }
      }
    } else {
      vs = v.toString();
    }
    QDomText text = doc.createCDATASection( vs );
    e.appendChild(text);
    root.appendChild(e);
  }
}

void ProcessingOptions::fromXml(const QDomElement& e)
{
  QLocale locale (QLocale::C);
  QDomNode n = e.firstChild();

  while (!n.isNull()) {
    QDomElement e = n.toElement();
    if (!e.isNull()) {
      if (e.tagName() == "param") {
        QString text = e.text();
        QVariant v;
        if( text.startsWith( "listpointf " ) )
        {
          QList<QVariant> list;
          text.remove(0, QString("listpointf ").size());
          foreach( QString onepoint, text.split( ";" ) )
          {
            QList<QString> elts = onepoint.split(",");
            list.push_back( QPointF( locale.toDouble( elts[0] ), locale.toDouble( elts[1] )));
          }
          
          v = list;
        } else {
          v= e.text();
        }
        d->values[e.attribute("name")] = v;
      }
    }
    n = n.nextSibling();
  }
}
void ProcessingOptions::fromXml( const QString& s)
{
  QDomDocument doc;
  doc.setContent( s );
  QDomElement e = doc.documentElement();
  fromXml(e);
  
}

