/***************************************************************************
 *   Copyright (C) 2005 by Adam Treat                                      *
 *   treat@kde.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef DATATABLEEDITORFACTORY_H
#define DATATABLEEDITORFACTORY_H

#include "kcombotable.h"

#include <qsqleditorfactory.h>
#include <klineedit.h>
#include <qmap.h>
#include <qvaluelist.h>
#include <qvbox.h>
#include <qhbox.h>
#include <qdatetime.h>

class DataTable;
class QSqlRecord;
class QSqlCursor;
class QLabel;
class QSqlField;
class DataField;
class DateEdit;
class TimeEdit;
class QPopupMenu;

typedef QMap<QString, QSqlRecord *> DataRecordList;

class DataTableEditorFactory : public QSqlEditorFactory
{
    Q_OBJECT
public:
    DataTableEditorFactory( DataTable *parent = 0, const char *name = 0 );
    virtual ~DataTableEditorFactory();

    virtual QWidget * createVirtualEditor( QWidget * parent, const QString &name );
    virtual QWidget * createEditor( QWidget * parent, const QVariant & variant );
    virtual QWidget * createEditor( QWidget * parent, const QSqlField * field );
    virtual QWidget * createEditor( QWidget * parent, DataField * field );

private:
    DataTable *m_dataTable;
};

class DataEditorBase : public QVBox
{
    Q_OBJECT
public:
    DataEditorBase( const QString &label, const QSqlField *field,
                    DataTableEditorFactory *factory, QWidget *parent = 0 );
    DataEditorBase( const QString &label, const QString &name,
                    DataTableEditorFactory *factory, QWidget *parent = 0 );
    DataEditorBase( const QString &label, DataField *dataField,
                    DataTableEditorFactory *factory, QWidget *parent = 0 );
    QWidget *getEditor() const;
    const QSqlField *getField() const;
    QSize getRecommendedSize() const;
    bool currentEdited() const;
    void setNullValue( const QString txt );
    bool calculateEditor();

signals:
    void editBufferChanged( QVariant value, bool currentEdited );
    void addConstraint( DataEditorBase *editor, const QString &field );

public slots:
    virtual void setFocus();
    void resetEditBuffer( QVariant value, bool currentEdited );

private slots:
    void editorChanged( const QString &txt );
    void editorChanged( const QDate &date );
    void editorChanged( const QTime &time );
    void editorChanged( const QDateTime &datetime );
    void slotAddConstraint( const QString &field );

private:
    QLabel *m_label;
    const QSqlField *m_field;
    QWidget *m_editor;
    bool m_currentEdited;
};

class DataLineEdit : public KLineEdit
{
    Q_OBJECT
    Q_PROPERTY( QString value READ value WRITE setValue )
public:
    DataLineEdit( DataField* field, DataTable *dataTable,
                  QWidget *parent = 0, const char *name = 0 );

    QString value() const;
    void setValue( const QString &value );
    bool calculateEditor();

public slots:
    void setReadOnly( bool readOnly );

private slots:
    void configureContextMenu( QPopupMenu *p );
    void configureDataField();
    void deleteDataField();

private:
    DataField *m_field;
    DataTable *m_dataTable;
};

class DataComboBox : public KComboBox
{
    Q_OBJECT
public:
    DataComboBox( DataField* field, DataTable *dataTable,
                  QWidget *parent = 0, const char *name = 0 );
    DataComboBox( DataField* field, DataTable *m_dataTable,
                  bool rw, QWidget *parent = 0, const char *name = 0 );

private slots:
    void configureContextMenu( QPopupMenu *p );
    void configureDataField();

private:
    DataField *m_field;
    DataTable *m_dataTable;
};

class DateTimeEdit : public QHBox
{
    Q_OBJECT
    Q_PROPERTY( QDateTime datetime READ dateTime WRITE setDateTime )
public:
    DateTimeEdit( DataField* field, DataTable *m_dataTable,
                  QWidget *parent, const char *name = 0 );

    QDateTime dateTime() const;
    void setDateTime( QDateTime dateTime );

private slots:
    void slotDateTimeChanged();

signals:
    void dateTimeChanged( const QDateTime &dateTime );

private:
    DateEdit *m_dateEdit;
    TimeEdit *m_timeEdit;
};

class RelationCombo : public KComboTable
{
    Q_OBJECT
    Q_PROPERTY( QString relationid READ relationId WRITE setRelationId )
public:
    RelationCombo( DataField* field, DataTable *dataTable,
                   QWidget *parent = 0,
                   const char *name = 0 );

    QString relationId() const;
    void setRelationId( const QString &id );
    void setNullValue( const QString &txt );
    void setIgnoreConstraints( bool ignore );
    void initializeFields( bool calculateWidths = false );

signals:
    void comboChanged( const QString &txt );
    void addConstraint( const QString &field );

public slots:
    void setCurrentId( int id );
    void setCurrentIndex( int index );
    void setCurrentTextId( const QString &txt );
    void completionMatch( const QString &txt );

private slots:
    void configureContextMenu( QPopupMenu *p );
    void configureDataField();
    void configureRelationField();

private:
    void setWidths();

private:
    int m_relationId;
    QMap< int, int > index2id;
    DataField *m_field;
    DataTable *m_dataTable;
    QStringList m_constraints;
    bool m_ignoreConstraints;
};

#endif
