/* 
** Utility file
**
** This library and program is free software; you can redistribute it and/or
** modify it under the terms of the GNU Library General Public
** License as published by the Free Software Foundation; either
** version 2 of the License, or (at your option) any later version.
**   
** This library is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** Library General Public License for more details.
**
** You should have received a copy of the GNU Library General Public
** License along with this library; if not, write to the Free
** Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
** MA 02111-1307, USA
*/

#include "config.h"

#include <argz.h>
#include <ctype.h>
#include <error.h>
#include <errno.h>
#include <fstab.h>
#include <getopt.h>
#include <grp.h>
#include <mntent.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <termios.h>
#include <unistd.h>
#include <sys/mount.h>
#include <sys/stat.h>

/* Linux coda head file */
/* avoid inclusion of kernel time.h, since it conflicts with userland time.h */
#define _LINUX_TIME_H
/* This macro is used in coda.h and causes compile errors.
   We don't need it and define it as empty. */
#define __user
#include <linux/coda.h>

/* Neon */
#include <ne_alloc.h>
#include <ne_string.h>

#include "util.h"


/* Global variables defined in davfsd.c */
/* ==================================== */
extern char *dav_url;
extern char *dav_mpoint;
extern char *dav_pidfile;
extern int dav_locks;
extern struct stat dav_file_stat;
extern struct stat dav_dir_stat;
/*2005-05-29, werner, security fix
extern int dav_noexec;*/


/* Global variables defined in webdav.c */
/* ==================================== */
extern char *dav_username;
extern char *dav_password;
extern char *dav_p_user;
extern char *dav_p_password;

/* Global variables used only in this file */
/* ======================================= */
/* Holds the filenames to look for mounted filesystems. */
char *dav_mounts[3] = { DAV_PROC_MOUNTS, _PATH_MOUNTED, NULL };


/* Functions used only in this file */
/* ================================ */
#ifdef DEBUG

#define DBG_ARGS(x) do { \
    DBG1("  Arguments after parsing %s:", x); \
    DBG1("    device = %s", args->device); \
    /*2005-05-29, werner, security fix, fixed mode
    DBG1("    uid = %i", args->uid); \
    DBG1("    gid = %i", args->gid); \
    DBG1("    file_mode = %o", args->file_mode); \
    DBG1("    dir_mode = %o", args->dir_mode); */ \
    DBG1("    ro = %lu", args->ro); \
    DBG1("    nodev = %lu", args->nodev); \
    DBG1("    noexec = %lu", args->noexec); \
    DBG1("    nosuid = %lu", args->nosuid); \
    DBG1("    user = %i", args->user); \
    DBG1("    p_host = %s", args->p_host); \
    DBG1("    p_port = %i", args->p_port); \
    DBG1("    askauth = %i", args->askauth); \
    DBG1("    useproxy = %i", args->useproxy); \
    DBG1("    locks = %i", args->locks); \
    DBG1("    mountanyway = %i", args->mountanyway); \
} while (0)

#define DBG_VARS(x) do { \
    DBG1("  Variables after parsing %s:", x); \
    DBG1("    mpoint = %s", dav_mpoint); \
    /*2005-05-29, werner, security fix, fixed mode
    DBG1("    file_uid = %i", dav_file_stat.st_uid); \
    DBG1("    file_gid = %i", dav_file_stat.st_gid); \
    DBG1("    file_mode = %o", dav_file_stat.st_mode); \
    DBG1("    dir_uid = %i", dav_dir_stat.st_uid); \
    DBG1("    dir_gid = %i", dav_dir_stat.st_gid); \
    DBG1("    dir_mode = %o", dav_dir_stat.st_mode); */ \
    DBG1("    url = %s", dav_url); \
    DBG1("    username = %s", dav_username); \
    DBG1("    passord = %s", dav_password); \
    DBG1("    p_user = %s", dav_p_user); \
    DBG1("    p_passord = %s", dav_p_password); \
    DBG1("    locks = %i", dav_locks); \
    DBG1("    pidfile = %s", dav_pidfile); \
} while (0)

#define DBG_CMDLINE(x) { \
    size_t len; \
    char * cmdline; \
    if (argz_create(x, &cmdline, &len) == 0) { \
        argz_stringify(cmdline, len, ' '); \
        DBG1("  %s", cmdline); \
        NE_FREE(cmdline); \
    } \
} while (0)

#else
#define DBG_ARGS(x)
#define DBG_VARS(x)
#define DBG_CMDLINE(x)
#endif


/* Allocates a new dav_args-structure and initializes it.
 * Defalt file-mode and dir_mode are derived from umask.
 * To delete the structure use dav_delete_args(dav_args *args). */
dav_args *new_args(void) {
    dav_args *args = ne_calloc(sizeof(*args));
    /*2005-05-29, werner, security fix, fixed mode
    args->uid = getuid();
    args->gid = getgid();
    args->file_mode = umask(0);
    umask(args->file_mode);
    args->file_mode = ~args->file_mode & (S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP
                                                  | S_IROTH | S_IWOTH);
    args->dir_mode = args->file_mode;
    if (args->dir_mode & S_IRUSR)
        args->dir_mode |= S_IXUSR;
    if (args->dir_mode & S_IRGRP)
        args->dir_mode |= S_IXGRP;
    if (args->dir_mode & S_IROTH)
        args->dir_mode |= S_IXOTH;*/
    args->nodev = MS_NODEV;
    args->noexec = MS_NOEXEC;
    args->nosuid = MS_NOSUID;
    args->useproxy = 1;
    args->askauth = 1;
    args->locks = 1; /* what about USE_DAV_LOCKS in config.h? */
    return args;
}


/* Prints version und help text.
 * prog : program name; used in version string */
void usage(const char *prog) {
    printf("%s: %s  <%s>\n", prog, PACKAGE_STRING, DAV_HOME); 
    printf("Usage: mount.davfs -V|--version   : print version string\n");
    printf("       mount.davfs -h|--help      : print this message\n");
    printf("  To mount a WebDAV-resource don't call %s directly,\n", prog);
    printf("  but use 'mount' instead.\n");
    printf("       mount <mountpoint>  : or\n");
    printf("       mount <server-url>  : mount the WebDAV-resource as specified\n");
    printf("                             in /etc/fstab. Filesystem type must be 'davfs'.\n");
    printf("                             For non-root users the option 'user' must be\n");
    printf("                             set in fstab.\n");
    printf("       mount -t davfs <server-url> <mountpoint> -o [options]\n");
    printf("                           : mount the WebDAV-resource <server-url>\n");
    printf("                             on mountpoint <mountpoint>. Only root\n");
    printf("                             is allowed to do this. Options is a\n");
    printf("                             comma separated list of options.\n");
    printf("  Recognised options:\n");
    /*2005-05-29, werner, security fix, fixed mode
    printf("       uid=         : user-id to use for the filesystem\n");
    printf("       gid=         : group-id to use for the filesystem\n");
    printf("       file_mode=   : default file mode (ocatal)\n");
    printf("       dir_mode=    : default directory mode (ocatal)\n");*/
    printf("       ro / rw      : mount read-only / read-write\n");
    /*2005-07-09, werner, security fix, no more sense in these options
    printf("       [no]dev      : (don't) interpret special devices on the file system\n");
    printf("       [no]exec     : (don't) allow execution of binaries on the file system\n");
    printf("       [no]suid     : (don't) allow suid- and sgid-bit to take effekt\n"); */
    printf("       [no]user     : (don't) allow normal users to mount\n");
    printf("       users        : same as user\n");
    printf("       proxy=       : name and -optional- port of the http-proxy\n");
    printf("       [no]useproxy : (don't) use proxy for connection\n");
    printf("       [no]askauth  : (don't) ask interactivly for credentials if not found\n");
    printf("                      in the secrets files.\n");
    printf("       [no]locks    : (don't) lock resources on the server\n\n");
    printf("       [no]mountanyway: (don't) mount even without connection\n\n");
}


/* Searches arg from the beginning for digits, valid in base, and converts
 * them to an integer. If arg does not start with valid digits an error
 * message is printed and exit(EXIT_FAILURE) is called.
 * Otherwise the integer is returned.
 * arg    : string to be converted
 * base   : radix of the number; value between 2 and 36
 * opt    : name of the option, arg belongs to. Used in the error message.
 * return value: the value of the integer number in arg */
int arg_to_int(const char *arg, int base, const char *opt) {
    char *tail = NULL;
    int n = strtol(arg, &tail, base);
    if ((n < 0) || (tail == NULL))
        error(EXIT_FAILURE, 0, "Argument must be a base-%i-number -%s", base,
              opt);
    return n;
}


/* Splits arg into hostname and port, if there is a colon in arg. If there is
 * no colon, arg is taken als hostname, and port is set to DEFAULT_PROXY_PORT.
 * The string host will be newly allacoated. The calling function is
 * responsible to free it.
 * *host : the hostname; will be set to the newly allacated string.
 * port  : portnumber; will be set by this function.
 * arg   : the string to be split. */
void split_proxy(char **host, int *port, const char *arg) {
    char *ps = strchr(arg, ':');
    if (ps == NULL) {
        *host = ne_strdup(arg);
        *port = DEFAULT_PROXY_PORT;
    } else {
        *host = ne_strndup(arg, strlen(arg) - strlen(ps));
        ps++;
        *port = arg_to_int(ps, 10, "proxy:port");
    }
}


/* Parses the string option and stores the values in the appropriate fields of
 * args. If an unknown option is found exit(EXIT_FAILURE) is called.
 * All strings returned in args are newly allocated, and the calling function
 * is responsible to free them.
 * args   : structure to store the options in.
 * option : a comma separated list of options (like the options in fstab and
 *          in the -o option of the mount-programm).
 *          For known options see the declaration at the beginning of the
 *          the function definition. */
void get_options(dav_args *args, char *option) {
    DBG0("    Parsing suboptions");
    enum {
        /*2005-05-29, werner, security fix, fixed mode
        UID = 0,
        GID,
        FILE_MODE,
        DIR_MODE,*/
        PROXY = 0,
        USEPROXY,
        NOUSEPROXY,
        ASKAUTH,
        NOASKAUTH,
        LOCKS,
        NOLOCKS,
        MOUNTANYWAY,
        NOMOUNTANYWAY,
        RO,
        RW,
        NODEV,
        DEV,
        NOEXEC,
        EXEC,
        NOSUID,
        SUID,
        USER,
        USERS,
        NOUSER,
        NOAUTO,
        DEFAULTS,
        END
    };
    char* suboptions[] = {
        /*2005-05-29, werner, security fix, fixed mode
        [UID] = "uid",
        [GID] = "gid",
        [FILE_MODE] = "file_mode",
        [DIR_MODE] = "dir_mode",*/
        [PROXY] = "proxy",
        [USEPROXY] = "useproxy",
        [NOUSEPROXY] = "nouseproxy",
        [ASKAUTH] = "askauth",
        [NOASKAUTH] = "noaskauth",
        [LOCKS] = "locks",
        [NOLOCKS] = "nolocks",
        [MOUNTANYWAY] = "mountanyway",
        [NOMOUNTANYWAY] = "nomountanyway",
        [RO] = "ro",
        [RW] = "rw",
        [NODEV] = "nodev",
        [DEV] = "dev",
        [NOEXEC] = "noexec",
        [EXEC] = "exec",
        [NOSUID] = "nosuid",
        [SUID] = "suid",
        [USER] = "user",
        [USERS] = "users",
        [NOUSER] = "nouser",
        [NOAUTO] = "noauto",
        [DEFAULTS] ="defaults",
        [END] = NULL
    };
    
    int so;
    char *argument = NULL;

    while (*option != 0) {
        so = getsubopt(&option, suboptions, &argument);
        if ((argument == NULL) && (so < USEPROXY))
            error(EXIT_FAILURE, 0,
                 "Option requires argument -%s", suboptions[so]);
        switch (so) {
        /*2005-05-29, werner, security fix, fixed mode
        case UID:
            struct passwd *pwd = getpwnam(argument);
            if (pwd == NULL) {
                args->uid = arg_to_int(argument, 10, suboptions[so]);
            } else {
                args->uid = pwd->pw_uid;
            }
            break;
        case GID:
            struct group *grp = getgrnam(argument);
            if (grp == NULL) {
                args->gid = arg_to_int(argument, 10, suboptions[so]);
            } else {
                args->gid = grp->gr_gid;
            }
            break;
        case FILE_MODE:
            args->file_mode = arg_to_int(argument, 8, suboptions[so]);
            break;
        case DIR_MODE:
            args->dir_mode = arg_to_int(argument, 8, suboptions[so]);
            break;*/
        case PROXY:
            split_proxy(&args->p_host, &args->p_port, argument);
            break;
        case USEPROXY:
            args->useproxy = 1;
            break;
        case NOUSEPROXY:
            args->useproxy = 0;
            break;
        case ASKAUTH:
            args->askauth = 1;
            break;
        case NOASKAUTH:
            args->askauth = 0;
            break;
        case LOCKS:
            args->locks = 1;
            break;
        case NOLOCKS:
            args->locks = 0;
            break;
        case MOUNTANYWAY:
            args->mountanyway = 1;
            break;
        case NOMOUNTANYWAY:
            args->mountanyway = 0;
            break;
        case RO:
            args->ro = MS_RDONLY;
            break;
        case RW:
            args->ro = 0;
            break;
        case USER:
        case USERS:
            args->user = 1;
            break;
        case NOUSER:
            args->user = 0;
            break;
        case NODEV:
    /*2005-07-09, werner, security fix,
      no more sense in these option, but mount may use it
            args->nodev = MS_NODEV;
            break; */
        case DEV:
    /*2005-07-09, werner, security fix,
      no more sense in these option, but mount may use it
            args->nodev = 0;
            break; */
        case NOEXEC:
    /*2005-07-09, werner, security fix,
      no more sense in these option, but mount may use it
            args->noexec = MS_NOEXEC;
            break; */
        case EXEC:
    /*2005-07-09, werner, security fix,
      no more sense in these option, but mount may use it
            args->noexec = 0;
            break; */
        case NOSUID:
    /*2005-07-09, werner, security fix,
      no more sense in these option, but mount may use it
            args->nosuid = MS_NOSUID;
            break; */
        case SUID:
     /*2005-07-09, werner, security fix,
      no more sense in these option, but mount may use it
           args->nosuid = 0;
            break; */
        case NOAUTO:
        case DEFAULTS:
            break;
        default:
            if (so == -1) {
                printf("Unknown suboption %s\n", argument);
                usage(PACKAGE_TARNAME);
                exit(EXIT_FAILURE);
            }
        }
    }
}


/* Checks fstab whether there is an entry for the mountpoint specified in args
 * and compares the values in args with the values in fstab.
 * If there is no such entry, or this entry does not allow user-mount, or the
 * values differ, an error message is printed and exit(EXIT_FAILURE) is called.
 * args : structure containing arguments and options from command line */
void check_fstab(const dav_args *args) {
    int err = 1;
    dav_args *n_args = new_args();
    setfsent();
    struct fstab *ft = getfsfile(dav_mpoint);
    if (ft == NULL)
        ft = getfsfile(basename(dav_mpoint));
    if (ft != NULL) {
        err = 0;
        err |= strcmp(dav_url, ft->fs_spec);
        err |= strcmp(DAV_VFS_TYPE, ft->fs_vfstype);
        get_options(n_args, ft->fs_mntops);
        /*2005-05-29, werner, security fix, fixed mode
        err |= (args->uid != n_args->uid);
        err |= (args->gid != n_args->gid);
        err |= (args->file_mode != n_args->file_mode);
        err |= (args->dir_mode != n_args->dir_mode);*/
        err |= (args->ro != n_args->ro);
        err |= (args->nodev != n_args->nodev);
        err |= (args->noexec != n_args->noexec);
        err |= (args->nosuid != n_args->nosuid);
        err |= (1 != n_args->user);
        err |= (args->useproxy != n_args->useproxy);
        err |= (args->askauth != n_args->askauth);
        err |= (args->locks != n_args->locks);
        err |= (args->mountanyway != n_args->mountanyway);
    }
    endfsent();
    dav_delete_args(n_args);
    if (err)
        error(EXIT_FAILURE, 0, "fstab does not allow you to mount\n"
                               "\t%s on %s.", dav_url, dav_mpoint);
}


/* Checks whether device is a character-oriented device special file with
 * major number CODA_PSDEV_MAJOR. If true, the minor number is returned,
 * else -1.
 * device : a filename to be tested
 * return value : minor number of the coda device file device
 *                -1 if device is not a coda device file or does not exist. */
int get_coda_minor(const char *device) {
    DBG1("      Checking %s: ", device);
    struct stat buf;
    if (stat(device, &buf) < 0) {
        DBG0("        no device\n");
        return -1;
    }
    if (S_ISCHR(buf.st_mode) && (major(buf.st_rdev) == CODA_PSDEV_MAJOR)) {
        DBG0("        davfs- or coda-device");
        return minor(buf.st_rdev);
    }
    DBG0("        other device");
    return -1;
}

 
/* Parses the string option for an option named dev or device and returns
 * the value of this option. If both options are present, the value of the
 * first one is returned. If none of this options is present, NULL is returned.
 * option : a comma separated list of options (like the options in fstab and
 *          in the -o option of the mount-programm).
 * return value : the value of the first option in option that is named
 *                dev or device, or NULL if none is found.
 * Note: The return value is just a pointer into the string pointed to by
 *       parameter option. If option is invalidated it gets invalid too. */
char *dev_from_options(char *option) {
    enum {
        DEV = 0,
        DEVICE,
        END
    };
    char *suboptions[] = {
        [DEV] = "dev",
        [DEVICE] = "device",
        [END] = NULL
    };
    int so;
    char *argument = NULL;
    while (*option != 0) {
        so = getsubopt(&option, suboptions, &argument);
        switch (so) {
        case DEV:
        case DEVICE:
            if (argument != NULL)
                return argument;
            break;
        default:
            break;
        }
    }
    return NULL;
}


/* Searches for a davfs-device that is not in use and stores a pointer to
 * its name-string in device.
 * To do this it scans the filesystems listet in dav_mounts for the names of
 * device special files that are mounted. As device special files that do not
 * represent real devices may not be listed in the device column of this
 * files, it also looks for options with the name dev or device to find the
 * names of devices in use.
 * All devices found are checked whether they are character special files
 * with major number CODA_PSDEV_MAJOR. If true their minor numbers are
 * marked as in use.
 * The first unused minor number that remains is returned. A string containig
 * the name of the davfs-device is build returned in *device.
 * device : after return it will point to the device name string.
 *          This string is newly allocated and the calling function is
 *          responsible to free it.
 * return value : the first free minor number */
int get_free_device(char **device) {
    char avail[MAX_CODADEVS];
    memset(avail, 0x00, sizeof(avail));
    char *dev;
    int minor;
    
    int i = 0;
    char *mounts = dav_mounts[i];
    while (mounts != NULL) {
        DBG1("    Scanning %s", mounts);
        FILE *mtab = setmntent(mounts, "r");
        if (mtab==NULL)
            error(EXIT_FAILURE, 0, "Could not open %s", mounts);
        struct mntent *mntent = getmntent(mtab);
        while (mntent != NULL) {
            minor = get_coda_minor(mntent->mnt_fsname);
            if (minor >= 0 && minor < MAX_CODADEVS)
                avail[minor] = 1;
            dev = dev_from_options(mntent->mnt_opts);
            if (dev != NULL) {
                minor = get_coda_minor(dev);
                if (minor >= 0 && minor < MAX_CODADEVS)
                    avail[minor] = 1;
            }
            mntent = getmntent(mtab);
        }
        endmntent(mtab);
        mounts = dav_mounts[++i];
    }

    for (minor=0; minor < MAX_CODADEVS ; minor++) {
        if(avail[minor] == 0) {
            if (asprintf(device, "%s%i", DAV_DEV_NAME, minor) < 0)
                abort();
            dav_pidfile = ne_concat(DAV_STATEDIR, "/",
                                    basename(*device), ".pid", NULL);
            if (access(dav_pidfile, F_OK) == 0) {
                NE_FREE(*device);
                NE_FREE(dav_pidfile);
            } else {
                return minor;
            }
        }
    }
    error(EXIT_FAILURE, 0, "No free davfs-device to mount");
    return 0;
}


/* If device node with name name does not exist, it is created with major
 * number CODA_PSDEV_MAJOR and minor number minor.
 * In case of an error a message is printed and exit(EXIT_FAILURE)
 * is called.
 * Owner and group of the device note is set to root and DAV_GROUP.
 * The device is made readable and writable by owner and group.
 * name  : name of the device node to be created.
 * minor : minor number of the device to dreate. */
void check_device(const char *name, int minor) {
    seteuid(0);
    if (access(name, F_OK) != 0) {
        DBG1("    Making device node %s", name);
        if (mknod(name, S_IFCHR, makedev(CODA_PSDEV_MAJOR, minor)) != 0)
            error(EXIT_FAILURE, errno, "Device %s could not be created", name);
    }
    struct group *grp = getgrnam(DAV_GROUP);
    if (grp == NULL)
        error(EXIT_FAILURE, 0, "Group %s does not exist.", DAV_GROUP);
    if (chown(name, 0, grp->gr_gid) < 0)
        error(EXIT_FAILURE, 0, "Could not change group of %s", name);
    if (chmod(name, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP) < 0)
        error(EXIT_FAILURE, 0, "Could not change file permissions of %s", name);
    seteuid(getuid());
}

/* Parses line for max. parmc white-space separated parameter tokens and
 * returns them in parmv[].
 * The '#' character marks the beginning of a comment and the rest of the line
 * is ignored. Parameters containing one of the characters ' ' (space), '\t'
 * (tab), '\', '"' or '#' must be enclosed in double quotes '"'. The characters
 * '\' and '"' within a parameter must be escaped by preceeding '\'.
 * line    : the line to be parsed
 * parmc   : the max. number of parameters. It is an error, if more than parmc
 *           parameters are found
 * parmv[] : the parameters found are returned in this array. They are newly
 *           allocated strings and must be freed by the calling function.
 * reurn value : the numer of parameters or -1 if an error occurs. */
int parse_line(char *line, int parmc, char *parmv[]) {
    enum {
        SPACE,
        SPACE_EXP,
        PARM,
        PARM_QUO,
        PARM_QUO_ESC,
        ERROR,
        END
    };
    int state = SPACE;
    int parm_no = 0;
    char buf[254];
    char *pos = buf;
    char *end = buf + 253;
    char *p = line;
    while ((state != END) && (state != ERROR)) {
        switch (state) {
        case SPACE:
            if ((*p == '\0') || (*p == '#')) {
                state = END;
            } else if (*p == '\"') {
                state = (parm_no < parmc) ? PARM_QUO : ERROR;
            } else if (*p == '\\') {
                state = ERROR;
            } else if (isspace(*p)) {
                ;
            } else if (isgraph(*p)) {
                *pos++ = *p;
                state = (parm_no < parmc) ? PARM : ERROR;
            } else {
                state = ERROR;
            }
            break;
        case SPACE_EXP:
            if (isspace(*p)) {
                state = SPACE;
            } else if (*p == '\0') {
                state = END;
            } else {
                state = ERROR;
            }
            break;
        case PARM:
            if ((*p == '#') || (*p == '\\') || (*p == '\"')) {
                state = ERROR;
            } else if (isspace(*p) || (*p == '\0')) {
                parmv[parm_no++] = ne_strndup(buf, pos - buf);
                pos = buf;
                state = SPACE;
            } else if (isgraph(*p)) {
                *pos++ = *p;
                state = (pos < end) ? PARM : ERROR;
            } else {
                state = ERROR;
            }
            break;
        case PARM_QUO:
            if (*p == '\\') {
                state = PARM_QUO_ESC;
            } else if (*p == '\"') {
                parmv[parm_no++] = ne_strndup(buf, pos - buf);
                pos = buf;
                state =  SPACE_EXP;
            } else if (isgraph(*p) || (*p == ' ') || (*p == '\t')) {
                *pos++ = *p;
                state = (pos < end) ? PARM_QUO : ERROR;
            } else {
                state = ERROR;
            }
            break;
        case PARM_QUO_ESC:
            if ((*p == '\"') || (*p == '\\')) {
                state = PARM_QUO;
                *pos++ = *p;
                state = (pos < end) ? PARM_QUO : ERROR;
            } else {
                state = ERROR;
            }
            break;
        }
        p++;
    }
    int i;
    if (state == END) {
        for (i = parm_no; i < parmc; i++)
            parmv[i] = NULL;
        return parm_no;
    } else {
        for (i = 0; i < parm_no; i++)
            NE_FREE(parmv[i]);
        return -1;
    }
}


/* Searches the file secrets for entries matching the proxy host or the url
 * in args and retrieves name and password from this entry and stores them
 * in the global variable dav_p_username or dav_username. If dav_p_username
 * or dav_username is not empty, the entry must match this name too. If name
 * is empty, the first matching entry is taken.
 * The uid of the file owner must equal the effective uid of the process
 * (root for system wide secrets file, the mounting user otherwise). It must
 * be read-writable by the owner only. Otherwise an error message is printed
 * and exit(EXIT_FAILURE) is called.
 * args : structure containig the arguments.
 * user : wether to search the secrets file in the home directory of the
 *        user (user = 1) or in the system configuration (user = 0).
 * return value : -1 if no secrets file could be found
 *                0 otherwise, even if no matching secrets was found in the
 *                secrets file. */
int read_secrets(dav_args *args, int user) {
    char *filename;
    if (user) {
        struct passwd *pw = getpwuid(getuid());
        filename = ne_concat(pw->pw_dir, "/.", PACKAGE_NAME, "/secrets", NULL);
    } else {
        filename = ne_concat(SYSCONFDIR, "/secrets", NULL);
    }
    struct stat st;
    if (stat(filename, &st) < 0) {
        DBG1("      %s could not be opened.", filename);
        NE_FREE(filename);
        return -1;
    }
    if (st.st_uid != geteuid())
        error(EXIT_FAILURE, 0, "%s has wrong owner.", filename);
    if ((st.st_mode & (S_IXUSR | S_IRWXG | S_IRWXO)) != 0)
        error(EXIT_FAILURE, 0, "%s has wrong permissions.", filename);
    DBG0("    Reading secrets from");
    DBG1("     %s.", filename);
    FILE *file = fopen(filename, "r");
    if (file == NULL) {
        DBG0("      could not be opened.");
        NE_FREE(filename);
        return -1;
    }
    
    size_t n = 0;
    char *line = NULL;
    int length = getline(&line, &n, file);
    int lineno = 1;
    int parmc = 3;
    char *parmv[parmc];
    int count;
    
    while (length > 0) {
        count = parse_line(line, parmc, parmv);
        /* A wellformed line must contain either
           - 0 parameters (empty line, comment) or
           - 2 parameters (resource name and user name) or
           - 3 parameters (resource name, user name and password). */
        if ((count != 0) && (count != 2) & (count != 3))
            error_at_line(EXIT_FAILURE, 0, filename, lineno,
                          "Malformed line");
        if ((count > 1) && (args->p_host != NULL)
                         && (strcmp(parmv[0], args->p_host) == 0)
                         && (dav_p_user == NULL)) {
            dav_p_user = parmv[1];
            dav_p_password = parmv[2];
        } else if ((count > 1) && (dav_url != NULL)
                               && (strcmp(parmv[0], dav_url) == 0)
                               && (dav_username == NULL)) {
            dav_username = parmv[1];
            dav_password = parmv[2];
        } else {
            NE_FREE(parmv[1]);
            NE_FREE(parmv[2]);
        }
        NE_FREE(parmv[0]);
        length = getline(&line, &n, file);
        lineno++;
    }
    
    NE_FREE(line);
    fclose(file);
    NE_FREE(filename);
    return 0;
}


/* Searches the config file for a proxy entry and stores it in args.
 * args : structure containing the arguments.
 * user : wether to search the config file in the home directory of the
 *        user (user = 1) or in the system configuration (user = 0).
 * return value: -1 if no config file could be found
 *               0 otherwise, even if no matching resource entry was found. */
int read_config(dav_args *args, int user) {
    char *filename;
    if (user) {
        struct passwd *pw = getpwuid(getuid());
        filename = ne_concat(pw->pw_dir, "/.",
                             PACKAGE_NAME, "/", PACKAGE_NAME, ".conf", NULL);
    } else {
        filename = ne_concat(SYSCONFDIR, "/", PACKAGE_NAME, ".conf", NULL);
    }
    DBG0("    Reading config from");
    DBG1("    %s", filename);
    FILE *file = fopen(filename, "r");
    if (file == NULL) {
        DBG0("      Could not be opened");
        NE_FREE(filename);
        return -1;
    }
    
    size_t n = 0;
    char *line = NULL;
    int length = getline(&line, &n, file);
    int lineno = 1;
    int parmc = 2;
    char *parmv[parmc];
    int count;
    
    while (length > 0) {
        count = parse_line(line, parmc, parmv);
        /* A wellformed line must contain either
           0 parameters (empty line, comment) or
           2 parameters ('proxy' and proxy-FQDN) */
        if ((count != 0) && (count != 2))
            error_at_line(EXIT_FAILURE, 0, filename, lineno,
                          "Malformed line");
        if ((count == 2) && (args->p_host == NULL)
                         && (strcmp(parmv[0], "proxy") == 0)) {
            split_proxy(&(args->p_host), &(args->p_port), parmv[1]); 
        }
        NE_FREE(parmv[0]);
        NE_FREE(parmv[1]);
        length = getline(&line, &n, file);
        lineno++;
    }
    
    NE_FREE(line);
    fclose(file);
    NE_FREE(filename);
    return 0;
}

/* Asks the user for the username.
 * return value : a newly allocated string, containing the username. */
char *ask_user() {
    char *s = NULL;
    size_t n = 0;
    ssize_t len = 0;
    printf("Username: ");
    len = getline(&s, &n, stdin);
    if (len < 0)
        abort();
    char *name = ne_strndup(s, len - 1);
    NE_FREE(s);
    return name;
}


/* Asks the user for the password. The input is not echoed to the
 * terminal.
 * return value : a newly allocated string, containing the password. */
char *ask_password() {
    char *s = NULL;
    size_t n = 0;
    ssize_t len = 0;
    printf("Password: ");
    struct termios termios;
    tcgetattr(STDOUT_FILENO, &termios);
    termios.c_lflag &= ~(ECHO | ICANON);
    tcsetattr(STDOUT_FILENO, TCSANOW, &termios);
    len = getline(&s, &n, stdin);
    termios.c_lflag |= (ECHO | ICANON);
    tcsetattr(STDOUT_FILENO, TCSANOW, &termios);
    if (len < 0)
        abort();
    printf("\n");
    char *password = ne_strndup(s, len - 1);
    NE_FREE(s);
    return password;
}


/* Get credentials for proxy or WebDAV-server from user interactivly-
 * user     : pointer to a string pointer, where a pointer to the newly
 *            allocated user string will be stored
 * passowrd : pointer to a string pointer, where a pointer to the newly
 *            allocated passowrd string will be stored
 * kind     : should be "server" or "proxy" to inform the user
 * url      : the url of the server or the FQDN of the proxy the
 *            credentials are needed for (to inform the user) */
void ask_auth(char **user, char **password, const char *kind, const char *url) {
    if (*user == NULL) {
        printf("Please enter the username for authentication with %s\n"
               "%s or hit enter for none.\n", kind, url);
        *user = ask_user();
    }
    printf("Please enter the password to authenticate %s with %s\n"
          "%s or hit enter for none.\n", *user, kind, url);
    *password = ask_password();
}


/* Creates the option string for the mtab-entry according the options stored
 * in args. The option string is newly allocated and the calling function is
 * responsible to free it.
 * args : structure containing the options
 * return value : the newly allocated option string */
char *get_mount_options(const dav_args *args) {
    DBG0("    Generate options string for real mount");
    char *o = ne_malloc(64);
    if (args->ro == 0) {
        strcpy(o,"rw");
    } else {
        strcpy(o,"ro");
    }
    if (args->nodev > 0)
        strcat(o, ",nodev");
    if (args->noexec > 0)
        strcat(o, ",noexec");
    if (args->nosuid > 0)
        strcat(o, ",nosuid");
    struct passwd *pw = getpwuid(getuid());
    char *opts;
    if (asprintf(&opts,
                 /*2005-05-29, werner, security fix, fixed mode
                 "%s,uid=%i,gid=%i,file_mode=%o,dir_mode=%o,device=%s,user=%s",
                 o, args->uid, args->gid, args->file_mode, args->dir_mode,*/
                 "%s,device=%s,user=%s",
                 o, args->device, pw->pw_name) < 0)
        abort();
    NE_FREE(o);
    return opts;
}


/* Functions for parsing command line and config-files and for mounting */
/* ==================================================================== */

void dav_delete_args(dav_args *args) {
    NE_FREE(args->device);
    NE_FREE(args->p_host);
    NE_FREE(args);
}


dav_args *dav_get_args(int argc, char *argv[]) {
    DBG_CMDLINE(argv);

    char *short_options = "Vho:";
    static const struct option options[] = {
        {"version", no_argument, NULL, 'V'},
        {"help", no_argument, NULL, 'h'},
        {"option", required_argument, NULL, 'o'},
        {0, 0, 0, 0}
    };

    DBG0("  Reading command line options");   
    /* char *optarg is a global variable supplied by getopt_long() */
    int o;
    dav_args *args = new_args();
    
    o = getopt_long(argc, argv, short_options, options, NULL);
    while (o != -1) {
        switch (o) {
        case 'V':
            printf("%s: %s  <%s>\n\n", argv[0], PACKAGE_STRING, DAV_HOME); 
            exit(EXIT_SUCCESS);
        case 'h':
            usage(argv[0]);
            exit(EXIT_SUCCESS);
        case 'o':
            get_options(args, optarg);
            break;
        default:
            error(EXIT_FAILURE, 0, "Unknown error parsing arguments");
        }
        o = getopt_long(argc, argv, short_options, options, NULL);
    }

    DBG0("  Reading command line arguments");
    /* The global variable optind is supplied by getopt_long(), and should now
       be the index of the first non-option argument in argv */
    int i = optind;
    switch (argc - i) {
    case 0:
    case 1:
        printf("Missing argument.\n");
        usage(PACKAGE_TARNAME);
        exit(EXIT_FAILURE);
    case 2:
        if (*argv[i] == '\"') {
            dav_url = ne_strndup(argv[i] + 1, strlen(argv[i]) -2);
        } else {
            dav_url = ne_strdup(argv[i]);
        }
        i++;
        dav_mpoint = canonicalize_file_name(argv[i]);
        break;
    default:
        printf("Too many arguments.\n");
        usage(PACKAGE_TARNAME);
        exit(EXIT_FAILURE);
    }
    if (dav_mpoint == NULL)
        error(EXIT_FAILURE, 0, "No mountpoint specified");
    if (dav_url == NULL)
        error(EXIT_FAILURE, 0, "No WebDAV-server specified");

    DBG_ARGS("command line");
    DBG_VARS("command line");
    return args;       
}


void dav_check_config(dav_args *args) {

    DBG0("  Checking for setuid");
    if (seteuid(0) < 0)
        error(EXIT_FAILURE, 0, "Can't get root permissions,"
                               " maybe program is not setuid");
    seteuid(getuid());
    

    if (getuid() != 0) {
        DBG0("  Checking group membership");
        struct group *grp = getgrnam(DAV_GROUP);
        if (grp == NULL)
            error(EXIT_FAILURE, 0, "Group %s does not exist.", DAV_GROUP);
        int is_member = (getgid() == grp->gr_gid);
        if (!is_member) {
            int ngroups = getgroups(0, NULL);
            gid_t *groups = (gid_t *) ne_calloc(ngroups * sizeof(gid_t));
            ngroups = getgroups(ngroups, groups);
            int i = 0;
            while (!is_member && (i < ngroups)) {
                is_member = (*(groups + i) == grp->gr_gid);
                ++i;
            }
            NE_FREE(groups);
        }
        if (!is_member)
            error(EXIT_FAILURE, 0, "You must be member of group %s",
                  DAV_GROUP);
        /* A non-root user might call mount.davfs directly and circumvent
         * the control of mount and fstab. So we better check args against 
         * fstab in the case of a non-root user. */
        DBG0("  Checking fstab for user mount");
        check_fstab(args);
    }

    DBG1("  Checking for %s", dav_mounts[0]);
    if (access(dav_mounts[0], F_OK) != 0) {
        dav_mounts[0] = dav_mounts[1];
        dav_mounts[1] = NULL;
    }

    DBG0("  Checking for double mounts");
    if (dav_still_mounted())
        error(EXIT_FAILURE, 0, "%s already mounted on %s", dav_url, dav_mpoint);

    DBG0("  Find a free davfs-device");
    int minor = get_free_device(&args->device);
    check_device(args->device, minor);

    DBG0("  Looking up proxy");
    if (args->useproxy) {
        if (getuid() != 0)
            read_config(args, 1);
        read_config(args, 0);
        if (args->p_host == NULL) {
            char *proxy = getenv("http_proxy");
            if (proxy != NULL)
                split_proxy(&(args->p_host), &(args->p_port), proxy);
        }
        if (args->p_host == NULL)
            args->useproxy = 0;
    }

    DBG0("  Looking up credentials");
    if (getuid() != 0)
        read_secrets(args, 1);
    seteuid(0);
    read_secrets(args, 0);
    seteuid(getuid());
    if (args->askauth && args->useproxy
                     && ((dav_p_user == NULL) || (dav_p_password == NULL)))
        ask_auth(&dav_p_user, &dav_p_password, "proxy", args->p_host);
    if (args->askauth && ((dav_username == NULL) || (dav_password == NULL)))
        ask_auth(&dav_username, &dav_password, "server", dav_url);

    DBG0("  Setting default file mode");
    /*2005-05-29, werner, security fix, fixed mode
    if (args->noexec)
        args->file_mode &= ~(S_IXUSR | S_IXGRP | S_IXOTH);
    if (args->nosuid)
        args->file_mode &= ~(S_ISUID | S_ISGID);
    dav_file_stat.st_uid = args->uid;
    dav_file_stat.st_gid = args->gid;
    dav_file_stat.st_mode = args->file_mode | S_IFREG;
    dav_file_stat.st_nlink = 1;
    dav_file_stat.st_blksize = 1024;
    dav_dir_stat.st_uid = args->uid;
    dav_dir_stat.st_gid = args->gid;
    dav_dir_stat.st_mode = args->dir_mode | S_IFDIR;
    dav_dir_stat.st_nlink = 1;
    dav_dir_stat.st_blksize = 1024;
    dav_dir_stat.st_size = 512;
    dav_noexec = args->noexec;*/
    dav_locks = args->locks;
    dav_file_stat.st_uid = getuid();
    dav_file_stat.st_gid = getgid();
    dav_file_stat.st_mode = S_IRUSR | S_IWUSR | S_IFREG;
    dav_file_stat.st_nlink = 1;
    dav_file_stat.st_blksize = 1024;
    dav_dir_stat.st_uid = getuid();
    dav_dir_stat.st_gid = getgid();
    dav_dir_stat.st_mode = S_IRWXU | S_IFDIR;
    dav_dir_stat.st_nlink = 1;
    dav_dir_stat.st_blksize = 1024;
    dav_dir_stat.st_size = 512;

    DBG_ARGS("config files");
    DBG_VARS("config files");
}


int dav_mount(int fd, const dav_args *args) {
    DBG0("  Mounting filesystem");
    struct coda_mount_data mountdata;
    mountdata.version = CODA_MOUNT_VERSION;
    mountdata.fd = fd;
    unsigned long int options = MS_MGC_VAL | args->ro | args->noexec
                                           | args->nosuid;
    seteuid(0);
    int ret = mount(dav_url, dav_mpoint, "coda", options,
                    (void *)&mountdata);
    seteuid(getuid());
    if (ret != 0) {
        error(0, 0, "Could not mount %s on %s", args->device, dav_mpoint);
        return -1;
    }

    DBG0("  Writing mtab-entry");
    struct mntent mntent;
    mntent.mnt_fsname = dav_url;
    mntent.mnt_dir = dav_mpoint;
    mntent.mnt_type = "coda";
    mntent.mnt_opts = get_mount_options(args);
    mntent. mnt_freq = 0;
    mntent. mnt_passno = 0;
    seteuid(0);
    FILE *mtab = setmntent(_PATH_MOUNTED, "a");
    if (mtab != NULL) {
        ret = addmntent(mtab, &mntent);
        endmntent(mtab);
    } else {
        ret = -1;
    }
    seteuid(getuid());
    NE_FREE(mntent.mnt_opts);
    if (ret != 0) {
        error(0, 0, "Could not write entry in mtab");
        return -1;
    }
    return 0;
}


void dav_save_mount_pid(void) {
    if (access(DAV_STATEDIR, F_OK) != 0) {
        seteuid(0);
        DBG1("  Creating %s", DAV_STATEDIR);
        if (mkdir(DAV_STATEDIR,
                  S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH | S_ISVTX) != 0)
            error(EXIT_FAILURE, 0, "Could not create %s", DAV_STATEDIR);
        struct group *grp = getgrnam(DAV_GROUP);
        if (grp == NULL)
            error(EXIT_FAILURE, 0, "Group %s does not exist.", DAV_GROUP);
        if (chown(DAV_STATEDIR, 0, grp->gr_gid) != 0)
            error(EXIT_FAILURE, 0, "Could not change group of %s", DAV_STATEDIR);
        if (chmod(DAV_STATEDIR,
                  S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH | S_ISVTX) != 0)
            error(EXIT_FAILURE, 0, "Could not change mode of %s", DAV_STATEDIR);
        seteuid(getuid());
    }
    DBG1("  Writing pid-file %s", dav_pidfile);
    FILE *fp = fopen(dav_pidfile, "w");
    if (fp == NULL)
        error(EXIT_FAILURE, 0, "Could not open pid-file");
    fprintf(fp, "%i\n", getpid());
    fclose(fp);
}


int dav_still_mounted() {
    DBG0("  Checking whether still mounted");
    FILE *mtab = setmntent(dav_mounts[0], "r");
    if (mtab == NULL)
        return 0;
    int found = 0;
    struct mntent *mt = getmntent(mtab);
    while ((mt != NULL) && (found == 0)) {
        if ((strcmp("coda", mt->mnt_type) == 0)
                    && (strcmp(dav_mpoint, mt->mnt_dir) == 0)
                    && (strcmp(dav_url, mt->mnt_fsname) == 0))
            found = 1;
        mt = getmntent(mtab);
    }
    endmntent(mtab);
    return found;
}


void dav_no_trail(char *str) {
    int len = strlen(str);
    while (len >= 1 && str[len - 1] == '/') {
        str[len - 1] = '\0';
        len = strlen(str);
    }
}


char *dav_get_tempnam(const char *prefix, int *fd) {
    char tmp[1024];
    snprintf(tmp, sizeof(tmp)-1, "%s/%s_%s_XXXXXX",
             DAV_TMP_DIR, PACKAGE_NAME, prefix);
    if((*fd=mkstemp(tmp))==-1) {
        perror(tmp);
        return NULL;
    }
    return ne_strdup(tmp);
}


/* start emacs stuff */
/* 
 * local variables: 
 * eval: (load-file "../tools/davfs-emacs.el") 
 * end: 
 */ 

