#!/usr/bin/perl

use 5.006;
use strict;
use warnings;

use Test::More;
$|++;

if( $> != 0 ) {
  plan skip_all => 'You must be root to run these tests';
} else {
  plan 'no_plan';
}

use File::Temp;

test_webdav("http://test.webdav.org/dav/");
test_webdav("http://test.webdav.org/auth-basic/","user3","user3");

sub test_webdav {
  my ($url,$user,$pass) = @_;
  my $mountpoint = mkdtemp( "/tmp/davtest_XXXXX" );
  my $pid;
  my $david;

  # need to NOT invoke the shell, or we won't know the pid of the mount
  # process
  my @mountargs = ($url,$mountpoint);
  push @mountargs, "-o", "username=$user,password=$pass"
   if ($user && $pass);

  if (($pid = fork()) == 0) {
    # child
    exec {"../mount.davfs"} ("mount.davfs",@mountargs)
      or print STDERR "couldn't exec foo: $!";

    # If we make it this far, there will be a spurious "No Tests Run"
    # message.  The way to fix this involves Test::Builder, and gets a
    # little complicated.
    exit 1;
  }
  sleep(1); # give dav a chance to warm up
  ok(defined $pid,"Created mount process (pid $pid)");

  # Make sure we're listed in the mtab
  # This might be a little linux specific
  {
    my $flag;
    open(my $mtab,"</etc/mtab") or die "Can't open mtab $!";
    while(<$mtab>) {
      if (m|^/dev/(davfs\d+)\s+$mountpoint|) {
	$flag++;
	$david = $1;
	last;
      }
    }
    close $mtab;
    ok($flag,"mtab says we're mounted at $mountpoint");
    exit unless $flag; # short circuit, or we'll get false successes
  }

  # Make a test directory to work in
  my $testpoint = mkdtemp( "$mountpoint/XXXXX" );

  # Try creating some directories
  ok(mkdir("$testpoint/$_")) for('a'..'d');

  # Test file creation and reading from
  for my $c ('A'..'D') {
    my $testfile = "$testpoint/$c";
    ok(open(my $f,">$testfile"),"file opened");
    my $line = $c x 1024;
    print $f $line."\n" for (1..10);
    close $f;
    ok(-e $testfile,"file exists");
    ok(open($f,"<$testfile"));
    my $lines = 0;
    my $flag = 0;
    while (<$f>) {
      $flag++ unless /^$c+$/;
      $lines++;
    }
    ok(!$flag); # lines match what we wrote out
    ok($lines==10,"right number of lines");
    ok(unlink("$testfile"),"unlink");
    ok(!-e $testfile,"notexist");
  }

  # create some well known files
  newfile("$testpoint/$_") for (1..4);

  # make sure the files we created, plus the directories from earlier,
  # including dot and dot dot exit.
  ok(opendir(my $dh, $testpoint));
  my @dots = readdir $dh;
  ok(eq_set(\@dots,[qw[. .. a b c d 1 2 3 4]]));
  closedir $dh;

  ok(rename("$testpoint/1","$testpoint/5"),"Renaming file 1 to file 5");
  ok(-e "$testpoint/5","File 5 exists");
  ok(!-e "$testpoint/1","File 1 does not exist");

  # Cleanup the files we just created
  ok(unlink("$testpoint/$_"),"unlink") for (2..5);

  # And those pesky directories
  ok(rmdir("$testpoint/$_"),"rmdir") for('a'..'d');

  # And the working directory we made
  ok(rmdir($testpoint));

  # The pid we get back from fork isn't the one that davfsd ends up
  # being because it forks itself.  Joy!
  open (my $davpid,"</tmp/$david.pid") or die $!;
  $pid = <$davpid>;
  close $davpid;

  # Now get rid of the mount.davfs and force the unmount
  is( kill(15, $pid), 1, "Sent SIGTERM to process $pid");

  sleep 1; # give it a second to calm down

  # If we can't remove the mountpoint, it probably failed.to unmount
  ok(rmdir($mountpoint),"Remove mountpoint ($mountpoint)");
}

sub newfile {
  my $file = shift;
  open(my $z,">$file");
  close $z;
}
