/* 
** DAV directory chache
 */

/*
 *  This file is part of davfs2.
 *
 *  davfs2 is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  davfs2 is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with davfs2; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */


#ifndef __DAV_CACHE_H__
#define __DAV_CACHE_H__


/* Constants */
/*===========*/

/* A mask to separate access bits from mode. */
#define DAV_A_MASK (S_IRWXU | S_IRWXG | S_IRWXO | S_ISUID | S_ISGID | S_ISVTX)


/* Data Types */
/*============*/

/* A node in the cache. It represents a directory or a regular file.
 * Nodes that are direct childs of the same directory-node are linked together
 * by the next-pointer to a linked list.
 * The hierarchical directory structure is formed by pointers to the parent
 * direcotry and to the first member in the linked list of direct childs.
 * Nodes are also stored in a hash table. The hash is derived from the nodes
 * address. Nodes with the same hash value are linked together by table_next. */
typedef struct dav_node dav_node;
struct dav_node {
    dav_node *parent;       /* The parent directory. */
    dav_node *childs;       /* The first node in the linked list of childs. */
    dav_node *next;         /* The next node in the linked list of childs. */
    dav_node *table_next;   /* The next node in the linked list of nodes with
                               the same hash value. */
    char *name;             /* The name of the node (not path) without leading
                               or trailing slash. The name of the root node is
                               the empty string. */
    char *cache_path;       /* Path of the file where the contents of the node
                               is cached. These files are only created when
                               needed, so cache_path may be NULL.
                               For directories: The file containing the
                               direntries must be created by the interface
                               (e.g. dav_coda), as its structure is defined by
                               the kernel. But the cache module will remove the
                               file, whenever the contents of a directory
                               changes.
                               For regular files: a local copy of the file. */
    off_t size;             /* Size of the contents of the node.
                               Regular files: Initially set to the value of the
                               content-lenght header. When opened for write it
                               is updated to the size of the cached file.
                               Directories: It must be computed by the interface
                               to the kernel module, when the cache file is
                               created. */
    mode_t mode;            /* mode, uid and gid are initially set to the   */
    uid_t uid;              /* default values, but may be changed by        */
    gid_t gid;              /* dav_setattr().                               */
    time_t atime;           /* Initially set to the value of mtime.
                               Dirs: Updated to system time by some upcalls.
                               Files: When a local copy exists updated to
                                      atime of this. */
    time_t mtime;           /* Initially set to the last-modified time from the
                               server, or the system time if created locally.
                               Files: When a local cache file exists updated to
                                      mtime of this cache file. 
                                      mtime > smtime indicates a dirty file. */
    time_t ctime;           /* Initially set to the value of mtime. Updated
                               by dav_setattr(). */
    union {
        int nref;           /* Dirs: Number of subdirectories, including "." and
                                     "..". */
        int locked;         /* Files: File is locked. */
    };
    int o_read;             /* Number of opens for read-only without close. */
    int o_write;            /* Number of opens for write without close. */
    int fd_read;            /* File descriptor for the local copy opened for read. */
    int fd_write;           /* File descriptor for the local copy opened for write. */
                            /* File descriptor > 0 indicate an open local copy.
                               fd_xxx will be maintaines by the interface
                               module, but before deleting a node the cache
                               module will close them if necessary. */ 
    union {
        time_t utime;       /* Dirs: Time when the node has to be updated. */
        time_t smtime;      /* Files: Last-modifies time from the server.
                                      0 for files created locally.
                                      If the last-modified time form the server
                                      is greater than smtime, the file has been
                                      changed remotely. If the file is dirty or
                                      opened for write we have got a lost
                                      update problem. */
    };
    char *etag;             /* Files: Etag from the server; used for
                                      conditional GET requests. */
    int created;            /* Files: Created localy and not yet stored on the
                                      server.
                                      Note: Some server create an empty file
                                            when locking. Maybe the RFC will
                                            be changed this way. */
};


/* Prototype of a function to inform the kernel about removed file nodes. The
   interface to the kernel should define such a function and register it using
   dav_register_kernel_interface(). */
typedef void (*dav_zap_file_fn)(dav_node *node);


/* Public global variables */
/*=========================*/

/* Alignment boudaries of dav_node in byte.
   Used to compute a hash value and file numbers from node pointers. */
extern int alignment;


/* Function prototypes */
/*=====================*/

/* Initializing and closing the cache. */

/* Sets a lot of private global variables that govern the begaviour of the
   cache, taking the values from args.
   If no dav_zap_file_fn-function has been registered it registers a dummy
   function instead.
   Creates the root node and restores the nodes from the permanent cache.
   Finally it retrieves the root directory entries from the server.
   If the connection to the server fails because of authentication failure
   it prints an error and terminates the programm. If the connection fails
   due to other reasons, it will nevertheless return with success, as the
   filesystem can be mounted, but will only get useable when the connection
   comes up.
   args : conatins all teh values necessary to initialize the cache. */
void dav_init_cache(const dav_args *args);


/* Tries to write back to the server all open files that have been changed or
   newly created. If a file from cache can not be stored back to the server,
   a local backup file is created. All local copies of files and the necessary
   directories are stored in the permanent cache. A new index file of the
   permanent cache is created.
   Finally it frees all nodes. */
void dav_close_cache();


/* Scans the hash table for file nodes to be restored and saves them back on
   the server. It must be called regularly.
   TODO: A more sophisticated restore strategy is needed as the file system
         will be blocked for a long time when many files are to be stored. */
void dav_idle();


/* Registers the kernel_interface.
   Sets the pointer to the zap_file_function.
   Return value: a string with the name of the cache directory. */
const char *dav_register_kernel_interface(dav_zap_file_fn fn);


/* Upcalls from the kernel, via the interface module. */

/* All these functions have some common behaviour. There are some exceptions
   (e.g. for dav_close() and dav_root()) that are described there.
   Except from dav_root() all this function take a parameter (*node or
   *parent) that is a pointer to a dav_node, and the uid of the user that
   tries to access the file system.

   The functions will test, whether the node exists and whether the user
   is allowed to perform the operation requested. While doing this all the
   directories above node up to root node will be updated if necessary. The
   same holds for node if it is a directory. Update is necessary if there
   are known changes since the last update or if expire seconds have
   elapsed since the last update.

   Common access permission requirements:
   The user uid must have execute permission on all the directories above
   the node up to root node.
   Access bits are allways tested for others, if uid is the owner of the node
   for owner, and if the user belongs to the group of node for group.

   Common parameters (not all of this must be present in one function):
   dav_node **nodep : Used to return a pointer to a node.
   dav_node *node   : A pointer to the node that shall be acted on.
   dav_node *parent : A pointer to the parent directory of the node in question.
   const char *name : Name of the node in question. It's is just one name, not
                      a path, and without any trailing or leading slashes.
   uid_t uid        : ID of the user that requested the operation.

   Common return values:
   0       Success.
   EACCES  Access is denied to user uid according to the acces bits.
   EAGAIN  A temporary failure in the connection to the WebDAV server.
   EBUSY   The action on the node can not be performed, as somebody else uses
           it allready in a way that would collide with the requested action.
   EEXIST  Cration of a new node is requested with flag O_EXCL, but it exists.
   EINVAL  One of the parameters has an invalid value.
   EIO     Error performing I/O-operation. Usually there are problems in the
           communication with the WebDAV server.
   EISDIR  The node is a directory but should not be.
   ENOENT  The file or directory does not exist.
   ENOSPC  There is not enough space on the server.
   ENOTDIR The node is not a directory but the requested action needs it to be.
   EPERM   The reuested action is not allowed to user uid. */


/* Tests whether user uid has access described by how to node.
   int how : A bit mask describing the kind of acces. It may be any combination
             of R_OK, W_OK, X_OK and F_OK. */
int dav_access(dav_node *node, uid_t uid, int how);


/* This function does not check the permissions of user uid. It assumes that the
   user has a valid file handle (file handles are handled by the coda kernel
   module, so davfs2 can't check this).
   If node is a directory, only the access time is set to the system time.
   If node is a file, the counters for pending opens are decremented. flags
   tells whether the file was opened for read or write. Access time and
   modification time are updated, taking the values from the cached file.
   It does not restore the file to the server. dav_idle() must be called
   from time to time to do this.*/
int dav_close(dav_node *node, uid_t uid, int flags);


/* Creates a new file node with name name in directory parent. The file is
   locked on the server allthough not existent there (lock-null resource).
   If the file allready exists, error EEXIST is returned,
   otherwise the file is locked on the server.
   uid and gid of the node are set to uid and the primary group of uid.
   mode is set to mode, but bits not allowed by file_umask are reset.
   Permissions: User uid must have write access to parent.
   If successful a pointer to the node is returned in nodep.
   int excl    : Ignored; the file must not yet exist, whatever excl says.
   mode_t mode : The mode of the newly created node. Will be anded with
                 ~file_umask. */
int dav_create(dav_node **nodep, dav_node *parent, const char *name,
               uid_t uid, int excl, mode_t mode);


/* Permissions: User uid must have read access to the parent of node.
   If node is a regular file, the file times are updated to the values of the
   cached file, if existent.
   It is the job of the interface module, to translate the attribute values
   from the node structure according the needs of the kernel module. */
int dav_getattr(dav_node *node, uid_t uid);


/* Permissions: User uid must have read access to parent.
   If successful a pointer to the node name in directory parent is returned
   in nodep. */
int dav_lookup(dav_node **nodep, dav_node *parent, const char *name, uid_t uid);


/* Permissions: User uid must have write access to parent.
   The directory must not yet exist.
   A new collection is created on the WebDAV server and a new node is created.
   The uid of the node is set to uid and the group to the primary group of uid.
   The mode is set to mode, but bits not allowed by dir_umask are reset.
   A pointer to the new directory node is returned in nodep.
   mode_t mode : The mode of the new directory. Will be anded with
                 ~dir_umask. */
int dav_mkdir(dav_node **nodep, dav_node *parent, const char *name, uid_t uid,
              mode_t mode);


/* flags must be one of O_RDONLY, O_WRONLY or O_RDWR, and may be ored with
   O_TRUNC.
   Permissions: User uid muss have acces to node according the action in flags.
   If node is a directory, th only action is to set the access time to the
   system time.
   For a regular file, the cache file is created and/or updated from the
   WebDAV server if necessary. If the file is opend for writing it is locked on
   the server. In case O_TRUNC is set in flags or the node has been created by
   dav_create(), no update of the cache file is done.
   The counters of pending opens are incremented according to the kind of
   access defined in flags. Access time is set to the system time.
   The cache module is not involved in the real read and write to the cached
   file. This will be done by the kernel module (coda) or by the interface
   module. It is also the job of the interface module to send a reference to
   the cache file to the kernel in a way appropriate for the kernel module.
   The interface module may store a file descriptor of the open cache file
   in node->fd_read or node->fd_write. */
int dav_open(dav_node *node, uid_t uid, int flags);


/* Permissions: User uid must have write access to parent.
   Unlocks and deletes the file on the server and removes the file node.
   If there are still pending opens, the node is removed but not deleted,
   so the kernel interface is still able to close regularly.
   If the file dan not be deleted on the server or unlocked, EIO is returned. */
int dav_remove(dav_node *parent, const char *name, uid_t uid);


/* Permissions: User uid must have write access to src-parent and dst_parent.
   There must be no pending opens on dst_name.
   Moves src_name in directory src_parent to dst_name in directory dst_parent
   on the server and in the cache. If dst_name exists it is deleted on the
   server and the node is removed.
   Moves into the local backup directory are not allowed. */
int dav_rename(dav_node *src_parent, const char *src_name,
               dav_node *dst_parent, const char *dst_name, uid_t uid);


/* Permissions: User uid must have write acces to parent.
   The directory name is removed on the server and the node is removed.
   The directory must be empty and not the local backup directory. */
int dav_rmdir(dav_node *parent, const char *name, uid_t uid);


/* Permissions: User uid must be root.
   This call may only be issued by the mount process and only root is allowed
   to mount. A pointer to the root node is returned in nodep. */
int dav_root(dav_node **nodep, uid_t uid);


/* Permissions: Only root and the owner are allowed to change owner, group
                or mode.
   Sets mode, owner, group, atime, mtime or ctime to new values. Which
   attributes are to be changed is defined by the parameters sx, that must be
   non zero in this case.
   There are some restrictions:
     - mode must not contain bits, prohibited by dir_umask or file_umask.
     - only root may change the owner.
     - if changing the gid, a non root user must be member of the new group.
   node->ctime will allways be set to the system time.
   If one of the parameters has an invalid value, none will be changed.
   The changes are only done in the cache not on the server. An exception
   is the execute bit which will be changed on the server too. */
int dav_setattr(dav_node *node, uid_t uid, int sm, mode_t mode,
                int so, uid_t owner, int sg, gid_t gid, int sat, time_t atime,
                int smt, time_t mtime);


int dav_sync(dav_node *node);

#endif
