/* 
** Interface to the Coda kernel module.
 */

/*
 *  This file is part of davfs2.
 *
 *  davfs2 is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  davfs2 is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with davfs2; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* Most of this file has been copied from the Coda kernel header coda.h,
 * Linux 2.6.8. So here is the original copyright notice from coda.h.
 */

/* Original copyright notice from coda.h. */
/* ====================================== */

/* 
   You may distribute this file under either of the two licenses that
   follow at your discretion.
*/

/* BLURB lgpl

                           Coda File System
                              Release 5

          Copyright (c) 1987-1999 Carnegie Mellon University
                  Additional copyrights listed below

This code is distributed "AS IS" without warranty of any kind under
the terms of the GNU Library General Public Licence Version 2, as
shown in the file LICENSE, or under the license shown below. The
technical and financial contributors to Coda are listed in the file
CREDITS.

                        Additional copyrights 
*/

/*

            Coda: an Experimental Distributed File System
                             Release 4.0

          Copyright (c) 1987-1999 Carnegie Mellon University
                         All Rights Reserved

Permission  to  use, copy, modify and distribute this software and its
documentation is hereby granted,  provided  that  both  the  copyright
notice  and  this  permission  notice  appear  in  all  copies  of the
software, derivative works or  modified  versions,  and  any  portions
thereof, and that both notices appear in supporting documentation, and
that credit is given to Carnegie Mellon University  in  all  documents
and publicity pertaining to direct or indirect use of this code or its
derivatives.

CODA IS AN EXPERIMENTAL SOFTWARE SYSTEM AND IS  KNOWN  TO  HAVE  BUGS,
SOME  OF  WHICH MAY HAVE SERIOUS CONSEQUENCES.  CARNEGIE MELLON ALLOWS
FREE USE OF THIS SOFTWARE IN ITS "AS IS" CONDITION.   CARNEGIE  MELLON
DISCLAIMS  ANY  LIABILITY  OF  ANY  KIND  FOR  ANY  DAMAGES WHATSOEVER
RESULTING DIRECTLY OR INDIRECTLY FROM THE USE OF THIS SOFTWARE  OR  OF
ANY DERIVATIVE WORK.

Carnegie  Mellon  encourages  users  of  this  software  to return any
improvements or extensions that  they  make,  and  to  grant  Carnegie
Mellon the rights to redistribute these changes without encumbrance.
*/

/*
 *
 * Based on cfs.h from Mach, but revamped for increased simplicity.
 * Linux modifications by 
 * Peter Braam, Aug 1996
 */

/* End of copyright notice form coda.h. */

/* This file merges data definitions from Coda Kernel Version 2 and Version 3.
 * To do this some structures are defined twice, as "coda2_xxx" and coda3_xxx".
 */


#ifndef _DAV_CODA_H_
#define _DAV_CODA_H_


/* Constants */
/*===========*/

/* This two constants are from coda_psdev.h */
#define CODA_PSDEV_MAJOR 67
#define MAX_CODADEVS  5

#define CIOC_KERNEL_VERSION _IOWR('c', 10, size_t)
#define CODA_MAXNAMLEN  255
#define CODA_MAXPATHLEN 1024

/* these are Coda's version of O_RDONLY etc combinations
 * to deal with VFS open modes */
#define	C_O_READ    0x001
#define	C_O_WRITE   0x002
#define C_O_TRUNC   0x010
#define C_O_EXCL    0x100
#define C_O_CREAT   0x200

/* these are to find mode bits in Venus */ 
#define C_M_READ  00400
#define C_M_WRITE 00200

/* for access Venus will use */
#define C_A_C_OK    8               /* Test for writing upon create.  */
#define C_A_R_OK    4               /* Test for read permission.  */
#define C_A_W_OK    2               /* Test for write permission.  */
#define C_A_X_OK    1               /* Test for execute permission.  */
#define C_A_F_OK    0               /* Test for existence.  */

/* File types */
#define	CDT_UNKNOWN 0
#define	CDT_FIFO    1
#define	CDT_CHR     2
#define	CDT_DIR     4
#define	CDT_BLK     6
#define	CDT_REG     8
#define	CDT_LNK    10
#define	CDT_SOCK   12
#define	CDT_WHT    14

/* lookup flags; not used by davfs2.
   davfs2 will do *no* case translation. */
#define CLU_CASE_SENSITIVE     0x01
#define CLU_CASE_INSENSITIVE   0x02


/* Data Types */
/*============*/

typedef dev_t cdev_t;
typedef u_int32_t vuid_t;
typedef u_int32_t vgid_t;
#define DAV_ID_MAX  0xffffffff  /* Coda likes to set variables of type
                                   vgid_t to -1 and we have to detect. */

struct coda_mount_data {
	int		version;
	int		fd;
};

struct venus_dirent {
    u_int32_t d_fileno;     /* file number of entry */
    u_int16_t d_reclen;     /* length of this record */
    u_int8_t  d_type;       /* file type, see below */
    u_int8_t  d_namlen;     /* length of string in d_name */
    char d_name[CODA_MAXNAMLEN + 1];    /* name must be no longer than this */
};
#undef DIRSIZ
#define DIRSIZ(dp)  ((sizeof (struct venus_dirent) - (CODA_MAXNAMLEN + 1)) + \
                     (((dp)->d_namlen + 1 + 3) & ~3))

/* CodaFid has changed from CODA_KERNEL_VERSION 2 to 3, so we have to
   define both types. davfs2 will insert:
     vol = DAV_VOL
     vnode = DAV_VNODE
     id: a pointer to the dav_node. */
struct Coda2Fid {
    u_int32_t vol;
    u_int32_t vnode;
    u_int32_t id;
};
struct Coda3Fid {
    u_int32_t vol;
    u_int32_t vnode;
    union {
        u_int32_t id;
        u_int64_t may_be_a_64_bit_pointer;
    };
};

struct coda_cred {
    vuid_t cr_uid, cr_euid, cr_suid, cr_fsuid;
    vgid_t cr_groupid, cr_egid, cr_sgid, cr_fsgid;
};

/* Vnode types.  VNON means no type.
   davfs2 currently only supports C_VREG and C_VDIR. */
enum coda_vtype	{ C_VNON, C_VREG, C_VDIR, C_VBLK,
                  C_VCHR, C_VLNK, C_VSOCK, C_VFIFO, C_VBAD };

struct coda_vattr {
    long            va_type;        /* vnode type (for create) */
    u_short         va_mode;        /* files access mode and type */
    short           va_nlink;       /* number of references to file */
    vuid_t          va_uid;         /* owner user id */
    vgid_t          va_gid;         /* owner group id */
    long            va_fileid;      /* file id */
    u_quad_t        va_size;        /* file size in bytes */
    long            va_blocksize;   /* blocksize preferred for i/o */
    struct timespec va_atime;       /* time of last access */
    struct timespec va_mtime;       /* time of last modification */
    struct timespec va_ctime;       /* time file changed */
    u_long          va_gen;         /* generation number of file */
    u_long          va_flags;       /* flags defined for file */
    dev_t           va_rdev;        /* device special file represents */
    u_quad_t        va_bytes;       /* bytes of disk space held by file */
    u_quad_t        va_filerev;     /* file modification number */
};

/* structure used by CODA_STATFS for getting cache information from venus */
struct coda_statfs {
    int32_t f_blocks;
    int32_t f_bfree;
    int32_t f_bavail;
    int32_t f_files;
    int32_t f_ffree;
};


/* Kernel <--> davfs2 communications. */
#define CODA_ROOT          2
#define CODA_OPEN_BY_FD    3 
#define CODA_OPEN          4
#define CODA_CLOSE         5
#define CODA_IOCTL         6 /* Not supported. */
#define CODA_GETATTR       7
#define CODA_SETATTR       8
#define CODA_ACCESS        9
#define CODA_LOOKUP       10
#define CODA_CREATE       11
#define CODA_REMOVE       12
#define CODA_LINK         13 /* Not supported. */
#define CODA_RENAME       14
#define CODA_MKDIR        15
#define CODA_RMDIR        16
#define CODA_SYMLINK      18 /* Not supported. */
#define CODA_READLINK     19 /* Not supported. */
#define CODA_FSYNC        20 /* Does nothing, returns success */
#define CODA_VGET         22 /* Not supported. */
#define CODA_SIGNAL       23 /* Not supported. */
#define CODA_REPLACE      24 /* Not used */
#define CODA_FLUSH        25 /* Not used */
#define CODA_PURGEUSER    26 /* Not used */
#define CODA_ZAPFILE      27
#define CODA_ZAPDIR       28 /* Not used */
#define CODA_PURGEFID     30 /* Not used */
#define CODA_OPEN_BY_PATH 31
#define CODA_RESOLVE      32 /* Not documented -> not supported. */
#define CODA_REINTEGRATE  33 /* Not documented -> not supported. */
#define CODA_STATFS       34 /* davfs2 fakes some big free space. */
#define CODA_STORE        35 /* Does nothing, returns success */
#define CODA_RELEASE      36 /* Seems to be same as CODA_CLOSE. */
#define CODA_NCALLS       37


/* This are the data structures for communication with coda. Due to
   changes in the coda interface, most are declared twice for different
   coda versions. dav_coda.c defines macros to access the memeber
   variables that are relevant for davfs2. The comments refer to
   the macros that use the declaration. */
struct coda2_in_hdr {
    u_int32_t opcode;         /* OC */
    u_int32_t unique;         /* ID */
    u_int16_t pid;
    u_int16_t pgid;
    u_int16_t sid;
    struct coda_cred cred;    /* UID */
};
struct coda3_in_hdr {
    u_int32_t opcode;         /* OC */
    u_int32_t unique;         /* ID */
    pid_t pid;
    pid_t pgid;
    vuid_t uid;               /* UID */
};

struct coda_out_hdr {         /* size RES */
    u_int32_t opcode;
    u_int32_t unique;	
    u_int32_t result;         /* RES */
};


struct coda_root_in {
    struct coda3_in_hdr in;
};

struct coda2_root_out {       /* size FID_O */
    struct coda_out_hdr oh;
    struct Coda2Fid VFid;
};
struct coda3_root_out {       /* size FID_O */
    struct coda_out_hdr oh;
    struct Coda3Fid VFid;
};


struct coda_open_by_fd_in {
    struct coda3_in_hdr ih;
    struct Coda3Fid VFid;
    int        flags;
};

struct coda_open_by_fd_out {  /* size FD */
    struct coda_out_hdr oh;
    int fd;                   /* FD */
};


struct coda2_open_in {
    struct coda2_in_hdr ih;
    struct Coda2Fid VFid;
    int	flags;
};
struct coda3_open_in {
    struct coda3_in_hdr ih;
    struct Coda3Fid VFid;
    int	flags;
};

struct coda_open_out {        /* size INO */
    struct coda_out_hdr oh;
    cdev_t	dev;              /* DEV */
    ino_t	inode;              /* INO */
};


struct coda_close_in {
    struct coda3_in_hdr ih;
    struct Coda3Fid VFid;
    int	flags;
};

struct coda_close_out {
    struct coda_out_hdr out;
};


struct coda_ioctl_in {
    struct coda3_in_hdr ih;
    struct Coda3Fid VFid;
    int	cmd;
    int	len;
    int	rwflag;
    char *data;
};

struct coda_ioctl_out {
    struct coda_out_hdr oh;
    int	len;
    caddr_t	data;
};


struct coda_getattr_in {
    struct coda3_in_hdr ih;
    struct Coda3Fid VFid;
};

struct coda_getattr_out {     /* size GET_ATTR */
    struct coda_out_hdr oh;
    struct coda_vattr attr;   /* GET_ATTR */
};


struct coda_setattr_in {
    struct coda3_in_hdr ih;
    struct Coda3Fid VFid;
    struct coda_vattr attr;
};

struct coda_setattr_out {
    struct coda_out_hdr out;
};


struct coda_access_in {
    struct coda3_in_hdr ih;
    struct Coda3Fid VFid;
    int	flags;
};

struct coda_access_out {
    struct coda_out_hdr out;
};


struct coda2_lookup_in {
    struct coda2_in_hdr ih;
    struct Coda2Fid VFid;     /* VOL_I, VN_I, FID_I */
    int name;                 /* FLAGS, NAME */
    int flags;
};
struct coda3_lookup_in {
    struct coda3_in_hdr ih;
    struct Coda3Fid VFid;     /* VOL_I, VN_I, FID_I */
    int name;                 /* FLAGS, NAME */
    int flags;
};

struct coda2_lookup_out {     /* size TYPE */
    struct coda_out_hdr oh;
    struct Coda2Fid VFid;     /* VOL_O, VN_O, FID_O */
    int	vtype;                /* TYPE */
};
struct coda3_lookup_out {     /* size TYPE */
    struct coda_out_hdr oh;
    struct Coda3Fid VFid;     /* VOL_O, VN_O, FID_O */
    int	vtype;                /* TYPE */
};


struct coda2_create_in {
    struct coda2_in_hdr ih;
    struct Coda2Fid VFid;
    struct coda_vattr attr;   /* ATTR_I */
    int excl;                 /* EXCL, DIRNAME */
    int mode;                 /* MODE */
    int name;                 /* C_NAME */
};
struct coda3_create_in {
    struct coda3_in_hdr ih;
    struct Coda3Fid VFid;
    struct coda_vattr attr;   /* ATTR_I */
    int excl;                 /* EXCL, DIRNAME */
    int mode;                 /* MODE */
    int name;                 /* C_NAME */
};

struct coda2_create_out {     /* size ATTR_O */
    struct coda_out_hdr oh;
    struct Coda2Fid VFid;
    struct coda_vattr attr;   /* ATTR_O */
};
struct coda3_create_out {     /* size ATTR_O */
    struct coda_out_hdr oh;
    struct Coda3Fid VFid;
    struct coda_vattr attr;   /* ATTR_O */
};


struct coda_remove_in {
    struct coda3_in_hdr ih;
    struct Coda3Fid VFid;
    int name;
};

struct coda_remove_out {
    struct coda_out_hdr out;
};


struct coda_link_in {
    struct coda3_in_hdr ih;
    struct Coda3Fid sourceFid;
    struct Coda3Fid destFid;
    int tname;
};

struct coda_link_out {
    struct coda_out_hdr out;
};


struct coda2_rename_in {
    struct coda2_in_hdr ih;
    struct Coda2Fid sourceFid;
    int 	srcname;
    struct Coda2Fid destFid;  /* DST_FID */
    int 	destname;           /* DST_NAME */
};
struct coda3_rename_in {
    struct coda3_in_hdr ih;
    struct Coda3Fid sourceFid;
    int 	srcname;
    struct Coda3Fid destFid;  /* DST_FID */
    int 	destname;           /* DST_NAME */
};

struct coda_rename_out {
    struct coda_out_hdr out;
};


struct coda_mkdir_in {
    struct coda3_in_hdr ih;
    struct Coda3Fid VFid;
    struct coda_vattr attr;
    int	   name;
};

struct coda_mkdir_out {
    struct coda_out_hdr oh;
    struct Coda3Fid VFid;
    struct coda_vattr attr;
};


struct coda_rmdir_in {
    struct coda3_in_hdr ih;
    struct Coda3Fid VFid;
    int name;
};

struct coda_rmdir_out {
    struct coda_out_hdr out;
};


struct coda_symlink_in {
    struct coda3_in_hdr ih;
    struct Coda3Fid VFid;
    int srcname;
    struct coda_vattr attr;
    int tname;
};

struct coda_symlink_out {
    struct coda_out_hdr out;
};


struct coda_readlink_in {
    struct coda3_in_hdr ih;
    struct Coda3Fid VFid;
};

struct coda_readlink_out {
    struct coda_out_hdr oh;
    int	count;
    caddr_t	data;
};


struct coda_fsync_in {
    struct coda3_in_hdr ih;
    struct Coda3Fid VFid;
};

struct coda_fsync_out {
    struct coda_out_hdr out;
};


struct coda_vget_in {
    struct coda3_in_hdr ih;
    struct Coda3Fid VFid;
};

struct coda_vget_out {
    struct coda_out_hdr oh;
    struct Coda3Fid VFid;
    int	vtype;
};


/* CODA_SIGNAL is out-of-band, doesn't need data. */


struct coda_replace_out { /* venus->kernel call */
    struct coda_out_hdr oh;
    struct Coda3Fid NewFid;
    struct Coda3Fid OldFid;
};


/* CODA_FLUSH is a venus->kernel call */


struct coda2_purgeuser_out { /* venus->kernel call */
    struct coda_out_hdr oh;
    struct coda_cred cred;
};
struct coda3_purgeuser_out { /* venus->kernel call */
    struct coda_out_hdr oh;
    vuid_t uid;
};


struct coda_zapfile_out { /* venus->kernel call */  
    struct coda_out_hdr oh;
    struct Coda3Fid CodaFid;
};


struct coda_zapdir_out { /* venus->kernel call */	  
    struct coda_out_hdr oh;
    struct Coda3Fid CodaFid;
};


struct coda_purgefid_out { /* venus->kernel call */ 
    struct coda_out_hdr oh;
    struct Coda3Fid CodaFid;
};


struct coda_open_by_path_in {
    struct coda3_in_hdr ih;
    struct Coda3Fid VFid;
    int	flags;
};

struct coda_open_by_path_out {
    struct coda_out_hdr oh;
    int path;
};


struct coda_statfs_in {
    struct coda3_in_hdr in;
};

struct coda_statfs_out {      /* size STAT */
    struct coda_out_hdr oh;
    struct coda_statfs stat;  /* STAT */
};


struct coda_store_in {
    struct coda3_in_hdr ih;
    struct Coda3Fid VFid;
    int	flags;
};

struct coda_store_out {
    struct coda_out_hdr out;
};


struct coda_release_in {
    struct coda3_in_hdr ih;
    struct Coda3Fid VFid;
    int	flags;
};

struct coda_release_out {
    struct coda_out_hdr out;
};


/* Public global variables */
/*=========================*/

/* A value of 1 tells dav_run_messageloop() to keep on running. It may be
   reset to 0 by e.g. signal handlers to tell dav_run_messageloop() to
   stop gracefully. Will be initialized to 1. */
extern volatile int keep_on_running;


/* Function prototypes */
/*=====================*/

/* Sets private global variables idle_time, device, version and cache_dir,
   opens the coda devices and returns a file descriptor of it in args
   and finally call dav_register_kernel_interface() to register the
   call-back function dav_coda_zap_file() with the cache.
   idle-time is taken from args. The coda device is created if necessary
   and opened. version is asked from the kernel.
   In case of an error it terminates the program.
   Note: The cache module must have been initialized before calling
         dav_init_coda().
   args : structure containing idle_time. A file descriptor of the open
          device will be returned in this structure. */
void dav_init_coda(dav_args *args);

/* Just closes the coda device. */
void dav_close_coda(void);

/* Runs an infinite loop that reads the upcalls from the coda kernel module,
   translates them into calls of cache functions and translates and returns
   the result to the kernel module. If there are no upcalls for more then
   idle_time seconds, it tests whether the file system is still
   mounted. If this test is negative it returns otherwise it will call
   dav_idle(). Setting global variable keep_on_running to 0 will terminate
   the loop too. */
void dav_run_messageloop(void);


#endif
